unit Zoomrec;
{ --------------------------------------------------------------------------
  WinCDR -
  Zoom module for Record to Disk module (rec.pas)
  Modal form which allows user to zoom in/out of display of recorded signals
  --------------------------------------------------------------------------}
interface

uses WinTypes, WinProcs, Classes, Graphics, Forms, Controls, Buttons,
  StdCtrls, ExtCtrls, Global, Shared, SysUtils, FileIO ;

type
  TZoomRecFrm = class(TForm)
    pbDisplay: TPaintBox;
    cbChannels: TComboBox;
    Label1: TLabel;
    lbTMin: TLabel;
    lbTMax: TLabel;
    bOK: TButton;
    bCancel: TButton;
    procedure FormPaint(Sender: TObject);
    procedure pbDisplayMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure pbDisplayMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure pbDisplayMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure cbChannelsChange(Sender: TObject);
    procedure bOKClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    ChOnDisplay : LongInt ;
    procedure ZoomOut ;
  end;

var
  ZoomRecFrm: TZoomRecFrm;

implementation

{$R *.DFM}

uses MDIform, maths, ViewCDR ;

type
TMousePos = ( TopLeft,
              TopRight,
              BottomLeft,
              BottomRight,
              MLeft,
              MRight,
              MTop,
              MBottom,
              MDrag ) ;

var
   ZoomCh : TChannel ;
   ZoomBox : TRect ;
   MousePos : TMousePos ;
   MoveZoomBox : Boolean ;
   XOld,YOld : Integer ;



procedure TZoomRecFrm.FormShow(Sender: TObject);
{ -------------------------------------------
  Initialisations each time form is displayed
  -------------------------------------------}
var
   ch : Integer ;
begin


     { Fill channel selection list }
     cbChannels.Clear ;
     for ch := 0 to CdrFH.NumChannels-1 do
          cbChannels.items.add( ' ' + Channel[ch].ADCName ) ;

    { Start with channel 0 selected }
    cbChannels.ItemIndex := ChOnDisplay ;

    { Set scaling for the channel to be worked on }

    ZoomCh := Channel[ChOnDisplay] ;
    ZoomCh.Left := 0 ;
    ZoomCh.Right := pbDisplay.Width ;
    ZoomCh.Top := pbDisplay.Height div 50 ;
    ZoomCh.Bottom := pbDisplay.Height - ZoomCh.Top ;

    ZoomCh.yMin := MinADCValue  ;
    ZoomCh.yMax := MaxADCValue  ;
    { Set scaling }
    ZoomCh.yScale := (ZoomCh.Bottom - ZoomCh.Top) / (ZoomCh.yMax - ZoomCh.yMin ) ;

    MoveZoomBox := False ;


     Ch := cbChannels.ItemIndex ;
     ChOnDisplay := Ch ;

     { Erase Display }
     pbDisplay.canvas.brush.color := clWhite ;
     pbDisplay.canvas.fillrect(pbDisplay.canvas.ClipRect);

     { Set trace colour }

     pbDisplay.canvas.pen.color := ZoomCh.color ;

     ZoomCh.xScale := (ZoomCh.Right - ZoomCh.Left) / (ZoomCh.xMax - ZoomCh.xMin ) ;
     ZoomCh.yScale := (ZoomCh.Bottom - ZoomCh.Top) / (ZoomCh.yMax - ZoomCh.yMin ) ;

     { Set size of zoom box }

     ZoomBox.Left :=  0 ;
     ZoomBox.Right := pbDisplay.Width ;
     ZoomBox.Top := ZoomCh.Bottom -
                     Trunc((Channel[ch].yMax - ZoomCh.yMin)*ZoomCh.yScale) ;
     ZoomBox.Bottom := ZoomCh.Bottom -
                     Trunc((Channel[ch].yMin - ZoomCh.yMin)*ZoomCh.yScale) ;

     { Display labels }

     lbTMin.caption := Format( '%.4g %s', [ZoomCh.xMin*Settings.TScale,Settings.TUnits] ) ;
     lbTMax.caption := Format( '%.4g %s', [ZoomCh.xMax*Settings.TScale,Settings.TUnits] ) ;

     lbTMin.left := pbDisplay.left ;
     lbTMin.Top := pbDisplay.Top + pbDisplay.Height + 1 ;
     lbTMax.Left := pbDisplay.Left + pbDisplay.width - lbTMax.Width ;
     lbTMax.Top := lbTMin.Top ;

     end;


procedure TZoomRecFrm.FormPaint(Sender: TObject);
{ ---------------------------------
  Point selected channel on display
  ---------------------------------}
var
   i,ChOffset,ch : LongInt ;

begin

     Ch := cbChannels.ItemIndex ;
     ChOnDisplay := Ch ;

     { Erase Display }
     pbDisplay.canvas.brush.color := clWhite ;
     pbDisplay.canvas.fillrect(pbDisplay.canvas.ClipRect);

     { Set trace colour }

     pbDisplay.canvas.pen.color := ZoomCh.color ;

     { Set size of zoom box }

     ZoomBox.Top := ZoomCh.Bottom -
                     Trunc((Channel[ch].yMax - ZoomCh.yMin)*ZoomCh.yScale) ;
     ZoomBox.Bottom := ZoomCh.Bottom -
                     Trunc((Channel[ch].yMin - ZoomCh.yMin)*ZoomCh.yScale) ;

     { Display labels }

     lbTMin.caption := Format( '0. %s', [Settings.TUnits] ) ;
     lbTMax.caption := Format( '%.4g %s', [ZoomCh.xMax*Settings.TScale,Settings.TUnits] ) ;

     lbTMin.left := pbDisplay.left ;
     lbTMin.Top := pbDisplay.Top + pbDisplay.Height + 1 ;
     lbTMax.Left := pbDisplay.Left + pbDisplay.width - lbTMax.Width ;
     lbTMax.Top := lbTMin.Top ;

     pbDisplay.canvas.DrawFocusRect( ZoomBox ) ;

     MoveZoomBox := False ;
     end ;


procedure TZoomRecFrm.pbDisplayMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
     MoveZoomBox := True ;
     end;

procedure TZoomRecFrm.pbDisplayMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
const
     Margin = 10 ;
     ZoomMin = 1 ;
var
   Height,Width : Integer ;
begin

     { Find MousePos of mouse relative to Zoom box
       and change mouse cursor shape appropriately }

     if not MoveZoomBox then begin

        if (Abs(Y - ZoomBox.Top) < Margin) and
              (X <= ZoomBox.Right) and (X >= ZoomBox.Left ) then begin
            pbDisplay.Cursor := crSizeNS ;
            MousePos := MTop ;
            end
        else if (Abs(Y - ZoomBox.Bottom) < Margin ) and
               (X <= ZoomBox.Right) and (X >= ZoomBox.Left ) then begin
            pbDisplay.Cursor := crSizeNS ;
            MousePos := MBottom ;
            end
        else if (ZoomBox.Bottom > Y) and (Y > ZoomBox.Top) then begin
            pbDisplay.Cursor := CrSize ;
            MousePos := MDrag ;
            XOld := X ;
            YOld := Y ;
            end
        else
            pbDisplay.Cursor := crDefault ;

        end
     else if MoveZoomBox then begin

          pbDisplay.canvas.DrawFocusRect( ZoomBox ) ;

          { Move the part of the zoom box which is under the mouse }

          case MousePos of
          MTop : Begin
              if (ZoomBox.Bottom-Y) > ZoomMin then ZoomBox.Top := Y ;
              end ;
          MBottom : Begin
              if (Y - ZoomBox.Top) > ZoomMin then ZoomBox.Bottom := Y ;
              end ;
          MDrag : begin
              Width := ZoomBox.Right - ZoomBox.Left ;
              Height := ZoomBox.Bottom - ZoomBox.Top ;
              ZoomBox.Left := MaxInt( [ZoomBox.Left + (X - XOld),ZoomCh.Left]) ;
              ZoomBox.Right := MinInt( [ZoomBox.Left + Width,ZoomCh.Right]) ;
              ZoomBox.Left := ZoomBox.Right - Width ;
              ZoomBox.Top := MaxInt( [ZoomBox.Top + (Y - YOld),ZoomCh.Top]) ;
              ZoomBox.Bottom := MinInt( [ZoomBox.Top + Height,ZoomCh.Bottom]) ;
              ZoomBox.Top := ZoomBox.Bottom - Height ;
              XOld := X ;
              YOld := Y ;
              end
          else
          end ;

          { Keep within bounds }

          ZoomBox.Top :=     MaxInt( [ZoomBox.Top,ZoomCh.Top] ) ;
          ZoomBox.Top :=     MinInt( [ZoomBox.Top,ZoomCh.Bottom] ) ;
          ZoomBox.Bottom :=  MaxInt( [ZoomBox.Bottom,ZoomCh.Top] ) ;
          ZoomBox.Bottom :=  MinInt( [ZoomBox.Bottom,ZoomCh.Bottom] ) ;

          pbDisplay.canvas.DrawFocusRect( ZoomBox ) ;
          end ;

     end ;

procedure TZoomRecFrm.pbDisplayMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
     MoveZoomBox := False ;
     end;


procedure TZoomRecFrm.cbChannelsChange(Sender: TObject);
begin

     Channel[ChOnDisplay].yMin := ((ZoomCh.Bottom - ZoomBox.Bottom) / ZoomCh.yScale)
                                  + ZoomCh.yMin ;
     Channel[ChOnDisplay].yMax := ((ZoomCh.Bottom - ZoomBox.Top) / ZoomCh.yScale)
                                  + ZoomCh.yMin ;

     { If the channel number has changed re-plot the display }
     pbDisplay.Refresh ;
     end;


procedure TZoomRecFrm.ZoomOut ;
{ ---------------------------------------------
  Set display channels to minimum magnification
  ---------------------------------------------}
var
   i : Integer ;
begin
     for i := 0 to CdrFH.NumChannels-1 do begin
         Channel[i].yMin := MinADCValue ;
         Channel[i].yMax := MaxADCValue ;
         end ;

    { Refresh child windows that exist }
     with Main do for i := 0 to MDIChildCount-1 do MDICHildren[i].Refresh ;
    end;


procedure TZoomRecFrm.bOKClick(Sender: TObject);
var
   ch,NumInUse : LongInt ;
   i : Integer ;
begin


     { Update magnification of selected channel }

     Ch := cbChannels.itemindex ;

     Channel[ch].yMin := ((ZoomCh.Bottom - ZoomBox.Bottom) / ZoomCh.yScale)
                           + ZoomCh.yMin ;
     Channel[ch].yMax := ((ZoomCh.Bottom - ZoomBox.Top) / ZoomCh.yScale)
                           + ZoomCh.yMin ;

     { Refresh all open windows }
     with Main do
          for i := 0 to MDIChildCount-1 do MDICHildren[i].Refresh ;

     end;



end.
