unit Zoomcdr;
{ --------------------------------------------------------------------------
  WinCDR -
  Zoom module for ViewCDR display module
  Modal form which allows user to zoom in/out of display of recorded signals
  --------------------------------------------------------------------------}
interface

uses WinTypes, WinProcs, Classes, Graphics, Forms, Controls, Buttons,
  StdCtrls, ExtCtrls, Global, Shared, SysUtils, FileIO ;

type
  TZoomCDRFrm = class(TForm)
    pbDisplay: TPaintBox;
    cbChannels: TComboBox;
    Label1: TLabel;
    lbTMin: TLabel;
    lbTMax: TLabel;
    ChannelsGrp: TGroupBox;
    ckInUse0: TCheckBox;
    ckInUse1: TCheckBox;
    ckInUse2: TCheckBox;
    ckInUse3: TCheckBox;
    ckInUse4: TCheckBox;
    ckInUse5: TCheckBox;
    bOK: TButton;
    bCancel: TButton;
    procedure FormPaint(Sender: TObject);
    procedure pbDisplayMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure pbDisplayMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure pbDisplayMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure cbChannelsChange(Sender: TObject);
    procedure bOKClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    { Private declarations }
    procedure SetChannelCheckBox( ckInUse : TCheckBox ; ChanNum : Integer ) ;
  public
    { Public declarations }
    ChOnDisplay : LongInt ;
    procedure ZoomOut ;
  end;

var
  ZoomCDRFrm: TZoomCDRFrm;

implementation

{$R *.DFM}

uses MDIform, maths, ViewCDR ;

type
TMousePos = ( TopLeft,
              TopRight,
              BottomLeft,
              BottomRight,
              MLeft,
              MRight,
              MTop,
              MBottom,
              MDrag ) ;

var
   ZoomCh : TChannel ;
   ZoomBox : TRect ;
   MousePos : TMousePos ;
   MoveZoomBox : Boolean ;
   XOld,YOld : Integer ;



procedure TZoomCDRFrm.FormShow(Sender: TObject);
{ -------------------------------------------
  Initialisations each time form is displayed
  -------------------------------------------}
var
   ch : Integer ;
begin

     { Set channel in use check boxes }
     SetChannelCheckBox( ckInUse0, 0 ) ;
     SetChannelCheckBox( ckInUse1, 1 ) ;
     SetChannelCheckBox( ckInUse2, 2 ) ;
     SetChannelCheckBox( ckInUse3, 3 ) ;
     SetChannelCheckBox( ckInUse4, 4 ) ;
     SetChannelCheckBox( ckInUse5, 5 ) ;

     { Fill channel selection list }
     cbChannels.Clear ;
     for ch := 0 to CdrFH.NumChannels-1 do
          cbChannels.items.add( ' ' + Channel[ch].ADCName ) ;

    { Start with channel 0 selected }
    cbChannels.ItemIndex := Main.ViewCDRChild.ZoomBuf^.ChannelSelected ;
    ChOnDisplay := Main.ViewCDRChild.ZoomBuf^.ChannelSelected ;

    { Set scaling for the channel to be worked on }

    ZoomCh := Channel[ChOnDisplay] ;
    ZoomCh.Left := pbDisplay.Width div 50 ;
    ZoomCh.Right := pbDisplay.Width - ZoomCh.Left;
    ZoomCh.Top := pbDisplay.Height div 50 ;
    ZoomCh.Bottom := pbDisplay.Height - ZoomCh.Top ;

    ZoomCh.xMin := Main.ViewCDRChild.ZoomBuf^.xMin ;
    ZoomCh.xMax := Main.ViewCDRChild.ZoomBuf^.xMax ;
    ZoomCh.yMin := MinADCValue  ;
    ZoomCh.yMax := MaxADCValue  ;
    { Set scaling }
    ZoomCh.xScale := (ZoomCh.Right - ZoomCh.Left) / (ZoomCh.xMax - ZoomCh.xMin ) ;
    ZoomCh.yScale := (ZoomCh.Bottom - ZoomCh.Top) / (ZoomCh.yMax - ZoomCh.yMin ) ;

    MoveZoomBox := False ;


     Ch := cbChannels.ItemIndex ;
     ChOnDisplay := Ch ;

     { Erase Display }
     pbDisplay.canvas.brush.color := clWhite ;
     pbDisplay.canvas.fillrect(pbDisplay.canvas.ClipRect);

     { Set trace colour }

     pbDisplay.canvas.pen.color := ZoomCh.color ;

     { Set scaling }
     ZoomCh.xScale := (ZoomCh.Right - ZoomCh.Left) / (ZoomCh.xMax - ZoomCh.xMin ) ;
     ZoomCh.yScale := (ZoomCh.Bottom - ZoomCh.Top) / (ZoomCh.yMax - ZoomCh.yMin ) ;

     { Set size of zoom box }

     ZoomBox.Left :=  Trunc((Channel[ch].xMin - ZoomCh.xMin)*ZoomCh.xScale)
                      + ZoomCh.Left ;
     ZoomBox.Right := Trunc((Channel[ch].xMax - ZoomCh.xMin)*ZoomCh.xScale)
                       + ZoomCh.Left ;
     ZoomBox.Top := ZoomCh.Bottom -
                     Trunc((Channel[ch].yMax - ZoomCh.yMin)*ZoomCh.yScale) ;
     ZoomBox.Bottom := ZoomCh.Bottom -
                     Trunc((Channel[ch].yMin - ZoomCh.yMin)*ZoomCh.yScale) ;

     { Display labels }

     lbTMin.caption := Format( '%.4g %s', [ZoomCh.xMin*Settings.TScale,Settings.TUnits] ) ;
     lbTMax.caption := Format( '%.4g %s', [ZoomCh.xMax*Settings.TScale,Settings.TUnits] ) ;

     lbTMin.left := pbDisplay.left ;
     lbTMin.Top := pbDisplay.Top + pbDisplay.Height + 1 ;
     lbTMax.Left := pbDisplay.Left + pbDisplay.width - lbTMax.Width ;
     lbTMax.Top := lbTMin.Top ;

     end;


procedure TZoomCDRFrm.FormPaint(Sender: TObject);
{ ---------------------------------
  Point selected channel on display
  ---------------------------------}
type
    TPointArray = Array[0..2000] of TPoint ;
var
   i,ChOffset,ch : LongInt ;
   x,y,dx : single ;
   n : Integer ;
   xy : ^TPointArray ;
   OK : Boolean ;

begin

     Ch := cbChannels.ItemIndex ;
     ChOnDisplay := Ch ;

     { Erase Display }
     pbDisplay.canvas.brush.color := clWhite ;
     pbDisplay.canvas.fillrect(pbDisplay.canvas.ClipRect);

     { Set trace colour }

     pbDisplay.canvas.pen.color := ZoomCh.color ;

     { Set size of zoom box }

     ZoomBox.Left :=  Trunc((Channel[ch].xMin - ZoomCh.xMin)*ZoomCh.xScale)
                      + ZoomCh.Left ;
     ZoomBox.Right := Trunc((Channel[ch].xMax - ZoomCh.xMin)*ZoomCh.xScale)
                       + ZoomCh.Left ;
     ZoomBox.Top := ZoomCh.Bottom -
                     Trunc((Channel[ch].yMax - ZoomCh.yMin)*ZoomCh.yScale) ;
     ZoomBox.Bottom := ZoomCh.Bottom -
                     Trunc((Channel[ch].yMin - ZoomCh.yMin)*ZoomCh.yScale) ;

     { Display labels }

     lbTMin.caption := Format( '%.4g %s', [ZoomCh.xMin*Settings.TScale,Settings.TUnits] ) ;
     lbTMax.caption := Format( '%.4g %s', [ZoomCh.xMax*Settings.TScale,Settings.TUnits] ) ;

     lbTMin.left := pbDisplay.left ;
     lbTMin.Top := pbDisplay.Top + pbDisplay.Height + 1 ;
     lbTMax.Left := pbDisplay.Left + pbDisplay.width - lbTMax.Width ;
     lbTMax.Top := lbTMin.Top ;

     { Plot channel}

     try
        New(xy) ;

        n := 0 ;
        for i := 0 to Main.ViewCDRChild.ZoomBuf^.nPoints-1 do begin
            xy^[n].y := ZoomCh.Bottom - Trunc(
                    ZoomCh.yScale*(Main.ViewCDRChild.ZoomBuf^.yMin[ch,i] - ZoomCh.yMin));
            xy^[n].x := Trunc(ZoomCh.xScale*(Main.ViewCDRChild.ZoomBuf^.x[i]
                      - ZoomCh.xMin) + ZoomCh.Left) ;
            Inc(n) ;
            xy^[n].y := ZoomCh.Bottom - Trunc(
                    ZoomCh.yScale*(Main.ViewCDRChild.ZoomBuf^.yMax[ch,i] - ZoomCh.yMin));
            xy^[n].x := xy^[n-1].x ;
            Inc(n) ;
            end ;
        OK := Polyline( pbDisplay.Canvas.Handle, xy^, n ) ;
     finally
        Dispose(xy) ;
        end ;

     pbDisplay.canvas.DrawFocusRect( ZoomBox ) ;

     MoveZoomBox := False ;
     end ;


procedure TZoomCDRFrm.pbDisplayMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
     MoveZoomBox := True ;
     end;

procedure TZoomCDRFrm.pbDisplayMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
const
     Margin = 10 ;
     ZoomMin = 1 ;
var
   Height,Width : Integer ;
begin

     { Find MousePos of mouse relative to Zoom box
       and change mouse cursor shape appropriately }

     if not MoveZoomBox then begin

        if (Abs(X - ZoomBox.Left) + Abs(Y - ZoomBox.Top)) < Margin then begin
            pbDisplay.Cursor := crSizeNWSE ;
            MousePos := TopLeft ;
            end
        else if (Abs(X - ZoomBox.Right) + Abs(Y - ZoomBox.Top)) < Margin then begin
            pbDisplay.Cursor := crSizeNESW ;
            MousePos := TopRight ;
            end
        else if (Abs(X - ZoomBox.Left) + Abs(Y - ZoomBox.Bottom)) < Margin then begin
            pbDisplay.Cursor := crSizeNESW ;
            MousePos := BottomLeft ;
            end
        else if (Abs(X - ZoomBox.Right) + Abs(Y - ZoomBox.Bottom)) < Margin then begin
            pbDisplay.Cursor := crSizeNWSE ;
            MousePos := BottomRight ;
            end
        else if (Abs(X - ZoomBox.Left) < Margin ) and
             (Y <= ZoomBox.Bottom) and (Y >= ZoomBox.Top ) then begin
            pbDisplay.Cursor := crSizeWE ;
            MousePos := MLeft ;
            end
        else if (Abs(X - ZoomBox.Right) < Margin) and
               (Y <= ZoomBox.Bottom) and (Y >= ZoomBox.Top ) then begin
            pbDisplay.Cursor := crSizeWE ;
            MousePos := MRight ;
            end
        else if (Abs(Y - ZoomBox.Top) < Margin) and
              (X <= ZoomBox.Right) and (X >= ZoomBox.Left ) then begin
            pbDisplay.Cursor := crSizeNS ;
            MousePos := MTop ;
            end
        else if (Abs(Y - ZoomBox.Bottom) < Margin ) and
               (X <= ZoomBox.Right) and (X >= ZoomBox.Left ) then begin
            pbDisplay.Cursor := crSizeNS ;
            MousePos := MBottom ;
            end
        else if (ZoomBox.Bottom > Y) and (Y > ZoomBox.Top) and
                (ZoomBox.Right > X) and (X > ZoomBox.Left) then begin
            pbDisplay.Cursor := CrSize ;
            MousePos := MDrag ;
            XOld := X ;
            YOld := Y ;
            end
        else
            pbDisplay.Cursor := crDefault ;

        end
     else if MoveZoomBox then begin

          pbDisplay.canvas.DrawFocusRect( ZoomBox ) ;

          { Move the part of the zoom box which is under the mouse }

          case MousePos of
          TopLeft : Begin
              if (ZoomBox.Bottom-Y) > ZoomMin then ZoomBox.Top := Y ;
              if (ZoomBox.Right-X) > ZoomMin then ZoomBox.Left := X ;
             end ;
          TopRight : Begin
              if (ZoomBox.Bottom-Y) > ZoomMin then ZoomBox.Top := Y ;
              if (X - ZoomBox.Left) > ZoomMin then ZoomBox.Right := X ;
              end ;
          BottomLeft : Begin
              if (ZoomBox.Bottom-Y) > ZoomMin then ZoomBox.Top := Y ;
              if (ZoomBox.Right-X) > ZoomMin then ZoomBox.Left := X ;
              end ;
          BottomRight : Begin
              if (ZoomBox.Bottom-Y) > ZoomMin then ZoomBox.Top := Y ;
              if (X - ZoomBox.Left) > ZoomMin then ZoomBox.Right := X ;
              end ;
          MTop : Begin
              if (ZoomBox.Bottom-Y) > ZoomMin then ZoomBox.Top := Y ;
              end ;
          MBottom : Begin
              if (Y - ZoomBox.Top) > ZoomMin then ZoomBox.Bottom := Y ;
              end ;
          MLeft : Begin
              if (ZoomBox.Right-X) > ZoomMin then ZoomBox.Left := X ;
              end ;
          MRight : Begin
              if (X - ZoomBox.Left) > ZoomMin then ZoomBox.Right := X ;

              end ;
          MDrag : begin
              Width := ZoomBox.Right - ZoomBox.Left ;
              Height := ZoomBox.Bottom - ZoomBox.Top ;
              ZoomBox.Left := MaxInt( [ZoomBox.Left + (X - XOld),ZoomCh.Left]) ;
              ZoomBox.Right := MinInt( [ZoomBox.Left + Width,ZoomCh.Right]) ;
              ZoomBox.Left := ZoomBox.Right - Width ;
              ZoomBox.Top := MaxInt( [ZoomBox.Top + (Y - YOld),ZoomCh.Top]) ;
              ZoomBox.Bottom := MinInt( [ZoomBox.Top + Height,ZoomCh.Bottom]) ;
              ZoomBox.Top := ZoomBox.Bottom - Height ;
              XOld := X ;
              YOld := Y ;
              end
          else
          end ;

          { Keep within bounds }

          ZoomBox.Left :=    MaxInt( [ZoomBox.Left,ZoomCh.Left] ) ;
          ZoomBox.Left :=    MinInt( [ZoomBox.Left,ZoomCh.Right] ) ;
          ZoomBox.Right :=   MaxInt( [ZoomBox.Right,ZoomCh.Left] ) ;
          ZoomBox.Right :=   MinInt( [ZoomBox.Right,ZoomCh.Right] ) ;
          ZoomBox.Top :=     MaxInt( [ZoomBox.Top,ZoomCh.Top] ) ;
          ZoomBox.Top :=     MinInt( [ZoomBox.Top,ZoomCh.Bottom] ) ;
          ZoomBox.Bottom :=  MaxInt( [ZoomBox.Bottom,ZoomCh.Top] ) ;
          ZoomBox.Bottom :=  MinInt( [ZoomBox.Bottom,ZoomCh.Bottom] ) ;

          pbDisplay.canvas.DrawFocusRect( ZoomBox ) ;
          end ;

     end ;

procedure TZoomCDRFrm.pbDisplayMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
     MoveZoomBox := False ;
     end;


procedure TZoomCDRFrm.cbChannelsChange(Sender: TObject);
begin

     Channel[ChOnDisplay].xMin := ((ZoomBox.Left - ZoomCh.Left) / ZoomCh.xScale)
                                  + ZoomCh.xMin ;
     Channel[ChOnDisplay].xMax := ((ZoomBox.Right - ZoomCh.Left) / ZoomCh.xScale)
                                  + ZoomCh.xMin ;
     Channel[ChOnDisplay].yMin := ((ZoomCh.Bottom - ZoomBox.Bottom) / ZoomCh.yScale)
                                  + ZoomCh.yMin ;
     Channel[ChOnDisplay].yMax := ((ZoomCh.Bottom - ZoomBox.Top) / ZoomCh.yScale)
                                  + ZoomCh.yMin ;

     { If the channel number has changed re-plot the display }
     pbDisplay.Refresh ;
     end;


procedure TZoomCDRFrm.ZoomOut ;
{ ---------------------------------------------
  Set display channels to minimum magnification
  ---------------------------------------------}
var
   i : Integer ;
begin
     for i := 0 to CdrFH.NumChannels-1 do begin
         Channel[i].xMin := Main.ViewCDRChild.ZoomBuf^.xMin ;
         Channel[i].xMax := Main.ViewCDRChild.ZoomBuf^.xMax ;
         Channel[i].yMin := MinADCValue ;
         Channel[i].yMax := MaxADCValue ;
         end ;

    { Refresh child windows that exist }
     with Main do for i := 0 to MDIChildCount-1 do MDICHildren[i].Refresh ;
    end;


procedure TZoomCDRFrm.bOKClick(Sender: TObject);
var
   ch,NumInUse : LongInt ;
   i : Integer ;
begin

     { Update channels in use }
     Channel[0].InUse := ckInUse0.checked ;
     Channel[1].InUse := ckInUse1.checked ;
     Channel[2].InUse := ckInUse2.checked ;
     Channel[3].InUse := ckInUse3.checked ;
     Channel[4].InUse := ckInUse4.checked ;
     Channel[5].InUse := ckInUse5.checked ;
     { Ensure at least one channel is displayed }
     NumInUse := 0 ;
     for ch := 0 to CdrFH.NumChannels-1 do
         if Channel[ch].InUse then NumInUse := NumInUse + 1;
     if NumInUse = 0 then Channel[0].InUse := True ;

     { Update magnification of selected channel }

     Ch := cbChannels.itemindex ;

     Channel[ch].xMin := ((ZoomBox.Left - ZoomCh.Left) / ZoomCh.xScale)
                           + ZoomCh.xMin ;
     Channel[ch].xMax := ((ZoomBox.Right - ZoomCh.Left) / ZoomCh.xScale)
                           + ZoomCh.xMin ;
     Channel[ch].yMin := ((ZoomCh.Bottom - ZoomBox.Bottom) / ZoomCh.yScale)
                           + ZoomCh.yMin ;
     Channel[ch].yMax := ((ZoomCh.Bottom - ZoomBox.Top) / ZoomCh.yScale)
                           + ZoomCh.yMin ;

     { Make sure X axis scaling is the same for all channels }
     for ch := 0 to CdrFH.NumChannels -1 do begin
         Channel[ch].XMin := Channel[cbChannels.itemindex].xMin ;
         Channel[ch].XMax := Channel[cbChannels.itemindex].xMax ;
         end ;

    Main.ViewCDRChild.ViewBuf^.tDisplay := Channel[0].xMax - Channel[0].xMin ;
     { Refresh all open windows }
     with Main do
          for i := 0 to MDIChildCount-1 do MDICHildren[i].Refresh ;

     end;


procedure TZoomCDRFrm.SetChannelCheckBox( ckInUse : TCheckBox ; ChanNum : Integer ) ;
begin
     if CdrFH.NumChannels > ChanNum then begin
          ckInUse.caption := format('Ch.%d %s',[ChanNum,Channel[ChanNum].ADCName])  ;
          ckInUse.enabled := True ;
          ckInUse.visible := True ;
          ckInUse.checked := Channel[ChanNum].InUse ;
          end
     else begin
          ckInUse.enabled := False ;
          ckInUse.visible := False ;
          ckInUse.checked := False ;
          end ;
     end ;

end.
