unit Viewwcp;
{ ========================================================
  WinCDR (c) John Dempster, University of Strathclyde 1998
  WCP DATA FILE (DETECTED SIGNALS) DISPLAY MODULE
  V0.99 0/3/98
  28/6/98 ... PrintRecord & CopyRecordImageToClipboard replace PlotRecords
  30/6/98 ... Ident line now updates correctly
  ========================================================}
interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, ExtCtrls, Global, Shared, FileIo, Zero, ClipBrd,
  PrintRec,Printers, PlotLib, menus ;

type
  TViewWCPFrm = class(TForm)
    CursorGrp: TGroupBox;
    RecordGrp: TGroupBox;
    pbDisplay: TPaintBox;
    edRecordNum: TEdit;
    Label2: TLabel;
    cbRecordType: TComboBox;
    ckBadRecord: TCheckBox;
    sbRecordNum: TScrollBar;
    mmCursor: TMemo;
    lbTMin: TLabel;
    lbTMax: TLabel;
    Timer: TTimer;
    lbCursor0: TLabel;
    edTime: TEdit;
    Label1: TLabel;
    edGroup: TEdit;
    Group: TLabel;
    FilterGrp: TGroupBox;
    cbLowPassFilter: TComboBox;
    edIdent: TEdit;
    Label3: TLabel;
    lbCursor1: TLabel;
    EdRecordIdent: TEdit;
    procedure pbDisplayPaint(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure sbRecordNumChange(Sender: TObject);
    procedure TimerTimer(Sender: TObject);
    procedure pbDisplayMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure pbDisplayMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure pbDisplayMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FormShow(Sender: TObject);
    procedure FormDeactivate(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure edRecordNumKeyPress(Sender: TObject; var Key: Char);
    procedure ckBadRecordClick(Sender: TObject);
    procedure cbRecordTypeChange(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure cbLowPassFilterChange(Sender: TObject);
    procedure pbDisplayDblClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure edIdentChange(Sender: TObject);
    procedure edIdentKeyPress(Sender: TObject; var Key: Char);
    procedure EdRecordIdentChange(Sender: TObject);
    procedure edGroupKeyPress(Sender: TObject; var Key: Char);
    procedure FormHide(Sender: TObject);
  private
    { Private declarations }
    procedure DrawHorizontalCursor(var pb : TPaintBox ; Const Gr : TChannel ;
                                    Level : Integer ) ;
    procedure PlotChannels( PB : TPaintBox ; var Buf : Array of Integer ) ;
    procedure UpdateLowPassFilterList(dt : single) ;
    procedure HeapBuffers( Operation : THeapBufferOp ) ;
    procedure InitializeDisplay( const PB : TPaintBox ) ;
  public
    { Public declarations }
   TimerBusy : boolean ;
   HardCopy : boolean ;
   NewFile : Boolean ;
   procedure PrintRecord ;
   procedure CopyRecordImageToClipboard ;
   procedure CopyDataToClipBoard ;
  end;


var
  ViewWCPFrm: TViewWCPFrm;

implementation

uses          MDIForm,ZoomWCP,copyrec, maths ;

type
    TCursorState = ( Cursor0,Cursor1,ZeroLevelCursor,NoCursor ) ;
    TReplayState = (PlotRecord,UpdateCursorReadout,Idle) ;
    TPointArray = Array[0..8192] of TPoint ;
var
   CurCh : LongInt ;
   MoveCursor : Boolean ;
   CursorState : TCursorState ;
   State : TReplayState ;
   RecHeader : TWCPRecHeader ;      { Signal record header block }
   ADC : ^TIntArray ;
   xy : ^TPointArray ;
   Olddt : single ;
   MouseOverChannel : Integer ;
   Destination : TDestination ;
   CursorChannel : TChannel ;
   

   BuffersAllocated : boolean ;{ Indicates if memory buffers have been allocated }

{$R *.DFM}

procedure TViewWCPFrm.FormCreate(Sender: TObject);
{ -----------------------------------------
  Initialisations done when form is created
  -----------------------------------------}
begin
     { Set flag in main form indicating that the replay form is open }
     Main.ViewWCPChildExists := True ;
     Main.mnViewWCP.visible := True ;
     Main.mnViewWCP.checked := True ;
     BuffersAllocated := False ;
     end;


procedure TViewWCPFrm.HeapBuffers( Operation : THeapBufferOp ) ;
{ -----------------------------------------------
  Allocate/deallocation dynamic buffers from heap
  -----------------------------------------------}
begin
     case Operation of
          Allocate : begin
             if not BuffersAllocated then begin
                New(ADC) ;
                New(xy) ;
                BuffersAllocated := True ;
                end ;
             end ;
          Deallocate : begin
             if BuffersAllocated then begin
                Dispose(ADC) ;
                Dispose(xy) ;
                BuffersAllocated := False ;
                end ;
             end ;
          end ;
     end ;



procedure TViewWCPFrm.FormShow(Sender: TObject);
{ ---------------------------------------
  Initialisations done when form is shown
  ---------------------------------------}
var
   i : LongInt ;
begin

     HeapBuffers( Allocate ) ;

     Main.ClearTicksInViewMenu ;
     Main.mnViewWCP.checked := True ;

     Height := CursorGrp.Top + CursorGrp.Height + 50 ;
     Width := edRecordIdent.Left + EdRecordIdent.Width + 30 ;
     cbRecordType.items := RecordTypes ;
     cbRecordType.items.delete(0) ; {Remove 'ALL' item}

     { Start scheduling timer }
     Timer.enabled := True ;
     TimerBusy := False ;

     WCPChannel[0].Cursor0 := 0 ;
     WCPChannel[0].Cursor1 := WCPfH.NumSamples div 2 ;

     { Request a record to be displayed }
     State := PlotRecord ;
     HardCopy := False ;
     NewFile := True ;
     Olddt := 0. ;

     end;


procedure TViewWCPFrm.pbDisplayPaint(Sender: TObject);
begin
     State := PlotRecord ;
     end ;


procedure TViewWCPFrm.FormResize(Sender: TObject);
{ ---------------------------
  Re-size components on form
  --------------------------}
begin
     pbDisplay.left := RecordGrp.Left + RecordGrp.width + 10 ;
     pbDisplay.width := Width - pbDisplay.left - 20 ;
     pbDisplay.top := EdIdent.Top + EdIdent.Height + 2 ;
     pbDisplay.Height := Height  - pbDisplay.Top -
                         (3*lbTMin.Height) div 2 - 40 ;

     lbTMin.Left := pbDisplay.left ;
     lbTMin.Top := pbDisplay.top +  pbDisplay.Height + 2 ;
     lbTMax.Left := pbDisplay.left + pbDisplay.width - lbTMax.Width ;
     lbTMax.Top := lbTMin.Top ;

     State := PlotRecord ;
     end;

procedure TViewWCPFrm.sbRecordNumChange(Sender: TObject);
begin
     { If the record number has changed ask for the display to be updated }
     State := PlotRecord ;
     end;


procedure TViewWCPFrm.TimerTimer(Sender: TObject);
{ -----------------------------------------------------------
  Process scheduler ... plots records, updates cursor readout
  -----------------------------------------------------------}
var
   ch,i,xPix,yPix,MaxCh,nOffScreen,rec,ChOffset : LongInt ;
   x,dx,y : Single ;

begin

     { If a new file has been opened reset display module }

     if not TimerBusy then  begin

        TimerBusy := True ;

        if NewFile then begin
           caption := 'View Detected ' + WCPfH.FileName ;

           if WCPfH.IdentLine = '' then WCPfH.IdentLine := CdrFH.IdentLine ;
           EdIdent.text := WCPfH.IdentLine ;

           { Clear readout cursor box }
           for i := 0 to mmCursor.Lines.Count-1 do mmCursor.Lines[i] := '' ;

           for ch := 0 to WCPfH.NumChannels-1 do begin
               WCPChannel[ch] := Channel[ch] ;
               WCPChannel[ch].xMin := 0 ;
               WCPChannel[ch].xMax := WCPFH.NumSamples-1 ;
               WCPChannel[ch].ADCZeroAt := 1 ;
               end ;

           if WCPfH.NumRecords > 0 then Main.mnPrint.enabled := True ;
           NewFile := False ;
           State := PlotRecord ;
           end ;

        case State of

            { Plot record(s) on display }
            PlotRecord : begin
                if WCPfH.NumRecords > 0 then begin

                   Main.SetCopyMenu( True, True ) ;
                   Main.mnPrint.Enabled := True ;

                   sbRecordNum.Max := WCPfH.NumRecords ;
                   sbRecordNum.Min := 1 ;
                   sbRecordNum.Enabled := True ;
                   WCPfH.RecordNum := SbRecordNum.position ;

                   { Read A/D data from file }
                   GetRecord( WCPfH, RecHeader, WCPChannel, WCPfH.RecordNum, ADC^ ) ;

                   InitializeDisplay( pbDisplay) ;

                   if (WCPChannel[0].Cursor1 < Trunc(WCPChannel[0].xMin)) or
                      (WCPChannel[0].Cursor1 > Trunc(WCPChannel[0].xMax)) then
                      WCPChannel[0].Cursor1 := Trunc( (WCPChannel[0].xMin +
                                                    WCPChannel[0].xMax)/2. ) ;

                   { Remove cursors before updating display }
                   CursorChannel := WCPChannel[0] ;
                   CursorChannel.Top := 0 ;
                   CursorChannel.Bottom := pbDisplay.Height ;
                   CursorChannel.color := clRed ;
                   DrawCursor( pbDisplay, WCPChannel[0].Cursor1, CursorChannel,lbCursor1 ) ;
                   CursorChannel.color := clPurple ;
                   DrawCursor( pbDisplay, WCPChannel[0].Cursor0, CursorChannel,lbCursor0 ) ;
                   for ch := 0 to WCPfH.NumChannels-1 do if WCPChannel[ch].InUse then
                       DrawHorizontalCursor(pbDisplay,WCPChannel[ch],WCPChannel[ch].ADCZero) ;


                   EraseDisplay(pbDisplay) ;
                   pbDisplay.canvas.pen.color := clBlue ;
                   PlotChannels( pbDisplay, ADC^ ) ;

                   edRecordNum.text := format('Rec. %d/%d',[sbRecordNum.position,
                                                            WCPfH.NumRecords]) ;
                   WCPfH.CurrentRecord := sbRecordNum.position ;

                   { Show whether record has been rejected by operator }
                   if RecHeader.Status = 'ACCEPTED' then ckBadRecord.checked := False
                                                    else ckBadRecord.checked := True ;
                   { Show type of record }
                   if cbRecordType.items.indexOf(RecHeader.RecType) >= 0 then
                      cbRecordType.ItemIndex := cbRecordType.items.indexOf(RecHeader.RecType) ;

                   edTime.text := format( '%f s', [RecHeader.Time] ) ;
                   edGroup.text := format( '%d', [ Trunc(RecHeader.Number) ] ) ;
                 { Problem with this line, long delay
                 edRecordIdent.text := RecHeader.Ident ; }

                   if RecHeader.dt <> Olddt then begin
                      UpdateLowPassFilterList( RecHeader.dt ) ;
                      Olddt := RecHeader.dt ;
                      end ;
                   MoveCursor := False ;
                   State := UpdateCursorReadout ;
                   end
                else begin
                   { No records available }
                   ch := 0 ;
                   EraseDisplay(pbDisplay) ;
                   InitializeDisplay( pbDisplay) ;
                   edRecordNum.text := 'No Records ' ;
                   sbRecordNum.Enabled := False ;
                   State := Idle ;
                   end ;

                { Restore cursors }
                CursorChannel.color := clRed ;
                DrawCursor( pbDisplay, WCPChannel[0].Cursor1, CursorChannel, lbCursor1 ) ;
                CursorChannel.color := clPurple ;
                DrawCursor( pbDisplay, WCPChannel[0].Cursor0, CursorChannel, lbCursor0 ) ;
                for ch := 0 to WCPfH.NumChannels-1 do if WCPChannel[ch].InUse then
                    DrawHorizontalCursor(pbDisplay,WCPChannel[ch],WCPChannel[ch].ADCZero) ;

                end ;


            UpdateCursorReadout : begin
               { ** Display time and sample values under cursor **}

               WCPChannel[0].CursorTime := (WCPChannel[0].Cursor1
                                        - WCPChannel[0].Cursor0)*RecHeader.dt ;
               mmCursor.lines[0] := Format( 'T=  %5.4g %s',
                                   [WCPChannel[0].CursorTime*Settings.TScale,
                                    Settings.TUnits] ) ;

               for ch := 0 to WCPfH.NumChannels-1 do if WCPChannel[ch].InUse then begin
                   WCPChannel[ch].CursorTime := WCPChannel[0].CursorTime ;
                   ChOffset := WCPChannel[ch].ChannelOffset ;
                   i := WCPChannel[0].Cursor1*WCPfH.NumChannels + ChOffset ;
                   WCPChannel[ch].CursorValue := (ADC^[i] - WCPChannel[ch].ADCZero)
                                               * WCPChannel[ch].ADCScale ;
                   mmCursor.lines[ch+1] := WCPChannel[ch].ADCName + '= ' +
                               Format( '%5.4g', [WCPChannel[ch].CursorValue] ) +
                               + ' ' + WCPChannel[ch].ADCUnits  ;
                   end ;
               mmCursor.Lines[WCPfH.NumChannels+1] := '' ;
               State := Idle ;
               end;

            Idle : begin
                     
                     end ;
            end ;
        TimerBusy := False ;
        end ;
     end ;



procedure TViewWCPFrm.DrawHorizontalCursor( var pb : TPaintBox ;
                                Const Gr : TChannel ;
                                Level : Integer ) ;
{ ----------------------------------------------------
  Draw dotted horizontal cursor at ADC level 'Level'
  in display area defined by  record data channel 'Gr'
  ----------------------------------------------------}
var
   yPix,xPix,TicSize : Integer ;
   OldColor : TColor ;
   OldStyle : TPenStyle ;
   OldMode : TPenMode ;
begin
     with pb.canvas do begin
          OldColor := pen.color ;
          OldStyle := pen.Style ;
          OldMode := pen.mode ;
          pen.mode := pmXor ;
          pen.color := clRed ;
          end ;

     yPix := Trunc( Gr.Bottom - (Level - Gr.yMin)*Gr.yScale ) ;
     pb.canvas.polyline([Point(Gr.Left,yPix),Point(Gr.Right,yPix)]);

     { if zero level has been calculated from record show the point used }

     if (Gr.xMin <= Gr.ADCZeroAt) and (Gr.ADCZeroAt <= Gr.xMax ) then begin
          xPix := Trunc((Gr.ADCZeroAt - Gr.xMin)*Gr.xScale - Gr.Left) ;
          TicSize := pb.Canvas.textheight('X') ;
          pb.canvas.polyline( [Point( xPix, yPix+TicSize ),
                               Point( xPix, yPix-TicSize )] );
          end ;

     with pb.canvas do begin
          pen.style := OldStyle ;
          pen.color := OldColor ;
          pen.mode := OldMode ;
          end ;
     { Request that the cursor readout memo box be updated }
     State := UpdateCursorReadout ;
    end ;

procedure TViewWCPFrm.pbDisplayMouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);
Const
     Margin = 4 ;
var
   X_Cursor,Y_Cursor,X_Zero,ch : LongInt ;
   OldIndex : Integer ;
begin
     { Make sure the the record number box has the focus
       to avoid unintended effect if arrow keys are used to move the cursors }
     edRecordNum.Setfocus ;

     { Find locations of measurement and zero time cursors }

     X_Cursor := Trunc( ((WCPChannel[0].Cursor1 - WCPChannel[0].xMin) *
                          WCPChannel[0].xScale ) + WCPChannel[0].Left ) ;
     X_Zero := Trunc( ((WCPChannel[0].Cursor0 - WCPChannel[0].xMin)*
                          WCPChannel[0].xScale) + WCPChannel[0].Left ) ;

     if not MoveCursor then begin

          { If not in the cursor move mode, check to see if the mouse
            is over any of the cursors and change its icon accordingly }

          CursorState := NoCursor ;
          pbDisplay.Cursor := crDefault ;

          if Abs(X - X_Cursor) < Margin then begin
                CursorState := Cursor1 ;
                pbDisplay.Cursor := crSizeWE ;
                end
          else if ( Abs(X - X_Zero) < Margin ) then begin
                CursorState := Cursor0 ;
                pbDisplay.Cursor := crSizeWE ;
                end
          else begin
                for ch := 0 to WCPfH.NumChannels-1 do begin
                    Y_Cursor := Trunc( WCPChannel[ch].Bottom -
                                ((WCPChannel[ch].ADCZero - WCPChannel[ch].yMin) *
                                WCPChannel[ch].yScale) ) ;
                    if Abs(Y-Y_Cursor) < Margin then begin
                       CurCh := ch ;
                       CursorState := ZeroLevelCursor ;
                       pbDisplay.Cursor := crSizeNS ;
                       end ;
                    end ;
                end ;
          end
      else begin

          { If in Move Cursor mode, move selected cursor to new position }

          case CursorState of
          Cursor1 : begin
              CursorChannel.color := clRed ;
              VerticalCursorScale( X,WCPChannel[0].Cursor1,OldIndex,CursorChannel);
              MoveVerticalCursor( pbDisplay, WCPChannel[0].Cursor1, OldIndex,
                                  CursorChannel, lbCursor1 ) ;
              end ;
          Cursor0 : begin
              CursorChannel.color := clPurple ;
              VerticalCursorScale( X,WCPChannel[0].Cursor0,OldIndex,CursorChannel);
              MoveVerticalCursor( pbDisplay, WCPChannel[0].Cursor0, OldIndex,
                                  CursorChannel, lbCursor0 ) ;
              end ;

          ZeroLevelCursor : Begin

              { Remove existing cursor }

              DrawHorizontalCursor( pbDisplay, WCPChannel[CurCh], WCPChannel[Curch].ADCZero ) ;

              WCPChannel[CurCh].ADCZero := Trunc( ((WCPChannel[CurCh].Bottom - Y) /
                                               WCPChannel[CurCh].yScale)
                                               + WCPChannel[CurCh].yMin ) ;

              { Keep cursor within display limits }

              if WCPChannel[CurCh].ADCZero < WCPChannel[CurCh].yMin then
                 WCPChannel[CurCh].ADCZero := Trunc(WCPChannel[CurCh].yMin) ;
              if WCPChannel[CurCh].ADCZero > WCPChannel[CurCh].yMax then
                 WCPChannel[CurCh].ADCZero := Trunc(WCPChannel[CurCh].yMax) ;

              { Draw new cursor }

              DrawHorizontalCursor( pbDisplay, WCPChannel[CurCh], WCPChannel[CurCh].ADCZero ) ;
              end ;

          else ;
          end ;
          State := UpdateCursorReadOut ;
          end ;
      end ;



procedure TViewWCPFrm.pbDisplayMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
   ch : Integer ;
begin
     if State = Idle then begin
        MoveCursor := True ;
        { Find the channel that the mouse is over }
        for ch := 0 to WCPfH.NumChannels-1 do
            if (WCPChannel[ch].Bottom >= Y) and (Y >= WCPChannel[ch].top) then
               MouseOverChannel := ch ;
        end ;

     end;


procedure TViewWCPFrm.pbDisplayMouseUp(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
     if State = Idle then begin
        if (Button = mbRight) and (CursorState = ZeroLevelCursor) then begin
           ZeroFrm.ChSel := CurCh ;
           ZeroFrm.NewZeroAt := Trunc( (X - WCPChannel[0].Left) / WCPChannel[0].xScale
                                        + WCPChannel[0].xMin ) ;
           ZeroFrm.ShowModal ;
           State := PlotRecord ;
           end ;
        end ;
     MoveCursor := False ;
     end;


procedure TViewWCPFrm.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
{ ------------------------
  Function key processing
  -----------------------}
type
    TAction = (MoveCursor,ChangeRecord,None) ;
var
   Action : TAction ;
   Step,OldPosition : Integer ;
   NewRecord : LongInt ;
begin
     case key of
          VK_LEFT : begin
             Action := MoveCursor ;
             Step := -1 ;
             end ;
          VK_RIGHT : begin
             Action := MoveCursor ;
             Step := 1 ;
             end ;
          VK_PRIOR : begin
             Action := ChangeRecord ;
             Step := -1 ;
             end ;
          VK_NEXT : begin
             Action := ChangeRecord ;
             Step := 1 ;
             end ;
          else Action := None ;
          end ;

     case Action of
        { Move vertical display cursor }
        MoveCursor : begin
            Case CursorState of
             Cursor0 : begin
                 OldPosition := WCPChannel[0].Cursor0 ;
                 WCPChannel[0].Cursor0 := OldPosition + Step ;
                 CursorChannel.color := clPurple ;
                 MoveVerticalCursor( pbDisplay,WCPChannel[0].Cursor0,OldPosition,
                                     CursorChannel, lbCursor0 ) ;
                 end ;
             Cursor1 : begin
                 OldPosition := WCPChannel[0].Cursor1  ;
                 WCPChannel[0].Cursor1 := OldPosition + Step ;
                 CursorChannel.color := clRed ;
                 MoveVerticalCursor( pbDisplay,WCPChannel[0].Cursor1,OldPosition,
                                     CursorChannel, lbCursor1 ) ;
                 end ;
             end ;
            State := UpdateCursorReadout ;
            end ;
        { Change record currently displayed }
        ChangeRecord : begin
            NewRecord := MinInt([MaxInt([WCPfH.RecordNum + Step,1]),WCPfH.NumRecords]) ;
            sbRecordNum.Position := NewRecord ;
            State := PlotRecord ;
            end ;
        end ;

     end;


procedure TViewWCPFrm.FormDeactivate(Sender: TObject);
begin
     Timer.Enabled := False ;
     TimerBusy := False ;
     HeapBuffers( Deallocate ) ;
     Main.SetCopyMenu( False, False ) ; { Disable copy menu options}
     Main.mnPrint.Enabled := False ;
     end;


procedure TViewWCPFrm.FormActivate(Sender: TObject);
begin

     Main.ClearTicksInViewMenu ;
     Main.mnViewWCP.checked := True ;

     HeapBuffers( Allocate ) ;{ Allocate memory buffers }
     TimerBusy := False ;
     Timer.Enabled := True ;
     end;


procedure TViewWCPFrm.edRecordNumKeyPress(Sender: TObject; var Key: Char);
{ ------------------------------------
  Go to record number entered by user
  -----------------------------------}
var
   i : LongInt ;
begin
     if key = chr(13) then begin
        i := ExtractInt( edRecordNum.text ) ;
        if ( i > 0 ) and ( i < WCPfH.NumRecords ) then sbRecordNum.position := i ;
        State := PlotRecord ;
        end ;
     end;

procedure TViewWCPFrm.ckBadRecordClick(Sender: TObject);
{ ------------------------------------------------
  Save new record ACCEPTED/REJECTED status to file
  ------------------------------------------------}
begin
     if ckBadRecord.checked then RecHeader.Status := 'REJECTED'
                            else RecHeader.Status := 'ACCEPTED' ;
     PutRecordHeaderOnly( fH, RecHeader, WCPfH.RecordNum ) ;
     end;


procedure TViewWCPFrm.cbRecordTypeChange(Sender: TObject);
{ -----------------------------
  Save new record type to file
  ----------------------------}
begin
     RecHeader.RecType := cbRecordType.text ;
     PutRecordHeaderOnly( fH, RecHeader, WCPfH.RecordNum ) ;
     end;

procedure TViewWCPFrm.CopyDataToClipBoard ;
{ --------------------------------------------------
  Copy the currently displayed record to the clipboard
  25/6/98 Clipboard buffer limit reduced to 31000
  --------------------------------------------------}
const
     BufSize = 31000 ;
     NumBytesPerNumber = 12 ;
var
   i,j,iSkip,x,ch,n,nc,NumBytesNeeded : LongInt ;
   Line : String ;
   CopyBuf0,Line0 : PChar ;
   OK : Boolean ;
   ClipRec : ^TWCPClipboardRecord ;

begin
     try
        OK := OpenClipboard( Handle ) ;
        OK := EmptyClipBoard ;

        { Allocate buffer to hold signal }
        CopyBuf0 := StrAlloc( BufSize ) ;
        StrPCopy( CopyBuf0, '' ) ;
        Line0 := StrAlloc(  256 ) ;

        { Determine sample skip factor to ensure that the compete record
          fits into the buffer }
        n := Trunc(WCPChannel[0].xMax - WCPChannel[0].xMin)+1 ;
        nc := 0 ;
        for ch := 0 to WCPfH.NumChannels-1 do if WCPChannel[ch].InUse then Inc(nc) ;
        NumBytesNeeded := n*(nc+1)*NumBytesPerNumber ;
        iSkip := MaxInt( [(NumBytesNeeded+BufSize-1) div BufSize,1]) ;

        i := Trunc(WCPChannel[0].xMin) ;
        screen.cursor := crHourglass ;
        while  i <= WCPChannel[0].xMax do begin
             { Create a line of <tab> separated ASCII data values terminated by <cr> <lf>
               Time <tab> Ch.0.Value <tab> Ch.1.Value .... <cr> <lf> }
             Line := format( '%5.4g', [i*RecHeader.dt*Settings.TScale] ) ;
             for ch := 0 to WCPfH.NumChannels-1 do if WCPChannel[ch].InUse then begin
                 j := i*WCPfH.NumChannels + WCPChannel[ch].ChannelOffset ;
                 Line := Line + chr(9) + Format('%10.4g',
                                         [(ADC^[j] - WCPChannel[ch].ADCZero)
                                               * WCPChannel[ch].ADCScale] ) ;
                 end ;
             Line := Line + chr(13) + chr(10) ;
             StrPCopy( Line0, Line ) ;
             CopyBuf0 := StrCat( CopyBuf0, Line0 ) ;
             i := i + iSkip ;
             end ;

        { Copy text accumulated in copy buffer to clipboard }
         ClipBoard.SetTextBuf( CopyBuf0 ) ;

     finally
         screen.cursor := crDefault ;
         { Dispose of buffers }
         StrDispose( Line0 ) ;
         StrDispose( CopyBuf0 ) ;
         OK := CloseClipboard ;
         end ;
     end ;


procedure TViewWCPFrm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
     HeapBuffers( Deallocate ) ;
     Main.ViewWCPChildExists := False ;
     Main.SetCopyMenu( False, False ) ; { Disable copy menu options}
     Main.mnPrint.Enabled := False ;
     Action := caFree ;
     end;


procedure TViewWCPFrm.CopyRecordImageToClipboard ;
{ ---------------------------------------------------------
  Plot currently displayed record on as bitmap in clipboard
  ---------------------------------------------------------}

var
    PrChan : array[0..ChannelLimit] of TChannel ;
   Page : TPlot ;
    TimeBarValue : single  ;
    PlotOK : boolean ;
    ch, i, Num, xPix, yPix : Integer ;
    ChanHeight,ChanTop,ChOffset,ChLast : Integer ;
    x,y,dx : single ;
    nOffScreen : Integer ;
    BMap : TBitmap ;

begin

     { If no previous settings ... define an usable set }
     if Settings.TimeBarValue = -1. then
        Settings.TimeBarValue := (WCPChannel[0].Xmax - WCPChannel[0].xMin) *
                                 RecHeader.dt*0.1  ;
     for ch := 0 to WCPfH.NumChannels-1 do if Settings.BarValue[ch] = -1. then
         Settings.BarValue[ch] := (WCPChannel[ch].yMax - WCPChannel[ch].yMin)
                                           * WCPChannel[ch].ADCScale * 0.1 ;

     { Get calibration bar sizes and other options from user }
     CopyRecDlg.ShowModal ;
     if CopyRecDlg.ModalResult = mrOK then begin

          Screen.Cursor := crHourglass ;

          { Create and size bitmap object }
          BMap := TBitmap.Create ;
          BMap.Height := Settings.BitmapHeight ;
          BMap.Width := Settings.BitmapWidth ;
          BMap.canvas.font.name := Settings.Plot.FontName ;
          BMap.canvas.font.size := Settings.Plot.FontSize ;
          BMap.Canvas.Pen.Width := Settings.Plot.LineThickness ;
          Page.Left := BMap.canvas.TextWidth('XXXXXXXXXX') ;
          Page.Right := BMap.Width - Page.Left ;
          Page.Top := BMap.canvas.TextHeight('X') ;
          Page.Bottom :=BMap.Height - 5*Page.Top ;

          { Make a copy of channel scaling information and find out
          how many channels are on display }
          Num := 0 ;
          for ch := 0 to WCPfH.NumChannels-1 do begin
              { Copy Printer channel from Display channel record }
              PrChan[ch] := WCPChannel[ch] ;
              { Use black if colour printing is not required }
              if not Settings.Plot.UseColor then PrChan[ch].Color := clBlack ;
              if PrChan[ch].InUse then Inc(Num) ;
              end ;
          ChanHeight := (Page.Bottom - Page.Top) div MaxInt( [Num,1] ) ;

          { Set up channel display areas on printed page }
          ChanTop := Page.Top ;
          for ch := 0 to WCPfH.NumChannels-1 do begin
              PrChan[ch].Left := Page.Left ;
              PrChan[ch].Right := Page.Right ;
              PrChan[ch].Top := ChanTop ;
              PrChan[ch].Bottom := PrChan[ch].Top + ChanHeight ;
              PrChan[ch].xScale := (PrChan[ch].Right - PrChan[ch].Left) /
                                   (PrChan[ch].xMax  - PrChan[ch].xMin ) ;
              PrChan[ch].yScale := (PrChan[ch].Bottom - PrChan[ch].Top) /
                                   (PrChan[ch].yMax   - PrChan[ch].yMin ) ;
              if PrChan[ch].InUse then ChanTop := ChanTop + ChanHeight ;
              end ;

          { Display Channel Name(s) }
          if Settings.ShowLabels then begin
             for ch := 0 to WCPfH.NumChannels-1 do if PrChan[ch].InUse then
                 BMap.Canvas.TextOut( PrChan[ch].Left -
                              printer.canvas.textwidth(prChan[ch].ADCName+' '),
                              (PrChan[ch].Top + PrChan[ch].Bottom) div 2,
                              PrChan[ch].ADCName ) ;
             end ;

          GetRecord( WCPfH, RecHeader, WCPChannel, WCPFH.RecordNum, ADC^ ) ;

          { Plot each record channel in turn }
          for ch := 0 to WCPfH.NumChannels-1 do if PrChan[ch].InUse then begin

              ChLast := ch ;
              dx := 1. ;
              x := 0. ;
              nOffScreen := 0 ;
              ChOffset := WCPChannel[ch].ChannelOffset ;
              for i := 0 to WCPfH.NumSamples-1 do begin

                  xPix := Trunc(PrChan[ch].xScale*(x - PrChan[ch].xMin)
                                 + PrChan[ch].Left) ;
                  y := ADC^[(i*WCPfH.NumChannels)+ChOffset] ;
                  yPix := Trunc(PrChan[ch].Bottom -
                                PrChan[ch].yScale*(y - PrChan[ch].yMin));

                  if (xPix < PrChan[ch].Left) or
                     (PrChan[ch].Right < xPix ) or
                     (yPix < PrChan[ch].Top) or
                     (PrChan[ch].Bottom < yPix ) then begin
                     xPix := MinInt( [ MaxInt( [xPix,PrChan[ch].Left]),
                                              PrChan[ch].Right] ) ;
                     yPix := MinInt( [ MaxInt( [yPix,PrChan[ch].Top]),
                                              PrChan[ch].Bottom] ) ;
                     Inc(nOffScreen) ;
                     end
                  else nOffScreen := 0 ;

                  if (i = 0) or (nOffScreen >1) then begin
                     BMap.canvas.pen.color := PrChan[ch].color ;
                     BMap.canvas.moveto(xPix,yPix) ;
                     end
                  else  BMap.canvas.LineTo(xPix,yPix) ;
                  x := x + dx ;

                  end ;
              end ;

          { Print zero level cursor(s) }
          If Settings.ShowZeroLevels then begin
             for ch := 0 to WCPfH.NumChannels-1 do if PrChan[ch].InUse then begin
                 BMap.canvas.pen.color := PrChan[ch].color ;
                 PrintHorizontalCursor(BMap.Canvas,PrChan[ch],WCPChannel[ch].ADCZero) ;
                 end ;
             end ;

          { Horizontal (time) calibration bar }
          BMap.canvas.pen.color := clBlack ;
          HorizontalBar(BMap.canvas,PrChan[ChLast],Settings.ShowLabels,
                        Settings.TimeBarValue,RecHeader.dt,Settings.TScale,
                        Settings.TUnits) ;

          { Vertical calibration bars }
          for ch := 0 to WCPfH.NumChannels-1 do if PrChan[ch].InUse then
              VerticalBar(BMap.canvas,PrChan[ch],Settings.ShowLabels,
                                  Settings.BarValue[ch],PrChan[ch].ADCUnits) ;

          { Close bitmap object }
          Clipboard.Assign(BMap) ;
          BMap.Free ;
          end ;

     Screen.Cursor := crDefault ;
     end;


procedure TViewWCPFrm.PrintRecord ;
{ --------------------------------------------
  Plot currently displayed records on printer
  --------------------------------------------}
var
    PrChan : array[0..ChannelLimit] of TChannel ;
    Page : TPlot ;
    TimeBarValue : single  ;
    ch, i, Num, xPix, yPix : Integer ;
    ChanHeight,ChanTop,ChOffset,ChLast : Integer ;
    x,y,dx : single ;
    OldPenWidth,nOffScreen : Integer ;

begin

     { If no previous settings ... define an usable set }
     if Settings.TimeBarValue = -1. then
        Settings.TimeBarValue := (WCPChannel[0].Xmax - WCPChannel[0].xMin) *
                                 RecHeader.dt*0.1  ;
     for ch := 0 to WCPfH.NumChannels-1 do if Settings.BarValue[ch] = -1. then
         Settings.BarValue[ch] := (WCPChannel[ch].yMax - WCPChannel[ch].yMin)
                                           * WCPChannel[ch].ADCScale * 0.1 ;

     { Get calibration bar sizes and other options from user }
     PrintRecFrm.ShowModal ;

     if PrintRecFrm.ModalResult = mrOK then begin

        Screen.Cursor := crHourglass ;

        { Set page size etc. }
        Printer.canvas.font.name := Settings.Plot.FontName ;
        Printer.canvas.font.size := PrinterPointsToPixels(Settings.Plot.FontSize) ;

        Printer.Canvas.Pen.Width := PrinterPointsToPixels(
                                    Settings.Plot.LineThickness) ;
        Page.Left := PrinterCmToPixels('H',Settings.Plot.LeftMargin) ;
        Page.Right := Printer.pagewidth -
                            PrinterCmToPixels('H',Settings.Plot.RightMargin) ;
        Page.Top := PrinterCmToPixels('V',Settings.Plot.TopMargin) ;
        Page.Bottom := Printer.pageheight -
                       PrinterCmToPixels('V',Settings.Plot.BottomMargin) ; ;

        Printer.BeginDoc ;
        Printer.Canvas.Pen.Color := clBlack ;

        PrintHeaderAndFooter ;

        { Make a copy of channel scaling information and find out
          how many channels are on display }
        Num := 0 ;
        for ch := 0 to WCPfH.NumChannels-1 do begin
            { Copy Printer channel from Display channel record }
            PrChan[ch] := WCPChannel[ch] ;
            { Use black if colour printing is not required }
            if not Settings.Plot.UseColor then PrChan[ch].Color := clBlack ;
            if PrChan[ch].InUse then Inc(Num) ;
            end ;
        ChanHeight := (Page.Bottom - Page.Top) div MaxInt( [Num,1] ) ;

        { Set up channel display areas on printed page }
        ChanTop := Page.Top ;
        for ch := 0 to WCPfH.NumChannels-1 do begin
              PrChan[ch].Left := Page.Left ;
              PrChan[ch].Right := Page.Right ;
              PrChan[ch].Top := ChanTop ;
              PrChan[ch].Bottom := PrChan[ch].Top + ChanHeight ;
              PrChan[ch].xScale := (PrChan[ch].Right - PrChan[ch].Left) /
                                   (PrChan[ch].xMax  - PrChan[ch].xMin ) ;
              PrChan[ch].yScale := (PrChan[ch].Bottom - PrChan[ch].Top) /
                                   (PrChan[ch].yMax   - PrChan[ch].yMin ) ;
              if PrChan[ch].InUse then ChanTop := ChanTop + ChanHeight ;
              end ;

        { Display Channel Name(s) }
        if Settings.ShowLabels then begin
             for ch := 0 to WCPfH.NumChannels-1 do if PrChan[ch].InUse then
                 Printer.Canvas.TextOut( PrChan[ch].Left -
                              printer.canvas.textwidth(prChan[ch].ADCName+' '),
                              (PrChan[ch].Top + PrChan[ch].Bottom) div 2,
                              PrChan[ch].ADCName )
             end ;

        GetRecord( WCPfH, RecHeader, WCPChannel, WCPFH.RecordNum, ADC^ ) ;

        { Plot each record channel in turn }
        for ch := 0 to WCPfH.NumChannels-1 do if PrChan[ch].InUse then begin

            ChLast := ch ;
            dx := 1. ;
            x := 0. ;
            nOffScreen := 0 ;
            ChOffset := WCPChannel[ch].ChannelOffset ;
            for i := 0 to WCPfH.NumSamples-1 do begin

                xPix := Trunc(PrChan[ch].xScale*(x - PrChan[ch].xMin)
                              + PrChan[ch].Left) ;
                y := ADC^[(i*WCPfH.NumChannels)+ChOffset] ;
                yPix := Trunc(PrChan[ch].Bottom -
                              PrChan[ch].yScale*(y - PrChan[ch].yMin));

                if (xPix < PrChan[ch].Left) or
                   (PrChan[ch].Right < xPix ) or
                   (yPix < PrChan[ch].Top) or
                   (PrChan[ch].Bottom < yPix ) then begin
                     xPix := MinInt( [ MaxInt( [xPix,PrChan[ch].Left]),
                                              PrChan[ch].Right] ) ;
                     yPix := MinInt( [ MaxInt( [yPix,PrChan[ch].Top]),
                                              PrChan[ch].Bottom] ) ;
                     Inc(nOffScreen) ;
                     end
                else nOffScreen := 0 ;

                if (i = 0) or (nOffScreen >1) then begin
                   Printer.canvas.pen.color := PrChan[ch].color ;
                   Printer.canvas.moveto(xPix,yPix) ;
                   end
                else Printer.canvas.lineto(xPix,yPix) ;
                x := x + dx ;

                end ;
            end ;

        { Print zero level cursor(s) }
        If Settings.ShowZeroLevels then begin
           for ch := 0 to WCPfH.NumChannels-1 do if PrChan[ch].InUse then begin
               Printer.canvas.pen.color := PrChan[ch].color ;
               PrintHorizontalCursor(Printer.Canvas,PrChan[ch],WCPChannel[ch].ADCZero) ;
               end ;
           end ;

        { Horizontal (time) calibration bar }
        Printer.canvas.pen.color := clBlack ;
        HorizontalBar(Printer.canvas,PrChan[ChLast],Settings.ShowLabels,
                      Settings.TimeBarValue,RecHeader.dt,Settings.TScale,
                      Settings.TUnits) ;

        { Vertical calibration bars }
        for ch := 0 to WCPfH.NumChannels-1 do if PrChan[ch].InUse then
            VerticalBar(Printer.canvas,PrChan[ch],Settings.ShowLabels,
                        Settings.BarValue[ch],PrChan[ch].ADCUnits) ;

        { Close down printer }
        Printer.EndDoc ;
        end ;

     Screen.Cursor := crDefault ;
     end;


procedure TViewWCPFrm.UpdateLowPassFilterList(dt : single) ;
{ -----------------------------------------------------------------------
  Update combo box containing list of low-pass filter cut-off frequencies
  -----------------------------------------------------------------------}
var
   cutoff,Diff,MinDiff : single ;
   i : Integer ;
begin
     { Add list of frequencies to box }
     { Note .. division by sampling rate to convert from samples^-1 units to Hz }
     cbLowPassFilter.Clear ;
     cbLowPassFilter.Items.Add( 'No Filter ') ;
     cbLowPassFilter.Items.Add( format( '%7.1f Hz', [0.1/dt] ) );
     cbLowPassFilter.Items.Add( format( '%7.1f Hz', [0.08/dt] ) );
     cbLowPassFilter.Items.Add( format( '%7.1f Hz', [0.05/dt] ) );
     cbLowPassFilter.Items.Add( format( '%7.1f Hz', [0.02/dt] ) );
     cbLowPassFilter.Items.Add( format( '%7.1f Hz', [0.01/dt] ) );
     cbLowPassFilter.Items.Add( format( '%7.1f Hz', [0.008/dt] ) );
     cbLowPassFilter.Items.Add( format( '%7.1f Hz', [0.005/dt] ) );

     { Find the entry which is closest to the current setting }
     if Settings.CutOffFrequency <= 0. then cbLowPassFilter.ItemIndex := 0
     else begin
          MinDiff := 1E30 ;
          for i := 1 to cbLowPassFilter.Items.Count-1 do begin
              Diff := Abs( ExtractFloat( cbLowPassFilter.text, 1. ) -
                           (Settings.CutOffFrequency/dt) );
              if Diff < MinDiff then begin
                 MinDiff := Diff ;
                 cbLowPassFilter.ItemIndex := i ;
                 end ;
              end ;
          end ;
     end ;


procedure TViewWCPFrm.cbLowPassFilterChange(Sender: TObject);
{ ----------------------------------------------------
  Request a new plot if the low pass filter is changed
  ----------------------------------------------------}
begin
     State := PlotRecord ;
     { Update LP filter setting }
     if cbLowPassFilter.ItemIndex <= 0 then Settings.CutOffFrequency := 0.
     else Settings.CutOffFrequency := ExtractFloat( cbLowPassFilter.text,
                                      Settings.CutOffFrequency )*RecHeader.dt ;
     end;

procedure TViewWCPFrm.pbDisplayDblClick(Sender: TObject);
{ -------------------------------
  Activate Zoom In/Out dialog box
  -------------------------------}
var
   i : Integer ;
begin
     ZoomWCPFrm.ChOnDisplay := MouseOverChannel ;
     Timer.Enabled := False ;
     ZoomWCPFrm.ShowModal ;
     Timer.Enabled := True ;
     State := PlotRecord ;
     MoveCursor := False ;
     {pbDisplay.Refresh ;}
     end;


procedure TViewWCPFrm.edIdentChange(Sender: TObject);
begin
     { Update ident line if it is changed }
     WCPfH.IdentLine := edIdent.text ;
     WCPfH.IdentLine := EdIdent.text ;
     SaveWCPHeader(WCPFH) ;
     end;

procedure TViewWCPFrm.edIdentKeyPress(Sender: TObject; var Key: Char);
{ -------------------------------------------
  Update identification string in file header
  -------------------------------------------}
begin
     if key = chr(13) then begin
        WcpFH.IdentLine := edIdent.Text ;
        SaveWCPHeader( WcpFH ) ;
        WriteToLogFile( edIdent.text ) ;
        end;
     end;


procedure TViewWCPFrm.EdRecordIdentChange(Sender: TObject);
begin
     RecHeader.Ident := EdRecordIdent.text ;
     PutRecordHeaderOnly( fH, RecHeader, WCPfH.RecordNum ) ;
     end;

procedure TViewWCPFrm.edGroupKeyPress(Sender: TObject; var Key: Char);
{ ------------------------------------
  Save new record group number to file
  ------------------------------------}
begin
     if Key = chr(13) then begin
        RecHeader.Number := ExtractInt( Edgroup.text ) ;
        PutRecordHeaderOnly( fH, RecHeader, WCPfH.RecordNum ) ;
        end ;
     end;

procedure TViewWCPFrm.FormHide(Sender: TObject);
begin
     HeapBuffers( Deallocate ) ;
     Timer.Enabled := False ;
     end;

procedure TViewWCPFrm.InitializeDisplay( const PB : TPaintBox ) ;
{ ----------------------------
  Initialise a display window
  ---------------------------}
var
   Height,ch,cTop,NumInUse : LongInt ;
begin

     { Set trace colour }

     PB.canvas.pen.color := WCPChannel[0].color ;

     { Determine number of channels in use and the height
       available for each channel }

     NumInUse := 0 ;
     for ch := 0 to WCPfH.NumChannels-1 do
         if WCPChannel[ch].InUse then NumInUse := NumInUse + 1 ;
     Height := PB.Height div MaxInt( [NumInUse,1] ) ;

     { Define display area for each channel in use }

     cTop := 0 ;
     for ch := 0 to WCPfH.NumChannels-1 do begin
          if WCPChannel[ch].InUse then begin
             WCPChannel[ch].Left := 0 ;
             WCPChannel[ch].Right := PB.width ;
             WCPChannel[ch].Top := cTop ;
             WCPChannel[ch].Bottom := WCPChannel[ch].Top + Height ;
             WCPChannel[ch].xMin := WCPChannel[0].xMin ;
             WCPChannel[ch].xMax := WCPChannel[0].xMax ;
             WCPChannel[ch].xScale := (WCPChannel[ch].Right - WCPChannel[ch].Left) /
                                (WCPChannel[ch].xMax - WCPChannel[ch].xMin ) ;
             WCPChannel[ch].yScale := (WCPChannel[ch].Bottom - WCPChannel[ch].Top) /
                                (WCPChannel[ch].yMax - WCPChannel[ch].yMin ) ;
             cTop := cTop + Height ;
             end ;
          end ;

     lbTMin.caption := Format( '%5.4g %s', [WCPChannel[0].xMin*RecHeader.dt*Settings.TScale,
                                               Settings.TUnits] ) ;
     lbTMax.caption := Format( '%5.4g %s', [WCPChannel[0].xMax*RecHeader.dt*Settings.TScale,
                                               Settings.TUnits] ) ;

     { Display Channel Name(s) }

     for ch := 0 to WCPfH.NumChannels-1 do
        if WCPChannel[ch].InUse then
           PB.Canvas.TextOut( WCPChannel[ch].Left,
           (WCPChannel[ch].Top + WCPChannel[ch].Bottom) div 2,' ' + WCPChannel[ch].ADCName ) ;

     end ;


procedure TViewWCPFrm.PlotChannels( PB : TPaintBox ; var Buf : Array of Integer ) ;
{ -------------------------------------
  Plot selected channels in a WCP record
  -------------------------------------}
var
   i,j,n,ch : Integer ;
   x : single ;
   OK : Boolean ;
begin
     { Clear display area }
     InitializeDisplay( PB ) ;
     EraseDisplay( PB ) ;

     for ch := 0 to WCPFH.NumChannels-1 do begin

         n := 0 ;
         x := WCPChannel[ch].xMin ;
         for i := 0 to WCPfH.NumSamples-1 do begin
             j := (i*WCPfH.NumChannels) + WCPChannel[ch].ChannelOffset ;
             xy^[n].y := WCPChannel[ch].Bottom - Trunc(
                         WCPChannel[ch].yScale*(Buf[j] - WCPChannel[ch].yMin));
             xy^[n].x := Trunc(WCPChannel[ch].xScale*(x -
                         WCPChannel[ch].xMin) + WCPChannel[ch].Left) ;
             Inc(n) ;
             x := x + 1.0 ;
             end ;
         OK := Polyline( PB.Canvas.Handle, xy^, n ) ;
         end ;


  {   DrawHorizontalCursor( PB , DetChannel, Detector.yBaseline, clRed ) ;
     DrawHorizontalCursor( PB , DetChannel,
                           Detector.yBaseline + Detector.yThreshold, clBlue ) ;}
     end ;


end.


