unit Viewcdr;
{ ========================================================
  WinCDR (c) John Dempster, University of Strathclyde 1998
  WCD DATA FILE DISPLAY MODULE
  V0.99 0/3/98
  28/6/98 ... PrintRecord & CopyRecordsImageToClipboard replace PlotRecords
  30/6/98 ... Ident line now updates correctly
  ========================================================}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, ExtCtrls, global, printers, plotlib, ClipBrd, fileio,
  Grids, CDRZero ;

type
  
  TViewCDRFrm = class(TForm)
    pbDisplay: TPaintBox;
    edIdent: TEdit;
    Label3: TLabel;
    CursorGrp: TGroupBox;
    RecordGrp: TGroupBox;
    Label1: TLabel;
    edTime: TEdit;
    lbTMin: TLabel;
    lbTMax: TLabel;
    lbCursor0: TLabel;
    lbCursor1: TLabel;
    sbDisplay: TScrollBar;
    Timer: TTimer;
    sgCursor: TStringGrid;
    lbFileSize: TLabel;
    procedure TimerTimer(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormResize(Sender: TObject);
    procedure sbDisplayChange(Sender: TObject);
    procedure pbDisplayDblClick(Sender: TObject);
    procedure pbDisplayMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FormCreate(Sender: TObject);
    procedure pbDisplayMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure pbDisplayMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure pbDisplayPaint(Sender: TObject);
    procedure edIdentKeyPress(Sender: TObject; var Key: Char);
  private
    { Private declarations }
    procedure HeapBuffers( Operation : THeapBufferOp ) ;
    procedure InitializeDisplay( PB : TPaintBox ) ;
    procedure DrawHorizontalCursor( Const Chan : TChannel ;
                                    Level : Integer ) ;
    procedure DrawVerticalCursor( Index : Integer ;
                               const Chan : TChannel ;
                               const Lab : TLabel ) ;
    procedure MoveVerticalCursor( var NewPosition, OldPosition : Integer ;
                                  const CursorChannel : TChannel ;
                                  var CursorLabel : TLabel ) ;
    procedure VerticalCursorScale( MouseXPos : Integer ;
                                   var CurrentIndex,OldIndex : Integer ;
                                   const Chan : TChannel ) ;
    procedure HorizontalCursorScale( MouseYPos : Integer ;
                                     var CurrentLevel,OldLevel : LongInt ;
                                     const Channel : TChannel ) ;
    procedure MoveHorizontalCursor( var pb : TPaintbox ;
                                    var NewLevel,OldLevel : LongInt ;
                                    const Channel : TChannel ) ;
    function OverHorizontalCursor( Y : Integer ;
                                   const Channel : TChannel ) : Boolean ;
    procedure CompressSignal ;

  public
    { Public declarations }
    NewFile : Boolean ;
    ViewBuf : ^TViewBuf ;
    ZoomBuf : ^TViewBuf ;
    procedure PrintRecord ;
    procedure CopyDataToClipBoard ;
    procedure CopyRecordsImageToClipboard ;
  end;



var
  ViewCDRFrm: TViewCDRFrm;

implementation

uses mdiform,maths,shared,zoomcdr,Printrec,copyrec ;

type
    TCursorState = ( Cursor0,Cursor1,ZeroLevelCursor,NoCursor ) ;
    TViewRecState = (PlotRecord,UpdateCursorReadout,Idle,
                   DoCopyToClipboardAsData) ;
    TPointArray = Array[0..2000] of TPoint ;
var
   TimerBusy : Boolean ;
   HardCopy : Boolean ;
   BuffersAllocated : Boolean ;
   State : TViewRecState ;
   ADC : ^TIntArray ;
   ViewBuf : ^TViewBuf ;
   MoveCursor : Boolean ;
   xy : ^TPointArray ;
   ScrollBarToBlockScale : LongInt ;
   CursorState : TCursorState ;
   CurCh : Integer ;
   CursorChannel : TChannel ;
{$R *.DFM}

procedure TViewCDRFrm.HeapBuffers( Operation : THeapBufferOp ) ;
{ -----------------------------------------------
  Allocate/deallocation dynamic buffers from heap
  -----------------------------------------------}
begin
     case Operation of
          Allocate : begin
             if not BuffersAllocated then begin
                New(ADC) ;
                New(ViewBuf) ;
                New(ZoomBuf) ;
                New(xy) ;
                BuffersAllocated := True ;
                end ;
             end ;
          Deallocate : begin
             if BuffersAllocated then begin
                Dispose(ADC) ;
                Dispose(ViewBuf) ;
                Dispose(ZoomBuf) ;
                Dispose(xy) ;
                BuffersAllocated := False ;
                end ;
             end ;
          end ;
     end ;


procedure TViewCDRFrm.FormCreate(Sender: TObject);
{ ------------------------------------
  Initialisations when form is created
  ------------------------------------}
begin
     BuffersAllocated := False ;
     NewFile := True ;

     end;


procedure TViewCDRFrm.FormShow(Sender: TObject);
{ ---------------------------------------
  Initialisations done when form is shown
  ---------------------------------------}
var
   i : Integer ;
begin

     HeapBuffers( Allocate ) ;

     Main.ViewCDRChildExists := True ;

     ClientHeight := CursorGrp.Top + CursorGrp.Height + 5 ;
     ClientWidth := edIdent.Left + EdIdent.Width + 10 ;

     { Start scheduling timer }
     Timer.enabled := True ;
     TimerBusy := False ;

     { Request a record to be displayed }
     State := PlotRecord ;
     HardCopy := False ;
     NewFile := True ;
     Timer.Enabled := True ;

     end;


procedure TViewCDRFrm.TimerTimer(Sender: TObject);
{ -----------------------------------------------------------
  Process scheduler ... plots records, updates cursor readout
  -----------------------------------------------------------}
var
   OK,Done : Boolean ;
   x,xStep,yMin,yMax : single ;
   i,FilePointer,nRead : LongInt ;
   y,ch,xPix,yPix,n,Row : Integer ;
   iSmall : Integer ;
begin
     if not TimerBusy then  begin

        TimerBusy := True ;

        if NewFile then begin
           { Get header data from file }
           GetCDRHeader( CdrFH ) ;

           { Display ident. info. }
           caption := 'View Recorded ' + CdrFH.FileName ;
           EdIdent.text := CdrFH.IdentLine ;

           { Clear readout cursor box }
           sgCursor.Cells[0,0] := 'Ch.' ;
           sgCursor.Cells[1,0] := 'Time=' ;
           sgCursor.RowCount := CdrFH.NumChannels*2 + 1 ;
           for ch := 0 to CdrFH.NumChannels-1 do begin
               Row := (ch*2) + 1 ;
               sgCursor.Cells[0,Row] := Channel[ch].ADCName ;
               sgCursor.Cells[1,Row] := 'Min.=' ;
               sgCursor.Cells[0,Row+1] := ' ' ;
               sgCursor.Cells[1,Row+1] := 'Max.=' ;
               end ;
           sgCursor.ColWidths[0] := sgCursor.Canvas.TextWidth('Ch.000') ;
           sgCursor.ColWidths[1] := sgCursor.Canvas.TextWidth('Time= ') ;
           sgCursor.ColWidths[2] := sgCursor.Canvas.TextWidth('XXXXXXXXXXX') ;

           if CdrFH.NumSamplesInFile > 0 then Main.mnPrint.enabled := True ;
           edTime.text := format( ' %.2f s',[ CdrFH.dt *
                          (CdrFH.NumSamplesInFile div CdrFH.NumChannels) ] ) ;
           { Size of data in file }
           lbFileSize.caption := format( ' %d Kb',
                           [Trunc((Settings.RecordDuration*CdrFH.NumChannels*2)
                                  /(CdrFH.dt*1024)) ] ) ;

           CdrFH.NumSamplesPerBlock := NumSamplesPerSector*CdrFH.NumChannels ;
           CdrFH.NumBytesPerBlock := CdrFH.NumSamplesPerBlock*2 ;
           CdrFH.NumBlocksInFile := CdrFH.NumSamplesInFile div CdrFH.NumSamplesPerBlock ;
           ScrollBarToBlockScale := MaxInt( [ ((CdrFH.NumBlocksInFile+High(iSmall))
                                                 div High(iSmall)),1]) ;
           sbDisplay.Max := (CdrFH.NumBlocksInFile div ScrollBarToBlockScale) - 1 ;
           sbDisplay.Position := 0 ;
           Channel[0].xMin := 0.0 ;
           Channel[0].xMax := (CdrFH.NumSamplesInFile div CdrFH.NumChannels)
                            *CdrFH.dt ;
           ViewBuf^.tDisplay := MaxFlt(
                 [(CdrFH.NumSamplesInFile div CdrFH.NumChannels)*CdrFH.dt,1.0]) ;


           { Create the compressed signal display }
           CompressSignal ;

           { Copy compressed image of whole file to zoom display buffer }
           for ch := 0 to CdrFH.NumChannels-1 do
               for i := 0 to ViewBuf^.nPoints-1 do begin
                   ZoomBuf^.x[i] := ViewBuf^.x[i] ;
                   ZoomBuf^.yMin[ch,i] := ViewBuf^.yMin[ch,i] ;
                   ZoomBuf^.yMax[ch,i] := ViewBuf^.yMax[ch,i] ;
                   end ;
           ZoomBuf^.xMin := ViewBuf^.xMin ;
           ZoomBuf^.xMax := ViewBuf^.xMax ;
           ZoomBuf^.tDisplay := ViewBuf^.tDisplay ;
           ZoomBuf^.nPoints := ViewBuf^.nPoints ;
           ZoomBuf^.ChannelSelected := 0 ;

           { Place cursor in middle of screen }
           Channel[0].Cursor0 := 0 ;
           Channel[0].Cursor1 := ViewBuf^.nPoints div 2 ;

           NewFile := False ;
           State := PlotRecord ;
           end ;

        case State of

             PlotRecord : begin
                 { Erase plotting area and update labels }

                 Channel[0].xMin := sbDisplay.position*ScrollBarToBlockScale ;
                 Channel[0].xMin := (Channel[0].xMin*CdrFH.NumSamplesPerBlock*CdrFH.dt) /
                                    CdrFH.NumChannels ;
                 Channel[0].xMax := Channel[0].xMin + ViewBuf^.tDisplay ;
                 InitializeDisplay( pbDisplay ) ;
                 EraseDisplay( pbDisplay ) ;

                 { Recreate  compressed signal block, if block has changed }
                 if (Channel[0].xMin <> ViewBuf^.xMin)
                    or (Channel[0].xMax <> ViewBuf^.xMax) then CompressSignal ;

                 { Plot digitised  signal on display }
                 for ch := 0 to CdrFH.NumChannels-1 do
                     if Channel[ch].InUse then begin
                     pbDisplay.canvas.pen.color := Channel[ch].color ;

                     n := 0 ;
                     for i := 0 to ViewBuf^.nPoints-1 do begin
                         xy^[n].y := Channel[ch].Bottom - Trunc(
                                     Channel[ch].yScale*(ViewBuf^.yMin[ch,i]
                                     - Channel[ch].yMin));
                         xy^[n].x := Trunc(Channel[ch].xScale*(ViewBuf^.x[i] -
                                     Channel[ch].xMin) + Channel[ch].Left) ;
                         Inc(n) ;
                         xy^[n].y := Channel[ch].Bottom - Trunc(
                                     Channel[ch].yScale*(ViewBuf^.yMax[ch,i] -
                                     Channel[ch].yMin));
                         xy^[n].x := xy^[n-1].x ;
                         Inc(n) ;
                         end ;
                     OK := Polyline( pbDisplay.Canvas.Handle, xy^, n ) ;

                     end ;

                 { Restore cursors }
                 CursorChannel := Channel[0] ;
                 CursorChannel.Top := 0 ;
                 CursorChannel.Bottom := pbDisplay.Height ;
                 CursorChannel.color := clRed ;

                 DrawVerticalCursor( Channel[0].Cursor1,CursorChannel,lbCursor1) ;
                 CursorChannel.color := clPurple ;
                 DrawVerticalCursor( Channel[0].Cursor0,CursorChannel,lbCursor0) ;

                 { Draw zero level cursors }
                 for ch := 0 to CdrFH.NumChannels-1 do if Channel[ch].InUse then
                     DrawHorizontalCursor(Channel[ch],Channel[ch].ADCZero) ;

                 { Channel names }
                 for ch := 0 to CdrFH.NumChannels-1 do if Channel[ch].InUse then
                     pbDisplay.canvas.textout( Channel[ch].Left,
                            (Channel[ch].Top + Channel[ch].Bottom) div 2,
                            Channel[ch].ADCName ) ;

                 Main.SetCopyMenu( True, True ) ;
                 Main.mnPrint.Enabled := True ;
                 State := UpdateCursorReadout ;
                 end ;

             { *** UPDATE CURSOR READOUT *** }
             UpdateCursorReadout : begin
                 { Time }
                 Channel[0].CursorTime := (ViewBuf^.x[Channel[0].Cursor1]
                                           - ViewBuf^.x[Channel[0].Cursor0]) ;

                 sgCursor.Cells[2,0] := Format( '%.3g %s',
                                   [Channel[0].CursorTime*Settings.TScale,
                                    Settings.TUnits] ) ;

                 for ch := 0 to CdrFH.NumChannels-1 do
                     if Channel[ch].InUse then begin
                        Channel[ch].CursorTime := Channel[0].CursorTime ;
                        i := Channel[0].Cursor1 ;
                        Row := (2*ch)+1 ;
                        yMin := (ViewBuf^.yMin[ch,i] - Channel[ch].ADCZero)
                             * Channel[ch].ADCScale ;
                        sgCursor.Cells[2,Row] := Format( '%.3g %s',
                                                [yMin,Channel[ch].ADCUnits] ) ;
                        yMax := (ViewBuf^.yMax[ch,i] - Channel[ch].ADCZero)
                             * Channel[ch].ADCScale ;
                        sgCursor.Cells[2,Row+1] := Format( '%.3g %s',
                                                [yMax,Channel[ch].ADCUnits] ) ;
                        end ;
                 State := Idle ;
                 end;


             end ;

        TimerBusy := False ;
        end ;
     end;


procedure TViewCDRFrm.CompressSignal ;
{ ---------------------------------------------------------
  Create a compressed extraction of a block of sampled data
  ---------------------------------------------------------}
var
   Done : Boolean ;
   x,xStep : single ;
   i,FilePointer,nRead,BlockSize : LongInt ;
   y,ch,ChCounter,nBlock,n : Integer ;
   yMin : Array[0..ChannelLimit] of Integer ;
   yMax : Array[0..ChannelLimit] of Integer ;
begin
     { Starting time }
     x := Channel[0].xMin ;

     { Calculate size of min./max. extraction block }
     Blocksize := Trunc( ViewBuf^.tDisplay/ (CdrFH.dt*600. ) )*CdrFH.NumChannels ;
     BlockSize := MaxInt( [BlockSize,CdrFH.NumChannels] ) ;
     xStep := ( BlockSize div CdrFH.NumChannels ) * CdrFH.dt ;

     { Set data file to start of block to be displayed }
     CdrFH.FilePointer := sbDisplay.position*ScrollBarToBlockScale ;
     CdrFH.FilePointer := (CdrFH.FilePointer*CdrFH.NumBytesPerBlock)
                          + CdrFH.NumBytesInHeader ;
     CdrFH.FilePointer := FileSeek( CdrFH.FileHandle, CdrFH.FilePointer, 0 ) ;

     { Initialise min./max. holders }
     for ch := 0 to CdrFH.NumChannels-1 do begin
         yMax[ch] := MinADCValue ;
         yMin[ch] := MaxADCValue ;
         end ;

     { Scan through selected block of samples, calculating and storing
       min. and max. values within each block of size BlockSize }

     Done := False ;
     ViewBuf^.nPoints := 0 ;
     nBlock := 0 ;
     ChCounter := 0 ;
     i := CdrFH.NumSamplesPerBlock ;

     while (not Done) do begin

           { Read next block of data from file when required }
           if i >= CdrFH.NumSamplesPerBlock then begin
              if FileRead(CdrFH.FileHandle,ADC^,CdrFH.NumBytesPerBlock)
                 < CdrFH.NumBytesPerBlock then Done := True ;
              i := 0 ;
              end ;

           { Get sample value }
           y := ADC^[i] ;
           { Determine which channel it is from }
           Ch := Channel[ChCounter].ChannelOffset ;
           { Find min./max. }
           If y <= YMin[Ch] Then YMin[Ch] := y ;
           If y >= YMax[Ch] Then YMax[Ch] := y ;
           { Increment counters & pointers }
           Inc(i) ;
           Inc(ChCounter) ;
           if ChCounter >= CdrFH.NumChannels then ChCounter := 0 ;
           nBlock := nBlock + 1 ;

           { Update compressed signal buffer, when full a block is in }

           if nBlock >= Blocksize Then begin
              x := x + xStep ;
              if x >= Channel[0].xMax then Done := True ;

              for ch := 0 to CdrFH.NumChannels-1 do begin

                  { Keep traces within limits of their part of display area }
                  if yMin[ch] < Trunc(Channel[ch].yMin) then yMin[ch] :=
                                                       Trunc(Channel[ch].yMin) ;
                  if yMax[ch] < Trunc(Channel[ch].yMin) then yMax[ch] :=
                                                       Trunc(Channel[ch].yMin) ;
                  if yMin[ch] > Trunc(Channel[ch].yMax) then yMin[ch] :=
                                                        Trunc(Channel[ch].yMax) ;
                  if yMax[ch] > Trunc(Channel[ch].yMax) then yMax[ch] :=
                                                        Trunc(Channel[ch].yMax) ;

                  { Store Min/Max. values in display buffer }
                  ViewBuf^.YMin[ch,ViewBuf^.nPoints] := YMin[ch] ;
                  ViewBuf^.YMax[ch,ViewBuf^.nPoints] := YMax[ch] ;

                  yMax[ch] := MinADCValue ;
                  yMin[ch] := MaxADCValue ;
                  end ;
              ViewBuf^.x[ViewBuf^.nPoints] := x ;
              ViewBuf^.nPoints := ViewBuf^.nPoints + 1 ;
              nBlock := 0 ;
              end ;
           end ;

     { Update min/max limits in compressed viewing buffer }
     ViewBuf^.xMin := Channel[0].xMin ;
     ViewBuf^.xMax := Channel[0].xMax ;

     Channel[0].Cursor0 := 0 ;
     Channel[0].Cursor1 := ViewBuf^.nPoints div 2 ;

     end ;


procedure TViewCDRFrm.FormClose(Sender: TObject; var Action: TCloseAction);
{ -------------------------
  Close and dispose of form
  -------------------------}
begin
     HeapBuffers( Deallocate ) ;
     Main.ViewCDRChildExists := False ;
    { Main.SetCopyMenu( False, False ) ; { Disable copy menu options}
     Main.mnPrint.Enabled := False ;
     Action := caFree ;
     end;


procedure TViewCDRFrm.FormResize(Sender: TObject);
{ --------------------------------------------
  Adjust size of display when form is resized
  --------------------------------------------}
var
   MinHeight : Integer ;
begin
    { sbDisplay.Top := CursorGrp.Top + CursorGrp.Height - sbDisplay.Height ;}
    MinHeight := CursorGrp.Top + CursorGrp.Height + lbTmin.Height +
                 sbDisplay.height ;

     if ClientHeight < MinHeight then Clientheight := MinHeight ;
     sbDisplay.Top := ClientHeight - sbDisplay.height - lbTMin.Height - 10 ;
     pbDisplay.Height := sbDisplay.Top - pbDisplay.Top ;
     lbTmin.Top := sbDisplay.Top + sbDisplay.Height + 2 ;
     lbTMax.Top := lbTmin.Top ;

     lbTMin.Left := pbDisplay.Left ;
     pbDisplay.Width := ClientWidth - pbDisplay.Left - 5 ;
     lbTMax.Left := pbDisplay.Left + pbDisplay.Width - lbTMax.Width ;
     sbDisplay.Width := pbDisplay.Width ;

     lbCursor0.Top := lbTmin.Top ;
     lbCursor1.Top := lbTmin.Top ;

     State := PlotRecord ;
     end;


procedure TViewCDRFrm.sbDisplayChange(Sender: TObject);
{ ---------------------------------------------
  Redisplay new block of data when slider moved
  ---------------------------------------------}
begin
     Channel[0].xMin := sbDisplay.position*ScrollBarToBlockScale ;
     Channel[0].xMin := (Channel[0].xMin*CdrFH.NumSamplesPerBlock*CdrFH.dt) /
                         CdrFH.NumChannels ;
     Channel[0].xMax := Channel[0].xMin + ViewBuf^.tDisplay ;
     State := PlotRecord ;
     end;


procedure TViewCDRFrm.InitializeDisplay( PB : TPaintBox ) ;
{ ----------------------------
  Initialise a display window
  ---------------------------}
var
   Height,ch,cTop,NumInUse : Integer ;
begin

     { Set trace colour }

     PB.canvas.pen.color := Channel[0].color ;

     { Determine number of channels in use and the height
       available for each channel }

     NumInUse := 0 ;
     for ch := 0 to CdrFH.NumChannels-1 do
         if Channel[ch].InUse then NumInUse := NumInUse + 1 ;
     Height := PB.Height div MaxInt( [NumInUse,1] ) ;

     { Define display area for each channel in use }

     cTop := 0 ;
     for ch := 0 to CdrFH.NumChannels-1 do begin
          if Channel[ch].InUse then begin
             Channel[ch].Left := 0 ;
             Channel[ch].Right := PB.width ;
             Channel[ch].Top := cTop ;
             Channel[ch].Bottom := Channel[ch].Top + Height ;
             Channel[ch].xMin := Channel[0].xMin ;
             Channel[ch].xMax := Channel[0].xMax ;
             Channel[ch].xScale := (Channel[ch].Right - Channel[ch].Left) /
                                (Channel[ch].xMax - Channel[ch].xMin ) ;
             Channel[ch].yScale := (Channel[ch].Bottom - Channel[ch].Top) /
                                (Channel[ch].yMax - Channel[ch].yMin ) ;
             cTop := cTop + Height ;
             end ;
          end ;

     lbTMin.caption := Format( '%5.4g %s', [Channel[0].xMin*Settings.TScale,
                                               Settings.TUnits] ) ;
     lbTMax.caption := Format( '%5.4g %s', [Channel[0].xMax*Settings.TScale,
                                               Settings.TUnits] ) ;

     { Display Channel Name(s) }

     for ch := 0 to CdrFH.NumChannels-1 do
        if Channel[ch].InUse then
           PB.Canvas.TextOut( Channel[ch].Left,
           (Channel[ch].Top + Channel[ch].Bottom) div 2,' ' + Channel[ch].ADCName ) ;

     end ;


procedure TViewCDRFrm.pbDisplayDblClick(Sender: TObject);
{ -------------------------------
  Activate Zoom In/Out dialog box
  -------------------------------}
begin
     Timer.Enabled := False ;
     ZoomCDRFrm.ShowModal ;
     sbDisplay.position := Trunc( (Channel[0].xMin*CdrFH.NumChannels)/
                         (CdrFH.NumSamplesPerBlock*CdrFH.dt*ScrollBarToBlockScale) );
     Timer.Enabled := True ;
     State := PlotRecord ;
     MoveCursor := False ;
     end;


procedure TViewCDRFrm.pbDisplayMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
{ ------------------------
  Mouse down event handler
  ------------------------}
var
   ch : Integer ;
begin
     if State = Idle then begin
        MoveCursor := True ;
        { Find the channel that the mouse is over }
        for ch := 0 to CdrFH.NumChannels-1 do
            if (Channel[ch].Bottom >= Y) and (Y >= Channel[ch].top) then
               ZoomBuf^.ChannelSelected := ch ;
        end ;

     end;

procedure TViewCDRFrm.DrawHorizontalCursor( Const Chan : TChannel ;
                                            Level : Integer ) ;
{ ----------------------------------------------------
  Draw dotted horizontal cursor at ADC level 'Level'
  in display area defined by  record data channel 'Chan'
  ----------------------------------------------------}
var
   yPix,xPix,TicSize : Integer ;
   OldColor : TColor ;
   OldStyle : TPenStyle ;
   OldMode : TPenMode ;
begin
     with pbDisplay.canvas do begin
          OldColor := pen.color ;
          OldStyle := pen.Style ;
          OldMode := pen.mode ;
          pen.mode := pmXor ;
          pen.color := clRed ;
          end ;

     yPix := Trunc( Chan.Bottom - (Level - Chan.yMin)*Chan.yScale ) ;
     pbDisplay.canvas.polyline([Point(Chan.Left,yPix),Point(Chan.Right,yPix)]);

     with pbDisplay.canvas do begin
          pen.style := OldStyle ;
          pen.color := OldColor ;
          pen.mode := OldMode ;
          end ;
     { Request that the cursor readout memo box be updated }
     State := UpdateCursorReadout ;
    end ;


procedure TViewCDRFrm.pbDisplayMouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);
{ ------------------------------------------------
  Operations when mouse is moved over display area
  ------------------------------------------------}
Const
     Margin = 4 ;
var
   X_Cursor,Y_Cursor,X_Zero,ch,OldIndex : Integer ;
   xScale : single ;
begin
     if ViewBuf^.nPoints > 0 then begin
        { Find locations of measurement and zero time cursors }
        xScale := pbDisplay.Width / ViewBuf^.nPoints ;
        X_Cursor := Trunc( Channel[0].Cursor1*xScale) + Channel[0].Left  ;
        X_Zero := Trunc( Channel[0].Cursor0*xScale) + Channel[0].Left  ;

        if not MoveCursor then begin

          { If not in the cursor move mode, check to see if the mouse
            is over any of the cursors and change its icon accordingly }

          CursorState := NoCursor ;
          pbDisplay.Cursor := crDefault ;

          if Abs(X - X_Cursor) < Margin then begin
                CursorState := Cursor1 ;
                pbDisplay.Cursor := crSizeWE ;
                end
          else if ( Abs(X - X_Zero) < Margin ) then begin
                CursorState := Cursor0 ;
                pbDisplay.Cursor := crSizeWE ;
                end
          else begin
                for ch := 0 to CdrFH.NumChannels-1 do begin
                    Y_Cursor := Trunc( Channel[ch].Bottom -
                                ((Channel[ch].ADCZero - Channel[ch].yMin) *
                                Channel[ch].yScale) ) ;
                    if Abs(Y-Y_Cursor) < Margin then begin
                       CurCh := ch ;
                       CursorState := ZeroLevelCursor ;
                       pbDisplay.Cursor := crSizeNS ;
                       end ;
                    end ;
                end ;
          end
        else begin

          { If in Move Cursor mode, move selected cursor to new position }

          case CursorState of
          Cursor1 : begin
              CursorChannel.color := clRed ;
              VerticalCursorScale( X,Channel[0].Cursor1,OldIndex,CursorChannel);
              MoveVerticalCursor( Channel[0].Cursor1, OldIndex,
                                  CursorChannel, lbCursor1 ) ;
              end ;
          Cursor0 : begin
              CursorChannel.color := clPurple ;
              VerticalCursorScale( X,Channel[0].Cursor0,OldIndex,CursorChannel);
              MoveVerticalCursor( Channel[0].Cursor0, OldIndex,
                                  CursorChannel, lbCursor0 ) ;
              end ;

          ZeroLevelCursor : Begin

              { Remove existing cursor }

              DrawHorizontalCursor( Channel[CurCh],Channel[Curch].ADCZero ) ;

              Channel[CurCh].ADCZero := Trunc( ((Channel[CurCh].Bottom - Y) /
                                               Channel[CurCh].yScale)
                                               + Channel[CurCh].yMin ) ;

              { Keep cursor within display limits }

              if Channel[CurCh].ADCZero < Channel[CurCh].yMin then
                 Channel[CurCh].ADCZero := Trunc(Channel[CurCh].yMin) ;
              if Channel[CurCh].ADCZero > Channel[CurCh].yMax then
                 Channel[CurCh].ADCZero := Trunc(Channel[CurCh].yMax) ;

              { Draw new cursor }

              DrawHorizontalCursor( Channel[CurCh], Channel[CurCh].ADCZero ) ;
              end ;

          else ;
          end ;
          State := UpdateCursorReadOut ;
          end ;
        end ;
      end ;


procedure TViewCDRFrm.pbDisplayMouseUp(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
{ -----------------------------------------------------
  Operations when mouse button is released over display
  -----------------------------------------------------}
begin
     if State = Idle then begin
        if (Button = mbRight) and (CursorState = ZeroLevelCursor) then begin
           CDRZeroFrm.ChSel := CurCh ;
           CDRZeroFrm.ShowModal ;
           State := PlotRecord ;
           end ;
        end ;
     MoveCursor := False ;
     end;


procedure TViewCDRFrm.DrawVerticalCursor( Index : Integer ;
                               const Chan : TChannel ;
                               const Lab : TLabel ) ;
{ --------------------------------------------------------
  Draw/Remove a vertical cursor on the record display area
  --------------------------------------------------------}
var
   Diam,xPos : Integer ;
   x : single ;
   OldColor : TColor ;
   OldStyle : TPenStyle ;
   OldMode : TPenMode ;
begin

     with pbDisplay.canvas do begin
          OldColor := pen.color ;
          OldStyle := pen.Style ;
          OldMode := pen.mode ;
          pen.mode := pmXor ;
          pen.color := Chan.color ;
          end ;

     { If cursor on screen, exclusive OR the line to remove an existing
       cursor / add a new one }
     if (Index >= 0) and (Index<ViewBuf^.nPoints) then begin
        x := ViewBuf^.x[Index] ;
        xPos := Trunc((x - Chan.xMin)*Chan.xScale - Chan.Left) ;
        pbDisplay.canvas.polyline([Point(xPos,Chan.Top),Point(xPos,Chan.Bottom)] ) ;

        if (xPos >= Chan.Left) and (xPos <= Chan.Right) then begin
              Lab.Left := pbDisplay.Left + xPos ;
              Lab.Top := sbDisplay.Top + sbDisplay.Height + 2 ;
              Lab.visible := True ;
              end
        else Lab.visible := False ;

        end
     else
         Lab.visible := False ;

     with pbDisplay.canvas do begin
          pen.style := OldStyle ;
          pen.color := OldColor ;
          pen.mode := OldMode ;
          end ;
     end ;


procedure TViewCDRFrm.VerticalCursorScale( MouseXPos : Integer ;
                               var CurrentIndex,OldIndex : Integer ;
                               const Chan : TChannel ) ;
{ -----------------------------------------------------------
  Get the index into the A/D channel from the mouse X position
  -----------------------------------------------------------}
var
   xScale : single ;
begin
     OldIndex := CurrentIndex ;
     xScale := ViewBuf^.nPoints / pbDisplay.Width ;
     CurrentIndex := Trunc( (MouseXPos - Chan.Left)*xScale  ) ;
     end ;


procedure TViewCDRFrm.MoveVerticalCursor( var NewPosition, OldPosition : Integer ;
                                          const CursorChannel : TChannel ;
                                          var CursorLabel : TLabel ) ;
{ ---------------------------------------
  Move vertical cursor on signal display
  --------------------------------------}
begin
     DrawVerticalCursor( OldPosition, CursorChannel, CursorLabel ) ;
     if NewPosition < 0 then NewPosition := 0 ;
     if NewPosition >= ViewBuf^.nPoints then NewPosition := ViewBuf^.nPoints-1 ;
     DrawVerticalCursor( NewPosition, CursorChannel, CursorLabel ) ;
     end ;


procedure TViewCDRFrm.HorizontalCursorScale( MouseYPos : Integer ;
                                 var CurrentLevel,OldLevel : LongInt ;
                                 const Channel : TChannel ) ;
{ -----------------------------------------------------------
  Get the A/D signal level from the mouse Y position
  -----------------------------------------------------------}
begin
     OldLevel := CurrentLevel ;
     CurrentLevel := Trunc((Channel.Bottom - MouseYPos)/Channel.yScale + Channel.yMin ) ;
     end ;


function TViewCDRFrm.OverHorizontalCursor( Y : Integer ;
                                           const Channel : TChannel ) : Boolean ;
const
     Margin = 4 ;
var
   YOfCursor : Integer ;
begin
     YofCursor := Trunc( Channel.Bottom -
                         ((Channel.ADCZero - Channel.yMin)*Channel.yScale) ) ;
     if Abs(Y - YofCursor) < Margin then OverHorizontalCursor := True
                                    else OverHorizontalCursor := False ;
     end ;


procedure TViewCDRFrm.MoveHorizontalCursor( var pb : TPaintbox ;
                                var NewLevel,OldLevel : LongInt ;
                                const Channel : TChannel ) ;
{ ------------------------------------------
  Move a horizontal cursor to a new position
  ------------------------------------------}
begin
     { Remove existing cursor }
     DrawHorizontalCursor( Channel, OldLevel ) ;

     { Keep cursor within display limits }
     if NewLevel < Channel.yMin then NewLevel := Trunc(Channel.yMin) ;
     if NewLevel > Channel.yMax then NewLevel := Trunc(Channel.yMax) ;

     { Draw new cursor }
     DrawHorizontalCursor( Channel, NewLevel ) ;
     end ;


procedure TViewCDRFrm.CopyRecordsImageToClipboard ;
{ ------------------------------------------------------
  Plot currently displayed record as bitmap in clipboard
  ------------------------------------------------------}
var
    PrChan : array[0..ChannelLimit] of TChannel ;
    Page : TPlot ;
    TimeBarValue : single  ;
    OK : boolean ;
    ch, i, n, Num : Integer ;
    ChanHeight,ChanTop,ChLast : Integer ;
    x,y,dx : single ;
    BMap : TBitmap ;
begin

     { If no previous settings ... define an usable set }
     if Settings.TimeBarValue = -1. then
        Settings.TimeBarValue := (Channel[0].Xmax - Channel[0].xMin)*0.1 ;

     for ch := 0 to CdrFH.NumChannels-1 do if Settings.BarValue[ch] = -1. then
         Settings.BarValue[ch] := (Channel[ch].yMax - Channel[ch].yMin)
                                           * Channel[ch].ADCScale * 0.1 ;

     { Get calibration bar sizes and other options from user }
     CopyRecDlg.ShowModal ;

     if CopyRecDlg.ModalResult = mrOK then begin

        Screen.Cursor := crHourglass ;

        BMap := TBitmap.Create ;
        BMap.Height := Settings.BitmapHeight ;
        BMap.Width := Settings.BitmapWidth ;
        BMap.Canvas.font.name := Settings.Plot.FontName ;
        BMap.Canvas.font.size := Settings.Plot.FontSize ;
        BMap.Canvas.Pen.Width := Settings.Plot.LineThickness ;
        Page.Left := BMap.Canvas.TextWidth('XXXXXXXXXX') ;
        Page.Right := BMap.Width - Page.Left ;
        Page.Top := BMap.Canvas.TextHeight('X') ;
        Page.Bottom :=BMap.Height - 5*Page.Top ;

        { Make a copy of channel scaling information and find out
          how many channels are on display }
        Num := 0 ;
        for ch := 0 to CdrFH.NumChannels-1 do begin
            { Copy Printer channel from Display channel record }
            PrChan[ch] := Channel[ch] ;
            { Use black if colour printing is not required }
            if not Settings.Plot.UseColor then PrChan[ch].Color := clBlack ;
            if PrChan[ch].InUse then Inc(Num) ;
            end ;
        ChanHeight := (Page.Bottom - Page.Top) div MaxInt( [Num,1] ) ;

        { Set up channel display areas on printed page }
        ChanTop := Page.Top ;
        for ch := 0 to CdrFH.NumChannels-1 do begin
            PrChan[ch].Left := Page.Left ;
            PrChan[ch].Right := Page.Right ;
            PrChan[ch].Top := ChanTop ;
            PrChan[ch].Bottom := PrChan[ch].Top + ChanHeight ;
            PrChan[ch].xScale := (PrChan[ch].Right - PrChan[ch].Left) /
                                 (PrChan[ch].xMax  - PrChan[ch].xMin ) ;
            PrChan[ch].yScale := (PrChan[ch].Bottom - PrChan[ch].Top) /
                                 (PrChan[ch].yMax   - PrChan[ch].yMin ) ;
            if PrChan[ch].InUse then ChanTop := ChanTop + ChanHeight ;
            end ;

        { Display Channel Name(s) }
        if Settings.ShowLabels then begin
             for ch := 0 to CdrFH.NumChannels-1 do if PrChan[ch].InUse then
                 BMap.Canvas.TextOut( PrChan[ch].Left -
                               printer.canvas.textwidth(prChan[ch].ADCName+' '),
                               (PrChan[ch].Top + PrChan[ch].Bottom) div 2,
                               PrChan[ch].ADCName ) ;
             end ;

        { Plot each record channel in turn }
        for ch := 0 to CdrFH.NumChannels-1 do if PrChan[ch].InUse then begin
            ChLast := ch ;
            n := 0 ;
            for i := 0 to ViewBuf^.nPoints-1 do begin
                xy^[n].y := PrChan[ch].Bottom - Trunc(
                            PrChan[ch].yScale*(ViewBuf^.yMin[ch,i]
                            - PrChan[ch].yMin));
                xy^[n].x := Trunc(PrChan[ch].xScale*(ViewBuf^.x[i] -
                                  PrChan[ch].xMin) + PrChan[ch].Left) ;
                Inc(n) ;
                if ViewBuf^.yMax[ch,i] <> ViewBuf^.yMin[ch,i] then begin
                   xy^[n].y := PrChan[ch].Bottom - Trunc(
                               PrChan[ch].yScale*(ViewBuf^.yMax[ch,i] -
                               PrChan[ch].yMin));
                   xy^[n].x := xy^[n-1].x ;
                   Inc(n) ;
                   end ;
                end ;

            OK := Polyline( BMap.Canvas.Handle, xy^, n ) ;
            end ;

        { Print zero level cursor(s) }
        If Settings.ShowZeroLevels then begin
           for ch := 0 to CdrFH.NumChannels-1 do if PrChan[ch].InUse then begin
                 BMap.Canvas.pen.color := PrChan[ch].color ;
                 PrintHorizontalCursor(BMap.Canvas,PrChan[ch],PrChan[ch].ADCZero) ;
                 end ;
           end ;

        { Horizontal (time) calibration bar }
        BMap.Canvas.pen.color := clBlack ;
        HorizontalBar(BMap.Canvas,PrChan[ChLast],Settings.ShowLabels,
                      Settings.TimeBarValue,1.0,Settings.TScale,Settings.TUnits) ;

        { Vertical calibration bars }
        for ch := 0 to CdrFH.NumChannels-1 do if PrChan[ch].InUse then
              VerticalBar(BMap.Canvas,PrChan[ch],Settings.ShowLabels,
                          Settings.BarValue[ch],PrChan[ch].ADCUnits) ;

        { Close bitmap object }
        Clipboard.Assign(BMap) ;
        BMap.Free ;
        end ;

     Screen.Cursor := crDefault ;
     end;


procedure TViewCDRFrm.PrintRecord ;
{ ------------------------------------------
  Plot currently displayed record on printer
  ------------------------------------------}
var
    PrChan : array[0..ChannelLimit] of TChannel ;
    Page : TPlot ;
    TimeBarValue : single  ;
    OK : boolean ;
    ch, i, n, Num  : Integer ;
    ChanHeight,ChanTop,ChLast : Integer ;
    x,y,dx : single ;
begin

     { If no previous settings ... define an usable set }
     if Settings.TimeBarValue = -1. then
        Settings.TimeBarValue := (Channel[0].Xmax - Channel[0].xMin)*0.1 ;
     for ch := 0 to CdrFH.NumChannels-1 do if Settings.BarValue[ch] = -1. then
         Settings.BarValue[ch] := (Channel[ch].yMax - Channel[ch].yMin)
                                           * Channel[ch].ADCScale * 0.1 ;

     { Get calibration bar sizes and other options from user }
     PrintRecFrm.ShowModal ;
     if PrintRecFrm.ModalResult = mrOK then begin

        Screen.Cursor := crHourglass ;

        { *** Print file name and ident line *** }

        Printer.Canvas.font.name := Settings.Plot.FontName ;
        Printer.BeginDoc ;
        Printer.Canvas.Pen.Color := clBlack ;

        { Print standard header and footer text }
        PrintHeaderAndFooter ;

        { *** Print signal channels *** }

        Printer.Canvas.font.size := PrinterPointsToPixels(Settings.Plot.FontSize) ;
        Printer.Canvas.Pen.Width := PrinterPointsToPixels(Settings.Plot.LineThickness) ;

        { Set plot size on page size }
        Page.Left := PrinterCmToPixels('H',Settings.Plot.LeftMargin) ;
        Page.Right := printer.pagewidth
                      - PrinterCmToPixels('H',Settings.Plot.RightMargin) ;
        Page.Top := PrinterCmToPixels('V',Settings.Plot.TopMargin) ;
        Page.Bottom := Printer.pageheight -
                       PrinterCmToPixels('V',Settings.Plot.BottomMargin) ; ;

        { Make a copy of channel scaling information and find out
          how many channels are on display }
        Num := 0 ;
        for ch := 0 to CdrFH.NumChannels-1 do begin
            { Copy Printer channel from Display channel record }
            PrChan[ch] := Channel[ch] ;
            { Use black if colour printing is not required }
            if not Settings.Plot.UseColor then PrChan[ch].Color := clBlack ;
            if PrChan[ch].InUse then Inc(Num) ;
            end ;
        ChanHeight := (Page.Bottom - Page.Top) div MaxInt( [Num,1] ) ;

        { Set up channel display areas on printed page }
        ChanTop := Page.Top ;
        for ch := 0 to CdrFH.NumChannels-1 do begin
            PrChan[ch].Left := Page.Left ;
            PrChan[ch].Right := Page.Right ;
            PrChan[ch].Top := ChanTop ;
            PrChan[ch].Bottom := PrChan[ch].Top + ChanHeight ;
            PrChan[ch].xScale := (PrChan[ch].Right - PrChan[ch].Left) /
                                 (PrChan[ch].xMax  - PrChan[ch].xMin ) ;
            PrChan[ch].yScale := (PrChan[ch].Bottom - PrChan[ch].Top) /
                                 (PrChan[ch].yMax   - PrChan[ch].yMin ) ;
            if PrChan[ch].InUse then ChanTop := ChanTop + ChanHeight ;
            end ;

        { Display Channel Name(s) }
        if Settings.ShowLabels then begin
             for ch := 0 to CdrFH.NumChannels-1 do if PrChan[ch].InUse then
                 Printer.Canvas.TextOut( PrChan[ch].Left -
                               printer.canvas.textwidth(prChan[ch].ADCName+' '),
                               (PrChan[ch].Top + PrChan[ch].Bottom) div 2,
                               PrChan[ch].ADCName ) ;
             end ;

        { Plot each record channel in turn }
        for ch := 0 to CdrFH.NumChannels-1 do if PrChan[ch].InUse then begin
              ChLast := ch ;
              n := 0 ;
              for i := 0 to ViewBuf^.nPoints-1 do begin
                  xy^[n].y := PrChan[ch].Bottom - Trunc(
                              PrChan[ch].yScale*(ViewBuf^.yMin[ch,i]
                              - PrChan[ch].yMin));
                  xy^[n].x := Trunc(PrChan[ch].xScale*(ViewBuf^.x[i] -
                                    PrChan[ch].xMin) + PrChan[ch].Left) ;
                  Inc(n) ;
                  if ViewBuf^.yMax[ch,i] <> ViewBuf^.yMin[ch,i] then begin
                     xy^[n].y := PrChan[ch].Bottom - Trunc(
                                 PrChan[ch].yScale*(ViewBuf^.yMax[ch,i] -
                                 PrChan[ch].yMin));
                     xy^[n].x := xy^[n-1].x ;
                     Inc(n) ;
                     end ;
                  end ;

              OK := Polyline( Printer.Canvas.Handle, xy^, n ) ;
              end ;

        { Print zero level cursor(s) }
        If Settings.ShowZeroLevels then begin
           for ch := 0 to CdrFH.NumChannels-1 do if PrChan[ch].InUse then begin
               Printer.Canvas.pen.color := PrChan[ch].color ;
               PrintHorizontalCursor(Printer.Canvas,PrChan[ch],PrChan[ch].ADCZero) ;
               end ;
           end ;

        { Horizontal (time) calibration bar }
        Printer.Canvas.pen.color := clBlack ;
        HorizontalBar(Printer.Canvas,PrChan[ChLast],Settings.ShowLabels,
                      Settings.TimeBarValue,1.0,Settings.TScale,Settings.TUnits) ;

          { Vertical calibration bars }
          for ch := 0 to CdrFH.NumChannels-1 do if PrChan[ch].InUse then
              VerticalBar(Printer.Canvas,PrChan[ch],Settings.ShowLabels,
                          Settings.BarValue[ch],PrChan[ch].ADCUnits) ;

          { Close down printer }
          Printer.EndDoc ;

          Screen.Cursor := crDefault ;
          end ;
     end;


procedure TViewCDRFrm.CopyDataToClipBoard ;
{ --------------------------------------------------
  Copy the currently displayed record to the clipboard
    25/6/98 Clipboard buffer limit reduced to 31000
  --------------------------------------------------}
const
     BufSize = 31000 ;
     NumBytesPerNumber = 12 ;
var
   i,j,iSkip,x,ch,n,nc,NumBytesNeeded : LongInt ;
   Line : String ;
   CopyBuf0,Line0 : PChar ;
   OK : Boolean ;
   ClipRec : ^TWCPClipboardRecord ;

begin
     try
        OK := OpenClipboard( Handle ) ;
        OK := EmptyClipBoard ;

        { Allocate buffer to hold signal }
        CopyBuf0 := StrAlloc( BufSize ) ;
        StrPCopy( CopyBuf0, '' ) ;
        Line0 := StrAlloc(  256 ) ;

        { Determine sample skip factor to ensure that the compete record
          fits into the buffer }
        n := ViewBuf^.nPoints*2 ;
        nc := 0 ;
        for ch := 0 to CdrFH.NumChannels-1 do if Channel[ch].InUse then Inc(nc) ;
        NumBytesNeeded := n*(nc+1)*NumBytesPerNumber ;
        iSkip := MaxInt( [(NumBytesNeeded+BufSize-1) div BufSize,1]) ;

        i := 0 ;
        screen.cursor := crHourglass ;
        while  i < ViewBuf^.nPoints do begin
             { Create a line of <tab> separated ASCII data values terminated by <cr> <lf>
               Time <tab> Ch.0.Value <tab> Ch.1.Value .... <cr> <lf> }
             { Write line of minimum values }
             Line := format( '%5.4g', [ViewBuf^.x[i]*Settings.TScale] ) ;
             for ch := 0 to CdrFH.NumChannels-1 do if Channel[ch].InUse then begin
                 Line := Line + chr(9) + Format('%10.4g',
                                         [(ViewBuf^.yMin[ch,i] - Channel[ch].ADCZero)
                                               * Channel[ch].ADCScale] ) ;
                 end ;
             Line := Line + chr(13) + chr(10) ;
             StrPCopy( Line0, Line ) ;
             CopyBuf0 := StrCat( CopyBuf0, Line0 ) ;
             { Write line of maximum values }
             Line := format( '%5.4g', [ViewBuf^.x[i]*Settings.TScale] ) ;
             for ch := 0 to CdrFH.NumChannels-1 do if Channel[ch].InUse then begin
                 Line := Line + chr(9) + Format('%10.4g',
                                         [(ViewBuf^.yMax[ch,i] - Channel[ch].ADCZero)
                                               * Channel[ch].ADCScale] ) ;
                 end ;
             Line := Line + chr(13) + chr(10) ;
             StrPCopy( Line0, Line ) ;
             CopyBuf0 := StrCat( CopyBuf0, Line0 ) ;
             i := i + iSkip ;
             end ;

        { Copy text accumulated in copy buffer to clipboard }
         ClipBoard.SetTextBuf( CopyBuf0 ) ;

     finally
         screen.cursor := crDefault ;
         { Dispose of buffers }
         StrDispose( Line0 ) ;
         StrDispose( CopyBuf0 ) ;
         OK := CloseClipboard ;
         end ;
     end ;

procedure TViewCDRFrm.pbDisplayPaint(Sender: TObject);
begin
     State := PlotRecord ;
     end;


procedure TViewCDRFrm.edIdentKeyPress(Sender: TObject; var Key: Char);
{ -------------------------------------------
  Update identification string in file header
  -------------------------------------------}
begin
     if key = chr(13) then begin
        CdrFH.IdentLine := edIdent.Text ;
        SaveCDRHeader( CdrFH ) ;
        WriteToLogFile( edIdent.text ) ;
        end ;
     end;

end.
