unit Simnoise;
{ ============================================================
  WinCDR - SimNoise.pas -Ion channel noise simulation
  (c) J. Dempster, University of Strathclyde 1998
  ============================================================
  29/6/98 Close button disabled during simulation run to avoid
          GPF due to closing form while Start button method running}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, ExtCtrls,
  global, shared, maths, fileio ;

type
  TSimNoiseFrm = class(TForm)
    GroupBox1: TGroupBox;
    bStart: TButton;
    bAbort: TButton;
    bClose: TButton;
    pbDisplay: TPaintBox;
    GroupBox4: TGroupBox;
    Label2: TLabel;
    Label11: TLabel;
    edBackgroundTime: TEdit;
    edTransientTime: TEdit;
    edSteadyStateTime: TEdit;
    GroupBox2: TGroupBox;
    Label1: TLabel;
    Label3: TLabel;
    edUnitCurrent: TEdit;
    edPOpen: TEdit;
    edTauOpen: TEdit;
    EdNumChannels: TEdit;
    Label4: TLabel;
    lbTMin: TLabel;
    lbTMax: TLabel;
    GroupBox5: TGroupBox;
    Label9: TLabel;
    Label6: TLabel;
    edNoiseRMS: TEdit;
    edLPFilter: TEdit;
    procedure bStartClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure bAbortClick(Sender: TObject);
    procedure bCloseClick(Sender: TObject);
    procedure edBackgroundTimeKeyPress(Sender: TObject; var Key: Char);
    procedure pbDisplayPaint(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
  private
    { Private declarations }
    procedure PlotChannel( xMin, xMax : single ) ;
    procedure InitializeDisplay ;
    procedure HeapBuffers( Operation : THeapBufferOp ) ;
    procedure UpdateEditBoxes ;
  public
    { Public declarations }
  end;

var
  SimNoiseFrm: TSimNoiseFrm;

implementation

{$R *.DFM}

uses mdiform ;

const
     NumSamplesPerBuffer = 512 ;
     ChSim = 0 ;
     MaxChannels = 100 ;
type
    TPointArray = Array[0..2000] of TPoint ;
    TChannelState = (Open,Closed) ;
    TChannelRecord = record
                  Time : Single ;
                  State : TChannelState ;
                  end ;
    TSim = record
         BackgroundTime : single ;
         TransientTime : single ;
         SteadyStateTime : single ;
         UnitCurrent : single ;
         NumChannels : Integer ;
         POpen : single ;
         TauOpen : single ;
         TauClosed : single ;
         NoiseRMS : Single ;
         LPFilter : single ;
         LPFilterFirstCall : Boolean ;
         t : double ;
         EndTime : double ;
         Channel : Array[0..MaxChannels-1] of TChannelRecord ;
         end ;
var
   GaussTemp : single ;
   ADC : ^TIntArray ; { Digitised signal buffer }
   xy : ^TPointArray ;
   Sim : TSim ;
   BuffersAllocated : boolean ;{ Indicates if memory buffers have been allocated }


procedure TSimNoiseFrm.HeapBuffers( Operation : THeapBufferOp ) ;
{ -----------------------------------------------
  Allocate/deallocation dynamic buffers from heap
  -----------------------------------------------}
begin
     case Operation of
          Allocate : begin
             if not BuffersAllocated then begin
                New(ADC) ;
                New(xy) ;
                BuffersAllocated := True ;
                end ;
             end ;
          Deallocate : begin
             if BuffersAllocated then begin
                Dispose(ADC) ;
                Dispose(xy) ;
                BuffersAllocated := False ;
                end ;
             end ;
          end ;
     end ;


procedure TSimNoiseFrm.FormShow(Sender: TObject);
{ --------------------------------------
  Initialisations when form is displayed
  --------------------------------------}
begin
     { Create buffers }
     HeapBuffers( Allocate ) ;

     bStart.Enabled := True ;
     bAbort.Enabled := False ;
     GaussTemp := 1.0 ;

     Channel[ChSim].ADCUnits := 'pA' ;
     Channel[ChSim].ADCName := 'Im' ;
     Channel[ChSim].ADCZero := 0 ;
     Channel[ChSim].yMin := MinADCValue ;
     Channel[ChSim].yMax := MaxADCValue ;
     Channel[ChSim].InUse := True ;

     lbTMin.Visible := False ;
     lbTMax.Visible := False ;

     { Clear display area }
     EraseDisplay( pbDisplay ) ;

     end ;


procedure TSimNoiseFrm.UpdateEditBoxes ;
{ --------------------------------------------------------------------------
  Update all edit boxes and transfer parameters to simulation control record
  --------------------------------------------------------------------------}
begin
     { Background noise period }
     Sim.BackgroundTime := GetFromEditBox(edBackgroundTime,
                                          10.0,0.0,1E6, ' %.1f ', 's' ) ;
     { Transient noise period }
     Sim.TransientTime := GetFromEditBox(edTransientTime,
                                         10.0,0.0,1E6,  ' %.1f ', 's' ) ;
     { Steady-state noise period }
     Sim.SteadyStateTime := GetFromEditBox(edSteadyStateTime,
                                           10.0,0.0,1E6, ' %.1f ', 's' ) ;
     { Standard deviation of quantum amplitude }
     Sim.UnitCurrent := GetFromEditBox(edUnitCurrent,
                                 1.0,-1E4,1E4, ' %.2f', Channel[ChSim].ADCUnits ) ;

     { Mean channel open time }
     Sim.TauOpen := GetFromEditBox(edTauOpen,1.0,0.1,1E6,' %.2f','ms')*MsToSecs ;

     { Channel open probability }
     Sim.POpen := GetFromEditBox(edPOpen,0.5,0.0,1.0,' %.2f','') ;

     { Number of channels }
     Sim.NumChannels := Trunc(GetFromEditBox(edNumChannels,10.,1.0,1000.0,' %.f','')) ;

     { Background noise RMS }
     Sim.NoiseRMS := GetFromEditBox(edNoiseRMS,
                             0., 0., 1E4, ' %.2f', Channel[ChSim].ADCUnits) ;

     { Low-pass filter cut-off frequency }
     Sim.LPFilter := GetFromEditBox(edLPFilter,0.1,0.,1E4, ' %.1f', 'Hz' ) ;

     end ;


procedure TSimNoiseFrm.bStartClick(Sender: TObject);
{ -------------------------
  Create simulated currents
  -------------------------}
var
   TBuffer,t : single ;
   Alpha, Beta, p : single ;
   NumOpenChannels,iZeroLevel,i,j,iCh : Integer ;
   NumBytesToWrite : LongInt ;
   Done,FirstTime : Boolean ;
begin

     bStart.Enabled := False ;
     bAbort.Enabled := True ;
     bClose.Enabled := False ;

     { Get data from edit boxes and put in Sim record }
     UpdateEditBoxes ;

     { Total simulation time }
     Sim.EndTime := Sim.BackgroundTime + Sim.TransientTime + Sim.SteadyStateTime ;

     { Set scaling factor if this is an empty file }
     if CdrFH.NumSamplesInFile = 0 then begin
        cdrFH.NumChannels := 1 ;
        Channel[ChSim].ADCCalibrationFactor := 1. ;
        Channel[ChSim].ADCScale := Abs(Sim.UnitCurrent*Sim.NumChannels*1.25) / MaxADCValue ;
        CdrFH.ADCVoltageRange :=  Channel[ChSim].ADCCalibrationFactor
                                  * ( Channel[ChSim].ADCScale * (MaxADCValue+1) ) ;

        if Sim.UnitCurrent > 0.0 then Channel[ChSim].ADCZero := -1024
                                 else Channel[ChSim].ADCZero := 1024 ;
        end ;

     Channel[ChSim].ChannelOffset := 0 ;

     { Position data file pointer at end of data in file }
     CdrFH.FilePointer := (CdrFH.NumSamplesInFile*2*CdrFH.NumChannels)
                          + CdrFH.NumBytesInHeader ;
     CdrFH.FilePointer := FileSeek(CdrFH.FileHandle,0,2) ;

     { *** Ion channel simulation loop *** }

     Done := False ;
     FirstTime := True ;
     TBuffer := NumSamplesPerBuffer*CdrFH.dt ;
     iZeroLevel := Channel[ChSim].ADCZero ;
     Sim.t := 0.0 ;
     while (not Done) do begin

         { Background noise }
         j := Channel[ChSim].ChannelOffset ;
         for i := 0 to NumSamplesPerBuffer do begin
             ADC^[j] := Trunc( (Sim.NoiseRMS*GaussianRandom(GaussTemp))
                               /Channel[ChSim].ADCScale ) + iZeroLevel ;
             j := j + CdrFH.NumChannels ;
             end ;

         { Ion channel noise }
         if Sim.t >= Sim.BackgroundTime then begin

            { Open channel probability for this time point }
            p := (1.0 - exp( -(Sim.t - Sim.BackgroundTime)/(Sim.TransientTime*0.25)))
                 * Sim.POPen ;

            { Calculate channel mean open and closed dwell times }
            Alpha := 1.0 / Sim.TauOpen ;
            if p > 0.0 then begin
               Beta :=(Alpha*p)/(1.0 - p) ;
               Sim.TauClosed := 1.0 / Beta ;
               end
            else Sim.TauClosed := 1E10 ;

            if FirstTime then begin
                 { Set all channels to closed }
                 for iCh := 0 to Sim.NumChannels-1 do begin
                     Sim.Channel[iCh].State := Closed ;
                     Sim.Channel[iCh].Time := 0.0 ;
                     Sim.Channel[iCh].Time := -Sim.TauClosed*ln(Random) ;
                     end ;
                 FirstTime := False ;
                 end ;

            { Calculate ionic current for each sample point in buffer }
            j := Channel[ChSim].ChannelOffset ;
            for i := 0 to NumSamplesPerBuffer do begin
                NumOPenChannels := 0 ;
                for iCh := 0 to Sim.NumChannels-1 do begin
                    { If at end dwell time in current channel state,
                      flip to other state and obtain a new dwell time }
                    if Sim.Channel[iCh].Time <= 0.0 then begin
                       if Sim.Channel[iCh].State = Open then begin
                          { Change to closed state }
                          Sim.Channel[iCh].Time := -Sim.TauClosed*ln(Random) ;
                          Sim.Channel[iCh].State := Closed ;
                          end
                       else begin
                          { Change to open state }
                          Sim.Channel[iCh].Time := -Sim.TauOpen*ln(Random) ;
                          Sim.Channel[iCh].State := Open ;
                          end ;
                       end
                    else begin
                         { Decrement time in this state }
                         Sim.Channel[iCh].Time := Sim.Channel[iCh].Time - CdrFH.dt ;
                         end ;
                    { If channel is open add it to open count }
                    if Sim.Channel[iCh].State = Open then Inc(NumOpenChannels) ;
                    end ;
                ADC^[j] := ADC^[j] + Trunc( (NumOpenChannels*Sim.UnitCurrent) /
                                             Channel[ChSim].ADCScale ) ;
                j := j + CdrFH.NumChannels ;
                end ;
            end ;

         { Display record }
         PlotChannel( Sim.t, Sim.t + TBuffer ) ;

         { Save data to file }
         NumBytesToWrite := NumSamplesPerBuffer*CdrFH.NumChannels*2 ;
         if FileWrite(CdrFH.FileHandle,ADC^,NumBytesToWrite)
            <> NumBytesToWrite then
            MessageDlg( ' File write file failed',mtWarning, [mbOK], 0 ) ;

         CdrFH.NumSamplesInFile := CdrFH.NumSamplesInFile
                                   + NumSamplesPerBuffer*CdrFH.NumChannels ;

         Sim.t := Sim.t + TBuffer ;
         if Sim.t > Sim.EndTime then Done := True ;
         if bStart.Enabled = True then Done := True ;

         Application.ProcessMessages ;

         end ;

     { Close form if simulation has not been aborted }
     {if not bStart.Enabled then close ;}
     bClose.Enabled := True ;
     bStart.Enabled := True ;
     bAbort.Enabled := False ;

     end;


procedure TSimNoiseFrm.PlotChannel( xMin, xMax : single ) ;
{ -----------------------------
  Plot a simulated noise record
  -----------------------------}
var
   x,dx : single ;
   i,j : Integer ;
   OK : Boolean ;
begin

     { Display simulated record }
     Channel[ChSim].xMin := xMin ;
     Channel[ChSim].xMax := xMax ;
     EraseDisplay(pbDisplay) ;
     InitializeDisplay ;

     dx := Channel[ChSim].xScale*cdrFH.dt ;
     x := Channel[ChSim].Left ;
     j := Channel[ChSim].ChannelOffset ;
     for i := 0 to NumSamplesPerBuffer-1 do begin
         xy^[i].x := Trunc(x) ;
         xy^[i].y := Trunc(Channel[ChSim].Bottom
                     - Channel[ChSim].yScale*(ADC^[j] - Channel[ChSim].yMin));
         xy^[i].y := IntLimitTo(xy^[i].y,Channel[ChSim].Top,Channel[ChSim].Bottom) ;
         j := j + CdrFH.NumChannels ;
         x := x + dx ;
         end ;

     OK := Polyline(pbDisplay.Canvas.Handle,xy^,NumSamplesPerBuffer) ;

     { Draw zero level cursor }
     DrawHorizontalCursor(pbDisplay,Channel[ChSim],Channel[ChSim].ADCZero) ;

     { Display Channel Name }
     pbDisplay.Canvas.TextOut( Channel[ChSim].Left,
                              (Channel[ChSim].Top + Channel[ChSim].Bottom) div 2,
                              ' ' + Channel[ChSim].ADCName ) ;
     end ;


procedure TSimNoiseFrm.InitializeDisplay ;
{ ----------------------------
  Initialise the display window
  ---------------------------}
begin

     { Set trace colour }

     pbDisplay.canvas.pen.color := Channel[ChSim].color ;

     { Define display area for each channel in use }

     Channel[ChSim].Left := 0 ;
     Channel[ChSim].Right := pbDisplay.width ;
     Channel[ChSim].Top := 0 ;
     Channel[ChSim].Bottom := pbDisplay.Height ;
     Channel[ChSim].xScale := (Channel[ChSim].Right - Channel[ChSim].Left) /
                           (Channel[ChSim].xMax - Channel[ChSim].xMin ) ;
     Channel[ChSim].yScale := (Channel[ChSim].Bottom - Channel[ChSim].Top) /
                           (Channel[ChSim].yMax - Channel[ChSim].yMin ) ;

     lbTMin.caption := Format( '%.3g %s', [Channel[ChSim].xMin*Settings.TScale,
                                               Settings.TUnits] ) ;
     lbTMax.caption := Format( '%.3g %s', [Channel[ChSim].xMax*Settings.TScale,
                                               Settings.TUnits] ) ;

     lbTMin.Top := pbDisplay.Top + pbDisplay.Height + 1 ;
     lbTmax.Top := lbTMin.Top ;
     lbTMin.Left := pbDisplay.Left ;
     lbTMax.Left := pbDisplay.Left + pbDisplay.Width - lbTMax.Width ;
     lbTMin.Visible := True ;
     lbTMax.Visible := True ;


     { Display Channel Name(s) }

     pbDisplay.Canvas.TextOut( Channel[ChSim].Left,
           (Channel[ChSim].Top + Channel[ChSim].Bottom) div 2,' ' + Channel[ChSim].ADCName ) ;

     end ;


procedure TSimNoiseFrm.FormCreate(Sender: TObject);
begin
     { Disable "Ion Channel Noise" item in "Simulation" menu }
     Main.mnSimNoise.enabled := false ;
     BuffersAllocated := False ;
     end;


procedure TSimNoiseFrm.FormClose(Sender: TObject;
  var Action: TCloseAction);
{ ------------------------
  Tidy up when form closed
  ------------------------}
begin

     if CdrFH.NumSamplesInFile > 0 then begin
        Main.mnViewCDR.Enabled := True ;
        Main.mnViewCDR.checked := True ;
        Main.Analysis.Enabled := True ;
        end ;

     HeapBuffers( Deallocate ) ;
     { Enable "Synaptic Current" item in "Simulation" menu }
     Main.mnSimNoise.enabled := true ;
     { Display results }
     if CdrFH.NumSamplesInFile > 0 then Main.mnViewCDR.Click ;
     Action := caFree ;
     SaveCDRHeader( CdrFH ) ;
     end;


procedure TSimNoiseFrm.bAbortClick(Sender: TObject);
begin
     bStart.Enabled := True ;
     bAbort.Enabled := False ;
     bClose.Enabled := True ;
     end;


procedure TSimNoiseFrm.bCloseClick(Sender: TObject);
begin
     Close ;
     end;

procedure TSimNoiseFrm.edBackgroundTimeKeyPress(Sender: TObject;
  var Key: Char);
begin
     if key = chr(13) then UpdateEditBoxes ;
     end;

procedure TSimNoiseFrm.pbDisplayPaint(Sender: TObject);
begin
     EraseDisplay( pbDisplay ) ;
     end;

procedure TSimNoiseFrm.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
     { Prevent form being closed if a simulation is running (Close button disabled) }
     if not bClose.Enabled then CanClose := False
                           else CanClose := True ;
     end;

end.
