unit Simmepc;
{ ===============================================================================
  WinCDR ... Miniature postsynaptic current simulation module
  (c) J. Dempster 1996-98 All Rights Reserved
  ===============================================================================
  24/6/98 ... MEPC inter-event interval distribution now corrected
  26/4/98 ... MEPC rising phase now exponential (rather than erf)
  29/6/98 ... Close button disabled during simulation run to avoid
              GPFs caused by closing form while bStart.Click running}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, StdCtrls, Shared, FileIo, Global, maths ;

type
  TMEPCSimFrm = class(TForm)
    pbDisplay: TPaintBox;
    lbTMin: TLabel;
    lbTMax: TLabel;
    GroupBox1: TGroupBox;
    Label7: TLabel;
    edSimDuration: TEdit;
    bStart: TButton;
    bAbort: TButton;
    bClose: TButton;
    GroupBox4: TGroupBox;
    Label1: TLabel;
    Label2: TLabel;
    edAmplitude: TEdit;
    edStDev: TEdit;
    GroupBox5: TGroupBox;
    edTauRise: TEdit;
    Label11: TLabel;
    Label3: TLabel;
    edTauDecay: TEdit;
    edTauInterval: TEdit;
    Label4: TLabel;
    edNoiseRMS: TEdit;
    Label9: TLabel;
    Label6: TLabel;
    edLPFilter: TEdit;
    GroupBox2: TGroupBox;
    Label10: TLabel;
    EdSineFrequency: TEdit;
    edSineAmplitude: TEdit;
    Label5: TLabel;
    edStatus: TEdit;
    procedure bStartClick(Sender: TObject);
    procedure bAbortClick(Sender: TObject);
    procedure bCloseClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormShow(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure edAmplitudeKeyPress(Sender: TObject; var Key: Char);
    procedure pbDisplayPaint(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
  private
    { Private declarations }
    procedure HeapBuffers( Operation : THeapBufferOp ) ;
    procedure InitializeDisplay ;
    procedure UpdateEditBoxes ;
    procedure MEPCSimulation ;
    procedure GaussianFilter( Var Buf,OldBuf : Array of Integer ;
                              NumSamples,NumChannels : Integer ;
                              CutOffFrequency : single ;
                              var FirstCall : Boolean ) ;
  public
    { Public declarations }
  end;

var
  MEPCSimFrm: TMEPCSimFrm;

implementation
uses
    SSQUnit,MDIForm ;
{$R *.DFM}
const
     NumSamplesPerBuffer = 512 ;
     ChSim = 0 ;
type
    TState = (Idle, DoSimulation, ClearDisplay, EndofSimulation) ;
    TPointArray = Array[0..2000] of TPoint ;
    TSim = record
         MeanAmplitude : single ;
         StDev : single ;
         NoiseRMS : Single ;
         TauRise : Single ;
         TauDecay : Single ;
         TauInterval : Single ;
         Amplitude : Array[0..200] of Single ;
         SineAmplitude : single ;
         SineFrequency : double ;
         LPFilter : single ;
         LPFilterFirstCall : Boolean ;
         tStart : Array[0..200] of double ;
         tLast : double ;
         tEndAt : Double ;
         t : double ;
         NumMEPCS : Integer ;
         end ;
var
   Sim : TSim ;
   GausTemp : single ;
   ADC : ^TIntArray ; { Digitised signal buffer }
   OldADC : ^TIntArray ;
   xy : ^TPointArray ;
   Units : string ;
   BuffersAllocated : boolean ;{ Indicates if memory buffers have been allocated }


procedure TMEPCSimFrm.HeapBuffers( Operation : THeapBufferOp ) ;
{ -----------------------------------------------
  Allocate/deallocation dynamic buffers from heap
  -----------------------------------------------}
begin
     case Operation of
          Allocate : begin
             if not BuffersAllocated then begin
                New(ADC) ;
                New(OldADC) ;
                New(xy) ;
                BuffersAllocated := True ;
                end ;
             end ;
          Deallocate : begin
             if BuffersAllocated then begin
                Dispose(ADC) ;
                Dispose(OldADC) ;
                Dispose(xy) ;
                BuffersAllocated := False ;
                end ;
             end ;
          end ;
     end ;


procedure TMEPCSimFrm.FormCreate(Sender: TObject);
begin
     { Disable "MEPC Simulation" item in "Simulation" menu }
     Main.mnMEPCSim.enabled := false ;
     BuffersAllocated := False ;
     end;


procedure TMEPCSimFrm.FormShow(Sender: TObject);
{ --------------------------------------
  Initialisations when form is displayed
  --------------------------------------}
begin
     { Create buffers }
     HeapBuffers( Allocate ) ;

     bStart.Enabled := True ;
     bAbort.Enabled := False ;

     Channel[ChSim].ADCUnits := 'nA' ;
     Channel[ChSim].ADCName := 'Im' ;

     GausTemp := 1.0 ;
     UpdateEditBoxes ;

     pbDisplay.Repaint ;

     end ;


procedure TMEPCSimFrm.bStartClick(Sender: TObject);
{ --------------------
  Start simulation run
  --------------------}
var

   i : Integer ;
begin
     { Update all edit boxes and transfer their parameters to
       simulation control record (SIM) }
     UpdateEditBoxes ;
     { Update log with parameters for this simulation run }
     WriteToLogFile( 'Miniature endplate current simulation') ;
     WriteToLogFile( 'Mean Amplitude = ' + edAmplitude.text ) ;
     WriteToLogFile( 'St. dev. = ' + edStDev.text ) ;
     WriteToLogFile( 'Tau(rise) = ' + edTauRise.text ) ;
     WriteToLogFile( 'Noise RMS = ' + edNoiseRMS.text ) ;
     WriteToLogFile( 'Tau(decay) = ' + edTauDecay.text ) ;
     WriteToLogFile( 'Tau(interval) = ' + edTauInterval.text ) ;
     WriteToLogFile( 'Sine wave amplitude = ' + edSineAmplitude.text ) ;
     WriteToLogFile( 'Sine wave frequency = ' + edSineFrequency.text ) ;
     WriteToLogFile( 'Low pass filter cut-off = ' + edLPFilter.text ) ;
     WriteToLogFile( 'Duration of simulation = ' + edSimDuration.text ) ;


     bStart.Enabled := False ;
     bClose.Enabled := False ;
     bAbort.Enabled := True ;

     MEPCSimulation ;
     end;


procedure TMEPCSimFrm.MEPCSimulation ;
{ ==================================
  Create simulated endplate currents
  ==================================}
var
   i,j,xPix,yPix,ch,ChOffset : Integer ;
   NumBytesToWrite : LongInt ;
   x,xLimit,y,yMEPC,dx,omega,MaxAmplitude : Single  ;
   t : double ;
   OK,Done : Boolean ;
begin

     Sim.t := 0.0 ;

     if CdrFH.NumSamplesInFile = 0 then begin
        { Set scaling factor }
        cdrFH.NumChannels := 1 ;
        Channel[ChSim].ADCCalibrationFactor := 1. ;
        MaxAmplitude := 5.0*Sim.MeanAmplitude*
                        MaxFlt([Sim.TauDecay/Sim.TauInterval,1.]) ;
        Channel[ChSim].ADCScale := MaxAmplitude / MaxADCValue ;
        CdrFH.ADCVoltageRange :=  Channel[ChSim].ADCCalibrationFactor
                                * ( Channel[ChSim].ADCScale * (MaxADCValue+1) ) ;
        Channel[0].ChannelOffset := 0 ;
        end ;

     Channel[ChSim].ADCUnits := 'nA' ;
     Channel[ChSim].ADCName := 'Im' ;
     Channel[ChSim].ADCZero := 0 ;
     Channel[ChSim].yMin := MinADCValue ;
     Channel[ChSim].yMax := MaxADCValue ;
     Channel[ChSim].InUse := True ;


     { Fill MEPC start times list }
     t := 0.0 ;
     for i := 0 to High(Sim.tStart) do begin
         t := t - Sim.TauInterval*ln(Random) ;
         Sim.tStart[i] := t ;
         Sim.Amplitude[i] := Sim.MeanAmplitude + GaussianRandom(GausTemp)*Sim.StDev ;
         end ;
     Sim.tLast := t ;
     Sim.NumMEPCs := 0 ;
     Sim.LPFilterFirstCall := True ;

     { Sampling interval }
     cdrFH.dt := Sim.TauDecay / 20.0 ;

     { Position data file pointer at end of data }
     CdrFH.FilePointer := FileSeek(CdrFH.FileHandle,
     (CdrFH.NumSamplesInFile*2) + CdrFH.NumBytesInHeader,0) ;

     ChOffset := Channel[ChSim].ChannelOffset ;
     xLimit := Sim.TauDecay*6.0 ;

     omega := 2.0*Pi*Sim.SineFrequency ;
     Done := False ;

     { Create simulated MEPCs }
     while not Done do begin

         Channel[ChSim].xMin := Sim.t ;

         { Clear all channels }
         for ch := 0 to cdrFH.NumChannels-1 do
             for i := 0 to cdrFH.NumSamples-1 do ADC^[i] := 0 ;

         { Create a buffer-ful of simulated samples }
         for i := 0 to NumSamplesPerBuffer-1 do begin

             { Create background noise with gaussian distribution
               and sine wave interference }
             y := GaussianRandom(GausTemp)*Sim.NoiseRMS
                  + Sim.SineAmplitude*sin(Sim.t*omega) ;

             for j := 0 to High(Sim.tStart) do
                 if Sim.t > Sim.tStart[j] then begin
                 x := Sim.t - Sim.tStart[j] ;
                 yMEPC := Sim.Amplitude[j]*exp(-x/Sim.TauDecay)  ;
                 if Sim.TauRise > 0. then
                    yMEPC := yMEPC*(1.0-exp(-x/Sim.TauRise)) ;
                    y := y + yMEPC ;
                    { If MEPC decayed to zero, put a new one into this slot }
                    if x >= xLimit then begin
                       Sim.tStart[j] := Sim.tLast - Sim.TauInterval*ln(Random) ;
                       Sim.tLast := Sim.tStart[j] ;
                       Sim.Amplitude[j] := Sim.MeanAmplitude
                                           + GaussianRandom(GausTemp)*Sim.StDev ;
                       Inc(Sim.NumMEPCs) ;
                       edStatus.text := format(' %d MEPCs',[Sim.NumMEPCs]) ;
                       end ;
                    end ;

                 Sim.t := Sim.t + cdrFH.dt ;
                 j := cdrFH.NumChannels*i + ChOffset ;
                 ADC^[j] := Trunc(y/Channel[ChSim].ADCScale) + Channel[ChSim].ADCZero ;
                 end ;

         { Apply a gaussian low pass filter }
         if Sim.LPFilter > 0.0 then GaussianFilter( ADC^, OldADC^,
                                    NumSamplesPerBuffer,cdrFH.NumChannels,
                                    Sim.LPFilter*CdrFH.dt,Sim.LPFilterFirstCall ) ;


         Channel[ChSim].xMax := Sim.t ;

         { Display simulated record }
         EraseDisplay(pbDisplay) ;
         InitializeDisplay ;

         dx := Channel[ChSim].xScale*cdrFH.dt ;
         x := Channel[ChSim].Left ;
         j := Channel[ChSim].ChannelOffset ;
         for i := 0 to NumSamplesPerBuffer-1 do begin
             xy^[i].x := Trunc(x) ;
             xy^[i].y := Trunc(Channel[ChSim].Bottom
                         - Channel[ChSim].yScale*(ADC^[j] - Channel[ChSim].yMin));
             xy^[i].y := IntLimitTo(xy^[i].y,Channel[ChSim].Top,Channel[ChSim].Bottom) ;
             j := j + CdrFH.NumChannels ;
             x := x + dx ;
             end ;

         OK := Polyline(pbDisplay.Canvas.Handle,xy^,NumSamplesPerBuffer) ;

         { Display Channel Name }
         pbDisplay.Canvas.TextOut( Channel[ChSim].Left,
                                  (Channel[ChSim].Top + Channel[ChSim].Bottom) div 2,
                                   ' ' + Channel[ChSim].ADCName ) ;

         { Draw cursors }
         DrawHorizontalCursor(pbDisplay,Channel[ChSim],Channel[ChSim].ADCZero) ;

         { Save data to file }
         NumBytesToWrite := NumSamplesPerBuffer*CdrFH.NumChannels*2 ;
         cdrFH.NumSamplesInFile := cdrFH.NumSamplesInFile
                                   + (NumSamplesPerBuffer*CdrFH.NumChannels) ;
        if FileWrite(CdrFH.FileHandle,ADC^,NumBytesToWrite) <> NumBytesToWrite then
           MessageDlg( ' File write file failed',mtWarning, [mbOK], 0 ) ;

        if (Sim.t >= Sim.tEndAt) or bStart.Enabled then begin
           Done := True ;
           bStart.Enabled := True ;
           bAbort.Enabled := False ;
           bClose.Enabled := True ;
           end ;

        Application.ProcessMessages ;
        end ;

     end ;


procedure TMEPCSimFrm.bAbortClick(Sender: TObject);
{ --------------------
  Abort simulation run
  --------------------}
begin
     bStart.Enabled := True ;
     bClose.Enabled := True ;
     bAbort.Enabled := False ;
     end;


procedure TMEPCSimFrm.bCloseClick(Sender: TObject);
begin
     close ;
     end;


procedure TMEPCSimFrm.FormClose(Sender: TObject; var Action: TCloseAction);
begin

     if CdrFH.NumSamplesInFile > 0 then begin
        Main.mnViewCDR.Enabled := True ;
        Main.mnViewCDR.checked := True ;
        Main.Analysis.Enabled := True ;
        end ;

     SaveCDRHeader( CdrFH ) ;
     WriteToLogFile( format( '%d MEPCs created',[Sim.NumMEPCs])) ;

     HeapBuffers( Deallocate ) ;
     { Enable "Synaptic Current" item in "Simulation" menu }
     Main.mnMEPCSim.enabled := true ;
     { Display results }
     if CdrFH.NumSamplesInFile > 0 then begin
        if Main.ViewCDRChildExists then Main.ViewCDRChild.NewFile := True ;
        Main.mnViewCDR.Click ;
        end ;
     Action := caFree ;
     end;


procedure TMEPCSimFrm.edAmplitudeKeyPress(Sender: TObject;
  var Key: Char);
begin
     if key = chr(13) then UpdateEditBoxes ;
     end;


procedure TMEPCSimFrm.InitializeDisplay ;
{ ----------------------------
  Initialise the display window
  ---------------------------}

begin

     { Set trace colour }

     pbDisplay.canvas.pen.color := Channel[ChSim].color ;

     { Define display area for each channel in use }

     Channel[ChSim].Left := 0 ;
     Channel[ChSim].Right := pbDisplay.width ;
     Channel[ChSim].Top := 0 ;
     Channel[ChSim].Bottom := pbDisplay.Height ;
     Channel[ChSim].xScale := (Channel[ChSim].Right - Channel[ChSim].Left) /
                           (Channel[ChSim].xMax - Channel[ChSim].xMin ) ;
     Channel[ChSim].yScale := (Channel[ChSim].Bottom - Channel[ChSim].Top) /
                           (Channel[ChSim].yMax - Channel[ChSim].yMin ) ;

     lbTMin.caption := Format( '%.3g %s', [Channel[ChSim].xMin*Settings.TScale,
                                               Settings.TUnits] ) ;
     lbTMax.caption := Format( '%.3g %s', [Channel[ChSim].xMax*Settings.TScale,
                                               Settings.TUnits] ) ;
     lbTMin.Left := pbDisplay.Left ;
     lbTMax.Left := pbDisplay.Left + pbDisplay.Width - lbTMax.Width ;


     { Display Channel Name(s) }

     pbDisplay.Canvas.TextOut( Channel[ChSim].Left,
           (Channel[ChSim].Top + Channel[ChSim].Bottom) div 2,' ' + Channel[ChSim].ADCName ) ;

     end ;


procedure TMEPCSimFrm.UpdateEditBoxes ;
{ --------------------------------------------------------------------------
  Update all edit boxes and transfer parameters to simulation control record
  --------------------------------------------------------------------------}
begin
     { Amplitude of MEPC }
     Sim.MeanAmplitude := GetFromEditBox(edAmplitude,
                                     1.,-1E4,1E4, '%.1f', Channel[ChSim].ADCUnits ) ;

     { Standard deviation of quantum amplitude }
     Sim.StDev := GetFromEditBox(edStDev,
                                 0.1,0.,1E4, '%.2f', Channel[ChSim].ADCUnits ) ;

     { Rise time constant of quantal event }
     Sim.TauRise := GetFromEditBox(edTauRise,0.1,0.0,100.0,'%.2f','ms')*MsToSecs ;

     { Background noise RMS }
     Sim.NoiseRMS := GetFromEditBox(edNoiseRMS,
                             0., 0., 1E4, '%.2f', Channel[ChSim].ADCUnits) ;

     { Decay time constant of quantal event }
     Sim.TauDecay := GetFromEditBox(edTauDecay,1.,0.01,10000.,'%.1f','ms')*MsToSecs ;

     { Inter-event interval time constant }
     Sim.TauInterval := GetFromEditBox(edTauInterval,100.,0.1,1E6,'%.1f','ms')*MsToSecs ;

     { Simulation time period }
     Sim.tEndAt := GetFromEditBox(edSimDuration,1.,1.0,10000.,'%.f','s');

     { Sine wave pk-pk amplitude }
     Sim.SineAmplitude := GetFromEditBox(edSineAmplitude,
                                 0.1,0.,1E4, '%.1f', Channel[ChSim].ADCUnits ) ;

     { Sine wave pk-pk amplitude }
     Sim.SineFrequency := GetFromEditBox(edSineFrequency,0.1,0.,1E4, '%.1f', 'Hz' ) ;

     { Low-pass filter cut-off frequency }
     Sim.LPFilter := GetFromEditBox(edLPFilter,0.1,0.,1E4, '%.1f', 'Hz' ) ;

     end ;


procedure TMEPCSimFrm.GaussianFilter( Var Buf,OldBuf : Array of Integer ;
                                      NumSamples,NumChannels : Integer ;
                                      CutOffFrequency : single ;
                                      var FirstCall : Boolean ) ;
{ --------------------------------------------------
  Gaussian digital filter. (based on Sigworth, 1983)
  --------------------------------------------------}
const
     MaxCoeff = 54 ;
type
    TWorkArray = Array[-2*MaxCoeff..MaxTBuf] of Integer ;
var
   a : Array[-MaxCoeff..MaxCoeff] of single ;
   b : Array[-2*MaxCoeff..0] of single ;
   Temp,sum,sigma,aScale : single ;
   i,iIn,iOut,j,j1,nca,ncb,Ch,ChOffset,EndOfData : Integer ;
   Work : ^TWorkArray ;
begin

      try

         New(Work) ;

         { Generate filter coefficients }
         sigma := 0.132505/CutOffFrequency ;
         if  sigma >= 0.62  then begin

	     aScale := -1./(2.*sigma*sigma) ;
	     nca := 0 ;
	     a[0] := 1.0 ;
	     sum := 1.0 ;
	     temp := 1.0 ;
	     while (temp >= 10.0*MinSingle) and (nca < MaxCoeff) do begin
	           Inc(nca) ;
	           temp := exp( nca*nca*aScale ) ;
	           a[nca] := temp ;
                   a[-nca] := Temp ;
	           sum := sum + 2.0*temp ;
                   end ;

             { Normalise coefficients so that they summate to 1. }
	     for i := -nca to nca do a[i] := a[i]/sum ;
             end
         else begin
            { Special case for very light filtering
              (See Colquhoun & Sigworth, 1983) }
            a[1] := (sigma*sigma)/2. ;
            a[-1] := a[1] ;
	    a[0] := 1.0 - 2.0*a[2] ;
	    nca := 1 ;
            end ;

         ncb := 1 ;
         for i := nca downto -nca do begin
             Dec(ncb) ;
             b[ncb] := a[i] ;
             end ;

         { Copy data into work array }
         for i := 0 to (NumSamples*NumChannels)-1 do Work^[i] := Buf[i] ;

         { If this is the first call to the filter, fill OldBuf
           buffer with the first samples in Buf }
         if FirstCall then begin
            for ch := 0 to NumChannels-1 do
                ChOffset := Channel[ch].ChannelOffset ;
                for i := 0 to NumSamples-1 do begin
                    j := i*NumChannels + ChOffset ;
                    OldBuf[j] := Buf[ch] ;
                    end ;
            FirstCall := False ;
            end ;

         { Apply filter to each channel }
         for Ch := 0 to NumChannels-1 do begin

             { Get samples at end of old buffer still to needed by filter }
             j := NumSamples - NumChannels + Channel[ch].ChannelOffset ;
             for i := -1 downto Low(Work^) do begin
                 Work^[i] := OldBuf[j] ;
                 j := j - NumChannels ;
                 end ;

             { Copy a channel from the new data to work buffer }
             j := Channel[ch].ChannelOffset ;
             for i := 0 to NumSamples-1 do begin
                 Work^[i] := Buf[j] ;
                 j := j + NumChannels ;
                 end ;

             { Apply gaussian filter to each point
               and store result back in buffer }
             iOut := Channel[ch].ChannelOffset ;
             EndOfData := NumSamples -1 ;
             for iIn := 0 to EndOfData do begin
	         sum := 0.0 ;
	         for j := ncb to 0 do begin
                     j1 := (j+iIn) ;
	             sum := sum + (Work^[j1])*b[j] ;
                     end ;
                 Buf[iOut] := Trunc(sum) ;
                 iOut := iOut + NumChannels ;
                 end ;
             end ;
      finally
             { Keep old data buffer because next call will need it }
             for i := 0 to (NumSamples*NumChannels)-1 do OldBuf[i] := Buf[i] ;
             Dispose(Work) ;
             end ;
      end ;


procedure TMEPCSimFrm.pbDisplayPaint(Sender: TObject);
begin
     { Erase display }
     lbTMin.Left := pbDisplay.Left ;
     lbTMin.Top := pbDisplay.Top + pbDisplay.Height + 2 ;
     lbTMax.Left := pbDisplay.Left + pbDisplay.Width
                    - canvas.TextWidth(lbTMax.caption) ;
     lbTMax.Top := lbTMin.Top ;
     lbTMin.caption := '' ;
     lbTMax.caption := '' ;

     edStatus.Left := pbDisplay.Left + (pbDisplay.Width - edStatus.Width) div 2 ;
     edStatus.Top := lbTmin.Top ;
     EraseDisplay(pbDisplay) ;

     end;

procedure TMEPCSimFrm.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
     { Prevent form being closed if a simulation is running (Close button disabled) }
     if not bClose.Enabled then CanClose := False
                           else CanClose := True ;
     end;

end.
