unit Setup;
{ ====================================================================
  WinCDR : Digital recording parameters setup dialog.
  (c) J. Dempster, University of Strathclyde 1996, All Rights Reserved
  ====================================================================}
interface

uses WinTypes, WinProcs, Classes, Graphics, Forms, Controls, Buttons,
  StdCtrls, Spin, ExtCtrls, shared, sysUtils, Grids, Dialogs, Global, FileIo,
  LabIo;

type
  TSetupDlg = class(TForm)
    GroupBox1: TGroupBox;
    lNumChannels: TLabel;
    EdNumChannels: TEdit;
    Label1: TLabel;
    EdSamplingInterval: TEdit;
    Channels: TGroupBox;
    Label5: TLabel;
    ChannelTable: TStringGrid;
    Label6: TLabel;
    cbADCVoltageRange: TComboBox;
    ColorBox: TPanel;
    ColorLabel: TLabel;
    Label2: TLabel;
    bOK: TButton;
    bCancel: TButton;
    GroupBox2: TGroupBox;
    cbLabInterface: TComboBox;
    TUnitsGrp: TGroupBox;
    rbTmsecs: TRadioButton;
    rbTSecs: TRadioButton;
    grp1902: TGroupBox;
    ckCED1902InUse: TCheckBox;
    PortsGrp: TGroupBox;
    lbIOPort: TLabel;
    edIOPort: TEdit;
    lbIRQ: TLabel;
    EdIRQ: TEdit;
    Label8: TLabel;
    EdDMA: TEdit;
    ckNIDisableDMA: TCheckBox;
    Label3: TLabel;
    ColorDialog1: TColorDialog;
    sbNumChannels: TSpinButton;
    procedure ColorBoxClick(Sender: TObject);
    procedure ColorBoxDblClick(Sender: TObject);
    procedure bOKClick(Sender: TObject);
    procedure rbTmsecsClick(Sender: TObject);
    procedure rbTSecsClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure bCancelClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure cbLabInterfaceChange(Sender: TObject);
    procedure EdSamplingIntervalKeyPress(Sender: TObject; var Key: Char);
    procedure sbNumChannelsDownClick(Sender: TObject);
    procedure sbNumChannelsUpClick(Sender: TObject);
  private
    { Private declarations }
    procedure UpdateChannels ;
  public
    { Public declarations }
  end;

const
     ChNum = 0 ;
     ChName = 1 ;
     ChCal = 2 ;
     ChAmp = 3 ;
     ChUnits = 4 ;
var
  SetupDlg: TSetupDlg;
  UpdateOK : boolean ;
  OKorCancelButtonClicked : boolean ;

implementation

{$R *.DFM}
Uses maths ;

procedure TSetupDlg.FormShow(Sender: TObject);
{ --------------------------------------------------------------------------
  Initialise setup's combo lists and tables with current recording parameters
  --------------------------------------------------------------------------}
var
   ch,i,NumOptions,Err : Integer ;
   RangeOptions : array[1..10] of TADCRange ;
   Buf : array[0..20] of char ;
   x,dx : Double ;
   Diff,MinDiff,CurrentDuration : single ;
   n : LongInt ;
   List : TStringList ;
begin


     cbLabInterface.Clear ;
     try
        List := TStringList.Create ;
        GetLabInterfaceList( List ) ;
        cbLabInterface.Items := List ;
        cbLabInterface.ItemIndex := Settings.LaboratoryInterface ;
        SelectLaboratoryInterface( Settings.LaboratoryInterface ) ;

        if TLaboratoryInterface(Settings.LaboratoryInterface) = DD then begin
           { If using Axon Digidata 1200 interface, show the port settings }
           PortsGrp.Visible := True ;
           edIoPort.text := format(' %xh', [Settings.DD1200IOPort] ) ;
           edIRQ.text := format( ' %d ', [Settings.DD1200IRQ] ) ;
           edDMA.text := format( ' %d ', [Settings.DD1200DMA] ) ;
           ckNIDisableDMA.visible := False ;
           end
        else if TLaboratoryInterface(Settings.LaboratoryInterface) = NI then begin
           PortsGrp.Visible := False ;
           ckNIDisableDMA.visible := True ;
           ckNIDisableDMA.Checked := Settings.NIDisableDMA ;
           end
        else begin
           PortsGrp.Visible := False ;
           ckNIDisableDMA.visible := False ;
           end ;

     finally
          List.Free ;
          end ;

     { Allow changes to no. of channels and/or sampling interval
       only if the data file is empty }
     if CdrFH.NumSamplesInFile > 0 then begin
        EdSamplingInterval.enabled := False ;
        sbNumChannels.enabled := False ;
        cbADCVoltageRange.Enabled := False ;
        end
     else begin
        EdSamplingInterval.enabled := True ;
        sbNumChannels.enabled := True ;
        cbADCVoltageRange.Enabled := True ;
        end ;

     { Set up A/D Converter voltage range selection box }
     GetADCVoltageRangeOptions( RangeOptions, NumOptions ) ;
     cbADCVoltageRange.clear ;
     for i := 1 to NumOptions do cbADCVoltageRange.items.add( RangeOptions[i] ) ;

     GetSamplingIntervalRange( Settings.MinSamplingInterval,
                               Settings.MaxSamplingInterval ) ;

     { Select the current A/D converter range in use }
     MinDiff := 1E30 ;
     for i := 1 to NumOptions do begin
         Diff := Abs(CdrFH.ADCVoltageRange-ExtractFloat(RangeOptions[i],1. )) ;
         if MinDiff >= Diff then begin
            MinDiff := Diff ;
            cbADCVoltageRange.ItemIndex := i-1
            end ;
         end ;


     { Number of channels }
     EdNumChannels.text := format( ' %d ', [CdrFH.NumChannels] ) ;
     { Number of samples per channel }
     EdSamplingInterval.text := format( '%.3g ms', [CdrFH.dt*1000.0] ) ;

     { If a CED 1902 programmable amplifier is available and
       in use ... set Channel 0 amplification factor to the gain of
       the CEd 1902 }
     if Settings.CED1902.InUse then begin
        ckCED1902InUse.checked := True ;
        Channel[0].ADCAmplifierGain := Settings.CED1902.GainValue ;
        end
     else ckCED1902InUse.checked := False ;


     { Set channel calibration table }
     ChannelTable.cells[ChNum,0] := 'Ch.' ;
     ChannelTable.colwidths[ChNum] := ChannelTable.DefaultColWidth div 2 ;
     ChannelTable.cells[ChName,0] := 'Name' ;
     ChannelTable.colwidths[ChName] := ChannelTable.DefaultColWidth ;
     ChannelTable.cells[ChCal,0] := 'V/Units' ;
     ChannelTable.colwidths[ChCal] := (5*ChannelTable.DefaultColWidth) div 4 ;
     ChannelTable.cells[ChAmp,0] := 'Amp. Gain' ;
     ChannelTable.colwidths[ChAmp] := (5*ChannelTable.DefaultColWidth) div 4 ;
     ChannelTable.cells[ChUnits,0] := 'Units' ;
     ChannelTable.colwidths[ChUnits] := ChannelTable.DefaultColWidth ;
     ChannelTable.RowCount := CdrFH.NumChannels + 1;
     ChannelTable.options := [goEditing,goHorzLine,goVertLine] ;

     UpdateChannels ;

     { Set trace colour box }
     ColorBox.color := Channel[0].color ;


     { Set time units radio buttons }
     if Settings.TUnits = 's' then rbTSecs.checked := true
                              else rbTmsecs.checked := true ;

     UpDateOK := True ;
     OKorCancelButtonClicked := False ;
     end;


procedure TSetupDlg.ColorBoxClick(Sender: TObject);
begin
     ColorDialog1.execute ;
     ColorBox.color := ColorDialog1.color ;
     end;


procedure TSetupDlg.ColorBoxDblClick(Sender: TObject);
begin
     ColorDialog1.execute ;
     ColorBox.color := ColorDialog1.color ;
     end;


procedure TSetupDlg.rbTmsecsClick(Sender: TObject);
begin
     Settings.TUnits := 'ms' ;
     Settings.TScale := SecsToms ;
     Settings.TUnscale := MsToSecs ;
     end;

procedure TSetupDlg.rbTSecsClick(Sender: TObject);
begin
     Settings.TUnits := 's' ;
     Settings.TScale := 1. ;
     Settings.TUnscale := 1. ;
     end;


procedure TSetupDlg.bOKClick(Sender: TObject);
{ ---------------------------------
  Exit setup dialog and update file
  ---------------------------------}
begin
     UpdateOK := True ;
     OKorCancelButtonClicked := True ;
     end ;


procedure TSetupDlg.bCancelClick(Sender: TObject);
{ ---------------------------------------
  Exit setup dialog but don't update file
  ---------------------------------------}
begin
     UpdateOK := False ;
     OKorCancelButtonClicked := True ;
     end;


procedure TSetupDlg.FormClose(Sender: TObject; var Action: TCloseAction);
{ ---------------------------------------------
  Update values in file header (CdrFH) and Exit
  ---------------------------------------------}
var
   ch : LongInt ;
begin

     if not OKorCancelButtonClicked then begin
        if MessageDlg( 'Save changes! Are you Sure? ', mtConfirmation,
           [mbYes,mbNo], 0 ) = mrYes then UpdateOK := True
                                     else UpdateOK := False ;
           end ;

     if UpdateOK then begin
        { Update file header block with new settings }

        Settings.LaboratoryInterface := cbLabInterface.ItemIndex ;

        { Use I/O port settings if the interface is a Digidata 1200 }
        if TLaboratoryInterface(Settings.LaboratoryInterface) = DD then begin
           if Contains( '100', edIOPort.text ) then Settings.DD1200IOPort := $100 ;
           if Contains( '120', edIOPort.text ) then Settings.DD1200IOPort := $120 ;
           if Contains( '140', edIOPort.text ) then Settings.DD1200IOPort := $140 ;
           if Contains( '160', edIOPort.text ) then Settings.DD1200IOPort := $160 ;
           if Contains( '180', edIOPort.text ) then Settings.DD1200IOPort := $180 ;
           if Contains( '1A0', edIOPort.text ) then Settings.DD1200IOPort := $1A0 ;
           if Contains( '1C0', edIOPort.text ) then Settings.DD1200IOPort := $1C0 ;
           if Contains( '1E0', edIOPort.text ) then Settings.DD1200IOPort := $1E0 ;
           if Contains( '200', edIOPort.text ) then Settings.DD1200IOPort := $200 ;
           if Contains( '220', edIOPort.text ) then Settings.DD1200IOPort := $220 ;
           if Contains( '240', edIOPort.text ) then Settings.DD1200IOPort := $240 ;
           if Contains( '300', edIOPort.text ) then Settings.DD1200IOPort := $300 ;
           if Contains( '320', edIOPort.text ) then Settings.DD1200IOPort := $320 ;
           if Contains( '340', edIOPort.text ) then Settings.DD1200IOPort := $340 ;
           Settings.DD1200IRQ := ExtractInt( edIRQ.text ) ;
           Settings.DD1200DMA := ExtractInt( edDMA.text ) ;
           end
        else if TLaboratoryInterface(Settings.LaboratoryInterface) = NI then begin
           Settings.NIDisableDMA := ckNIDisableDMA.Checked ;
           end ;

        CdrFH.NumBytesInHeader := 512 ;

        { Recording parameters (only safe to update these if no data in file) }
        if CdrFH.NumSamplesInFile = 0 then begin
           CdrFH.dt := ExtractFloat( edSamplingInterval.text,CdrFH.dt*1000.0 )
                       /1000.0 ;
           CdrFH.NumChannels := ExtractInt( edNumChannels.text ) ;
           CdrFH.ADCVoltageRange := ExtractFloat(
                                    cbADCVoltageRange.items[cbADCVoltageRange.ItemIndex],
                                    CdrFH.ADCVoltageRange ) ;
           end ;

        { Channel calibration }
        for ch := 0 to CdrFH.NumChannels-1 do begin
            Channel[ch].ADCName := ChannelTable.cells[ChName,ch+1] ;
            Channel[ch].ADCCalibrationFactor := ExtractFloat(
                                             ChannelTable.cells[ChCal,ch+1],
                                             Channel[ch].ADCCalibrationFactor);
            Channel[ch].ADCAmplifierGain := ExtractFloat(
                                         ChannelTable.cells[ChAmp,ch+1],
                                         Channel[ch].ADCAmplifierGain);
            Channel[ch].ADCUnits := ChannelTable.cells[ChUnits,ch+1] ;
            end ;

        { Set channel colours }
        for ch := 0 to ChannelLimit do Channel[ch].color := ColorBox.color ;

        { Update CED 1902 amplifier settings }
        if ckCED1902InUse.checked then begin
           Channel[ch].ADCAmplifierGain := Settings.CED1902.GainValue ;
           Settings.CED1902.InUse := True ;
           end
        else Settings.CED1902.InUse := False ;

        { Save data to file header }
        SaveCDRHeader( CdrFH ) ;
        end ;
     end ;



procedure TSetupDlg.cbLabInterfaceChange(Sender: TObject);
var
   i,NumOptions : Integer ;
   RangeOptions : array[1..10] of TADCRange ;
begin
     SelectLaboratoryInterface( cbLabInterface.ItemIndex ) ;
     GetADCVoltageRangeOptions( RangeOptions, NumOptions ) ;
     cbADCVoltageRange.clear ;
     for i := 1 to NumOptions do cbADCVoltageRange.items.add( RangeOptions[i] ) ;
     cbADCVoltageRange.ItemIndex := 0 ;

     if TLaboratoryInterface(cbLabInterface.ItemIndex) = DD then begin
        PortsGrp.Visible := True ;
        edIoPort.text := format(' %xh', [Settings.DD1200IOPort] ) ;
        edIRQ.text := format( ' %d ', [Settings.DD1200IRQ] ) ;
        edDMA.text := format( ' %d ', [Settings.DD1200DMA] ) ;
        end
     else PortsGrp.Visible := False ;
     end;


procedure TSetupDlg.UpdateChannels ;
{ ----------------------------------------------------------------
  Update channels and sampling interval for new number of channels
  ----------------------------------------------------------------}
var
   nChannels,nChannelsOld,ch : Integer ;
   x : single ;
begin
     nChannelsOld := nChannels ;
     { Update number of channels and keep within valid limits }
     nChannels := ExtractInt( edNumChannels.text ) ;
     nChannels := MinInt([MaxInt( [nChannels, 1] ),ChannelLimit+1]) ;
     edNumChannels.text := format( ' %d ',[nChannels] ) ;

     { Make sure sampling interval is kept within valid limits }
     x := MsToSecs*(ExtractFloat( edSamplingInterval.text, CdrFH.dt*SecsToMs ) /
                    nChannels) ;
     x := MinFlt([MaxFlt( [x, Settings.MinSamplingInterval] ),
                           Settings.MaxSamplingInterval]) ;
     edSamplingInterval.text := format( ' %.3g ms',[x*SecsToMs*nChannels] ) ;

     ChannelTable.RowCount := nChannels + 1 ;
     { Add lines to channel table if new channels have been added }

     for ch := 0 to nChannels-1 do begin
         ChannelTable.cells[ChNum,ch+1] := format('%d',[ch]) ;
         ChannelTable.cells[ChName,ch+1] := Channel[ch].ADCName ;
         ChannelTable.cells[ChCal,ch+1] := Format( '%5.4g',[Channel[ch].ADCCalibrationFactor] ) ;
         ChannelTable.cells[ChAmp,ch+1] := Format( '%5.4g',[Channel[ch].ADCAmplifierGain] ) ;
         ChannelTable.cells[ChUnits,ch+1] := Channel[ch].ADCUnits ;
         end ;

     end ;


procedure TSetupDlg.EdSamplingIntervalKeyPress(Sender: TObject;
  var Key: Char);
var
   x : single ;
   nChannels : Integer ;
begin
     if key = chr(13) then begin

        nChannels := ExtractInt( edNumChannels.text ) ;

        { Make sure sampling interval is kept within valid limits }
        x := MsToSecs*(ExtractFloat( edSamplingInterval.text, CdrFH.dt*SecsToMs ) /
                       nChannels) ;
        x := MinFlt([MaxFlt( [x, Settings.MinSamplingInterval] ),
                             Settings.MaxSamplingInterval]) ;
        edSamplingInterval.text := format( ' %.3g ms',[x*SecsToMs*nChannels] ) ;
        end ;
     end;

procedure TSetupDlg.sbNumChannelsDownClick(Sender: TObject);
{ ---------------------------------------------
  Decrement required number of recording sweeps
  ---------------------------------------------}
var
   n : Integer ;
begin
     n := MaxInt( [ExtractInt(edNumChannels.text)-1,1] ) ;
     edNumChannels.text := format(' %d ',[n] ) ;
     UpdateChannels ;
     end;

procedure TSetupDlg.sbNumChannelsUpClick(Sender: TObject);
{ ---------------------------------------------
  Increment required number of recording sweeps
  ---------------------------------------------}
var
   n : Integer ;
begin
     n := MinInt( [ExtractInt(edNumChannels.text)+1,ChannelLimit+1] ) ;
     edNumChannels.text := format(' %d ',[n] ) ;
     UpdateChannels ;
     end;
end.
