unit Import;
{ ==============================================================
  WinCDR - General purpose Binary/ASCII data file import module
  (c) J. Dempster, University of Strathclyde, 1998
  25/6/98 Binary file import now works,
          Bug in text file import fixed
  ==============================================================}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, TabNotBk, maths, shared, global, fileio;

type
  TImportFrm = class(TForm)
    bOK: TButton;
    bCancel: TButton;
    tnImportType: TTabbedNotebook;
    mePreview: TMemo;
    GroupBox1: TGroupBox;
    Label1: TLabel;
    Label2: TLabel;
    Label4: TLabel;
    Label6: TLabel;
    edIgnore: TEdit;
    EdDt: TEdit;
    edNumColumns: TEdit;
    EdTimeColumn: TEdit;
    cbTUnits: TComboBox;
    Label5: TLabel;
    GroupBox2: TGroupBox;
    Label7: TLabel;
    Label8: TLabel;
    Label9: TLabel;
    Label10: TLabel;
    Label11: TLabel;
    edNumFileHeaderBytes: TEdit;
    edDTBinary: TEdit;
    EdNumChannelsBinary: TEdit;
    EdScaleBy: TEdit;
    cbTUnitsBinary: TComboBox;
    Label13: TLabel;
    EdOffsetBy: TEdit;
    lbNumBytesToImport: TLabel;
    EdNumBytesToImport: TEdit;
    procedure FormShow(Sender: TObject);
    procedure bOKClick(Sender: TObject);
    procedure cbTUnitsChange(Sender: TObject);
    procedure edNumColumnsKeyPress(Sender: TObject; var Key: Char);
    procedure edNumFileHeaderBytesKeyPress(Sender: TObject; var Key: Char);
    procedure cbTUnitsBinaryChange(Sender: TObject);
  private
    { Private declarations }
    procedure InspectFile ;
    procedure ImportText ;
    procedure ImportBinary ;
    procedure UpdateImportTextParameters ;
    procedure UpdateImportBinaryParameters ;
  public
    { Public declarations }
    FileName : string ;
  end;

var
  ImportFrm: TImportFrm;

implementation

{$R *.DFM}
type
    TTextImport = record
            LastColumn : Integer ;
            TCol : Integer ;
            NumRowsIgnored : Integer ;
            dt : single ;
            TScale : single ;
            MaxValues : Array[0..20] of single ;
            Initialised : Boolean ;
            end ;

    TBinaryImport = record
            NumChannels : Integer ;
            NumBytesInFileHeader : Integer ;
            NumBytesToImport : longInt ;
            NumBytesInFile : longInt ;
            ScaleBy : single ;
            OffsetBy : single ;
            dt : single ;
            TScale : single ;
            Initialised : Boolean ;
            end ;

var
   ImpText : TTextImport ;
   ImpBinary : TBinaryImport ;


procedure TImportFrm.FormShow(Sender: TObject);
{ -----------------------------------
  Display first 10 lines of data file
  -----------------------------------}
begin

     cbTUnits.Clear ;
     cbTUnits.Items.Add('s') ;
     cbTUnits.Items.Add('ms') ;
     cbTUnits.Items.Add('us') ;
     cbTUnits.ItemIndex := 1 ;

     cbTUnitsBinary.Clear ;
     cbTUnitsBinary.Items.Add('s') ;
     cbTUnitsBinary.Items.Add('ms') ;
     cbTUnitsBinary.Items.Add('us') ;
     cbTUnitsBinary.ItemIndex := 1 ;

     { Initialise settings }
     InspectFile ;

     end ;


procedure TImportFrm.InspectFile ;
{ ----------------------------------------------------------
  Inspect the data file to determine initial import settings
  ----------------------------------------------------------}
var
   F : TextFile ;
   i,Col,nCols,nColsOld,nSamples,MaxSamples,FileHandle : Integer ;
   nLines : LongInt ;
   T,TOld : single ;
   Values : Array[0..20] of Single ;
   s : string ;
begin

     Screen.Cursor := crHourglass ;

     { Open file for reading as text }
     AssignFile( F, ImportFrm.FileName ) ;
     Reset(F) ;

     { Initialise counters }
     for Col := 0 to High(ImpText.MaxValues) do
         ImpText.MaxValues[Col] := -1E30 ;
     nSamples := 0 ;
     MaxSamples := 0 ;
     nLines := 0 ;
     nColsOld := 0 ;

     mePreview.Clear ;
     while not EOF(F) do begin
         { Read a line of text from file }
         ReadLn( F,s ) ;
         { Display first 10 lines into memo box }
         if nLines < 10 then mePreview.Lines.Add( s ) ;
         Inc(nLines) ;

         { Number of columns }
         nCols := ExtractListOfFloats( s, Values, False ) ;
         if nCols = nColsOld then begin
            ImpText.LastColumn := nCols - 1;
            end ;
         nColsOld := nCols ;

         { Determine maximum absolute values within data columns }
         for Col := 0 to nCols-1 do begin
              ImpText.MaxValues[Col] := MaxFlt( [Abs(Values[Col]),
                                               ImpText.MaxValues[Col]] ) ;
              end ;

         { Sampling interval }
         if nCols > 1 then begin
            T := Values[0] ;
            ImpText.DT := T-TOld ;
            TOld := T ;
            end ;

         end ;

     ImpText.NumRowsIgnored := 0 ;

     ImpText.Initialised := False ;
     UpdateImportTextParameters ;

     CloseFile(F) ;

     { Determine number of bytes in file (for binary import) }
     FileHandle := FileOpen( ImportFrm.FileName, fmOpenRead ) ;
     ImpBinary.NumBytesInFile := FileSeek( FileHandle, 0, 2 ) ;
     FileClose( FileHandle ) ;
     ImpBinary.Initialised := False ;
     UpdateImportBinaryParameters ;

     Screen.Cursor := crDefault ;
     end;


procedure TImportFrm.bOKClick(Sender: TObject);
{ ---------------------
  Import data from file
  ---------------------}
begin

     if tnImportType.Pages[tnImportType.PageIndex] = 'ASCII' then ImportText
                                                             else ImportBinary ;
     end;


procedure TImportFrm.ImportText ;
{ ----------------------
  Import ASCII data file
  ----------------------
  25/6/98 ... Imported data now placec correctly in data file }

var
   F : TextFile ;
   Values,Scale : Array[0..20] of Single ;
   s : string ;
   Col,nCols,n,nSamples,NumSamplesPerBuffer,NumBytesPerBuffer,ch,i,nw : Word ;
   TScale : single ;
   Buf : ^TIntArray ;
begin
     try

        Screen.Cursor := crHourglass ;

        { Create buffer to hold samples }
        New(Buf) ;

        { Update settings in Import record from text boxes on form }
        UpdateImportTextParameters ;

        { Close existing WCD data file }
        if CdrFH.FileHandle > 0 then FileClose( CdrFH.FileHandle ) ;
        { Create name of WCD file to hold ASCII data file }
        CdrFH.FileName := ReplaceFileEnding( ImportFrm.FileName, '.wcd' ) ;
        { Create a new WCD file to hold converted data }
        CdrFH.FileHandle := FileCreate( CdrFH.FileName ) ;

        CdrFH.NumSamplesInFile := 0 ;

        { Open file for reading as text }
        AssignFile( F, ImportFrm.FileName ) ;
        Reset(F) ;

        { Number of signal channels }
        CdrFH.NumChannels := ImpText.LastColumn + 1 ;
        { If one of the data columns is time data, reduce channel count }
        if ImpText.TCol >= 0 then Dec(CdrFH.NumChannels) ;

        { Set sampling interval }
        if ImpText.dt = 0.0 then begin
           MessageDlg( ' Sampling interval not defined!',mtWarning, [mbOK], 0 ) ;
           ImpText.dt := 1.0 ;
           end ;
        CdrFH.dt := ImpText.dt*ImpText.TScale ;

        CdrFH.ADCVoltageRange := 1.0 ;
        Ch := 0 ;
        for Col := 0 to ImpText.LastColumn do
          if Col <> ImpText.TCol then begin
            Scale[Col] := MaxADCValue / (1.1*ImpText.MaxValues[Col]) ;
            Channel[Ch].ADCScale := 1.0 / Scale[Col] ;
            Channel[ch].ADCCalibrationFactor := CdrFH.ADCVoltageRange /
                          ( Channel[ch].ADCScale * (MaxADCValue+1) ) ;
            Inc(Ch) ;
            end ;

        { Channel calibration and scale factor settings }
        for ch := 0 to CdrFH.NumChannels-1 do begin
            Channel[ch].ChannelOffset := ch ;
            Channel[ch].ADCAmplifierGain := 1. ;
            Channel[ch].ADCZero := 0 ;
            Channel[ch].ADCUnits := '' ;
            Channel[ch].ADCName := Format( 'Ch.%d', [ch] ) ;
            end ;

        { Import ASCII data into .WCD file format }
        Reset(F) ;
        nSamples := 0 ;
        NumSamplesPerBuffer := CdrFH.NumChannels*NumSamplesPerSector ;
        NumBytesPerBuffer := NumSamplesPerBuffer*2 ;
        { Move destination file pointer to end of file header of CDR file }
        CdrFH.FilePointer := FileSeek(CdrFH.FileHandle,CdrFH.NumBytesInHeader,0) ;

        while (not EOF(F)) do begin

          { Read in a row of text }
          ReadLn( F,s ) ;
          { Extract samples from  row }
          nCols := ExtractListOfFloats( s, Values, False ) ;
          nCols := MinInt( [nCols,ImpText.LastColumn+1] ) ;

          { If at end of file, put last samples into buffer and request save }
          if EOF(F) then begin
             for Col := 0 to nCols-1 do if (Col <> ImpText.TCol) then begin
                 Buf^[nSamples] := Trunc( Values[Col]*Scale[Col] ) ;
                 Inc(nSamples) ;
                 end ;
             end
          else begin
               { Normal update of binary data buffer }
               for Col := 0 to nCols-1 do if (Col <> ImpText.TCol) then begin
                   Buf^[nSamples] := Trunc( Values[Col]*Scale[Col] ) ;
                   Inc(nSamples) ;
                   end ;
               end ;

          { Copy to file when buffer is full }
          if (nSamples >= NumSamplesPerBuffer) then begin
             nw := FileWrite( CdrFH.FileHandle, Buf^, NumBytesPerBuffer ) ;
             CdrFH.NumSamplesInFile := CdrFH.NumSamplesInFile + NumSamplesPerBuffer ;
             nSamples := 0 ;
             end ;

          end ;

        SaveCDRHeader(CdrFH) ;
        { Close and re-open WCD file }
        FileClose( CdrFH.FileHandle ) ;
        CdrFH.FileHandle := FileOpen( CdrFH.FileName, fmOpenReadWrite ) ;

        WriteToLogFile( 'ASCII Data File : ' + ImportFrm.FileName ) ;
        WriteToLogFile( 'converted to WCD file : ' + CdrFH.FileName ) ;

     finally
          Dispose(Buf) ;
          Screen.Cursor := crDefault ;
          end ;

     end ;


procedure TImportFrm.ImportBinary ;
{ ----------------------
  Import binary data file
  ----------------------}
const
     NumBytesPerBuf = 512 ;
     NumSamplesperBuf = NumBytesPerBuf div 2 ;
var
   ch,i : Word ;
   Filehandle : Integer ;
   FilePointer,NumBytesCopied : LongInt ;
   Buf : ^TIntArray ;
   Done : Boolean ;
begin
     try
        screen.cursor := crHourglass ;

        { Update parameters settings }
        UpdateImportBinaryParameters ;

        { Create buffer to hold samples }
        New(Buf) ;

        { Close existing WCD data file }
        if CdrFH.FileHandle > 0 then FileClose( CdrFH.FileHandle ) ;
        { Create name of WCD file to hold ASCII data file }
        CdrFH.FileName := ReplaceFileEnding( ImportFrm.FileName, '.WCD' ) ;
        { Create a new WCD file to hold converted data }
        CdrFH.FileHandle := FileCreate( CdrFH.FileName ) ;

        { Number of signal channels }
        CdrFH.NumChannels := ImpBinary.NumChannels ;
        CdrFH.ADCVoltageRange := 1.0 ;

        { Channel calibration and scale factor settings }
        for ch := 0 to CdrFH.NumChannels-1 do begin
            Channel[ch].ChannelOffset := ch ;
            Channel[Ch].ADCScale := 1.0 ;
            Channel[ch].ADCAmplifierGain := 1. ;
            Channel[ch].ADCCalibrationFactor := CdrFH.ADCVoltageRange /
                          ( Channel[ch].ADCScale * (MaxADCValue+1) ) ;
            Channel[ch].ADCZero := 0 ;
            Channel[ch].ADCUnits := '' ;
            Channel[ch].ADCName := Format( 'Ch.%d', [ch] ) ;
            end ;

        { Set sampling interval }
        CdrFH.dt := ImpBinary.dt*ImpBinary.TScale ;

        { Open import file }
        FileHandle := FileOpen( ImportFrm.FileName, fmOpenRead ) ;
        { Move file pointer to start of data in source file }
        FilePointer := FileSeek( FileHandle, ImpBinary.NumBytesInFileHeader, 0 ) ;
        { Move file pointer to start of data in destination file }
        CdrFH.FilePointer := FileSeek( CdrFH.FileHandle, CdrFH.NumBytesInHeader, 0 ) ;

        { Copy samples from binary file into WCD file }
        Done := False ;
        CdrFH.NumSamplesInFile := 0 ;
        NumBytesCopied := 0 ;
        while not Done do begin

            { Read sample data }
            if FileRead(FileHandle,Buf^,NumBytesPerBuf) = NumBytesPerBuf then begin
               { Do scaling and offset }
               for i := 0 to NumSamplesPerBuf-1 do
                   Buf^[i] := Trunc(Buf^[i]*ImpBinary.ScaleBy + ImpBinary.OffsetBy) ;
               NumBytesCopied := NumBytesCopied + NumBytesPerBuf ;
               if NumBytesCopied >= ImpBinary.NumBytesToImport then Done := True ;
               end
            else Done := True ;

            { Write data to file }
            if FileWrite(CdrFH.FileHandle,Buf^,NumBytesPerBuf)
               = NumBytesPerBuf then begin
               CdrFH.NumSamplesInFile := CdrFH.NumSamplesInFile + NumSamplesPerBuf ;
               end
            else begin
               MessageDlg(' File Write failed! ',mtWarning, [mbOK], 0 ) ;
               Done := True ;
               end ;
            end ;

        { Close import file }
        FileClose( FileHandle ) ;
        { Update header information }
        SaveCDRHeader(CdrFH) ;
        { Close and re-open WCD file }
        FileClose( CdrFH.FileHandle ) ;
        CdrFH.FileHandle := FileOpen( CdrFH.FileName, fmOpenReadWrite ) ;

        WriteToLogFile( 'Binary Data File : ' + ImportFrm.FileName ) ;
        WriteToLogFile( 'converted to WCD file : ' + CdrFH.FileName ) ;

     finally
          Dispose(Buf) ;
          screen.cursor := crdefault ;
          end ;

     end ;


procedure TImportFrm.cbTUnitsChange(Sender: TObject);

begin
     UpdateImportTextParameters ;
     end;


procedure TImportFrm.edNumColumnsKeyPress(Sender: TObject; var Key: Char);

begin
     if key = chr(13) then UpdateImportTextParameters ;
     end;


procedure TImportFrm.UpdateImportTextParameters ;
{ -------------------------------------------
  Update ASCII text import parameter settings
  -------------------------------------------}

begin
     { Number of signal channels }
     if ImpText.Initialised then
        ImpText.LastColumn := ExtractInt(edNumColumns.text) - 1 ;
     if ImpText.LastColumn > (ChannelLimit+2) then begin
           MessageDlg( format(' Only %d channels allowed',[ChannelLimit+2]),
                           mtWarning, [mbOK], 0 ) ;
           end ;
     ImpText.LastColumn := MaxInt( [0,MinInt([ChannelLimit+1,ImpText.LastColumn])] ) ;
     edNumColumns.text := format( ' %d ', [ImpText.LastColumn+1] ) ;

     { Sampling interval }
     if ImpText.Initialised then ImpText.dt := ExtractFloat( edDT.text, 1.0 ) ;
     edDT.text := format( ' %.4g %s', [ImpText.dt,cbTUnits.text] ) ;

     { Set time conversion factor }
     if cbTUnits.Items[cbTUnits.ItemIndex] = 's' then ImpText.TScale := 1.0
     else if cbTUnits.Items[cbTUnits.ItemIndex] = 'ms' then ImpText.TScale := 1E-3
     else ImpText.TScale := 1E-6 ;

     { Time column (-1 = no time column) }
     if ImpText.Initialised then ImpText.TCol := ExtractInt(edTimeColumn.text) ;
     if ImpText.LastColumn = 0 then begin
        ImpText.TCol := 0 ;
        edTimeColumn.Enabled := False ;
        end
     else edTimeColumn.Enabled := True ;
     ImpText.TCol := MaxInt( [0,MinInt([ImpText.LastColumn,ImpText.TCol])]) - 1;
     edTimeColumn.text := format( ' %d ', [ImpText.TCol] ) ;

     if ImpText.Initialised then
        ImpText.NumRowsIgnored := ExtractInt( edIgnore.text ) ;
     edIgnore.text := format( ' %d ', [ImpText.NumRowsIgnored] ) ;

     ImpText.Initialised := True ;

     end ;

procedure TImportFrm.UpdateImportBinaryParameters ;
{ -------------------------------------------
  Update binary import parameter settings
  -------------------------------------------}

begin

     { Number of bytes in file header }
     if ImpBinary.Initialised then
        ImpBinary.NumBytesInFileHeader :=  ExtractInt(edNumFileHeaderBytes.text) ;
     ImpBinary.NumBytesInFileHeader := MaxInt( [0,ImpBinary.NumBytesInFileHeader]) ;
     edNumFileHeaderBytes.text := format( ' %d ',
                                  [ImpBinary.NumBytesInFileHeader] ) ;

     if ImpBinary.Initialised then
        ImpBinary.NumBytesToImport :=  ExtractInt(edNumBytesToImport.text)
     else ImpBinary.NumBytesToImport := ImpBinary.NumBytesInFile ;
     ImpBinary.NumBytesToImport := MaxInt( [0,ImpBinary.NumBytesToImport]) ;
     edNumBytesToImport.text := format( ' %d ',[ImpBinary.NumBytesToImport] ) ;
     lbNumBytesToImport.Caption := format('No. bytes to import (%d)',
                                   [ImpBinary.NumBytesInFile]);

     { Number of signal channels }
     if ImpBinary.Initialised then
        ImpBinary.NumChannels := ExtractInt(edNumChannelsBinary.text)
     else ImpBinary.NumChannels := 1 ;
     if ImpBinary.NumChannels > (ChannelLimit+1) then begin
           MessageDlg( format(' Only %d channels allowed',[ChannelLimit+1]),
                           mtWarning, [mbOK], 0 ) ;
           end ;
     ImpBinary.NumChannels := MaxInt( [0,MinInt([ChannelLimit+1,ImpBinary.NumChannels])] ) ;
     edNumChannelsBinary.text := format( ' %d ', [ImpBinary.NumChannels] ) ;

     { Sampling interval }
     if ImpBinary.Initialised then
        ImpBinary.dt := ExtractFloat( edDTBinary.text, ImpBinary.dt )
     else ImpBinary.dt := 1.0 ;
     edDTBinary.text := format( ' %.4g %s', [ImpBinary.dt,cbTUnits.text] ) ;

     { Set time conversion factor }
     if cbTUnitsBinary.Items[cbTUnitsBinary.ItemIndex] = 's' then
        ImpBinary.TScale := 1.0
     else if cbTUnitsBinary.Items[cbTUnitsBinary.ItemIndex] = 'ms' then
        ImpBinary.TScale := 1E-3
     else ImpBinary.TScale := 1E-6 ;

     { Scaling factor }
     if ImpBinary.Initialised then
        ImpBinary.ScaleBy := ExtractFloat( edScaleBy.text, ImpBinary.ScaleBy )
     else ImpBinary.ScaleBy := 1.0 ;
     edScaleBy.text := format( ' %f ', [ImpBinary.ScaleBy] ) ;

     { Offset }
     if ImpBinary.Initialised then
        ImpBinary.OffsetBy := ExtractFloat( edOffsetBy.text, ImpBinary.OffsetBy )
     else ImpBinary.OffsetBy := 0.0 ;
     edOffsetBy.text := format( ' %f ', [ImpBinary.OffsetBy] ) ;

     ImpBinary.Initialised := True ;

     end ;






procedure TImportFrm.edNumFileHeaderBytesKeyPress(Sender: TObject;
  var Key: Char);
begin
     if key = chr(13) then UpdateImportBinaryParameters ;
     end;

procedure TImportFrm.cbTUnitsBinaryChange(Sender: TObject);
begin
     UpdateImportBinaryParameters ;
     end;

end.
