	subroutine quantal_content_analysis(ifile,i_start,i_end)
$include:'wcpcom.for'
c
c	Calculate quantal content and quantal size
c	from series of evoked synaptic currents (or potentials)
c	=======================================================
c	See "Non-linear summation of end-plate potentials in the
c	frog and mouse" E.M. McLachlan & A.R. Martin J. Physiol 311 307-324
c	(1981)
c
	character key

	parameter(nmenu=7)
	character*36 menu(nmenu) /
     &	' Source of nerve-evoked records F1',
     &	' Source of miniature records    F2',
     &	' Record range                   F3',
     &	' Non-linear summation           F4',
     &	' Do quantal analysis            F5',
     &	' Print results                  F6 ',
     &	' Exit                          ESC' /

	parameter(nmenu1=4)
	character*44 menu1(nmenu1) /
     &	' Correct for non-linear summation (Y/N)',
     &	' Membrane potential (mV) ',
     &	' Reversal potential (mV) ',
     &	' Correction factor ' /

	character*6 list(nmenu)
	character*48 title
	logical quit,new_menu

	integer*2 ir_s /0/, ir_e /0/
	real vm /-90./, vr /0./
	character nls_corrected /'N'/
	real f /1./
	character*4 evoked_type / 'EVOK' /
	character*4 mini_type / 'MINI' /
	character*4 failure_type / 'FAIL' /

	character*58 report(20)
	equivalence( iwork(max_points+1), report )

c
c	code
c	====
c
	if( ir_s .le. 0 ) ir_s = 1
	if( ir_e .le. 0 ) ir_e = n_records
	ir_e = min(n_records,ir_e)

	quit = .false.
	do while( .not. quit )

	    new_menu = .true.
	    iop = Iwait_MENU_VERTICAL1(menu,'123456$'
     &	    ,nmenu,2,2,new_menu,ich,' Quantal Analysis ',key)

	    select case ( iop )
	    case( 1 )
c
c		Select channel and record type for evoked records
c
		new_menu = .true.
		call select_channel(3,3+iop,' Channel No. ',ichan)
		call select_record_type( 4,4+ichan+iop, evoked_type )

	    case( 2 )
c
c		Select channel and record type for miniature records
c
		new_menu = .true.
		call select_channel(3,3+iop,' Channel No. ',ichan)
		call select_record_type( 4,4+ichan+iop, mini_type )

	    case( 3 )
c
c		Set range of records to be analysed

		call set_record_range(ir_s,ir_e,3,4+iop,
     &		' Records to be analysed ')

	    case( 4 )
c
c		Non-linear summation correction
c
		i = 1
		list(i) = nls_corrected
		i = i + 1
		write(list(i),'(F6.1)') vm
		i = i + 1
		write(list(i),'(F6.1)') vr
		i = i + 1
		write(list(i),'(f6.2)') f

		title = ' '
10		if(title .eq. ' ') title = ' Non-linear summation '
		call text_window(menu1,list,nmenu1,3,3+iop,title)

		i = 1
		nls_corrected = check_letter(list,'YN',i,title)
		if( title .ne. ' ' ) goto 10

		i = i + 1
		vm = check_limits(list,-200.,200.,i,title)
		if( title .ne. ' ' ) goto 10

		i = i + 1
		vr = check_limits(list,-200.,200.,i,title)
		if( title .ne. ' ' ) goto 10

		i = i + 1
		f = check_limits(list,-200.,200.,i,title)
		if( title .ne. ' ' ) goto 10

	    case( 5 )
c
c		Do quantal analysis
c
		call do_quantal_analysis(ifile,ir_s,ir_e,ichan,
     &		i_start,i_end,
     &		evoked_type,mini_type,failure_type,nls_corrected,
     &		vm,vr,f,report,nlines)

	    case( 6 )
c
c		Print results
c
		call print_results( 3,3+iop, report, nlines )
	    case( 7 )
		quit = .true.
	    end select
	end do

	return
	end

	subroutine do_quantal_analysis(ifile,ir_s,ir_e,ichan,
     &	i_start,i_end,
     &	evoked_type,mini_type,failure_type,nls_corrected,
     &	vm,vr,f,report,nl)
c
c	Do quantal analysis calculations
c
$include:'wcpcom.for'
	integer*2 ifile 	    ! Unit No. of data file
	integer*2 ir_s,ir_e	    ! Start/end of range of records
	integer*2 ichan 	    ! Channel be analysed
	character*(*) evoked_type   ! Record type containing nerve-evoked
	character*(*) mini_type     ! Record type containing minis
	character*(*) failure_type  ! Record type containing minis
	character*(*) nls_corrected ! 'Y' = correct for non-linear summation
	real*4 vm		    ! Membrane potential (mV)
	real*4 vr		    ! Reversal potential (mV)
	real*4 f		    ! nls correction factor
	character*(*) report(1)     ! Results
	integer*2 nl		    ! No. of lines in report

	character*20 string
	character key
	logical mini_data_available
c
c	code
c

	nevoked = 0
	avg_evoked = 0.
	var_evoked = 0.
	skew_evoked = 0.
	nfailures = 0
	nmini = 0
	avg_mini = 0.
	var_mini = 0.
	skew_mini = 0.
	background_variance_evoked = 0.
	background_variance_mini = 0.
	nvar = 0
	vd = vm - vr

	do ir = ir_s,ir_e

	    call get_record( ifile, ir, rec.buf, iwork )
c
c	    Copy selected channel into start of buffer
c
	    j = ichan
	    do i = 1,n_points
		iwork(i) = iwork(j)
		j = j + n_channels
	    end do

	    if( rec.status .eq. 'ACCEPTED' ) then
c
c		Calculate signal variance from portion of
c		signal record used to compute zero level
c
		if( izero_sample(ichan) .gt. 0 ) then
		    i0 = izero_sample(ichan)
		    i1 = i0 + nzero - 1

		    avg = 0.
		    do i = i0,i1
			avg = avg + float(iwork(i))
		    end do
		    avg = avg/float(max(nzero,1))

		    var = 0.
		    do i = i0,i1
			r = float(iwork(i)) - avg
			var = var + r*r
		    end do
		    var = var/float(max(nzero-1,1))
		    var = var*y_scale(ichan)*y_scale(ichan)
		else
		    var = 0.
		end if


C
C		Find largest absolute value within analysis range
C
		iz = iy_zero(ichan)
		imax = 0
		do i = i_start,i_end
		    if( abs(iwork(i) - iz) .gt. abs(imax) ) then
			imax = iwork(i) - iz
		    end if
		end do
		peak = float(imax)*y_scale(ichan)
		rec.results(ichan,ixpeak) = peak

		if( (rec.type .ne. mini_type) .or.
     &		    (evoked_type .eq. 'ALL') ) then

		    nevoked = nevoked + 1

		    if(nls_corrected.eq.'Y') peak =
     &		     peak/(1. - f*abs(peak/vd))

		    if( rec.type .eq. failure_type ) then
			nfailures = nfailures + 1
			peak = 0.
		    end if

		    avg_evoked = peak + avg_evoked

		    background_variance_evoked =
     &		    background_variance_evoked + var

		elseif( rec.type .eq. mini_type ) then
		    nmini = nmini + 1
		    avg_mini = peak + avg_mini
		    background_variance_mini =
     &		    background_variance_mini + var
		endif

		call put_analysis_block( ifile, ir, rec.buf )


		if( mod(ir,10) .eq. 0 ) then
		    call display_progress(2,14,
     &		    ' Averages (Press ESC to abort) ',ir,ir_e,key)
		    if( key .eq. '$' ) goto 101
		end if

	    end if
	end do

	avg_evoked = avg_evoked / float(max(nevoked,1))
	background_variance_evoked = background_variance_evoked /
     &	float(max(nevoked,1))
	avg_mini = avg_mini / float(max(nmini,1))
	background_variance_mini = background_variance_mini /
     &	float(max(nmini,1))


c
c	Calc. variance of evoked signals
c
	if( nevoked .gt. 2 ) then

	    do ir = ir_s,ir_e

		call get_analysis_block( ifile, ir, rec.buf )

		if( rec.status .eq. 'ACCEPTED' ) then

		    peak = rec.results(ichan,ixpeak)

		    if( (rec.type    .ne. mini_type) .or.
     &			(evoked_type .eq. 'ALL')  ) then

			if(nls_corrected.eq.'Y') peak =
     &			 peak/(1. - f*abs(peak/vd))

			r = peak - avg_evoked
			var_evoked = var_evoked + r*r
			skew_evoked = skew_evoked + r*r*r

		    else
			r = peak - avg_mini
			var_mini = var_mini + r*r
			skew_mini = skew_mini + r*r*r
		    endif

		end if

		if( mod(ir,10) .eq. 0 ) then
		    call display_progress(2,14,
     &		    ' Variances (Press ESC to abort) ',ir,ir_e,key)
		    if( key .eq. '$' ) goto 101
		end if

	    end do

c
c	    Calculate variance of evoked records
c
	    var_evoked = var_evoked/float(nevoked-1) -
     &	    background_variance_evoked
c
c	    Calculate skew (3rd moment) of evoked records
c
	    skew_evoked = skew_evoked/float(nevoked-2)

c
c	    Calculate quantal content by variance method
c	    (NB only valid for poisson distributions)
c
	    qc_variance = (avg_evoked*avg_evoked) / var_evoked

	    if( nfailures .gt. 0 )
     &	    qc_failures = log( float(nevoked) / float(nfailures) )

	else
	    qc_variance = 0.
	endif

	mini_data_available = .false.
	if( nmini .lt. 1 ) then
c
c	    If no mini information available from the records
c	    in the data file ask the user.
c
	    CALL QUERY_BOX(2,14,
     &	    ' No data on minis! Enter directly? (Y/N)',key)
	    if( key .eq. 'Y') then
		call get_mini_data(avg_mini,var_mini,
     &		y_units(ichan), 2, 17 )
		mini_data_available = .true.
		mini_type = 'USER'
	    end if
	else
	    if( nmini .gt. 1 ) then
c
c		If more than one mini record in file, calculate
c		the variance and skew
c
		var_mini = var_mini / float(nmini-1) -
     &		background_variance_mini
		skew_mini = skew_mini / float(nmini-2)
	    else
		var_mini = 0.
	    end if
	    mini_data_available = .true.
	end if


c
c ===	If miniature synaptic current/potentials
c	available, also used the direct method to calculate
c	quantal content.
c
	if( mini_data_available ) then
	    qc_direct = avg_evoked / avg_mini
	else
	    qc_direct = 0.
	endif
c
c ---	Compile report on quantal content analysis
c

	nl = 1
	write(report(nl),'(a,'' Records '',i5,''-'',i5)')
     &	file_name(1:len_trim(file_name)),ir_s,ir_e

	nl = nl + 1
	report(nl) = id
c
c	Include details of non-linear summation correction
c	(If it has been used )
c
	if( nls_corrected .eq. 'Y' ) then

	    nl = nl + 1
	    report(nl) = '(Corrected for non-linear summation)'

	    nl = nl + 1
	    write(report(nl),
     &	    '(''Vm: '',f8.2,''mV  Vr: '',f8.2
     &	    ,''mV  Corr. fact. '',f5.2)') vm,vr,f

	endif


	nl = nl + 1
	write(report(nl),'(a4,a24,f8.2,a2,a6,i5)')
     &	evoked_type,' Mean amplitude =',
     &	avg_evoked,y_units(ichan),' n= ',nevoked

	nl = nl + 1
	string = y_units(ichan)
	string(3:4) = '^2'
	write(report(nl),'(a4,a24,f8.2,a4,a6,f8.2)')
     &	' ',' Variance =',var_evoked,string(1:4),' Skew= ',skew_evoked

	nl = nl + 1
	write(report(nl),'(a4,a24,f8.2,a14)')
     &	' ','Quantal content =',qc_variance,' (variance)'

	nl = nl + 1
	write(report(nl),'(a4,a24,f8.2,a14)')
     &	' ','Quantal content =',qc_failures,' (failures)'

c	Results from miniature synaptic signals
c	(if they are present)

	if( mini_data_available ) then

	    nl = nl + 1
	    write(report(nl),'(a4,a24,f8.2,a2,a6,i5)')
     &	    mini_type,' Mean amplitude =',
     &	    avg_mini,y_units(ichan),' n= ',nmini

	    nl = nl + 1
	    string = y_units(ichan)
	    string(3:4) = '^2'
	    write(report(nl),'(a4,a24,f8.4,a4,a6,f10.6)')
     &	    ' ',' Variance =',var_mini,string(1:4),
     &	     ' Skew= ',skew_mini

	    nl = nl + 1
	    write(report(nl),'(a4,a24,f8.2,a14)')
     &	    ' ','Quantal content =',qc_direct,' (direct)'
c
c	    Binomial analysis to get releasable pool (n)
c	    and release probability (p)

	    if( qc_direct .gt. 0. .and. abs(avg_mini).gt.0. ) then
		bp = 1. - var_evoked/(qc_direct*avg_mini*avg_mini)
     &		     + var_mini / (avg_mini*avg_mini)
		bn = qc_direct / bp
	    else
		bp = 0.
		bn = 0.
	    end if

	    nl = nl + 1
	    write(report(nl),'(a4,a24,f6.0,a4,f7.4)')
     &	    ' ','Binomial analysis: n = ',bn,' p= ',bp

	endif

	call display_results( report, 1, nl )

101	continue

	return
	end

	subroutine get_mini_data( avg, var, units, ix, iy )
	real*4 avg	     ! Average mini amplitude (Returned)
	real*4 var	     ! Mini variance (Returned)
	character*(*) units  ! Signal units (In)
	integer*2 ix,iy ! Position of top/left of table on screen (In)
c
c	Get miniature signal average and variance
c
	parameter(nmenu=2)
	character*26 menu(nmenu)
	character*12 list(nmenu)
	character*36 title

	i = 1
	write( menu(i), '('' Average amplitude  ('',a,'')'')') units
	write(list(i),'(f10.3)') avg
	i = i + 1
	write( menu(i), '('' Variance           ('',a,'')^2'')') units
	write(list(i),'(f10.3)') var

	title = ' '
10	if(title .eq. ' ') title = ' Enter miniature data '
	call text_window(menu,list,nmenu,ix,iy,title)

	i = 1
	avg = check_limits(list,-1E30,1E30,i,title)
	if( title .ne. ' ' ) goto 10
	i = i + 1
	var = check_limits(list,0.,1E30,i,title)
	if( title .ne. ' ' ) goto 10

	return
	end
