c	Routines to handle Axon Instrument's Digidata 1200
c	laboratory interface
c	(c) J. Dempster  1994
c	5/6/95 ... set_dacs_digidata now works
c
	subroutine open_digidata( ierr )
$INCLUDE: 'labdigc.FOR'

c
c	Reset all board functions
c
	call outw (iBoard_ResetControl, iRESETWHOLEBOARD )

c	 Set up the 9513 timer chip: master reset. Do I/O in 8-bit mode.
c	 Set to 16-bit mode.

	call OutB (iBoard_TimerControl, iMASTERRESET)
c	call OutB (iBoard_TimerControl, iSET16BITMODE)

c	Point the Data Pointer register at the Master Mode register.

	call outb (iBoard_TimerControl, iMASTERMODE)

c	  Set up the Master Mode register:
c	      15  scaler control = BCD division
c	      14  enable data pointer auto-increment
c	      13  data-bus width = 16 bits
c	      12  FOUT gate OFF
c	    11-8  FOUT divider = divide by 16
c	     7-4  FOUT source = F1 (1 MHz oscillator)
c	       3  comparator 2 disabled
c	       2  comparator 1 disabled
c	     1-0  Time-of-Day disabled

	iword = 2#0110000000000000
	call outb (iBoard_TimerData, iword )
	call outb (iBoard_TimerData, iword/16#100 )

c
c	Configure board to use DMA channels 5(DAC) & 6(ADC) and
c	to use DMA single transfer mode (bit4=1)
c
	call outw( iBoard_InterruptDMAControl, 2#10110 )
c
c	Enable D/A 0 and use COUNTER 1 to control it
c	(Note the use of the common variable iADCDAC to keep
c	 the port settings)
c
	iADCDAC = iADCASYNCDAC .or. iDACCHAN0ENABLE
	call outw (iBoard_ADCDACControl, iADCDAC )
c
c	Clear D/A outputs
c
	call outw( iBoard_DACData, 0 )
c
	ierr = 0
	return
	end

	subroutine set_adc_clock_digidata( dt, itrigger )
$include: 'labdigc.for'
C
C	Set ADC sampling clock
c	----------------------
c	A/D samples are timed using a 16 bit counter fed via dividers
c	from a 0.25us clock. (The digidata 1200's 9513A time channel No.2
c	is used)
C
c	code
c	i
c	Convert A/D sampling period from <dt> (in ms) into
c	clocks ticks, using a clock period which
c	ensures that the number of ticks fits into the 9513A's
c	16 bit counter register.
c
	call check_dt_digidata( dt, iticks, ifreq )

c
c	Set counter No. 2 mode register to:- repeated counts,
c	frequency source period (set by ifreq_source):
c	ACTIVE-HIGH terminal count toggled On/Off pulse
c
c
	imode2 = (ifreq+16#A)*16#100 .or.
     &		 iREPEAT_COUNT .or. iACTIVE_HIGH_TC
c
c	If external triggering requested, set Counter 2 (A/D sampling timer)
c	to be gated by an ACTIVE-HIGH level from the terminal count of
c	Counter 3. Set Counter 3 for a single count, triggered by an
c	ACTIVE-HIGH LEVEL pulse on GATE 3.
c
	if( itrigger .ne. 0 ) then
c
c	    Set up Channel 3 to to a single 2us count when
c	    the GATE 3 goes high and to toggle its OUT 3 line
c	    high when the terminal count is reached
c
	    call outw (iBoard_TimerControl, iDISARM    .or. iCOUNTER3 )
	    call outw (iBoard_TimerControl, iCLEAROUT .or. 3 )

	    imode3 = 16#B00 .or. iTOGGLE .or. iACTIVE_HIGH_LEVEL_GATE
	    call outw (iBoard_TimerControl, iMODE_REG .or. iCTR3_GRP )
	    call outw (iBoard_TimerData, imode3 )
	    call outw (iBoard_TimerControl, iLOAD_REG .or. iCTR3_GRP )
	    call outw (iBoard_TimerData, 3 )
	    call outw (iBoard_TimerControl, iLOADCOUNT .or. iCOUNTER3 )

	    imode2 = imode2 .or. iACTIVE_HIGH_LEVEL_GATE

	endif

c
c	Set Counter 2's mode and load registers and initialise counter
c	(If in External Trigger mode, gate Counter 2 with the GATE 2
c	input,
c
	call outw (iBoard_TimerControl, iMODE_REG .or. iCTR2_GRP )
	call outw (iBoard_TimerData, imode2 )
	call outw (iBoard_TimerControl, iLOAD_REG .or. iCTR2_GRP )
	call outw (iBoard_TimerData, iticks )
	call outw (iBoard_TimerControl, iLOADCOUNT .or. iCOUNTER2 )

	return
	end

	subroutine set_dac_clock_digidata( dt )
$include: 'labdigc.for'
C
C	Set D/A output clock
c	----------------------
c	D/A outputs are timed using a 16 bit counter fed via dividers
c	from a 0.25us clock. (The digidata 1200's 9513A time channel No.1
c	is used)
C
c	code
c
c	Convert A/D sampling period from <dt> (in ms) into
c	clocks ticks, using a clock period which
c	ensures that the number of ticks fits into the 9513A's
c	16 bit counter register.
c
	call check_dt_digidata( dt, iticks, ifreq )

c
c	Set Counter 1's mode and load registers and initialise counter
c
	imode1 = (ifreq+16#A)*16#100 .or.
     &		 iREPEAT_COUNT .or. iACTIVE_HIGH_TC
	call outw (iBoard_TimerControl, iMODE_REG .or. iCTR1_GRP )
	call outw (iBoard_TimerData, imode1 )
	call outw (iBoard_TimerControl, iLOAD_REG .or. iCTR1_GRP )
	call outw (iBoard_TimerData, iticks )
c
c	Note clock does not start yet, ARM command needed
c
	return
	end


	subroutine check_dt_digidata( dt, iticks, ifreq_out )
$include: 'labdigc.for'
C
c	Adjust the floating point value in the a/d sampling interval
c	<dt> so that it matches a valid 9513A clock setting
C
c	code
C
c	Convert A/D sampling period from <dt> (in ms) into
c	clocks ticks, using a clock period which
c	ensures that the no. of ticks fits into the 9513A's
c	16 bit counter register.
c
	scale = 1.
	do ifreq = 1,5
	    ticks = dt / (clock_period*scale)
	    if( ticks .le. 32767. ) goto 11
	    scale = scale*16.
	end do
11	continue
	dt = ticks*scale*clock_period
	iticks = int( ticks )
	ifreq_out = min(ifreq,5)
	return
	end


	subroutine adc_stop_digidata
$include: 'labdigc.for'
c
c	Stop COUNTER 2 which times A/D samples
c
	call outw (iBoard_TimerControl, iDISARM .or. ICOUNTER2 )
	call outw (iBoard_TimerControl, iCLEAROUT .or. 2 )
c
c	Stop A/D conversions
c
	CALL OUTB(IDMA_MASK, ICH6_OFF )
	return
	end

	subroutine adc_to_memory_digidata(dt,n_channels,n_points,
     &	itrigger,ibuffer,istart,imode,adc_range,key_returned)
$INCLUDE: 'labdigc.for'

	INTEGER IBUFFER(1)
C
C	Input arguments
C	DT =         Group sampling period (ms)
C	N_CHANNELS = No. A/D channels to be scanned
C	N_POINTS =   No. of sample groups held by IBUFFER
C	ITRIGGER =   1 Wait for external trigger before starting sampling
C	             0 Start sampling immediately
C	IBUFFER =    DMA Memory buffer to hold ADC samples
C	ISTART =     Starting point in IBUFFER
C			   either 1, or (N_POINTS*N_CHANNELS)/2 + 1
C	IMODE =      0 Wait till IBUFFER is full before returning
C	             1 Return once sampling has started, collect
C	               a single buffer-ful
C	             2 Return once sampling has started, keep
C	               filling IBUFFEr circulerly until told to stop
C	ADC_RANGE =  Input voltage range
C	KEY_RETURNED = Key pressed by user during sampling

	CHARACTER*(*) KEY_RETURNED
	CHARACTER KEY
	LOGICAL  end_of_sweep
	parameter( nranges=4 )
	real*4 range(nranges) / 10., 5., 2.5, 1.25 /

C
C	CODE
C
C	Inter-sample interval is channel group sampling interval
C	divided by number of channels in group. Note that DT1 and DT
C	are modified after SET_ADC_CLOCK_DIGIDATA to be precisely equal to an
C	interval supported by the interface.
C
	dt1 = dt / float(n_channels)
	call set_adc_clock_digidata( dt1, itrigger )
	dt = dt1 * float(n_channels)

c
c
c	Select a gain setting
c
	igain = 0
	do while( range(igain+1) .gt. adc_range )
	    igain = igain + 1
	end do
	adc_range = range(igain+1)
c
c	Program channel gain/select list
c
	iADCDAC = iADCDAC .or. iADCSCANLISTENABLE
	call outw(iBoard_ADCDACControl,iADCDAC)

	do i = 1,n_channels
	    iword = (i-1) .or. 16#100*(i-1) .or. igain*16#2000
	    if( i .eq. n_channels ) iword = iword .or. 16#8000
	    call outw(iBoard_ChannelScanList, iword )
	end do
	iADCDAC = iADCDAC .and. iADCSCANLISTDISABLE
	call outw(iBoard_ADCDACControl,iADCDAC)
c
c	Reset A/D FIFO & scan list pointer (bit 4)
c	      DMA transfer done flag (bit 5)
c	      9513 DONE3 flag (bit 1)
c
	call outw(iBoard_ResetControl, 2#110010 )

C
C --	Set-up DMA controller for transfer to memory --
C
C	Select auto-initialise or single sweep DMA mode
C
	CALL OUTB(IDMA_MASK, ICH6_OFF )
	IF(IMODE.EQ.2) THEN
	    CALL OUTB(IDMA_MODE,IWRITE_MODE6A)	 ! DMA in auto-initialise mode
	ELSE
	    CALL OUTB(IDMA_MODE,IWRITE_MODE6)	 ! DMA in single sweep mode
	ENDIF
	nwords = N_POINTS*N_CHANNELS
	CALL set_dma16_ADDRESS(IBUFFER,ISTART,nwords,6)
	CALL set_dma16_COUNT(nwords-1,6)
	CALL OUTB( IDMA_MASK, ICH6_ON )

c
c	Add marker to buffer if in single sweep modes
c
	if( imode .le. 1 ) call mark_buffer( ibuffer(istart), nwords )

c
c	Start clock if in FREE RUN trigger mode
c	(otherwise wait for pulse on GATE 3)
c
	if( itrigger .ne. 0 ) then
	    call outw (iBoard_TimerControl, iLOADCOUNT .or. ICOUNTER2 )
	    call outw (iBoard_TimerControl, iARM .or. ICOUNTER2 )
c
c	    Enable split-clock mode (bit5) which connects OUT 3 to GATE 2
c
	    iADCDAC = iADCDAC .or. iAdcSplitClockEnable
	    call outw(iBoard_ADCDACControl, iADCDAC )
	    call outw (iBoard_TimerControl, iARM .or. ICOUNTER3 )
	else
	    iADCDAC = iADCDAC .and. iAdcSplitClockDisable
	    call outw(iBoard_ADCDACControl, iADCDAC )
	    call outw (iBoard_TimerControl, iARM .or. ICOUNTER2 )
	end if

C
C	If IMODE = 0 wait here till conversions are completed
C	which is indicated by the terminal count flag ITC in the
C	DMA status register being set

	clock_start = time_in_secs()

	IERR = 0
	if( IMODE .EQ. 0 ) then
	    end_of_sweep = .false.
	    do while( .not. end_of_sweep )
		call check_sweep(
     &		ibuffer(istart), nwords, end_of_sweep, key )
		IF( KEY .NE. CHAR(0) ) KEY_RETURNED = KEY
	    end do

	    call adc_stop_digidata
	    call convert_adc_values_digidata( ibuffer(istart), nwords )

	end if

	return
	end

	subroutine memory_to_dac_digidata( dt, nchan, npoints, ibuffer )
$include:'labdigc.for'
	integer*2 ibuffer(1)

c	Generate a D/A output waveform
c	==============================
c	When one D/A channel is selected (nchan=1), D/A values
c	are output to DAC0 of the Digidata 1200. When 2 channels
c	are selected (nchan=2), D/A values are sent to DAC 0, but
c	the second D/A channel is mapped to DIGITAL OUT 0.
c	(Both DAC0 and DIG0 are encoded into a single 16 bit
c	 output word, so that they are updated synchronously)

c
	integer*4 idac4
c
c	code
c
c
c	Clear D/A FIFO and DONE 1 latch
c
	call outw(iBoard_ResetControl, 2#1001 )

c
c	Set DAC clock period
c
	call set_dac_clock_digidata( dt )

c
c	Pack 2 D/A channels into one word. D/A 0 in upper 12 bits
c	D/A 1 into digital output 0 (bit 0)
c	values > 2048 = 1, values <=2048 = 0
c
	do i = 1,npoints
	    j = (i-1)*nchan + 1
	    idac4 = (ibuffer(j) - 2048)*16
	    if( nchan .gt. 1 ) then
		if( ibuffer(j+1) .gt. 2048 ) idac4 = idac4 .or. 1
	    end if
	    ibuffer(i) = idac4
	end do

c
c	Find part of ibuffer which is contained within
c	a 64Kb memory page and copy data to it, if necessary
c
	CALL OUTB( IDMA_MASK, ICH5_off )
	CALL OUTB(IDMA_MODE,IREAD_MODE5)    ! DMA in single sweep READ mode
	CALL set_dma16_ADDRESS(IBUFFER,ISTART,npoints,5)

	if( istart .ne. 1 ) then
	    j = istart
	    do i = 1,npoints
		ibuffer(j) = ibuffer(i)
		j = j + 1
	    end do
	end if

	CALL set_dma16_COUNT(npoints-1,5)
	CALL OUTB( IDMA_MASK, ICH5_ON )
c
c	Start D/A output
c
	call outw (iBoard_TimerControl, iLOADCOUNT .or. ICOUNTER1 )
	call outw (iBoard_TimerControl, iARM .or. iCOUNTER1 )

	return
	end

	subroutine set_dacs_digidata( vdac, ndacs )
$include:'labdigc.for'
	real vdac(ndacs)
	parameter( bitv = 10.24 / 2048. )
	integer*2 iword
	integer*1 ibyte(2)
	equivalence( iword,ibyte )
	integer*4 iword4
c
c	code
c
c
c	Clear D/A FIFO buffer
c
	call dac_stop_digidata()
c
c	Write to D/A 0 directly, resetting, all digital bits to zero
c
	iword4 = int4( vdac(1)/ bitv )
	iword4 = (iword4 * 16#10) .and. 16#fff0
c
c	Enable DAC #0 and allow direct access to it via I/O port
c
	call outw( iBoard_ADCDACControl, 2#101 )
c
c	Write to DAC 0
c
	call outw( iBoard_DACData, iword4 )

	return
	end

	subroutine dac_stop_digidata
$include:'labdigc.for'
c
c	Stop COUNTER 1 which times D/A updates
c
	call outw (iBoard_TimerControl, iDISARM .or. ICOUNTER1 )
	call outw (iBoard_TimerControl, iCLEAROUT .or. 1 )
c
c	Clear D/A FIFO buffer
c
	call outw(iBoard_ResetControl, iRESETDACFLAGS )
c
c	Disable DMA channel 5
c
	CALL OUTB( IDMA_MASK, ICH5_off )

	return
	end

	SUBROUTINE set_dma16_ADDRESS(IBUF,IS,NWORDS,idma_chan)
$INCLUDE: 'LABdigc.FOR'
C
C	Set DMA controller page and address register to point
C	to a region within IBUF (size = 2*NWORDS) which is completely
C	contained within a 128Kb DMA memory page.
C	IS is returned = 1 or NWORDS+1 depending on which half of
C	IBUF fulfils this condition.
C
	INTEGER IBUF(1),IERR2(2)
	INTEGER*4 ADDRESS_32,OFFSET_32,SEGMENT_32,PAGE_32
C
C	CODE
C
	IS = 1
	CALL CHECK_DMAW(IERR2,IBUF(IS),NWORDS)
	IF(IERR2(1).NE.0) IS = IS + NWORDS
C
C	Find address of IBUF(IS) and split into 128Kb page No. and offset
C
	CALL VARPTR(ISEG,IOFFSET,IBUF(IS))
C
C	Make linear byte address from SEG/OFFSET
C
	SEGMENT_32 = ISEG
	OFFSET_32 = IOFFSET
	ADDRESS_32 = SEGMENT_32*16 + OFFSET_32
C
C	Convert to 16 bit word address
C
	ADDRESS_32 = ADDRESS_32/2
C
C	Make 64Kw DMA PAGE/OFFSET address
C
	PAGE_32 = ADDRESS_32/16#10000
	OFFSET_32 = ADDRESS_32 - PAGE_32*16#10000

C
C
C	Write DMA page register
C	(**** NOTE ******)
C	Although the 16 bit 8237A DMA controller in the AT works
C	in pages on 128Kb, its page register value MUST still be
C	in 64Kb blocks. Therefore PAGE_32 is multiplied by 2.
C	See page 1-14 of the the IBM AT technical reference manual)

	PAGE_32 = PAGE_32*2

	select case (idma_chan)
	case (5)
	    ipage_reg = iDMA_PAGE5
	    iaddr_reg = iDMA_ADDRESS5
	case (6)
	    ipage_reg = iDMA_PAGE6
	    iaddr_reg = iDMA_ADDRESS6
	case (7)
	    ipage_reg = iDMA_PAGE7
	    iaddr_reg = iDMA_ADDRESS7
	end select

	CALL OUTB(ipage_reg,PAGE_32)
	CALL OUTB(IDMA_FFLOP,1)
	CALL OUTB(iaddr_reg,OFFSET_32)
	CALL OUTB(iaddr_reg,OFFSET_32/16#100)

	RETURN
	END

	SUBROUTINE set_dma16_COUNT(NWORDS,idma_chan)
$INCLUDE: 'LABdigc.FOR'
C
C	Set the DMA word transfer counter for 16 bit DMA channel
C
	select case (idma_chan)
	case (5)
	    ireg = iDMA_COUNT5
	case (6)
	    ireg = iDMA_COUNT6
	case (7)
	    ireg = iDMA_COUNT7
	end select

	CALL OUTB(IDMA_FFLOP,1)
	CALL OUTB(ireg, NWORDS )
	CALL OUTB(ireg, NWORDS/16#100 )
	RETURN
	END

	subroutine convert_adc_values_digidata( ibuf, np )
	integer*2 ibuf(np),np
	integer*4 i4
c
c	Convert Digidata A/D values to 0-4095 range
c
	do i = 1,np
	    i4 = ibuf(i)
	    ibuf(i) = (i4/16) + 2048
	end do
	return
	end
