
	TITLE LABat16a
;
;	Single channel spontaneous event detection routine
;	for National Instruments ATMIO-16F
;	Uses 16 bit DMA channel 6
;	(c) J. Dempster April 1994
;
; DMA channel registers
;
BASE_ADDRESS_REG = 0c8H
BYTE_COUNT_REG = 0caH
CLEAR_BYTE_REG = 0d8H


LABdiga   SEGMENT PARA PUBLIC 'CODE'
	ASSUME CS:LABdiga,DS:NOTHING
;
; --	Macros ------------------------
;
SAVE_REGS     MACRO
              PUSH DS
              PUSH BP
              MOV BP,SP
		    SUB SP,8
			MOV AX,0
			MOV [BP]-2,AX
              ENDM
;
RESTORE_REGS  MACRO
              MOV SP,BP
              POP BP
              POP DS
              ENDM
;
UNPACK	      MACRO PNARGS,P1
	      MOV SI,[BP]+(4+4*(PNARGS-P1+1))
	      MOV DS,[BP]+(6+4*(PNARGS-P1+1))
              ENDM

;
ADC_MAX = 2047
ADC_MIN = -2048
LIMITS_MASK = -4
;
;	-- Spontaneous event collection routine ----------------------
;
;	CALL DETECT_EVENT_atmio16f(iwork_BUFFER,IBUFFER_OUT,NSAMPLES,ITRIGGER
;				,NPRE,IESC,IKEY)
;
;	IWORK_BUFFER	Work buffer used for continuous collection
;	IBUFFER_OUT	Contains NSAMPLES of an event after it has
;			been detected.
;	NSAMPLES	No. of samples required
;			NB. IWORK_BUFFER =2*NSAMPLES words
;			IBUFFER_OUT = NSAMPLE words in length
;	ITRIGGER	Event detection trigger level
;	NPRE		No. of pre-trigger samples
;	IKEY -		Returns Function key pressed during routine
;
DETECT_EVENT_atmio16f	     PROC FAR
;
	NARGS = 7
	PUBLIC DETECT_EVENT_atmio16f
;
	SAVE_REGS

;	Set ES = IESC, If ES=1 then pressing ESC key allows escape
;	from detection loop.
;
	UNPACK NARGS   6
	MOV AX,DS:[SI]
	MOV ES,AX

;
;	Get trigger level
;
	UNPACK NARGS   4
	MOV DI,DS:[SI]			;DI = trigger level
;
;	Put address of end of IWORK_BUFFER into SI
;
	UNPACK NARGS   3				;AX = N_SAMPLES
	MOV AX,DS:[SI]
	SAL AX,1				;X4
	SAL AX,1
	DEC AX				;-1

	mov [bp]-4,ax

	UNPACK NARGS   1	;Get addr. of start of IWORK_BUFFER
	ADD SI,AX		;DS:SI = end
;
;	Set initial values of running average (DX) and last sample
;	collected (CX) ADC_MAX for positive going trigger and
;	ADC_MIN for negative going
;
	CMP DI,0
	JGE ADC_DET2
	    MOV DX,ADC_MIN
	    MOV CX,DX
	    JMP ADC_DET3
ADC_DET2:
	    MOV DX,ADC_MAX
	    MOV CX,DX
ADC_DET3:
	    MOV DS:[SI]-1,CX
;
; -- Event detection phase ------------------------------------------
;
;	-----------------------------------------------------
;	Get address of last sample written to memory
;	Use the byte count in BX as a negative offset from the end of 
;	IWORK_BUFFER to find the address of the latest sample
;	-----------------------------------------------------
;	DX = running average, CX = last sample collected
;	DI = trigger level, SI points to end of IWORK_BUFFER
;
	MOV AX,0					;Put a dummy DMA count on stack
	PUSH AX

DETECT:
;
;	Set BX = current contents of DMA word register
;
	OUT CLEAR_BYTE_REG,AL	;Get DMA current count pointer
	IN AL,BYTE_COUNT_REG		;Get lo byte
	MOV BL,AL
	IN AL,BYTE_COUNT_REG		;Get hi byte
	MOV BH,AL

	cmp bl,0
	jz detect

	sal bx,1			;Convert to byte count
	inc bx				; (2 * word count) + 1

;	If DMA count has not changed since last read, keep
;	waiting

	POP AX					;Get old count
	PUSH BX					;save current count
	CMP AX,BX
	JE DETECT					;Keep waiting if count not changed

	INC BX				; +2 to point to
	INC BX				; last sample written
	AND BX,[bp]-4	      ;Keep count within buffer limits
;
;	Get latest sample collected using BX as negative offset
;	from end of IWORK_BUFFER
;
	NEG BX	
	MOV AX,DS:[SI][BX]		;AX = value of last sample
;
;    ------------------------------------------
;	Update running average baseline level (DX)
;	DX = (7*DX + AX)/8
;    ------------------------------------------
;
	MOV CX,AX				;Update latest sample in CX
	MOV AX,BX				;DMA count to AX
	MOV BX,DX				;Old baseline to BX
;
	AND AX,6				;Skip if bits 1,2 of DMA cnt are non-zero
	JNZ ADC_DET15			;so that average only updated every
;						four samples
;
;
	MOV AX,DX				;AX = DX = running average
	SAL AX,1				;AX = DX*8
	SAL AX,1
	SAL AX,1
	SUB AX,DX				;AX = DX*7
	ADD AX,CX				;AX = DX*7 + CX
	SAR AX,1				;AX = AX/8
	SAR AX,1
	SAR AX,1
	MOV DX,AX				;New baseline to DX
	
;
;	Look for keypress, exit if ESC pressed
;	======================================
;
	MOV AX,ES
	CMP AX,1				; test for keypress only if ES=1
	JNE DET_NO_KEY_PRESSED

	MOV AH,0BH		;Has a key been pressed ?
	INT 21H
	CMP AL,0FFH		;AL=FF if a key has been pressed

	JNE DET_NO_KEY_PRESSED
		MOV AH,7H		;Key pressed: Read character
		INT 21H
		MOV AH,0		;Make sure upper byte is zero
		MOV [BP]-2,AX	;Store until exit in stack work area

		CMP AL,27		;Is it ESC
		JNE DET_NO_KEY_PRESSED

;			EXIT if ESC key pressed

			MOV AL,'$'
			POP BX				;Clear stack
			UNPACK NARGS   7		;RETurn IKEY = '$'
			MOV DS:[SI],AX
			RESTORE_REGS
			RET 4*NARGS

DET_NO_KEY_PRESSED:


;
ADC_DET15:
;
;	-------------------------------------------------------------
;	Does this sample exceed trigger level. An event is detected
;	when the difference between the signal level and the baseline
;	level exceeds the trigger level, in the negative direction if
;	the trigger level is negative, and positive if otherwise.
;	-------------------------------------------------------------
;
	MOV AX,CX				;Let AX = latest sample
	SUB AX,BX				;AX = sample - old baseline level
	CMP DI,0				;Positive trigger
	JGE ADC_DET20
		CMP DI,AX			;Negative trigger level test
		JMP ADC_DET30
ADC_DET20:
		CMP AX,DI			;Positive trigger level test
ADC_DET30:
;
	JG COLLECT_START
	JMP DETECT			;Continue detection phase if no event yet
;
;
; -- Event collection phase ------------------------------------------
;
COLLECT_START:

;
;	Get position of DMA byte counter where signal was detected
;
	POP AX
	MOV DX,AX				;Copy into DX
;
;	Get No. of samples to be collected after an event
;	is detected (NSAMPLES - NPRE)
;
	PUSH DS				;Save DS:SI
	PUSH SI
	UNPACK NARGS   3		;CX = NSAMPLES
	MOV CX,DS:[SI]
	UNPACK NARGS   5		; - NPRE
	SUB CX,DS:[SI]
	SAL CX,1			;X2 to get bytes
	POP SI				;Restore DS:SI
	POP DS
	SUB AX,CX			;AX = End of signal record
	AND AX,[bp]-4	      ;Keep count within buffer limits
	PUSH AX 	      ;   = DMA count - No. post-trigger bytes
;
;	Wait for post-trigger bytes to be collected
;	(Enter with AH = last DMA byte count)
;
	MOV AH,DL		;Get DMA lo byte from DX
COLLECT:
	OUT CLEAR_BYTE_REG,AL	;Get latest DMA byte count (lo byte)
	IN AL,BYTE_COUNT_REG
	SUB AH,AL		;AH = No. of bytes transferred
	MOV DL,AH		;DX = AH
	MOV DH,0
	SUB CX,DX		;Subtract no. bytes from loop counter
	JLE COLLECT_END
	MOV AH,AL		;AH = latest DMA lo byte
	JMP COLLECT
;
COLLECT_END:
;
;	------------------------------------------------------
;	Copy 2X NSAMPLES bytes from IWORK_BUFFER to BUFFER_OUT
;	------------------------------------------------------
;
	POP BX				;Get end of signal record from STACK
	INC BX				;Don't use last point
	INC BX				;Don't use last point
	AND BX,0FFFEH			;Make sure count is EVEN
	NEG BX				;Make negative to use as offset
;
	PUSH DS				;Save DS:SI pointer to end of samples
	PUSH SI
;
	UNPACK NARGS   3		;Get NSAMPLES
	MOV CX,DS:[SI]
	SAL CX,1			;X2 to get bytes to transfer
;
	UNPACK NARGS   2		;Get address of BUFFER_OUT
	ADD SI,CX			;add bytes offset
	DEC SI				;and subtract 1
	MOV DI,SI			;DI points to end of BUFFER_OUT
	MOV DX,DS
	MOV ES,DX
	POP SI				;SI points to end of IWORK_BUFFER
	POP DS
;
TRANSFER:
	MOV AL,DS:[SI][BX]		;Get byte from IWORK_BUFFER
	MOV ES:[DI],AL			;Put into IBUFFER_OUT
	DEC DI				;Decrement IBUFFER_OUT pointer
	DEC BX				;Decrement IWORK_BUFFER index
	NEG BX
	AND BX,[bp]-4		   ;Keep within limits
	NEG BX
	LOOP TRANSFER
;
;
;
;	Normal EXIT
;
	UNPACK NARGS   7		;RETurn key pressed by user
	MOV AX,[BP]-2			;Get from stack frame
	MOV DS:[SI],AX

	RESTORE_REGS
	RET 4*NARGS

DETECT_EVENT_atmio16f	     ENDP

get_dma_count_atmio16f	    PROC FAR
;
	NARGS = 1
	PUBLIC get_dma_count_atmio16f
;
	SAVE_REGS

get_count:
	 OUT CLEAR_BYTE_REG,AL	 ;Get DMA current count pointer
	 IN AL,BYTE_COUNT_REG		 ;Get lo byte
	 MOV BL,AL
	 IN AL,BYTE_COUNT_REG		 ;Get hi byte
	 MOV BH,AL
	cmp bl,0
	je get_count

	unpack nargs,1
	MOV DS:[SI],bx

	restore_regs
	ret 4*nargs

get_dma_count_atmio16f	    endp
LABdiga ENDS

END

