unit Zoomwcp;
{ --------------------------------------------------------------------------
  WinCDR - WCP file ZOOM MODULE V1.0  (c) J. Dempster 1997
  Modal form which allows user to zoom in/out of display of recorded signals
  --------------------------------------------------------------------------}
interface

uses WinTypes, WinProcs, Classes, Graphics, Forms, Controls, Buttons,
  StdCtrls, ExtCtrls, Global, Shared, SysUtils, FileIO ;

type
  TZoomWCPFrm = class(TForm)
    pbDisplay: TPaintBox;
    cbChannels: TComboBox;
    Label1: TLabel;
    lbTMin: TLabel;
    lbTMax: TLabel;
    bOK: TButton;
    bCancel: TButton;
    ChannelsGrp: TGroupBox;
    ckInUse0: TCheckBox;
    ckInUse1: TCheckBox;
    ckInUse2: TCheckBox;
    ckInUse3: TCheckBox;
    ckInUse4: TCheckBox;
    ckInUse5: TCheckBox;
    procedure FormActivate(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure pbDisplayMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure pbDisplayMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure pbDisplayMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure cbChannelsChange(Sender: TObject);
    procedure FormDeactivate(Sender: TObject);
    procedure bOKClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormHide(Sender: TObject);
    procedure FormCreate(Sender: TObject);
  private
    { Private declarations }
    procedure HeapBuffers( Operation : THeapBufferOp ) ;
    procedure SetChannelCheckBox( ckInUse : TCheckBox ; ChanNum : Integer ) ;
  public
    { Public declarations }
    ChOnDisplay : LongInt ;
    procedure ZoomOut ;
  end;

var
  ZoomWCPFrm: TZoomWCPFrm;

implementation

{$R *.DFM}

uses MDIform, maths ;

type
TMousePos = ( TopLeft,
              TopRight,
              BottomLeft,
              BottomRight,
              MLeft,
              MRight,
              MTop,
              MBottom,
              MDrag ) ;

var
   ZoomCh : TChannel ;
   ZoomBox : TRect ;
   MousePos : TMousePos ;
   MoveZoomBox : Boolean ;
   XOld,YOld : Integer ;
   RecHeader : ^TWCPRecHeader ;
   ADC : ^TIntArray ;
   BuffersAllocated : boolean ;{ Indicates if memory buffers have been allocated }


procedure TZoomWCPFrm.HeapBuffers( Operation : THeapBufferOp ) ;
{ -----------------------------------------------
  Allocate/deallocation dynamic buffers from heap
  -----------------------------------------------}
begin
     case Operation of
          Allocate : begin
             if not BuffersAllocated then begin
                New(ADC) ;
                New(RecHeader) ;
                BuffersAllocated := True ;
                end ;
             end ;
          Deallocate : begin
             if BuffersAllocated then begin
                Dispose(ADC) ;
                Dispose(RecHeader) ;
                BuffersAllocated := False ;
                end ;
             end ;
          end ;
     end ;


procedure TZoomWCPFrm.FormActivate(Sender: TObject);
{ -------------------------------
  Initialisation when form opens
  ------------------------------}
var
   ch : Longint ;
begin
     { Create buffers }
     HeapBuffers( Allocate ) ;

     { Activate channel in use check boxes }
     { Set channel in use check boxes }
     SetChannelCheckBox( ckInUse0, 0 ) ;
     SetChannelCheckBox( ckInUse1, 1 ) ;
     SetChannelCheckBox( ckInUse2, 2 ) ;
     SetChannelCheckBox( ckInUse3, 3 ) ;
     SetChannelCheckBox( ckInUse4, 4 ) ;
     SetChannelCheckBox( ckInUse5, 5 ) ;

     { Fill channel selection list }
     cbChannels.Clear ;
     for ch := 0 to wcpFH.NumChannels-1 do
          cbChannels.items.add( ' ' + WCPChannel[ch].ADCName ) ;

    { Start with channel 0 selected }
    chOnDisplay := MinInt( [MaxInt([ChOnDisplay,0]),wcpFH.NumChannels-1]) ;
    cbChannels.ItemIndex := ChOnDisplay ;

    { Set scaling for the channel to be worked on }

    ZoomCh := WCPChannel[ChOnDisplay] ;
    ZoomCh.Left := pbDisplay.Width div 50 ;
    ZoomCh.Right := pbDisplay.Width - ZoomCh.Left;
    ZoomCh.Top := pbDisplay.Height div 50 ;
    ZoomCh.Bottom := pbDisplay.Height - ZoomCh.Top ;

    ZoomCh.xMin := 0. ;
    ZoomCh.xMax := wcpFH.NumSamples  ;
    ZoomCh.yMin := MinADCValue  ;
    ZoomCh.yMax := MaxADCValue  ;

    pbDisplay.Refresh ;
    MoveZoomBox := False ;

    end;

procedure TZoomWCPFrm.FormPaint(Sender: TObject);
var
   i,ChOffset,ch : LongInt ;
   x,y,dx : single ;
   xPix,yPix : Integer ;

begin
     { Load record from file }
     GetRecord( WCPFH, RecHeader^, WCPChannel, wcpFH.CurrentRecord, ADC^ ) ;

     Ch := cbChannels.ItemIndex ;
     ChOnDisplay := Ch ;

     { Erase Display }
     pbDisplay.canvas.brush.color := clWhite ;
     pbDisplay.canvas.fillrect(pbDisplay.canvas.ClipRect);

     { Set trace colour }

     pbDisplay.canvas.pen.color := ZoomCh.color ;

     { Set scaling }
     ZoomCh.xScale := (ZoomCh.Right - ZoomCh.Left) / (ZoomCh.xMax - ZoomCh.xMin ) ;
     ZoomCh.yScale := (ZoomCh.Bottom - ZoomCh.Top) / (ZoomCh.yMax - ZoomCh.yMin ) ;

     { Set size of zoom box }

     ZoomBox.Left :=  Trunc((WCPChannel[ch].xMin - ZoomCh.xMin)*ZoomCh.xScale)
                      + ZoomCh.Left ;
     ZoomBox.Right := Trunc((WCPChannel[ch].xMax - ZoomCh.xMin)*ZoomCh.xScale)
                       + ZoomCh.Left ;
     ZoomBox.Top := ZoomCh.Bottom -
                     Trunc((WCPChannel[ch].yMax - ZoomCh.yMin)*ZoomCh.yScale) ;
     ZoomBox.Bottom := ZoomCh.Bottom -
                     Trunc((WCPChannel[ch].yMin - ZoomCh.yMin)*ZoomCh.yScale) ;

     { Display labels }

     lbTMin.caption := Format( '%5.4g %s', [0.,Settings.TUnits] ) ;
     lbTMax.caption := Format( '%5.4g %s',
                               [wcpFH.NumSamples*RecHeader^.dt*Settings.TScale,
                                Settings.TUnits] ) ;
     lbTMin.left := pbDisplay.left ;
     lbTMin.Top := pbDisplay.Top + pbDisplay.Height + 1 ;
     lbTMax.Left := pbDisplay.Left + pbDisplay.width - lbTMax.Width ;
     lbTMax.Top := lbTMin.Top ;

     { Plot channel}

     dx := 1. ;
     x := 0. ;
     ChOffset := WCPChannel[ch].ChannelOffset ;
     for i := 0 to wcpFH.NumSamples-1 do begin

             { Note that channels are stored in the A/D buffer
               in reverse order. i.e. with the highest channel no.
               first (e.g. C3,C2,C1,C0,C3,C2,C1,.... }
             y := ADC^[(i*wcpFH.NumChannels) + ChOffset ] ;
             xPix := Trunc(ZoomCh.xScale*(x - ZoomCh.xMin) + ZoomCh.Left) ;
             yPix := Trunc(ZoomCh.Bottom - ZoomCh.yScale*(y - ZoomCh.yMin));
             if i = 0 then pbDisplay.canvas.moveto(xPix,yPix)
                      else pbDisplay.canvas.lineto(xPix,yPix);
             x := x + dx ;
             end ;

     pbDisplay.canvas.DrawFocusRect( ZoomBox ) ;

     MoveZoomBox := False ;
     end ;


procedure TZoomWCPFrm.pbDisplayMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
     MoveZoomBox := True ;
     end;

procedure TZoomWCPFrm.pbDisplayMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
const
     Margin = 10 ;
     ZoomMin = 10 ;
var
   Height,Width : LongInt ;
begin

     { Find MousePos of mouse relative to Zoom box
       and change mouse cursor shape appropriately }

     if not MoveZoomBox then begin

        if (Abs(X - ZoomBox.Left) + Abs(Y - ZoomBox.Top)) < Margin then begin
            pbDisplay.Cursor := crSizeNWSE ;
            MousePos := TopLeft ;
            end
        else if (Abs(X - ZoomBox.Right) + Abs(Y - ZoomBox.Top)) < Margin then begin
            pbDisplay.Cursor := crSizeNESW ;
            MousePos := TopRight ;
            end
        else if (Abs(X - ZoomBox.Left) + Abs(Y - ZoomBox.Bottom)) < Margin then begin
            pbDisplay.Cursor := crSizeNESW ;
            MousePos := BottomLeft ;
            end
        else if (Abs(X - ZoomBox.Right) + Abs(Y - ZoomBox.Bottom)) < Margin then begin
            pbDisplay.Cursor := crSizeNWSE ;
            MousePos := BottomRight ;
            end
        else if (Abs(X - ZoomBox.Left) < Margin ) and
             (Y <= ZoomBox.Bottom) and (Y >= ZoomBox.Top ) then begin
            pbDisplay.Cursor := crSizeWE ;
            MousePos := MLeft ;
            end
        else if (Abs(X - ZoomBox.Right) < Margin) and
               (Y <= ZoomBox.Bottom) and (Y >= ZoomBox.Top ) then begin
            pbDisplay.Cursor := crSizeWE ;
            MousePos := MRight ;
            end
        else if (Abs(Y - ZoomBox.Top) < Margin) and
              (X <= ZoomBox.Right) and (X >= ZoomBox.Left ) then begin
            pbDisplay.Cursor := crSizeNS ;
            MousePos := MTop ;
            end
        else if (Abs(Y - ZoomBox.Bottom) < Margin ) and
               (X <= ZoomBox.Right) and (X >= ZoomBox.Left ) then begin
            pbDisplay.Cursor := crSizeNS ;
            MousePos := MBottom ;
            end
        else if (ZoomBox.Bottom > Y) and (Y > ZoomBox.Top) and
                (ZoomBox.Right > X) and (X > ZoomBox.Left) then begin
            pbDisplay.Cursor := CrSize ;
            MousePos := MDrag ;
            XOld := X ;
            YOld := Y ;
            end
        else
            pbDisplay.Cursor := crDefault ;

        end
     else if MoveZoomBox then begin

          pbDisplay.canvas.DrawFocusRect( ZoomBox ) ;

          { Move the part of the zoom box which is under the mouse }

          case MousePos of
          TopLeft : Begin
              if (ZoomBox.Bottom-Y) > ZoomMin then ZoomBox.Top := Y ;
              if (ZoomBox.Right-X) > ZoomMin then ZoomBox.Left := X ;
             end ;
          TopRight : Begin
              if (ZoomBox.Bottom-Y) > ZoomMin then ZoomBox.Top := Y ;
              if (X - ZoomBox.Left) > ZoomMin then ZoomBox.Right := X ;
              end ;
          BottomLeft : Begin
              if (ZoomBox.Bottom-Y) > ZoomMin then ZoomBox.Top := Y ;
              if (ZoomBox.Right-X) > ZoomMin then ZoomBox.Left := X ;
              end ;
          BottomRight : Begin
              if (ZoomBox.Bottom-Y) > ZoomMin then ZoomBox.Top := Y ;
              if (X - ZoomBox.Left) > ZoomMin then ZoomBox.Right := X ;
              end ;
          MTop : Begin
              if (ZoomBox.Bottom-Y) > ZoomMin then ZoomBox.Top := Y ;
              end ;
          MBottom : Begin
              if (Y - ZoomBox.Top) > ZoomMin then ZoomBox.Bottom := Y ;
              end ;
          MLeft : Begin
              if (ZoomBox.Right-X) > ZoomMin then ZoomBox.Left := X ;
              end ;
          MRight : Begin
              if (X - ZoomBox.Left) > ZoomMin then ZoomBox.Right := X ;

              end ;
          MDrag : begin
              Width := ZoomBox.Right - ZoomBox.Left ;
              Height := ZoomBox.Bottom - ZoomBox.Top ;
              ZoomBox.Left := MaxInt( [ZoomBox.Left + (X - XOld),ZoomCh.Left]) ;
              ZoomBox.Right := MinInt( [ZoomBox.Left + Width,ZoomCh.Right]) ;
              ZoomBox.Left := ZoomBox.Right - Width ;
              ZoomBox.Top := MaxInt( [ZoomBox.Top + (Y - YOld),ZoomCh.Top]) ;
              ZoomBox.Bottom := MinInt( [ZoomBox.Top + Height,ZoomCh.Bottom]) ;
              ZoomBox.Top := ZoomBox.Bottom - Height ;
              XOld := X ;
              YOld := Y ;
              end
          else
          end ;

          { Keep within bounds }

          ZoomBox.Left :=    MaxInt( [ZoomBox.Left,ZoomCh.Left] ) ;
          ZoomBox.Left :=    MinInt( [ZoomBox.Left,ZoomCh.Right] ) ;
          ZoomBox.Right :=   MaxInt( [ZoomBox.Right,ZoomCh.Left] ) ;
          ZoomBox.Right :=   MinInt( [ZoomBox.Right,ZoomCh.Right] ) ;
          ZoomBox.Top :=     MaxInt( [ZoomBox.Top,ZoomCh.Top] ) ;
          ZoomBox.Top :=     MinInt( [ZoomBox.Top,ZoomCh.Bottom] ) ;
          ZoomBox.Bottom :=  MaxInt( [ZoomBox.Bottom,ZoomCh.Top] ) ;
          ZoomBox.Bottom :=  MinInt( [ZoomBox.Bottom,ZoomCh.Bottom] ) ;

          pbDisplay.canvas.DrawFocusRect( ZoomBox ) ;
          end ;

     end ;

procedure TZoomWCPFrm.pbDisplayMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
     MoveZoomBox := False ;
     end;


procedure TZoomWCPFrm.cbChannelsChange(Sender: TObject);
begin

     WCPChannel[ChOnDisplay].xMin := ((ZoomBox.Left - ZoomCh.Left) / ZoomCh.xScale)
                                  + ZoomCh.xMin ;
     WCPChannel[ChOnDisplay].xMax := ((ZoomBox.Right - ZoomCh.Left) / ZoomCh.xScale)
                                  + ZoomCh.xMin ;
     WCPChannel[ChOnDisplay].yMin := ((ZoomCh.Bottom - ZoomBox.Bottom) / ZoomCh.yScale)
                                  + ZoomCh.yMin ;
     WCPChannel[ChOnDisplay].yMax := ((ZoomCh.Bottom - ZoomBox.Top) / ZoomCh.yScale)
                                  + ZoomCh.yMin ;

     { If the channel number has changed re-plot the display }
     pbDisplay.Refresh ;
     end;

procedure TZoomWCPFrm.FormDeactivate(Sender: TObject);
begin
     HeapBuffers( Deallocate ) ;
     end;

procedure TZoomWCPFrm.ZoomOut ;
{ ---------------------------------------------
  Set display channels to minimum magnification
  ---------------------------------------------}
var
   i : Integer ;
begin
     for i := 0 to wcpFH.NumChannels-1 do begin
         WCPChannel[i].xMin := 0. ;
         WCPChannel[i].xMax := wcpFH.NumSamples-1 ;
         WCPChannel[i].yMin := MinADCValue ;
         WCPChannel[i].yMax := MaxADCValue ;
         end ;
    { Refresh child windows that exist }
     with Main do for i := 0 to MDIChildCount-1 do MDICHildren[i].Refresh ;
    end;

procedure TZoomWCPFrm.bOKClick(Sender: TObject);
var
   ch,NumInUse : LongInt ;
   i : Integer ;
begin

     { Update channels in use }
     WCPChannel[0].InUse := ckInUse0.checked ;
     WCPChannel[1].InUse := ckInUse1.checked ;
     WCPChannel[2].InUse := ckInUse2.checked ;
     WCPChannel[3].InUse := ckInUse3.checked ;
     WCPChannel[4].InUse := ckInUse4.checked ;
     WCPChannel[5].InUse := ckInUse5.checked ;
     { Ensure at least one channel is displayed }
     NumInUse := 0 ;
     for ch := 0 to wcpFH.NumChannels-1 do
         if WCPChannel[ch].InUse then NumInUse := NumInUse + 1;
     if NumInUse = 0 then WCPChannel[0].InUse := True ;

     { Update magnification of selected channel }

     Ch := cbChannels.itemindex ;

     WCPChannel[ch].xMin := ((ZoomBox.Left - ZoomCh.Left) / ZoomCh.xScale)
                           + ZoomCh.xMin ;
     WCPChannel[ch].xMax := ((ZoomBox.Right - ZoomCh.Left) / ZoomCh.xScale)
                           + ZoomCh.xMin ;
     WCPChannel[ch].yMin := ((ZoomCh.Bottom - ZoomBox.Bottom) / ZoomCh.yScale)
                           + ZoomCh.yMin ;
     WCPChannel[ch].yMax := ((ZoomCh.Bottom - ZoomBox.Top) / ZoomCh.yScale)
                           + ZoomCh.yMin ;

     { Make sure X axis scaling is the same for all channels }
     for ch := 0 to wcpFH.NumChannels -1 do begin
         WCPChannel[ch].XMin := WCPChannel[cbChannels.itemindex].xMin ;
         WCPChannel[ch].XMax := WCPChannel[cbChannels.itemindex].xMax ;
         end ;
     { Refresh all open windows }
     with Main do
          for i := 0 to MDIChildCount-1 do MDICHildren[i].Refresh ;

     end;


procedure TZoomWCPFrm.FormShow(Sender: TObject);
begin
     HeapBuffers( Allocate ) ;
     end;

procedure TZoomWCPFrm.FormHide(Sender: TObject);
begin
     HeapBuffers( Deallocate ) ;
     end;

procedure TZoomWCPFrm.FormCreate(Sender: TObject);
begin
     BuffersAllocated := False ;
     end;

procedure TZoomWCPFrm.SetChannelCheckBox( ckInUse : TCheckBox ; ChanNum : Integer ) ;
begin
     if wcpFH.NumChannels > ChanNum then begin
          ckInUse.caption := format('Ch.%d %s',[ChanNum,WCPChannel[ChanNum].ADCName])  ;
          ckInUse.enabled := True ;
          ckInUse.visible := True ;
          ckInUse.checked := WCPChannel[ChanNum].InUse ;
          end
     else begin
          ckInUse.enabled := False ;
          ckInUse.visible := False ;
          ckInUse.checked := False ;
          end ;
     end ;

end.
