unit Rec;
{ ==============================================================================
  WinCDR - Analogue Recording module (c) J. Dempster 1998, All Rights Reserved.
  ==============================================================================}
interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, ExtCtrls, Global, Shared, FileIO, Labio, vds,
  Spin, wavrun, zoomrec ;

type
  TRecordFrm = class(TForm)
    ControlGrp: TGroupBox;
    bRecord: TButton;
    bStop: TButton;
    lbTMin: TLabel;
    lbTMax: TLabel;
    TriggerGrp: TGroupBox;
    rbFreeRun: TRadioButton;
    Timer1: TTimer;
    edStatus: TEdit;
    edIdent: TEdit;
    Label4: TLabel;
    pbDisplay: TPaintBox;
    TimerGrp: TGroupBox;
    edTimeOfDay: TEdit;
    bResetTimer: TButton;
    Bevel1: TBevel;
    edRecordsRequired: TEdit;
    EdRecordDuration: TEdit;
    Label3: TLabel;
    Label1: TLabel;
    rbExtTrigger: TRadioButton;
    edTDisplay: TEdit;
    lbTDisplay: TLabel;
    rbStimulusProgram: TRadioButton;
    cbStimulusProgram: TComboBox;
    Shape1: TShape;
    lbFileSize: TLabel;
    sbRecordDuration: TSpinButton;
    sbRecordsRequired: TSpinButton;
    procedure Timer1Timer(Sender: TObject);
    procedure bRecordClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure bStopClick(Sender: TObject);
    procedure cbPulseProgramChange(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure bResetTimerClick(Sender: TObject);
    procedure rbFreeRunClick(Sender: TObject);
    procedure rbExtTriggerClick(Sender: TObject);
    procedure rbPulseProgramClick(Sender: TObject);
    procedure FormDeactivate(Sender: TObject);
    procedure edIdentKeyPress(Sender: TObject; var Key: Char);
    procedure FormActivate(Sender: TObject);
    procedure pbDisplayDblClick(Sender: TObject);
    procedure pbDisplayMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure edTDisplayKeyPress(Sender: TObject; var Key: Char);
    procedure cbStimulusProgramChange(Sender: TObject);
    procedure sbRecordDurationUpClick(Sender: TObject);
    procedure sbRecordDurationDownClick(Sender: TObject);
    procedure sbRecordsRequiredDownClick(Sender: TObject);
    procedure sbRecordsRequiredUpClick(Sender: TObject);
    procedure EdRecordDurationKeyPress(Sender: TObject; var Key: Char);
    procedure edRecordsRequiredKeyPress(Sender: TObject; var Key: Char);
  private
    { Private declarations }
    procedure DrawHorizontalCursor( var pb : TPaintBox ;
                                    Const Gr : TChannel ;
                                     Level : Integer ) ;
    procedure UpdateStopwatch ;

    procedure HeapBuffers( Operation : THeapBufferOp ) ;
    procedure InitializeDisplay( PB : TPaintBox ) ;
    procedure StartADC ;
    procedure StartDAC ;
    procedure StopRecording ;
    procedure SaveBufferToFile( StartAt, EndAt : Integer ) ;
    procedure DisplayTraces( StartAt, EndAt : Integer ) ;
    procedure SetRecordDuration ;
  public
    { Public declarations }
    TimerBusy : boolean ;
    ReStartADC : Boolean ;
    procedure ShutDownLabInterface ;
  end;

  TState = ( Idle, SaveLoBuffer, SaveHiBuffer, Display, None ) ;

var
  RecordFrm: TRecordFrm;


implementation

{$R *.DFM}

uses MDiform, maths ;
type

    TStopwatch = record
               ElapsedTime : TDateTime ;
               Hour : Word ;
               Minute : Word ;
               Second : Word ;
               MSec : Word ;
               TickStart : single ;
               OldSeconds : single ;
               end ;
     TRecordingState = record
                     Recording : Boolean ;
                     NextSample : Integer ;
                     WaitingForLoBuffer : Boolean ;
                     NumBytesToAcquire : longInt ;
                     NumRecordsToAcquire : LongInt ;
                     x : single ;
                     xyOld : Array[0..ChannelLimit] of TPoint ;
                     end ;
const

     StimDACChannel = 0 ;
     SyncDACChannel = 1 ;
     NumDACChannels = 2 ;
     NumPoints = 2048*6 ;
     EndOfADC = NumPoints-1 ;
     StartOfLoBuf = 0 ;
     EndOfLoBuf = (NumPoints div 2) - 1 ;
     StartOfHiBuf = EndOfLoBuf + 1 ;
     EndOfHiBuf = EndOfADC ;
     NumBytesInBuf = NumPoints ;

var
  ADC : ^TIntArray ; { A/D data buffer }
  ADCHnd : THandle ;
  DAC : ^TIntArray ; { D/A data buffer }
  DACHnd : THandle ;
  SaveBuf : ^TIntArray ; {Save data to file buffer }
  DigBuf : ^TIntArray ; { Digital port output array }
  VProgram : ^TWaveform ; { Voltage program in current use }

  State : TState ;   { Current operational state of this module }
  RecordingState : TRecordingState ;

  NumRecordsDone,EndAtRecord : LongInt ;
  TAcquired,TBlock : single ;
  HeightMin : LongInt ;
  ADCActive : Boolean ; { A/D input sub-system active }
  DACActive : Boolean ; { D/A output sub-system active }
  DigActive : Boolean ; { Digital data output sub-system active }
  NewDigitalPattern : Boolean ;
  Stopwatch : TStopwatch ;
  MouseOverChannel : Integer ;
  BuffersAllocated : boolean ;{ Indicates if memory buffers have been allocated }
  ADCPointer : LongInt ;
  SweepDescription : string ;


procedure TRecordFrm.HeapBuffers( Operation : THeapBufferOp ) ;
{ -----------------------------------------------
  Allocate/deallocation dynamic buffers from heap
  -----------------------------------------------}
var
   Err : Word ;
begin
     case Operation of
          Allocate : begin
             if not BuffersAllocated then begin
                { Create A/D and D/A data buffers }
                ADCHnd := GlobalAlloc( GMEM_FIXED, Sizeof(TIntArray) ) ;
                GlobalFix( ADCHnd ) ;
                ADC := GlobalLock( ADCHnd ) ;
                New(DAC) ;
                { Create buffers for event detection procedures }
                New(SaveBuf) ;
                New(DigBuf) ;
                New(VProgram) ;
                BuffersAllocated := True ;
                end ;
             end ;
          Deallocate : begin
             if BuffersAllocated then begin
                GlobalUnlock( ADCHnd ) ;
                GlobalUnFix( ADCHnd ) ;
                GlobalFree( ADCHnd ) ;
                Dispose( DAC ) ;
                Dispose(SaveBuf) ;
                Dispose(DigBuf) ;
                Dispose(VProgram) ;
                BuffersAllocated := False ;
                end ;
             end ;
          end ;
     end ;


procedure TRecordFrm.FormCreate(Sender: TObject);
{ ------------------------------------
  Initialisation when form is created
  ----------------------------------- }

begin

     { Allocate memory buffers }
     BuffersAllocated := False ;
     HeapBuffers( Allocate ) ;

     { Select which kind of lab. interface hardware is to be used }
     SelectLaboratoryInterface( Settings.LaboratoryInterface ) ;

     { Set size of window }
     HeightMin := TimerGrp.Top + TimerGrp.Height + 10 ;

     Caption := 'Record ' + CdrFH.FileName ;
     edIdent.text := CdrFH.IdentLine ;
     ADCActive := False ;
     DACActive := False ;
     DigActive := False ;
     RestartADC := False ;

     edTDisplay.text := format( ' %6.1f s', [Settings.DisplayDuration]) ;

     { Set trigger mode }
     if Settings.TriggerMode = 'E' then rbExtTrigger.checked := True
                                   else rbFreeRun.checked := True ;

     { Fill combo box with list of available stimulus programs }
     CreateVProgramList( cbStimulusProgram ) ;
     if Settings.VProgramFileName <> '' then
        cbStimulusProgram.ItemIndex := cbStimulusProgram.Items.IndexOf(
                                 ExtractFileNameOnly(Settings.VProgramFileName))
     else cbStimulusProgram.ItemIndex := 0 ;

     { Force an update to command voltage and digital port settings }
     Settings.UpdateOutputs := True ;

     { Initialise stopwatch }
     Stopwatch.TickStart := GetTickCount*1.0 ;
     Stopwatch.OldSeconds := 0. ;

     { If a CED 1902 is in use, update its settings }
     if Settings.CED1902.InUse then SetCED1902( Settings.CED1902 ) ;

     edRecordsRequired.text := format( ' %d ',[Settings.NumRecordsRequired] ) ;
     SetRecordDuration ;

     edStatus.font.color := clBlack ;
     TimerBusy := False ;
     State := Idle ;
     RecordingState.Recording := False ;
     RecordingState.NextSample := 1 ;
     RecordingState.WaitingForLoBuffer := True ;
     RecordingState.NumRecordsToAcquire := 0 ;
     Timer1.Enabled := True ;

     Main.RecordExists := True ;


     end;


procedure TRecordFrm.Timer1Timer(Sender: TObject);
{ ---------------------
  Timed Event scheduler
  ---------------------}
var
   StartAt,EndAt,n : Integer ;
   Quit : Boolean ;
   NumSamplesIn : LongInt ;
begin

     if not TimerBusy then begin

        TimerBusy := True ;

        { If a re-start requested, shut down interface
          to force a re-start next clock tick }
        if (not RecordingState.Recording ) and RestartADC then begin
           ShutDownLabInterface ;
           RestartADC := False ;
           end ;

        { Start continuous A/D conversion into cyclic buffer }
        if not ADCActive then begin

            { Start A/D conversion }
            StartADC ;

            { Start D/A conversion if in stimulus program mode
             AND recording to disk is enabled }
            if rbStimulusProgram.checked
              and RecordingState.Recording then StartDAC ;

            RecordingState.WaitingForLoBuffer := True ;
            ADCPointer := 0 ;
            RecordingState.x := Channel[0].xMax + 1. ;
            State := Idle ;
            End
        else begin
            { This routine ONLY needed when using CED 1401 }
            GetADCSamples( @ADC^[ADCPointer], NumSamplesIn ) ;
            ADCPointer := ADCPointer + NumSamplesIn ;
            if ADCPointer > EndofADC then ADCPointer := ADCPointer - EndofADC - 1 ;

            { Find block of samples which have been acquired since the last timer tick }
            StartAt := RecordingState.NextSample ;
            n := 0 ;
            While ((ADC^[RecordingState.NextSample] <> EmptyFlag))
                  and ( n < EndOfADC) do begin
                  Inc(RecordingState.NextSample) ;
                  if RecordingState.NextSample > EndOfADC then
                     RecordingState.NextSample := 0 ;
                  Inc(n) ;
                  end ;
            EndAt := RecordingState.NextSample  ;

            { Determine whether a buffer save or a display operation is required }
            if RecordingState.Recording
               and (RecordingState.NumBytesToAcquire <=0) then begin
                  { Stop recording and shut down A/D & D/A conversions }
                  StopRecording ;
                  State := Idle ;

                  if not rbStimulusProgram.checked then begin
                     { IN FREE RUN or EXT TRIGGER mode ...
                       Restart recording if more records to be done }
                     Dec(RecordingState.NumRecordsToAcquire) ;
                     if RecordingState.NumRecordsToAcquire > 0 then bRecord.Click ;
                     end
                  else begin
                     { IN STIMULUS PROGRAM mode ...
                       Re-start recording if another stimulus program is
                       linked to the current one }
                     if FileExists(VProgram^.NextProtocolFileName) then begin
                        cbStimulusProgram.ItemIndex := cbStimulusProgram.Items.IndexOf(
                              ExtractFileNameOnly(VProgram^.NextProtocolFileName)) ;
                        bRecord.Click ;
                        end ;
                     end ;
                  end

            else if RecordingState.WaitingForLoBuffer
                 and (ADC^[EndOfLoBuf] <> EmptyFlag) then begin
                 { *** Save lower half-buffer to file *** }
                 SaveBufferToFile( StartOfLoBuf,EndOfLoBuf ) ;

                 RecordingState.WaitingForLoBuffer := False ;
                 if RecordingState.NextSample <= EndOfLoBuf then
                    RecordingState.NextSample := StartOfHiBuf ;
                 end

            else if (not RecordingState.WaitingForLoBuffer)
                 and (ADC^[EndOfHiBuf] <> EmptyFlag) then begin
                 { *** Save upper half-buffer to file *** }
                 SaveBufferToFile( StartOfHiBuf,EndOfHiBuf ) ;

                 RecordingState.WaitingForLoBuffer := True ;
                 if RecordingState.NextSample >= StartOfHiBuf then
                    RecordingState.NextSample := StartOfLoBuf ;
                 end
            else begin
                 { *** Display latest parts of signal traces *** }
                 DisplayTraces( StartAt, EndAt ) ;
                 end ;

            end ;

        if RestartADC then begin
           ShutDownLabInterface ;
           RestartADC := False ;
           end ;

        UpdateStopwatch ;
        TimerBusy := False ;

        end ;
     end ;


procedure TRecordFrm.StartADC ;
{ ---------------------
  Start A/D conversions
  ---------------------}
var
   Offsets : Array[0..ChannelLimit] of LongInt ;
   ExtTrigger,OK : Boolean ;
   RangeOptions : Array[0..10] of TADCRange ;
   NumOptions,i,ch : Integer ;
begin

     { If the A/D converter voltage range setting is not
       valid for the laboratory interface, close the form and exit. }
     GetADCVoltageRangeOptions( RangeOptions, NumOptions ) ;
     OK := False ;
     for i := 0 to NumOptions-1 do begin
         if Abs(CdrFH.ADCVoltageRange - ExtractFloat(RangeOptions[i],1.0)) <= 0.01 then
            OK := True ;
         end ;
     if not OK then edStatus.text := 'ERROR! A/D converter voltage range not valid ' ;
     

     GetChannelOffsets( Offsets, CdrFH.NumChannels ) ;
     for ch := 0 to CdrFH.NumChannels-1 do begin
         Channel[ch].ChannelOffset := Offsets[ch] ;
         end ;

     Channel[0].xMin := 0. ;
     Channel[0].xMax := Settings.DisplayDuration / CdrFH.dt ;

     InitializeDisplay(pbDisplay) ;


     Channel[0].xScale := pbDisplay.Width /
                          (Channel[0].xMax - Channel[0].xMin) ;

     RecordingState.NextSample := 1 ;

     for ch := 0 to CdrFH.NumChannels -1 do begin
         RecordingState.xyOld[ch].x := Channel[0].Left ;
         RecordingState.xyOld[ch].y := (Channel[ch].Top + Channel[ch].Bottom) div 2 ;
         end ;

     { Define buffer limits }
     RecordingState.WaitingForLoBuffer := True ;

     { If in pulse program or external trigger modes ...
       Request A/D converter to wait for a trigger pulse }
     if rbExtTrigger.checked  then ExtTrigger := True
                              else ExtTrigger := False ;

     { Start A/D converter sampling }
     ADCToMemory( ADC^, CdrFH.NumChannels,
                  (EndOfADC+1) div CdrFH.NumChannels,CdrFH.dt,
                  CdrFH.ADCVoltageRange,ExtTrigger,True) ;
     ADCPointer := 0 ;
     ADCActive := True ;
     end ;


procedure TRecordFrm.StartDAC ;
{ ---------------------
  Start D/A conversions
  ---------------------}
begin
     { Create D/A voltage and digital waveform data }
     CreateWaveform( VProgram^, DAC^, DigBuf^ ) ;

     { Convert to appropriate binary form for laboratory interface }
     ConvertToDACCodes(DAC^,NumDACChannels,VProgram^.EndOfInterval) ;
     { Initiate digital data output (if required) }
     if VProgram^.DigitalInUse then begin
        NewDigitalPattern := True ;
        MemoryToDigitalPort( Handle, DigBuf^, VProgram^.EndOfInterval,
                             DAC^, VProgram^.RecordingStart,
                             VProgram^.DACdt, NewDigitalPattern ) ;
        DIGActive := True ;
        end ;

     { Start D/A conversion (in continuous cycle mode) }
     MemoryToDAC( DAC^,NumDACChannels,VProgram^.EndofInterval,
                  VProgram^.DACdt, 0, DACActive ) ;
     DACActive := True ;
     end ;


procedure TRecordFrm.SaveBufferToFile( StartAt, EndAt : Integer ) ;
{ -------------------------------------------------------
  Save A/D samples between StartAt and EndAt to data file
  -------------------------------------------------------}
var
   OverFlow : Boolean ;
begin
     { Overflow test }
     if (ADC^[EndOfHiBuf] <> EmptyFlag)
        and (ADC^[EndOfLoBuf] <> EmptyFlag) then OverFlow := True
                                            else OverFlow := False ;

     { Copy A/D samples from circular buffer into save buffer.
       Samples are converted to standard 12 bit integer and replaced
       with the empty data flag }
     CopyADCNativeTo12Bit( StartAt, EndAt, EmptyFlag, ADC^, SaveBuf^ ) ;

     { Write data to file }
     if RecordingState.Recording then begin
        if FileWrite(CdrFH.FileHandle,SaveBuf^,NumBytesInBuf) <> NumBytesInBuf then
           edStatus.Text := 'ERROR: File write operation failed!'
        else RecordingState.NumBytesToAcquire := RecordingState.NumBytesToAcquire
                                                 - NumBytesInBuf ;
        CdrFH.NumSamplesInFile := CdrFH.NumSamplesInFile +
                                  (NumBytesInBuf div 2) ;
        TAcquired := TAcquired + TBlock ;

        if OverFlow then
           edStatus.text := SweepDescription + format(' %.1f of %.1f s Overflow',
                                  [TAcquired,Settings.RecordDuration])
        else
           edStatus.text := SweepDescription + format(' %.1f of %.1f s',
                                  [TAcquired,Settings.RecordDuration]) ;

        end ;
     if OverFlow then edStatus.text := 'Overflow' ;
     end ;


procedure TRecordFrm.DisplayTraces( StartAt, EndAt : Integer ) ;
{ ----------------------------------------------------
  Display traces of the A/D samples in the ADC^ buffer
  between StartAt and EndAt
  ----------------------------------------------------}
var
   nPoints,nSkip,ch,i,ich,n,nDone : Integer ;
   xKeep,dx : single ;
   OK : Boolean ;
   xy : Array[0..100] of TPoint ;
begin
     { Determine how many points are available to be displayed
       and how many should be skipped due to time limitations }

     nPoints := EndAt - StartAt - CdrFH.NumChannels ;
     if nPoints > 0 then begin
        EndAt := EndAt - 1 ;
        if EndAt < 0 then EndAt := EndOfADC ;
        nSkip := MaxInt([(nPoints div (10*CdrFH.NumChannels)),1])*CdrFH.NumChannels ;

     { Display the points }


        { Erase display once sweep has started (if required) }
        if RecordingState.x > Channel[0].xMax then begin
           EraseDisplay( pbDisplay ) ;
           InitializeDisplay(pbDisplay) ;

           RecordingState.x := Channel[0].xMin ;

           { Draw horizontal zero level cursor(s) }
           for ch := 0 to CdrFH.NumChannels-1 do if Channel[ch].InUse then begin
               DrawHorizontalCursor( pbDisplay,Channel[ch],Channel[ch].ADCZero ) ;
               RecordingState.xyOld[ch].x := Channel[0].Left ;
               end ;
           end ;

        { Plot latest segment of traces for each channel in use }

        xKeep := RecordingState.x ;
        for iCh := 0 to CdrFH.NumChannels-1 do begin
            { Determine channel number from position of sample }
            ch := Channel[(StartAt+iCh) mod CdrFH.NumChannels].ChannelOffset ;
            if Channel[ch].InUse then begin
               { Start new trace segment from end of last one for this channel }
               xy[0].x := RecordingState.xyOld[ch].x ;
               xy[0].y := RecordingState.xyOld[ch].y ;
               { Step through the available samples for this channel
                 and create a line for plotting }
               n := 1 ;
               nDone := 0 ;
               RecordingState.x := xKeep ;
               dx := nSkip div CdrFH.NumChannels ;
               i := StartAt + iCh ;
               repeat
                  {Scale to paintbox coordinates}
                  xy[n].x := Trunc(RecordingState.x*Channel[ch].xScale) ;
                  xy[n].y := Channel[ch].Bottom - Trunc( Channel[ch].yScale*
                                                 ( ADCNativeTo12Bit(ADC^[i])
                                                  - Channel[ch].yMin) );
                  Inc(n) ;
                  nDone := nDone + nSkip ;
                  RecordingState.x := RecordingState.x + dx ;
                  i := i + nSkip ;
                  if i > EndOfADC then i := i - EndOfADC - 1 ;
                  until (nDone >= nPoints) or (n > High(xy) )  ;

               { Plot trace segment in paintbox }
               OK := Polyline( pbDisplay.Canvas.Handle, xy, n-1 ) ;
               { Save coordinates of end of this trace segment }
               RecordingState.xyOld[ch].x := xy[n-1].x ;
               RecordingState.xyOld[ch].y := xy[n-1].y ;
               end ;
            end ;
        end ;
     end ;


procedure TRecordFrm.StopRecording ;
{ ----------------------
  Stop recording to disk
  ----------------------}
begin
     bRecord.Enabled := True ;
     bStop.Enabled := False ;

     { Save data to file header }
     SaveCDRHeader( CdrFH ) ;

     if Main.ViewCDRChildExists then Main.ViewCDRChild.NewFile := True ;
     edStatus.font.color := clBlack ;
     edStatus.Text := ' Done' ;
     RecordingState.Recording := False ;

     { Shut down lab. interface and re-start
       (This turns off any stimulus pattern that is in progress) }
     ShutDownLabInterface ;

     { Indicate when recording stopped on log file }
     WriteToLogFile( SweepDescription + format(
                    ' Stopped at %.4g s',[(CdrFH.NumSamplesInFile*CdrFH.dt)/
                                           CdrFH.NumChannels])  ) ;

     Main.SetMenus ;
     end ;


procedure TRecordFrm.ShutDownLabInterface ;
{ ---------------------------------------------
  Shut down A/D, D/A and digital O/P subsystems
  ---------------------------------------------}
begin
     { Disable A/D }
     if ADCActive then begin
        StopADC ;
        ADCActive := False ;
        end ;
     { Disable D/A waveform generation }
     if DACActive then begin
        { Return voltage command to holding voltage and Sync. O/P to 5V }
        StopDAC ;
        WriteOutputPorts( Settings.VCommand.HoldingVoltage *
                          Settings.VCommand.DivideFactor,
                          Settings.DigitalPort.Value ) ;
        DACActive := False ;
        end ;
     { Disable digital waveform generation }
     if DIGActive then begin
        StopDIG( Settings.DigitalPort.Value ) ;
        DIGActive := False ;
        end ;

     end ;


procedure TRecordFrm.DrawHorizontalCursor( var pb : TPaintBox ;
                                Const Gr : TChannel ;
                                Level : Integer ) ;
{ -----------------------------------------------
  Draw dotted horizontal cursor at ADC level 'Level'
  in display area defined by  record data channel 'Gr'
  ----------------------------------------------------}
var
   yPix,xPix,TicSize : Integer ;
   OldColor : TColor ;
   OldStyle : TPenStyle ;
   OldMode : TPenMode ;
begin
     with pb.canvas do begin
          OldColor := pen.color ;
          OldStyle := pen.Style ;
          OldMode := pen.mode ;
          pen.mode := pmXor ;
          pen.color := clRed ;
          end ;

     yPix := Trunc( Gr.Bottom - (Level - Gr.yMin)*Gr.yScale ) ;
     pb.canvas.polyline([Point(Gr.Left,yPix),Point(Gr.Right,yPix)]);

     { if zero level has been calculated from record show the point used }

     if (Gr.xMin <= Gr.ADCZeroAt) and (Gr.ADCZeroAt <= Gr.xMax ) then begin
          xPix := Trunc((Gr.ADCZeroAt - Gr.xMin)*Gr.xScale - Gr.Left) ;
          TicSize := pb.Canvas.textheight('X') ;
          pb.canvas.polyline( [Point( xPix, yPix+TicSize ),
                               Point( xPix, yPix-TicSize )] );
          end ;

     with pb.canvas do begin
          pen.style := OldStyle ;
          pen.color := OldColor ;
          pen.mode := OldMode ;
          end ;
    end ;


procedure TRecordFrm.bRecordClick(Sender: TObject);
{ -----------------------
  Start recording session
  ----------------------- }
var
   FilePointer,OldPointer : LongInt ;
begin

     bRecord.Enabled := False ;
     bStop.Enabled := True ;
     edStatus.font.color := clRed ;

     { Set number of recording sweeps to be collected }
     Settings.NumRecordsRequired := ExtractInt( edRecordsRequired.text ) ;
     { Update number of recording sweeps to be done }
     if RecordingState.NumRecordsToAcquire = 0 then
        RecordingState.NumRecordsToAcquire := Settings.NumRecordsRequired ;
        { Note. A non-zero value indicates that a repeated series of
          sweeps are in progress }

     { Number of bytes to be acquired from A/D converter }
     RecordingState.NumBytesToAcquire := Settings.NumBytesRequired ;

     TAcquired := 0.0 ;
     TBlock := (NumPoints div ( 2*CdrFH.NumChannels )) * CdrFH.dt ;

     { Load stimulus program }
     if rbStimulusProgram.checked then begin

        Settings.VProgramFileName := Settings.VProtDirectory
                                     + cbStimulusProgram.text + '.vpr' ;
        if FileExists( Settings.VProgramFileName ) then begin
           LoadVProgramFromFile(Settings.VProgramFileName,VProgram^) ;
           WriteToLogFile( 'Stimulus Program = ' + cbStimulusProgram.text ) ;
           end
        else begin
             MessageDlg( ' No Stimulus Program - Recording Aborted! ',
                         mtWarning, [mbOK], 0 ) ;
             State := Idle ;
             end ;
        { Update status box }
        SweepDescription := ExtractFileName(Settings.VProgramFileName) ;
        if VProgram^.NextProtocolFileName <> '' then
        SweepDescription := SweepDescription + ' ( f/b '
                        + ExtractFileName(VProgram^.NextProtocolFileName) + ') : ';
        end
     else begin
        { Sweep count if in Free Run or Ext. Trigger mode }
        SweepDescription := format( 'Record %d/%d : ', [Settings.NumRecordsRequired-
                                     RecordingState.NumRecordsToAcquire+1,
                                     Settings.NumRecordsRequired] ) ;
        end ;

     edStatus.text := SweepDescription + format( '%.1f of %.1f s',
                              [TAcquired,Settings.RecordDuration]);

     WriteToLogFile( SweepDescription + format(
                    ' Started at %.4g s',[(CdrFH.NumSamplesInFile*CdrFH.dt)/
                                           CdrFH.NumChannels])  ) ;

     { Position data file pointer at end of A/D data }
     FilePointer := (CdrFH.NumSamplesInFile*2) + CdrFH.NumBytesInHeader ;
     FilePointer := FileSeek(CdrFH.FileHandle,FilePointer,0) ;

     { Prewrite at expected end of file to force allocation of disk space
       to file }
     OldPointer := FilePointer ;
     FilePointer := FileSeek(CdrFH.FileHandle,
                    FilePointer+RecordingState.NumBytesToAcquire-1,0) ;
     if FileWrite(CdrFH.FileHandle,ADC^,1) <> 1 then
        MessageDlg( ' Error : Cannot write to data file',mtWarning, [mbOK], 0 ) ;

     FilePointer := FileSeek(CdrFH.FileHandle,OldPointer,0) ;

     { Stop A/D conversions to force a re-start of A/D sampling }
     ShutDownLabInterface ;
     ADCActive := False ;
     RecordingState.Recording := True ;

     end;


procedure TRecordFrm.FormDestroy(Sender: TObject);
{ ------------------------------
  Tidy up when form is destroyed
  ------------------------------}
begin
     { Release memory used by objects created in FORMCREATE}
     HeapBuffers( Deallocate ) ;
     Main.RecordExists := False ;
     ShutDownLabInterface ;
     end;


procedure TRecordFrm.FormPaint(Sender: TObject);
begin
    { if State <> SweepInProgress then State := BlankDisplay ;}
     end;


procedure TRecordFrm.FormResize(Sender: TObject);
{ ------------------------------------------
  Adjust components when display is re-sized
  ------------------------------------------ }
begin
     if ClientHeight < HeightMin then ClientHeight := HeightMin ;

     edStatus.Top := ClientHeight - 10 - EdStatus.Height ;
     lbTMin.Top := edStatus.Top ;
     lbTMax.Top := edStatus.Top ;


     pbDisplay.Height := edStatus.Top - pbDisplay.Top  - 10 ;
     pbDisplay.Width := Width - pbDisplay.Left - 20 ;
     lbTMin.left := pbDisplay.Left ;
     lbTMax.Left := pbDisplay.Left + pbDisplay.Width - lbTMax.width ;

     edStatus.Left := lbTMin.Left + lbTMin.Width + 10 ;

     edTDisplay.Top := EdStatus.Top ;
     edTDisplay.Left := pbDisplay.Left + pbDisplay.Width - edTDisplay.Width ;
     lbTDisplay.Top :=  EdStatus.Top ;
     lbTDisplay.Left := edTDisplay.Left - lbTDisplay.Width - 2 ;
     lbTMax.Visible := False ;

     if ADCActive then begin
        ShutDownLabInterface ;
        ADCActive := False ;
        end ;

     if WindowState = wsMinimized then Timer1.Enabled := False
                                  else Timer1.Enabled := True ;

     end;


procedure TRecordFrm.bStopClick(Sender: TObject);
{ ---------------
  Stop recording
  --------------- }
begin
     { Set number of records to be acquired to zero }
     RecordingState.NumRecordsToAcquire := 0 ;
     { Stop recording to disk and shutdown A/D & D/A conversion }
     StopRecording ;

     end;


procedure TRecordFrm.cbPulseProgramChange(Sender: TObject);
{ ---------------------------
  Load a new voltage program
  --------------------------}

begin
     { Update voltage program file name stored in 'settings' when
       program combo box is changed }

     end;


procedure TRecordFrm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
     { Shut down Lab. interface hardware ... essential to avoid program crash }
     ShutDownLabInterface ;
     Main.SetMenus ;
     Action := caFree ;
     end;


procedure TRecordFrm.UpdateStopwatch ;
{ -------------------------
  Update stop-watch display
  -------------------------}
var
   Seconds : single ;
begin

        Seconds := (GetTickCount - Stopwatch.TickStart) / 1000. ;
        if (Seconds - Stopwatch.OldSeconds) > 1. then begin
           Stopwatch.OldSeconds := Seconds ;
           Stopwatch.Minute := Trunc(Seconds / 60. ) ;
           Stopwatch.Second := Trunc( Seconds - Stopwatch.Minute*60. ) ;
           Stopwatch.Hour := Stopwatch.Minute div 60 ;
           Stopwatch.Minute := Stopwatch.Minute mod 60 ;
           Stopwatch.MSec := 0 ;

           Stopwatch.ElapsedTime := EncodeTime(Stopwatch.Hour,Stopwatch.Minute,
                                            Stopwatch.Second,Stopwatch.Msec ) ;
           edTimeofDay.text := TimetoStr(Time)
                               + format( ' (%d:%d:%d)',[StopWatch.Hour,
                                 StopWatch.Minute,StopWatch.Second]) ;
           end ;
        end ;


procedure TRecordFrm.bResetTimerClick(Sender: TObject);
{ ---------------------------------------
  Reset elapsed time counter on stopwatch
  ---------------------------------------}
begin
     Stopwatch.TickStart := (GetTickCount*1. ) ;
     Stopwatch.OldSeconds := 0. ;

     end;


procedure TRecordFrm.rbFreeRunClick(Sender: TObject);
{ ----------------------------
  Set trigger mode to free run
  ----------------------------}
begin
     Settings.TriggerMode := 'F' ;
     end;


procedure TRecordFrm.rbExtTriggerClick(Sender: TObject);
{ ------------------------------------
  Set trigger mode to External trigger
  ------------------------------------}
begin
     Settings.TriggerMode := 'E' ;
     end;

procedure TRecordFrm.rbPulseProgramClick(Sender: TObject);
{ ------------------------------------------
  Set trigger mode to stimulus pulse program
  ------------------------------------------}
begin
     Settings.TriggerMode := 'P' ;
     end;


procedure TRecordFrm.FormDeactivate(Sender: TObject);
{ ---------------------------------------
  Actions to be taken if form loses focus
  ---------------------------------------}
begin
     { Release lab. interface to avoid conflicts with other windows/programs }
     ShutDownLabInterface ;
     { If we're recording give up (in disgust!!) }
    { bStop.Click ;}
     Timer1.Enabled := False ;
     Main.SetMenus ;
     end;


procedure TRecordFrm.edIdentKeyPress(Sender: TObject; var Key: Char);
{ -------------------------------
  Update identification text line
  -------------------------------}
begin
     { Update ident line if it is changed }
     CdrFH.IdentLine := edIdent.text ;
     { Only save to file if recording is not in progress }
     if bRecord.Enabled then begin
        SaveCDRHeader(CdrFH) ;
        if key = chr(13) then WriteToLogFile( CdrFH.IdentLine ) ;
        end ;
     end;


procedure TRecordFrm.FormActivate(Sender: TObject);
{ --------------------------------------------
  Activities required when form becomes active
  --------------------------------------------}
begin
     Timer1.Enabled := True ;
     end;


procedure TRecordFrm.pbDisplayDblClick(Sender: TObject);
{ -------------------------------
  Activate Zoom In/Out dialog box
  -------------------------------}
var
   i : Integer ;
begin
     if not RecordingState.Recording then begin
        Timer1.Enabled := False ;
         ShutDownLabInterface ;
        ZoomRecFrm.ChOnDisplay := MouseOverChannel ;
        ZoomRecFrm.ShowModal ;
        Timer1.Enabled := True ;
        end ;
     end;


procedure TRecordFrm.pbDisplayMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
{ ----------------------------------------
  Find the channel that the mouse is over
  ---------------------------------------}
var
   ch : Integer ;
begin
     if State = Idle then begin
        { Find the channel that the mouse is over }
        for ch := 0 to CdrFH.NumChannels-1 do
            if (Channel[ch].Bottom >= Y) and (Y >= Channel[ch].top) then
               MouseOverChannel := ch ;
        end ;
     end;

procedure TRecordFrm.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
{ --------------
  Function keys
  -------------}
begin
     case Key of
          { F1 starts recording }
          VK_F1 : if bRecord.enabled then bRecord.Click ;
          { F2 stops recording }
          VK_F2 : if bStop.enabled then bStop.click ;
          end ;
     end;


procedure TRecordFrm.edTDisplayKeyPress(Sender: TObject;
  var Key: Char);
begin
     if key = chr(13) then begin
        Settings.DisplayDuration := ExtractFloat( TEdit(Sender).text,
                                                  Settings.DisplayDuration ) ;
        TEdit(Sender).text := format( ' %6.1f s', [Settings.DisplayDuration] ) ;
        RestartADC := True ;
        end ;
     end;


procedure TRecordFrm.InitializeDisplay( PB : TPaintBox ) ;
{ ----------------------------
  Initialise a display window
  ---------------------------}
var
   Height,ch,cTop,NumInUse : Integer ;
begin

     { Set trace colour }

     PB.canvas.pen.color := Channel[0].color ;

     { Determine number of channels in use and the height
       available for each channel }

     NumInUse := 0 ;
     for ch := 0 to CdrFH.NumChannels-1 do
         if Channel[ch].InUse then NumInUse := NumInUse + 1 ;
     Height := PB.Height div MaxInt( [NumInUse,1] ) ;

     { Define display area for each channel in use }

     cTop := 0 ;
     for ch := 0 to CdrFH.NumChannels-1 do begin
          if Channel[ch].InUse then begin
             Channel[ch].Left := 0 ;
             Channel[ch].Right := PB.width ;
             Channel[ch].Top := cTop ;
             Channel[ch].Bottom := Channel[ch].Top + Height ;
             Channel[ch].xMin := Channel[0].xMin ;
             Channel[ch].xMax := Channel[0].xMax ;
             Channel[ch].xScale := (Channel[ch].Right - Channel[ch].Left) /
                                (Channel[ch].xMax - Channel[ch].xMin ) ;
             Channel[ch].yScale := (Channel[ch].Bottom - Channel[ch].Top) /
                                (Channel[ch].yMax - Channel[ch].yMin ) ;
             cTop := cTop + Height ;
             end ;
          end ;

     lbTMin.caption := Format( '%5.4g %s', [Channel[0].xMin*CdrFH.dt*Settings.TScale,
                                               Settings.TUnits] ) ;
     lbTMax.caption := Format( '%5.4g %s', [Channel[0].xMax*CdrFH.dt*Settings.TScale,
                                               Settings.TUnits] ) ;

     { Display Channel Name(s) }

     for ch := 0 to CdrFH.NumChannels-1 do if Channel[ch].InUse then
         PB.Canvas.TextOut( Channel[ch].Left,
         (Channel[ch].Top + Channel[ch].Bottom) div 2,' ' + Channel[ch].ADCName ) ;

     end ;


procedure TRecordFrm.cbStimulusProgramChange(Sender: TObject);
{ ---------------------------
  Load a new voltage program
  --------------------------}
begin
     { Update stimulus program file name stored in 'settings' when
       program combo box is changed }
     Settings.VProgramFileName := Settings.VProtDirectory
                                  + cbStimulusProgram.text + '.vpr' ;
     if FileExists( Settings.VProgramFileName ) then begin
         LoadVProgramFromFile(Settings.VProgramFileName,VProgram^) ;
         end ;
     end;

procedure TRecordFrm.sbRecordDurationUpClick(Sender: TObject);
begin
     Settings.NumBytesRequired := Settings.NumBytesRequired
                                   + NumBytesInBuf ;
     SetRecordDuration ;
     end;

procedure TRecordFrm.sbRecordDurationDownClick(Sender: TObject);
begin
     Settings.NumBytesRequired := Settings.NumBytesRequired
                                     - NumBytesInBuf ;
     Settings.NumBytesRequired := MaxInt( [Settings.NumBytesRequired,
                                              NumBytesInBuf] ) ;
     SetRecordDuration ;

     end;


procedure TRecordFrm.SetRecordDuration ;
{ --------------------------------------------
  Set the record duration & file size data
  --------------------------------------------}
begin

     Settings.RecordDuration := (Settings.NumBytesRequired
                                 div (2*CdrFH.NumChannels))*CdrFH.dt ;
     edRecordDuration.text := format( ' %.2f s', [Settings.RecordDuration] );
     lbFileSize.caption := format( ' %d Kb',
                           [Trunc((Settings.RecordDuration*CdrFH.NumChannels*2)
                                  /(CdrFH.dt*1024)) ] ) ;
     end ;


procedure TRecordFrm.sbRecordsRequiredDownClick(Sender: TObject);
{ ---------------------------------------------
  Decrement required number of recording sweeps
  ---------------------------------------------}
begin
     Settings.NumRecordsRequired := MaxInt( [Settings.NumRecordsRequired-1,1] ) ;
     edRecordsRequired.text := format(' %d ',[Settings.NumRecordsRequired] ) ;
     end;

procedure TRecordFrm.sbRecordsRequiredUpClick(Sender: TObject);
{ ---------------------------------------------
  Increment required number of recording sweeps
  ---------------------------------------------}
begin
     Settings.NumRecordsRequired := MinInt( [Settings.NumRecordsRequired+1,
                                             High(Settings.NumRecordsRequired)] ) ;
     edRecordsRequired.text := format(' %d ',[Settings.NumRecordsRequired] ) ;
     end;

procedure TRecordFrm.EdRecordDurationKeyPress(Sender: TObject;
  var Key: Char);
{ -------------------------------
  Change recording sweep duration
  -------------------------------}
var
   t :single ;
begin
     if key = chr(13) then begin
        t := ExtractFloat( edRecordDuration.text, Settings.RecordDuration ) ;
        Settings.NumBytesRequired := Trunc((t*2.0*CdrFH.NumChannels)/CdrFH.dt) ;
        Settings.NumBytesRequired := ((Settings.NumBytesRequired div NumBytesInBuf)
                                       + 1) * NumBytesInBuf  ;
        Settings.NumBytesRequired := MaxInt( [Settings.NumBytesRequired,NumBytesInBuf]) ;
        SetRecordDuration ;
        end ;
     end;

procedure TRecordFrm.edRecordsRequiredKeyPress(Sender: TObject;
  var Key: Char);
{ ------------------------------------------
  Change required number of recording sweeps
  ------------------------------------------}
begin
     if key = chr(13) then begin
        Settings.NumRecordsRequired := ExtractInt( edRecordsRequired.text ) ;
        Settings.NumRecordsRequired := MaxInt( [Settings.NumRecordsRequired,1] ) ;
        edRecordsRequired.text := format(' %d ',[Settings.NumRecordsRequired] ) ;
        end ;
     end;

end.
