unit Pwrspec;
{ ======================================================================
  WinCDR (c) J. Dempster, University of Strathclyde, All Rights Reserved
  Variance / Spectral Analysis module
  Computes mean level, variance, skew for blocks of data.
  Computes power spectrum and fits Lorentzians
  Computes MEPC frequency using Segal et al method
  ======================================================================
    25/6/98 Clipboard buffer limit reduced to 31000
    29/6/98 MEPC Frequency button added (and removed from variable list) }

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, StdCtrls, Spin, TabNotBk, ClipBrd,
  global, shared, maths, plotlib, Grids, ssqunit, setfitpa, printers, fileio ;
const
     MaxRecordSize = 8192 ;
type

   TCursor = record
           Selected : Boolean ;
           Active : Boolean ;
           xPix : Integer ;
           xVal : single ;
           index : Integer ;
           Lab : TLabel ;
           end ;

   TSpectrum = record
              RecordNum : LongInt ;
              RecordSize : LongInt ;
              MaxRecord : LongInt ;
              StartAt : Integer ;
              EndAt : Integer ;
              NumAveraged : Integer ;
              NumPoints : Integer ;
              Available : Boolean ;
              Frequency : Array[0..(MaxRecordSize div 2)+1] of single ;
              Power : Array[0..(MaxRecordSize div 2)+1] of single ;
              Equation : TEquation ;
              Variance : single ;
              AvgDCMean : single ;
              MedianFrequency : single ;
              Cursor0 : TCursor ;
              Cursor1 : TCursor ;
              end ;

  TPwrSpecFrm = class(TForm)
    TabbedNotebook: TTabbedNotebook;
    GroupBox1: TGroupBox;
    edRecord: TEdit;
    sbRecord: TScrollBar;
    ckRejected: TCheckBox;
    GroupBox2: TGroupBox;
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    edRecordSize: TEdit;
    spRecordSize: TSpinButton;
    cbACChannel: TComboBox;
    cbDCChannel: TComboBox;
    pbACDisplay: TPaintBox;
    pbDCDisplay: TPaintBox;
    lbTMin: TLabel;
    lbTMax: TLabel;
    Label4: TLabel;
    Label5: TLabel;
    GroupBox3: TGroupBox;
    edMean: TEdit;
    Label6: TLabel;
    Label7: TLabel;
    edVariance: TEdit;
    VarGrp: TGroupBox;
    GroupBox5: TGroupBox;
    Label9: TLabel;
    cbVarYAxis: TComboBox;
    bDoVariance: TButton;
    cbVarXAxis: TComboBox;
    Label8: TLabel;
    GroupBox6: TGroupBox;
    rbVarAllRecords: TRadioButton;
    rbVarRange: TRadioButton;
    EdVarRange: TEdit;
    bVarSetAxes: TButton;
    pbVarDisplay: TPaintBox;
    pbSpecDisplay: TPaintBox;
    specGrp: TGroupBox;
    GroupBox8: TGroupBox;
    bDoSpectrum: TButton;
    GroupBox9: TGroupBox;
    rbSpecAllRecords: TRadioButton;
    rbSpecRange: TRadioButton;
    edSpecRange: TEdit;
    bSpecSetAxes: TButton;
    rbNoWindow: TRadioButton;
    rbCosineWindow: TRadioButton;
    GroupBox10: TGroupBox;
    edNumFreqAveraged: TEdit;
    edVarStatus: TEdit;
    edSpecStatus: TEdit;
    GroupBox11: TGroupBox;
    ckSubtractTrends: TCheckBox;
    ckSpecSubtractBackground: TCheckBox;
    cbRecType: TComboBox;
    Label10: TLabel;
    rbNoFreqAveraging: TRadioButton;
    rbLogFreqAveraging: TRadioButton;
    rbLinFreqAveraging: TRadioButton;
    Label11: TLabel;
    GroupBox12: TGroupBox;
    ckVarSubtractBackground: TCheckBox;
    SpecFitGrp: TGroupBox;
    bFitLorentzian: TButton;
    cbSpecEquation: TComboBox;
    sgSpecResults: TStringGrid;
    lbSpecCursor0: TLabel;
    lbSpecCursor1: TLabel;
    VarFitGrp: TGroupBox;
    bVarFit: TButton;
    cbVarEquation: TComboBox;
    sgVarResults: TStringGrid;
    lbVarCursor0: TLabel;
    lbVarCursor1: TLabel;
    bDataClose: TButton;
    bVarClose: TButton;
    bSpecClose: TButton;
    edRecordOverlap: TEdit;
    Label12: TLabel;
    sbRecordOverlap: TSpinButton;
    bMEPCFrequency: TButton;
    Panel1: TPanel;
    procedure FormShow(Sender: TObject);
    procedure spRecordSizeUpClick(Sender: TObject);
    procedure spRecordSizeDownClick(Sender: TObject);
    procedure edRecordSizeKeyPress(Sender: TObject; var Key: Char);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure pbACDisplayPaint(Sender: TObject);
    procedure sbRecordChange(Sender: TObject);
    procedure pbDCDisplayPaint(Sender: TObject);
    procedure bDoVarianceClick(Sender: TObject);
    procedure pbVarDisplayPaint(Sender: TObject);
    procedure bDoSpectrumClick(Sender: TObject);
    procedure pbSpecDisplayPaint(Sender: TObject);
    procedure bSpecSetAxesClick(Sender: TObject);
    procedure bVarSetAxesClick(Sender: TObject);
    procedure cbRecTypeChange(Sender: TObject);
    procedure ckRejectedClick(Sender: TObject);
    procedure edRecordKeyPress(Sender: TObject; var Key: Char);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure bFitLorentzianClick(Sender: TObject);
    procedure pbSpecDisplayMouseMove(Sender: TObject; Shift: TShiftState;
      X, Y: Integer);
    procedure pbSpecDisplayMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure pbSpecDisplayMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure cbSpecEquationChange(Sender: TObject);
    procedure bVarFitClick(Sender: TObject);
    procedure pbVarDisplayMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure pbVarDisplayMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure pbVarDisplayMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FormResize(Sender: TObject);
    procedure bDataCloseClick(Sender: TObject);
    procedure sbRecordOverlapDownClick(Sender: TObject);
    procedure sbRecordOverlapUpClick(Sender: TObject);
    procedure cbVarEquationChange(Sender: TObject);
    procedure bMEPCFrequencyClick(Sender: TObject);
  private
    { Private declarations }


    procedure HeapBuffers( Operation : THeapBufferOp ) ;
    procedure SetRecordSize( NewSize : Integer ) ;
    procedure InitializeDisplay( const PB : TPaintBox ;
                                 var Chan : TChannel ) ;
    procedure PlotChannel( PB : TPaintBox ;
                           var Chan : TChannel ;
                           var Buf : Array of Integer ) ;
    procedure DrawHorizontalCursor( pb : TPaintBox ;
                                               Const Chan : TChannel ;
                                               Level : Integer ;
                                               Colour : TColor ) ;
    procedure GetRecord ;
    procedure ReadRecord( Rec : Integer ;
                           var Buf : Array of SmallInt ;
                           var xMin,xMax : single ) ;
    procedure CopyRecordToClipBoard ;
    function GetVariable( ItemIndex, Rec : Integer ) : single ;
    function GetVariableUnits( ItemIndex : Integer ) : string ;

    procedure PlotGraph ;
    procedure PrintVarianceGraph ;
    procedure PrintPageTitle( const Canvas : TCanvas ;
                              ResultsAvailable : Boolean ;
                              const Results : TStringGrid ;
                              var YEndOfText : Integer ) ;

    procedure CopyVarianceDataToClipboard ;

    procedure ComputePowerSpectrum( RecType : TRecType ;
                                    StartAt,EndAt : Integer ;
                                    var Spectrum : TSpectrum ) ;
    procedure AverageFrequencies( var Spectrum : TSpectrum ) ;
    procedure SubtractLinearTrend( var Y : Array of single ;
                                   iStart,iEnd : Integer ) ;
    procedure CosineWindow( iStart,iEnd : Integer ;
                            var Y : Array of single ;
                            var VarianceCorrection : single ) ;
    procedure SpectralVariance( var Spectrum : TSpectrum ) ;

    procedure PlotSpectrum ;
    procedure PrintSpectrum ;
    procedure CopySpectrumDataToClipBoard ;

    procedure DrawVerticalCursor( var pb : TPaintBox ;
                                  var Plot : TPlot ;
                                  X : single ;
                                  const Lab : TLabel ) ;
    procedure MoveVerticalCursor( var pb : TPaintbox ;
                                          var Plot : TPlot ;
                                          var Cursor : TCursor ;
                                          XPix : Integer ) ;

  public
    { Public declarations }
    procedure PrintGraph ;
    procedure CopyDataToClipboard ;
  end;

var
  PwrSpecFrm: TPwrSpecFrm;

implementation

uses mdiform,setaxes,SetPage,MEPCFreq ;

const
     RecordLimit = 16383 ;
     MinRecordSize = 32 ;
     vRecordNum = 0 ;
     vTime = 1 ;
     vMeanDC = 2 ;
     vStDev = 3 ;
     vVariance = 4 ;
     vSkew = 5 ;
     vMedianFrequency = 6 ;

     DataPage = 0 ;
     VariancePage = 1 ;
     SpectrumPage = 2 ;

type
    TPointArray = Array[0..MaxRecordSize] of TPoint ;

    TData = record
          RecordNum : LongInt ;
          RecordSize : LongInt ;
          RecordOffset : LongInt ;
          RecordOverlap : LongInt ;
          MaxRecord : LongInt ;
          end ;

    TVariance = record
              RecordNum : LongInt ;
              RecordSize : LongInt ;
              MaxRecord : LongInt ;
              StartAt : LongInt ;
              EndAt : LongInt ;
              nPoints : LongInt ;
              Available : Boolean ;
              Equation : TEquation ;
              Cursor0 : TCursor ;
              Cursor1 : TCursor ;
              NewPlot : Boolean ;
              end ;

var
   Data : TData ; { Data description record }
   { Power spectral data arrays }
   PowerSpectrum : ^TSpectrum ;
   BackgroundSpectrum : ^TSpectrum ;
   TempSpectrum : ^TSpectrum ;
   Variance : TVariance ; { Variance analysis control record }
   ADC : ^TIntArray ; { A/D sample data array }
   xy : ^TPointArray ; { Plotting data array }
   DCMean : ^TSingleArray ; { DC (mean) signal level array }
   ACVariance : ^TSingleArray ; { AC channel signal variance array }
   ACSkew : ^TSingleArray ; { AC channel signal skew array }
   MedFreq : ^TSingleArray ; { AC channel median power frequency }
   RecordStatus : ^TRecordStatusArray ; { Record accepted/rejected/type array }

   BuffersAllocated : Boolean ;
   ACChannel : TChannel ;
   DCChannel : TChannel ;
   VarPlot : TPlot ; { Signal mean/variance plot }
   SpecPlot : TPlot ; { Power spectrum plot }

{$R *.DFM}


procedure TPwrSpecFrm.HeapBuffers( Operation : THeapBufferOp ) ;
{ -----------------------------------------------
  Allocate/deallocation dynamic buffers from heap
  -----------------------------------------------}
begin
     case Operation of
          Allocate : begin
             if not BuffersAllocated then begin
                New(ADC) ;
                New(xy) ;
                New(DCMean) ;
                New(ACVariance) ;
                New(ACSkew) ;
                New(MedFreq) ;
                New(RecordStatus) ;
                New(PowerSpectrum) ;
                New(BackgroundSpectrum) ;
                New(TempSpectrum) ;
                BuffersAllocated := True ;
                end ;
             end ;
          Deallocate : begin
             if BuffersAllocated then begin
                Dispose(ADC) ;
                Dispose(xy) ;
                Dispose(DCMean) ;
                Dispose(ACVariance) ;
                Dispose(ACSkew) ;
                Dispose(MedFreq) ;
                Dispose(RecordStatus) ;
                Dispose(PowerSpectrum) ;
                Dispose(BackgroundSpectrum) ;
                Dispose(TempSpectrum) ;
                BuffersAllocated := False ;
                end ;
             end ;
          end ;
     end ;


procedure TPwrSpecFrm.FormShow(Sender: TObject);
{ --------------------------------------
  Initialisations when form is displayed
  --------------------------------------}
var
   i,ch : Integer ;
   FileName : string ;
   FileHandle : Integer ;
begin
     { Allocate working buffers from heap }
     HeapBuffers( Allocate ) ;

     { Make sure no more forms can be created }
     Main.mnPwrSpec.Enabled := False ;

     { Fill AC and DC channel selection lists }
     cbACChannel.Clear ;
     cbDCChannel.Clear ;
     for ch := 0 to CdrFH.NumChannels-1 do begin
          cbACChannel.items.add( format('Ch.%d %s',[ch,Channel[ch].ADCName]) ) ;
          cbDCChannel.items.add( format('Ch.%d %s',[ch,Channel[ch].ADCName]) ) ;
          end ;
     cbACChannel.ItemIndex := 0 ;
     cbDCChannel.ItemIndex := 0 ;
     ACChannel := Channel[cbACChannel.ItemIndex] ;
     DCChannel := Channel[cbDCChannel.ItemIndex] ;

     { Set size of variance/FFT record }
     Data.RecordOverlap := Settings.Variance.RecordOverlap ;
     Data.RecordSize := Settings.Variance.RecordSize ;
     SetRecordSize( Data.RecordSize ) ;

     { Initial axis variable selections for variance plot }
     cbVarXAxis.ItemIndex := vRecord ;
     cbVarYAxis.ItemIndex := vVariance ;

     { Clear display areas }
     EraseDisplay( pbACDisplay ) ;
     EraseDisplay( pbDCDisplay ) ;

     sbRecord.Position := 0 ;
     Data.RecordNum := sbRecord.Position + 1 ;

     { Initial settings for variance/mean fitted curve }
     Variance.Available := False ;
     Variance.Equation.Available := False ;
     Variance.Cursor0.Lab := lbVarCursor0 ;
     Variance.Cursor0.Active := False ;
     Variance.Cursor1.Lab := lbVarCursor1 ;
     Variance.Cursor1.Active := False ;
     { Set initial variance/mean equation to None }
     cbVarEquation.ItemIndex := 0 ;

     { Initial settings for power spectrum fitted curve }
     PowerSpectrum^.Available := False ;
     PowerSpectrum^.Equation.Available := False ;
     PowerSpectrum^.Cursor0.Lab := lbSpecCursor0 ;
     PowerSpectrum^.Cursor0.Active := False ;
     PowerSpectrum^.Cursor1.Lab := lbSpecCursor1 ;
     PowerSpectrum^.Cursor1.Active := False ;
     { Set initial power spectrum equation to None }
     cbSpecEquation.ItemIndex := 0 ;

     VarPlot.xAxis.Log := False ;
     VarPlot.yAxis.Log := False ;
     SpecPlot.xAxis.Log := True ;
     SpecPlot.yAxis.Log := True ;

     bMEPCFrequency.Enabled := False ;

     { Load record status data }
     FileName := ReplaceFileEnding( CdrFH.FileName, '.REC' ) ;
     if FileExists( FileName ) then begin
        { Load record status data from file }
        FileHandle := FileOpen( FileName, fmOpenReadWrite ) ;
        if FileRead( FileHandle, RecordStatus^, SizeOf(TRecordStatusArray) )
           <> SizeOf(TRecordStatusArray) then
           MessageDlg( 'Error reading record status data',
                        mtWarning, [mbOK], 0 ) ;
        FileClose( FileHandle ) ;
        end
     else begin
          { Initialise new record status array }
          for i := 0 to High(RecordStatus^) do begin
              RecordStatus^[i].Valid := True ;
              RecordStatus^[i].RecType := Test ;
              end ;
          end ;

     GetRecord ;

     end;

procedure TPwrSpecFrm.PrintGraph ;
{ -----------------------------------
  Print the currently displayed graph
  -----------------------------------}
begin
     if (TabbedNotebook.PageIndex = VariancePage) and Variance.Available then begin
        { Variance/mean plot }
        SetPageFrm.ShowModal ;
        if (SetPageFrm.ModalResult = mrOK) then PrintVarianceGraph ;
        end
     else if (TabbedNotebook.PageIndex = SpectrumPage)
        and PowerSpectrum^.Available then begin
        { Power spectrum plot }
        SetPageFrm.ShowModal ;
        if (SetPageFrm.ModalResult = mrOK) then PrintSpectrum ;
        end ;
     end ;


procedure TPwrSpecFrm.CopyDataToClipboard ;
{ -----------------------------------------------------------
  Copy the data in currently displayed graph to the clipboard
  -----------------------------------------------------------}
begin
     if (TabbedNotebook.PageIndex = VariancePage) and Variance.Available then begin
        { Variance/mean plot }
        CopyVarianceDataToClipboard ;
        end
     else if (TabbedNotebook.PageIndex = SpectrumPage)
        and PowerSpectrum^.Available then begin
        CopySpectrumDataToClipboard ;
        end
     else CopyRecordToClipboard ;
     end ;


{ *******************************************************
  Data record display procedures
  *******************************************************}

procedure TPwrSpecFrm.PlotChannel( PB : TPaintBox ;
                                   var Chan : TChannel ;
                                   var Buf : Array of Integer ) ;
{ -----------------------------------
  Plot a record from a signal channel
  -----------------------------------}
var
   i,j,n,ch : Integer ;
   x : single ;
   OK : Boolean ;
begin
     { Clear display area }
     InitializeDisplay( PB, Chan ) ;
     EraseDisplay( PB ) ;

     n := 0 ;
     x := Chan.xMin ;
     for i := 0 to Data.RecordSize-1 do begin
         j := (i*CdrFH.NumChannels) + Chan.ChannelOffset ;
         xy^[n].y := Chan.Bottom - Trunc(
                     Chan.yScale*(Buf[j] - Chan.yMin));
         xy^[n].x := Trunc(Chan.xScale*(x - Chan.xMin) + Chan.Left) ;
         Inc(n) ;
         x := x + 1.0 ;
         end ;
     OK := Polyline( PB.Canvas.Handle, xy^, n ) ;

     DrawHorizontalCursor( PB , Chan, Chan.ADCZero, clRed ) ;
     end ;

procedure TPwrSpecFrm.InitializeDisplay( const PB : TPaintBox ;
                                            var Chan : TChannel ) ;
{ ----------------------------
  Initialise a display window
  ---------------------------}
var
   ch,cTop : Integer ;
begin

     { Set trace colour }

     PB.canvas.pen.color := Channel[0].color ;

     { Define display area for each channel in use }

     Chan.Left := 0 ;
     Chan.Right := PB.width ;
     Chan.Top := 0 ;
     Chan.Bottom := Chan.Top + PB.Height ;
     Chan.xScale := (Chan.Right - Chan.Left) / (Chan.xMax - Chan.xMin ) ;
     Chan.yScale := (Chan.Bottom - Chan.Top) / (Chan.yMax - Chan.yMin ) ;

     lbTMin.Top := pbDCDisplay.Top + pbDCDisplay.Height + 1 ;
     lbTMax.Top := lbTMin.Top ;
     lbTMin.caption := Format( '%5.4g %s', [Chan.xMin*CdrFH.dt*Settings.TScale,
                                               Settings.TUnits] ) ;
     lbTMax.caption := Format( '%5.4g %s', [Chan.xMax*CdrFH.dt*Settings.TScale,
                                               Settings.TUnits] ) ;
     lbTMax.Left := PB.Left + PB.Width - lbTMax.Width ;

     { Display Channel Name(s) }

     PB.Canvas.TextOut( Chan.Left,
                        (Chan.Top + Chan.Bottom) div 2,' ' + Chan.ADCName ) ;

     end ;

procedure TPwrSpecFrm.DrawHorizontalCursor( pb : TPaintBox ;
                                               Const Chan : TChannel ;
                                               Level : Integer ;
                                               Colour : TColor ) ;
{ ----------------------------------------------------
  Draw dotted horizontal cursor at ADC level 'Level'
  in display area defined by  record data channel 'Gr'
  ----------------------------------------------------}
var
   yPix,xPix,TicSize : Integer ;
   OldColor : TColor ;
   OldStyle : TPenStyle ;
   OldMode : TPenMode ;
begin
     with pb.canvas do begin
          OldColor := pen.color ;
          OldStyle := pen.Style ;
          OldMode := pen.mode ;
          pen.mode := pmXor ;
          pen.color := Colour ;
          end ;

     yPix := Trunc( Chan.Bottom - (Level - Chan.yMin)*Chan.yScale ) ;
     pb.canvas.polyline([Point(Chan.Left,yPix),Point(Chan.Right,yPix)]);

     with pb.canvas do begin
          pen.style := OldStyle ;
          pen.color := OldColor ;
          pen.mode := OldMode ;
          end ;
     { Request that the cursor readout memo box be updated }
    { State := UpdateCursorReadout ;}
    end ;


procedure TPwrSpecFrm.GetRecord ;
{ -----------------------------------------
  Get A/D samples for record from data file
  -----------------------------------------}
var

   Sum,MeanAC,MeanDC,VarianceAC : single ;
   i,j : Integer ;
begin

    edRecord.text := format( ' %d/%d ',[Data.RecordNum,Data.MaxRecord]) ;

    ReadRecord( Data.RecordNum, ADC^, ACChannel.xMin, ACChannel.xMax ) ;
    { Set Min/Max limits of DC axis }
    DCChannel.xMin := ACChannel.xMin ;
    DCChannel.xMax := ACChannel.xMax ;

    { Calculate mean signal level of DC Channel }
    Sum := 0.0 ;
    j := DCChannel.ChannelOffset ;
    for i := 0 to Data.RecordSize-1 do begin
        Sum := Sum + (ADC^[j] - DCChannel.ADCZero) ;
        j := j + CdrFH.NumChannels ;
        end ;

    MeanDC := (Sum/Data.RecordSize) * DCChannel.ADCScale ;
    edMean.text := format( ' %.4g %s',[MeanDC,DCChannel.ADCUnits] ) ;

    { Calculate signal variance of AC Channel }
    { First, calculate mean level }
    Sum := 0.0 ;
    j := ACChannel.ChannelOffset ;
    for i := 0 to Data.RecordSize-1 do begin
        Sum := Sum + (ADC^[j]) ;
        j := j + CdrFH.NumChannels ;
        end ;
    MeanAC := Sum / Data.RecordSize ;
    ACChannel.ADCZero := Trunc(MeanAC) ;

    { Next, calculate variance }
    Sum := 0.0 ;
    j := ACChannel.ChannelOffset ;
    for i := 0 to Data.RecordSize-1 do begin
        Sum := Sum + (ADC^[j] - MeanAC)*(ADC^[j] - MeanAC) ;
        j := j + CdrFH.NumChannels ;
        end ;
    VarianceAC := (Sum/Data.RecordSize)*ACChannel.ADCScale*ACChannel.ADCScale ;
    edVariance.text := format( ' %.4g %s^2', [VarianceAC,DCChannel.ADCUnits] ) ;

    { Record accepted/rejected status }
    if RecordStatus^[Data.RecordNum].Valid then ckRejected.Checked := False
                                           else ckRejected.Checked := True ;
    { Type of record }
    if RecordStatus^[Data.RecordNum].RecType = Background then
       cbRecType.ItemIndex := 1
    else
       cbRecType.ItemIndex := 0 ;

    end ;

procedure TPwrSpecFrm.ReadRecord( Rec : Integer ;
                                  var Buf : Array of SmallInt ;
                                  var xMin,xMax : single ) ;
{ --------------------------------------
  Read record from A/D sample data file
  -------------------------------------}
var
   NumBytesPerRecord,ByteOffset : LongInt ;
begin
    { Move file point to beginning of data to be read }
    NumBytesPerRecord := Data.RecordSize*CdrFH.NumChannels*2 ;
    ByteOffset := Data.RecordOffset*CdrFH.NumChannels*2 ;
    CdrFH.FilePointer := (Rec-1)*ByteOffset
                         + CdrFH.NumBytesInHeader ;
    CdrFH.FilePointer := FileSeek( CdrFH.FileHandle, CdrFH.FilePointer, 0 ) ;
    { Read data record }
    FileRead(CdrFH.FileHandle,Buf,NumBytesPerRecord) ;

    xMin := (Rec-1)*Data.RecordOffset ;
    xMax := xMin + Data.RecordSize ;
    end ;


procedure TPwrSpecFrm.SetRecordSize( NewSize : Integer ) ;
{ -------------------------------
  Set size of variance/FFT record
  -------------------------------}
var
   n,Rec : Integer ;
   QuarterRecord,OldMaxRecord,OldRecordOffset : LongInt ;
   OldStatus : ^TRecordStatusArray ;
   Src,SrcStep,Dest,DestStep : single ;
   Valid : Boolean ;
begin

     try
        New(OldStatus) ;

        { Ensure record size is a power of 2 and within valid limits }
        n := MinRecordSize div 2 ;
        repeat
           n := n*2 ;
           until (n >= NewSize) or (n = MaxRecordSize) ;

        { Set record size }
        edRecordSize.text := format(' %d ',[n] ) ;
        PowerSpectrum^.RecordSize := n ;
        BackgroundSpectrum^.RecordSize := n ;
        Data.RecordSize := n ;

        OldMaxRecord := Data.MaxRecord ;
        OldRecordOffset := Data.RecordOffset ;

        { Determine number of records in data file }
        Data.MaxRecord := CdrFH.NumSamplesInFile div
                          (Data.RecordSize*CdrFH.NumChannels) ;

        { Keep record overlapping with limits }
        if (Data.RecordOverlap < 0) or  (Data.RecordOverlap > 3)then
           Data.RecordOverlap := 0 ;

        { Calculate record offset to provide required degree of overlap }
        QuarterRecord := Data.RecordSize div 4 ;
        Data.RecordOffset := Data.RecordSize - (Data.RecordOverlap*QuarterRecord) ;
        { Increase number of record to account for record overlapping }
        Data.MaxRecord := (Data.MaxRecord*Data.RecordSize) div Data.RecordOffset ;

        edRecordOverlap.text := format( ' %.0f%%',[Data.RecordOverlap*25.0] ) ;

        PowerSpectrum^.MaxRecord := Data.MaxRecord ;
        BackgroundSpectrum^.MaxRecord := Data.MaxRecord ;
        Variance.MaxRecord := Data.MaxRecord ;

        { Warn user if working limits exceeded }
        if Data.MaxRecord > MaxSingle then begin
           MessageDlg( format(' Record storage capacity exceeded (%d)',[MaxSingle]),
                       mtWarning, [mbOK], 0 ) ;
           end ;

        sbRecord.Max := Data.MaxRecord  - 1 ;
        sbRecord.Position := 0 ;

        { Set range of records for variance and spectra }
        edVarRange.text := format( ' 1 - %d', [Data.MaxRecord] ) ;
        edSpecRange.text := format( ' 1 - %d', [Data.MaxRecord] ) ;

        { Make a copy of existing status array }
        for Rec := 0 to OldMaxRecord do OldStatus^[Rec] := RecordStatus^[Rec] ;


        if Data.RecordOffset < OldRecordOffset then begin
           { Record size has decreased / No. of data records has increased
             - Expand status array }
           Src := 1.0 ;
           SrcStep := (Data.RecordOffset) / (OldRecordOffset) ;
           for Rec := 1 to Data.MaxRecord do begin
               RecordStatus^[Rec] := OldStatus^[Trunc(Src)] ;
               Src := Src + SrcStep ;
               end ;
           end
        else if OldRecordOffset > 0 then begin
           { Record size has increased / No. of data records has decreased
           - Contract status array }
           for Rec := 1 to Data.MaxRecord do RecordStatus^[Rec].Valid := True ;
           Dest := 1.0 ;
           DestStep := (OldRecordOffset) / (Data.RecordOffset) ;
           for Rec := 1 to OldMaxRecord do begin
               Valid := RecordStatus^[Trunc(Dest)].Valid ;
               RecordStatus^[Trunc(Dest)] := OldStatus^[Rec] ;
               if not Valid then RecordStatus^[Trunc(Dest)].Valid := False ;
               Dest := Dest + DestStep ;
               end ;
           end ;

     finally ;
             Dispose(OldStatus) ;
             end ;

     end ;


procedure TPwrSpecFrm.spRecordSizeUpClick(Sender: TObject);
{ -----------------------------------------
  Increase size of variance/FFT record (X2)
  -----------------------------------------}
begin
     Data.RecordSize := Data.RecordSize*2 ;
     SetRecordSize( Data.RecordSize ) ;
     GetRecord ;
     pbACDisplay.RePaint ;
     pbDCDisplay.RePaint ;
     end;


procedure TPwrSpecFrm.spRecordSizeDownClick(Sender: TObject);
{ -----------------------------------------
  Decrease size of variance/FFT record (X2)
  -----------------------------------------}
begin
     Data.RecordSize :=Data.RecordSize div 2 ;
     SetRecordSize( Data.RecordSize ) ;
     GetRecord ;
     pbACDisplay.RePaint ;
     pbDCDisplay.RePaint ;
     end;


procedure TPwrSpecFrm.edRecordSizeKeyPress(Sender: TObject; var Key: Char);
{ ---------------------------------
  Set size of variance/FFT record
  ---------------------------------}
begin
     if key = chr(13) then begin
        Data.RecordSize := ExtractInt( edRecordSize.text ) ;
        SetRecordSize( Data.RecordSize ) ;
        GetRecord ;
        pbACDisplay.RePaint ;
        pbDCDisplay.RePaint ;
        end ;
     end;


procedure TPwrSpecFrm.FormClose(Sender: TObject; var Action: TCloseAction);
{ -------------------------
  Close and dispose of form
  -------------------------}
var
   FileName : string ;
   FileHandle : Integer ;
begin

     { Save record status data }
     FileName := ReplaceFileEnding( CdrFH.FileName, '.REC' ) ;
     if FileExists(FileName) then FileHandle := FileOpen(FileName,fmOpenReadWrite)
                             else FileHandle := FileCreate( FileName ) ;
     if FileWrite( FileHandle, RecordStatus^, SizeOf(TRecordStatusArray) )
        <> SizeOf(TRecordStatusArray) then begin
        MessageDlg( 'Error writing record status data',mtWarning, [mbOK], 0 ) ;
        FileClose( FileHandle ) ;
        end ;

     Settings.Variance.RecordSize := Data.RecordSize ;
     Settings.Variance.RecordOverlap := Data.RecordOverlap ;
     SaveCDRHeader( cdrFH ) ;

     HeapBuffers( Deallocate ) ;

     Main.mnPwrSpec.Enabled := True ;
     Main.CopyAndPrintMenus( False, False ) ;
     Action := caFree ;
     end;


procedure TPwrSpecFrm.sbRecordChange(Sender: TObject);
{ -----------------------------------------------
  Data record slider bar changed - Update display
  -----------------------------------------------}
begin
     Data.RecordNum := sbRecord.Position + 1 ;
     GetRecord ;
     pbACDisplay.RePaint ;
     pbDCDisplay.RePaint ;
     end;


procedure TPwrSpecFrm.pbDCDisplayPaint(Sender: TObject);
{ -------------------------------------
  Update DC Channel data record display
  -------------------------------------}
begin
     PlotChannel( pbDCDisplay, DCCHannel, ADC^ ) ;
     Main.CopyAndPrintMenus( True, False ) ;
     end;


procedure TPwrSpecFrm.pbACDisplayPaint(Sender: TObject);
{ -------------------------------------
  Update AC Channel data record display
  -------------------------------------}
begin
     PlotChannel( pbACDisplay, ACChannel, ADC^) ;
     Main.CopyAndPrintMenus( True, False ) ;

     end;

procedure TPwrSpecFrm.cbRecTypeChange(Sender: TObject);
{ ---------------------
  Change type of record
  ---------------------}
begin
     if cbRecType.ItemIndex = 0 then
        RecordStatus^[Data.RecordNum].RecType := Test
     else
        RecordStatus^[Data.RecordNum].RecType := Background ;
     end;


procedure TPwrSpecFrm.ckRejectedClick(Sender: TObject);
{ -----------------------------------------
  Change accepted/rejected status of record
  -----------------------------------------}
begin
     if ckRejected.checked then RecordStatus^[Data.RecordNum].Valid := False
                           else RecordStatus^[Data.RecordNum].Valid := True ;
     end;


procedure TPwrSpecFrm.edRecordKeyPress(Sender: TObject; var Key: Char);
{ -----------------------------------------
  Let user enter the record number directly
  -----------------------------------------}
begin
     if key = chr(13) then begin
        sbRecord.Position := ExtractInt( edRecord.text ) - 1 ;
        GetRecord ;
        end ;
     end;


procedure TPwrSpecFrm.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
{ --------------------------------
  Process special accelerator keys
  --------------------------------}
begin
     case key of
          VK_SUBTRACT : begin { - key }
                if sbRecord.Position > 0 then begin
                   sbRecord.Position := sbRecord.Position - 1 ;
                   GetRecord ;
                   end ;
                end ;
          VK_ADD : begin { + key }
                if sbRecord.Position < (Data.MaxRecord-1) then begin
                   sbRecord.Position := sbRecord.Position + 1 ;
                   GetRecord ;
                   end ;
                end ;
          $54,$4c,$45,$4d,$46 : begin
              { if (Shift = [ssCtrl]) then Action := NewRecordType ;}
               end ;
          $52 : begin
               if (Shift = [ssCtrl]) then begin
                   ckRejected.Checked := not ckRejected.Checked ;
                   end ;
               end ;
          else ;
          end ;
     end;


{ ***********************************************************************
  Variance/mean plot procedures
  ***********************************************************************}

procedure TPwrSpecFrm.bDoVarianceClick(Sender: TObject);
{ ------------------------------------------
  Calculate record DC means and AC variances
  ------------------------------------------}
var
   Rec,i,j,nAvg,Row : Integer ;
   Sum,Sum2,Sum3,MeanAC,xMin,xMax,y,Average : single ;
   SumMean,SumVar,AvgBackgroundMean,AvgBackgroundVar : single ;
begin

     if rbVarAllRecords.Checked then begin
        { Use all records }
        Variance.StartAt := 1 ;
        Variance.EndAt := Data.MaxRecord ;
        end
     else begin
        { Use selected range of records }
        GetIntRangeFromEditBox( edVarRange, Variance.StartAt, Variance.EndAt,
                                1,Data.MaxRecord) ;
        end ;

     Variance.Available := False ;
     for Rec := Variance.StartAt to Variance.EndAt do
         if RecordStatus^[Rec].Valid then begin

         { Read record from file }
         ReadRecord(Rec, ADC^, xMin, xMax ) ;

         { Calculate mean signal level of DC Channel }
         Sum := 0.0 ;
         j := DCChannel.ChannelOffset ;
         for i := 0 to Data.RecordSize-1 do begin
             Sum := Sum + (ADC^[j] - DCChannel.ADCZero) ;
             j := j + CdrFH.NumChannels ;
             end ;
         DCMean^[Rec] := (Sum/Data.RecordSize) * DCChannel.ADCScale ;

         { Calculate signal variance of AC Channel }
         { First, calculate mean level }
         Sum := 0.0 ;
         j := ACChannel.ChannelOffset ;
         for i := 0 to Data.RecordSize-1 do begin
             Sum := Sum + (ADC^[j]) ;
             j := j + CdrFH.NumChannels ;
             end ;
         MeanAC := Sum / Data.RecordSize ;
         ACChannel.ADCZero := Trunc(MeanAC) ;

         { Next, calculate variance and skew }
         Sum2 := 0.0 ;
         Sum3 := 0.0 ;
         j := ACChannel.ChannelOffset ;
         for i := 0 to Data.RecordSize-1 do begin
             y := (ADC^[j] - MeanAC) ;
             Sum2 := Sum2 + y*y ;
             Sum3 := Sum3 + y*y*y ;
             j := j + CdrFH.NumChannels ;
             end ;
         ACVariance^[Rec] :=
         (Sum2/Data.RecordSize)*ACChannel.ADCScale*ACChannel.ADCScale ;
         ACSkew^[Rec] := (Sum3/Data.RecordSize)*ACChannel.ADCScale
                          *ACChannel.ADCScale*ACChannel.ADCScale ;

         { Compute the median power frequency of the fluctuations }
         ComputePowerSpectrum( Test, Rec, Rec, TempSpectrum^ ) ;
         MedFreq^[Rec] := TempSpectrum^.MedianFrequency ;

         edVarStatus.text := format( ' %d/%d', [Rec,Variance.EndAt] ) ;

         application.ProcessMessages ;

         Variance.Available := True ;
         end ;

     { Compute and subtract background AC variance & DC mean signals }

     if ckVarSubtractBackground.checked then begin
        nAvg := 0 ;
        SumMean := 0.0 ;
        SumVar := 0.0 ;
        for Rec := 1 to Variance.MaxRecord do
            if (RecordStatus^[Rec].Valid)
               and (RecordStatus^[Rec].RecType = Background) then begin
               SumMean := SumMean + DCMean^[Rec] ;
               SumVar := Sumvar + ACVariance^[Rec] ;
               Inc(nAvg) ;
               end ;
        { Subtract mean background values }
        if nAvg > 0 then begin
           AvgBackgroundMean := SumMean / nAvg ;
           AvgBackgroundMean := SumVar / nAvg ;
           for Rec := Variance.StartAt to Variance.EndAt do
               if (RecordStatus^[Rec].Valid)
                  and (RecordStatus^[Rec].RecType = Test) then begin
                  DCMean^[Rec] := DCMean^[Rec] - AvgBackgroundMean ;
                  ACVariance^[Rec]  := ACVariance^[Rec] - AvgBackgroundMean ;
                  end ;
           end ;
        end ;

     Variance.NewPlot := True ;
     { Remove any fitted equation }
     Variance.Equation.Available := False ;

     { Standard settings for mean/variance plot }
     VarPlot.xAxis.AutoRange := True ;
     VarPlot.yAxis.AutoRange := True ;
     VarPlot.xAxis.Lab := cbVarXAxis.text + ' '
                          + GetVariableUnits(cbVarXAxis.ItemIndex) ;
     VarPlot.yAxis.Lab := cbVarYAxis.text + ' '
                          + GetVariableUnits(cbVarYAxis.ItemIndex) ;

     { Calculate average of Y variable }
     nAvg := 0 ;
     Sum := 0.0 ;
     for Rec := Variance.StartAt to Variance.EndAt do
         if RecordStatus^[Rec].Valid and
            (RecordStatus^[Rec].RecType = Test) then begin
               Sum := Sum + GetVariable( cbVarYAxis.ItemIndex, Rec ) ;
               Inc(nAvg) ;
               end ;
     if nAvg > 0 then Average := Sum / nAvg ;

     { Display average and number of records in results table }
     sgVarResults.RowCount := 2 ;
     Row := 0 ;
     sgVarResults.Cells[0,Row] := format(' Avg. %s = %.4g %s',
                                  [cbVarYAxis.text,Average,
                                   GetVariableUnits(cbVarYAxis.ItemIndex)]) ;
     Inc(Row) ;
     sgVarResults.Cells[0,Row] := format(' No. records = %d',[nAvg]) ;

     bMEPCFrequency.Enabled := True ;

     { Request plot }
     pbVarDisplay.RePaint ;
     end;


procedure TPwrSpecFrm.PlotGraph ;
{ ------------------------
  Plot variance/mean graph
  ------------------------}
var
   Rec,i,Row : Integer ;
   xy : TxyBuf ;
   x,dx,xOffset : single ;
begin
     try
        { Create x,y plot data object }
        xy := TxyBuf.Create ;
        { Read selected X and Y axis data from arrays into xy }
        xy.NumPoints := 0 ;
        for Rec := Variance.StartAt to Variance.EndAt do
            if RecordStatus^[Rec].Valid and
               (RecordStatus^[Rec].RecType = Test) then begin
               xy.x[xy.NumPoints] := GetVariable( cbVarXAxis.ItemIndex, Rec ) ;
               xy.y[xy.NumPoints] := GetVariable( cbVarYAxis.ItemIndex, Rec ) ;
               Inc(xy.NumPoints) ;
               end ;

        { Sort data into ascending order of x }
        Sort( xy.x, xy.y, xy.NumPoints ) ;

        { Set plot size/fonts for screen }
        VarPlot.Left :=   pbVarDisplay.Width div 6 ;
        VarPlot.Right :=  pbVarDisplay.Width - (pbVarDisplay.Width div 10) ;
        VarPlot.Top :=    pbVarDisplay.Height div 8 ;
        VarPlot.Bottom := pbVarDisplay.Height - VarPlot.Top*2 ;
        VarPlot.MarkerSize := Settings.Plot.MarkerSize ;
        xy.MarkerSize := VarPlot.MarkerSize ;
        xy.Color := clRed ;
        VarPlot.LineThickness := Settings.Plot.LineThickness ;
        VarPlot.NumGraphs := 1 ;
        VarPlot.Title := ' ' ;

        pbVarDisplay.canvas.font.name := Settings.Plot.FontName ;
        pbVarDisplay.canvas.font.size := Settings.Plot.FontSize ;
        { ** Plot graph on screen ** }
        pbVarDisplay.Canvas.Brush.color := clWhite ;
        pbVarDisplay.Canvas.Pen.Width := VarPlot.LineThickness ;
        pbVarDisplay.canvas.FillRect( pbVarDisplay.canvas.ClipRect ) ;

        { Plot line joining data points }
        DrawAxes( pbVarDisplay.canvas, VarPlot, xy ) ;
        if Settings.Plot.ShowLines and (not Variance.Equation.Available) then
           DrawLine(pbVarDisplay.canvas,VarPlot,xy) ;
        { Plot data points }
        if Settings.Plot.ShowMarkers then DrawMarkers(pbVarDisplay.canvas,VarPlot,xy) ;

        { Plot fitted curve (if one available) }
        if Variance.Equation.Available then begin
           xy.NumPoints := 200 ;

           if Variance.Equation.EqnType = Exponential then begin
                xOffset := MinFlt([Variance.Cursor0.xVal,Variance.Cursor1.xVal]) ;
                x := xOffset ;
                end
           else begin
                xOffset := 0.0 ;
                x := VarPlot.xAxis.Lo ;
                end ;
           dx := (VarPlot.xAxis.Hi - x ) / xy.NumPoints ;

           for i := 0 to xy.NumPoints-1 do begin
               xy.x[i] := x ;
               xy.y[i] := MathFunc( Variance.Equation, xy.x[i] - xOffset ) ;
               x := x + dx ;
               end ;
           DrawLine(pbVarDisplay.canvas,VarPlot,xy ) ;
           end ;

        { Set position of cursor 0 to nearest data point }
        if Variance.NewPlot then Variance.Cursor0.xVal := xy.x[0] ;
        DrawVerticalCursor( pbVarDisplay, VarPlot,
                            Variance.Cursor0.xVal,
                            Variance.Cursor0.Lab) ;

        { Set position of cursor 1 to nearest data point}
        if Variance.NewPlot then Variance.Cursor1.xVal := xy.x[xy.NumPoints-1] ;
        DrawVerticalCursor( pbVarDisplay, VarPlot,
                            Variance.Cursor1.xVal,
                            Variance.Cursor1.Lab) ;

        Variance.NewPlot := False ;
     finally
        xy.Free ;
        end ;

     end ;


procedure TPwrSpecFrm.PrintVarianceGraph ;
{ ------------------------
  Print variance/mean graph
  ------------------------}
var
   Rec,i,Row,YEndOfText : Integer ;
   xy : TxyBuf ;
   x,dx,xOffset : single ;
   Canvas : TCanvas ;
begin
     try

         Screen.Cursor := crHourglass ;

        { Create x,y plot data object }
        xy := TxyBuf.Create ;
        { Read selected X and Y axis data from arrays into xy }
        xy.NumPoints := 0 ;
        for Rec := Variance.StartAt to Variance.EndAt do
            if RecordStatus^[Rec].Valid and
               (RecordStatus^[Rec].RecType = Test) then begin
               xy.x[xy.NumPoints] := GetVariable( cbVarXAxis.ItemIndex, Rec ) ;
               xy.y[xy.NumPoints] := GetVariable( cbVarYAxis.ItemIndex, Rec ) ;
               Inc(xy.NumPoints) ;
               end ;

        { Sort data into ascending order of x }
        Sort( xy.x, xy.y, xy.NumPoints ) ;

        Canvas := Printer.Canvas ;

        { Define position of plot on page }
        VarPlot.Left :=   PrinterCmToPixels('H',Settings.Plot.LeftMargin) ;
        VarPlot.Right :=  Printer.PageWidth -
                          PrinterCmToPixels('H',Settings.Plot.RightMargin) ;
        VarPlot.Top :=    PrinterCmToPixels('V',Settings.Plot.TopMargin) ;
        VarPlot.Bottom := Printer.PageHeight -
                          PrinterCmToPixels('V',Settings.Plot.BottomMargin) ;

        Printer.BeginDoc ;

        Canvas.Pen.Color := clBlack ;
        VarPlot.MarkerSize := Settings.Plot.MarkerSize ;
        xy.MarkerSize := PrinterPointsToPixels(VarPlot.MarkerSize) ;
        xy.Color := clRed ;
        VarPlot.LineThickness := PrinterPointsToPixels(Settings.Plot.LineThickness) ;
        VarPlot.NumGraphs := 1 ;
        VarPlot.Title := ' ' ;

        Canvas.font.name := Settings.Plot.FontName ;
        Canvas.font.size := PrinterPointsToPixels(Settings.Plot.FontSize) ;

        { ** Print graph ** }
        Canvas.Brush.color := clWhite ;
        Canvas.Pen.Width := VarPlot.LineThickness ;

        { Print file/experiment identification information }
        PrintPageTitle( Canvas, Variance.Equation.Available,
                        sgVarResults, YEndofText ) ;
        if VarPlot.Top < YEndofText then VarPlot.Top := YEndofText ;

        { Plot axes }
        DrawAxes( Canvas, VarPlot, xy ) ;

        { Plot line joining data points }
        if Settings.Plot.ShowLines and (not Variance.Equation.Available) then
           DrawLine(Canvas,VarPlot,xy) ;

        { Plot data points }
        if Settings.Plot.ShowMarkers then DrawMarkers(Canvas,VarPlot,xy) ;

        { Plot fitted curve (if one available) }
        if Variance.Equation.Available then begin
           xy.NumPoints := 200 ;

           if Variance.Equation.EqnType = Exponential then begin
                xOffset := MinFlt([Variance.Cursor0.xVal,Variance.Cursor1.xVal]) ;
                x := xOffset ;
                end
           else begin
                xOffset := 0.0 ;
                x := VarPlot.xAxis.Lo ;
                end ;
           dx := (VarPlot.xAxis.Hi - x ) / xy.NumPoints ;

           for i := 0 to xy.NumPoints-1 do begin
               xy.x[i] := x ;
               xy.y[i] := MathFunc( Variance.Equation, xy.x[i] - xOffset ) ;
               x := x + dx ;
               end ;
           DrawLine(Canvas,VarPlot,xy ) ;
           end ;

     finally
        Printer.EndDoc ;
        xy.Free ;
        Screen.Cursor := crDefault ;
        end ;

     end ;

procedure TPwrSpecFrm.PrintPageTitle( const Canvas : TCanvas ;
                                      ResultsAvailable : Boolean ;
                                      const Results : TStringGrid ;
                                      var YEndOfText : Integer ) ;
{ -----------------------------------------------------
  Print experiment identification and other information
  -----------------------------------------------------}
var
   xPix,yPix,CharWidth,LineHeight,Row : Integer ;
   OldFontName : String ;
   OldFontSize : Integer ;

begin
     { Save the current font settings }
     OldFontName := Canvas.Font.Name ;
     OldFontSize := Canvas.Font.Height ;

     { Select standard font name and size for text information }
     Canvas.Font.Name := 'Arial' ;
     {Canvas.Font.Size := PrinterPointsToPixels(10) ;}

     CharWidth := Canvas.TextWidth('X') ;
     LineHeight := (Canvas.TextHeight('X')*12) div 10 ;

     { Start printing a top-left of page }
     xPix := Printer.PageWidth div 10 ;
     yPix := Printer.PageHeight div 60 ;

     { File Name }
     Canvas.TextOut(xPix,yPix, 'File ... ' + CdrfH.FileName ) ;
     { Ident line }
     yPix := yPix + LineHeight ;
     Canvas.TextOut( xPix, yPix, CdrfH.IdentLine ) ;

     { If a curve has been fitted, print the best fit parameters }
     if ResultsAvailable then begin
        for Row := 0 to Results.RowCount-1 do begin
            Canvas.TextOut( xPix, yPix, Results.Cells[0,Row] ) ;
            yPix := yPix + LineHeight ;
            end ;
        end ;

     { Return the vertical position of the bottom of the area used for text }
     YEndOfText := yPix + LineHeight ;

     { Restore the old font settings }
     Canvas.Font.Name := OldFontName ;
     Canvas.Font.Height := OldFontSize ;
     end ;


procedure TPwrSpecFrm.CopyVarianceDataToClipBoard ;
{ ----------------------------------------------------------------
  Copy the data points in the variance/mean graph to the clipboard
  ----------------------------------------------------------------
    25/6/98 Clipboard buffer limit reduced to 31000}
const
     BufSize = 31000 ;
     NumBytesPerNumber = 12 ;
var
   Rec,i,iSkip,NumBytesNeeded,NumColumns : LongInt ;
   xOffset : single ;
   Line : String ;
   CopyBuf0,Line0 : PChar ;
   xy : TXYBuf ;
   OK : Boolean ;

begin
     try
        OK := OpenClipboard( Handle ) ;
        OK := EmptyClipBoard ;
        { Create x/y array for graph }
        xy := TXYBuf.Create ;
        { Allocate ASCII text buffer to hold graph }
        CopyBuf0 := StrAlloc( BufSize ) ;
        StrPCopy( CopyBuf0, '' ) ;
        Line0 := StrAlloc(  256 ) ;

        { Get data }
        for Rec := Variance.StartAt to Variance.EndAt do
            if RecordStatus^[Rec].Valid and
               (RecordStatus^[Rec].RecType = Test) then begin
               xy.x[xy.NumPoints] := GetVariable( cbVarXAxis.ItemIndex, Rec ) ;
               xy.y[xy.NumPoints] := GetVariable( cbVarYAxis.ItemIndex, Rec ) ;
               Inc(xy.NumPoints) ;
               end ;

        { Sort data into ascending order of x }
        Sort( xy.x, xy.y, xy.NumPoints ) ;

        { Determine starting point for an exponential fit }
        if Variance.Equation.EqnType = Exponential then
           xOffset := MinFlt([Variance.Cursor0.xVal,Variance.Cursor1.xVal])
        else xOffset := 0.0 ;

        { Determine sample skip factor to ensure that the compete record
          fits into the buffer }
        NumColumns := 2 ;
        if Variance.Equation.Available then NumColumns := NumColumns + 1 ;
        NumBytesNeeded := xy.NumPoints*NumBytesPerNumber*NumColumns ;
        iSkip := MaxInt( [(NumBytesNeeded+BufSize-1) div BufSize,1]) ;

        i := 0 ;
        screen.cursor := crHourglass ;
        repeat
              Line := format( '%8.5g', [xy.x[i]] ) + chr(9) ;
              Line := Line + format( '%8.5g', [xy.y[i]] ) ;
              if Variance.Equation.Available then begin
                 Line := Line + chr(9) + format( '%8.5g',
                 [MathFunc(Variance.Equation,xy.x[i] - xOffset )]) ;
                 end ;
             Line := Line + chr(13) + chr(10) ;
             StrPCopy( Line0, Line ) ;
             CopyBuf0 := StrCat( CopyBuf0, Line0 ) ;
             i := i + iSkip ;
             until i >= xy.NumPoints ;

        { Copy text accumulated in copy buffer to clipboard }
         ClipBoard.SetTextBuf( CopyBuf0 ) ;

     finally
         screen.cursor := crDefault ;
         { Dispose of buffers }
         StrDispose( Line0 ) ;
         StrDispose( CopyBuf0 ) ;
         xy.Free ;
         OK := CloseClipboard ;
         end ;
     end ;


procedure TPwrSpecFrm.CopyRecordToClipBoard ;
{ -------------------------------------------------------------
  Copy the data points in the displayed record to the clipboard
  -------------------------------------------------------------}
const
     BufSize = 31000 ;
     NumBytesPerNumber = 12 ;
var
   Rec,i,j,iSkip,NumBytesNeeded,NumColumns : LongInt ;
   x : single ;
   Line : String ;
   CopyBuf0,Line0 : PChar ;
   OK : Boolean ;

begin
     try
        screen.cursor := crHourglass ;

        OK := OpenClipboard( Handle ) ;
        OK := EmptyClipBoard ;

        { Allocate ASCII text buffer to hold graph }
        CopyBuf0 := StrAlloc( BufSize ) ;
        StrPCopy( CopyBuf0, '' ) ;
        Line0 := StrAlloc(  256 ) ;


        { Determine number of samples to skip to keep data within
          clipboard buffer limit  }
        if cbACChannel.ItemIndex <> cbACChannel.ItemIndex then NumColumns := 3
                                                          else NumColumns := 2 ;
        NumBytesNeeded := Data.RecordSize*NumBytesPerNumber*NumColumns ;
        iSkip := MaxInt( [(NumBytesNeeded+BufSize-1) div BufSize,1]) ;

        { ** Copy AC and DC channel data to clipboard **
          (NOTE. Only copy one channel if AC=DC channel}
        x := 0.0 ;
        i := 0 ;
        while i < Data.RecordSize-1 do begin
            {Create line of text }
            j := i*CdrFH.NumChannels + DCChannel.ChannelOffset ;
            Line := format( '%8.5g', [x] )  ;
            Line := Line + chr(9) + format( '%8.5g', [ADC^[j]*DCChannel.ADCScale] ) ;
            if cbACChannel.ItemIndex <> cbDCChannel.ItemIndex then begin
               Line := Line + chr(9) + format( '%8.5g', [ADC^[j]*ACChannel.ADCScale] ) ;
               end ;
            Line := Line + chr(13) + chr(10) ;
            { Add line to copy buffer }
            StrPCopy( Line0, Line ) ;
            CopyBuf0 := StrCat( CopyBuf0, Line0 ) ;
            { Increment to next sample }
            x := x + CdrFH.dt*iSkip ;
            i := i + iSkip ;
            end ;

        { Copy text accumulated in copy buffer to clipboard }
         i := strlen( COpyBuf0) ;
         ClipBoard.SetTextBuf( CopyBuf0 ) ;

     finally
         screen.cursor := crDefault ;
         { Dispose of buffers }
         StrDispose( Line0 ) ;
         StrDispose( CopyBuf0 ) ;
         OK := CloseClipboard ;
         end ;
     end ;



function TPwrSpecFrm.GetVariable( ItemIndex, Rec : Integer ) : single ;
{ ---------------------------------------------------------------------------
  Get the variable selected by the combo box "ItemIndex" for the record "Rec"
  ---------------------------------------------------------------------------}
begin
     case ItemIndex of
          vRecordNum : Result := Rec ;
          vTime : Result := (((Rec-1)*Data.RecordOffset) + Data.RecordSize)
                            *CdrFH.dt*Settings.TScale ;
          { Signal mean DC level }
          vMeanDC : Result := DCMean^[Rec] ;
          { Standard Deviation of signal }
          vStDev : begin
                   if ACVariance^[Rec] > 0.0 then Result := Sqrt(ACVariance^[Rec])
                                             else Result := 0.0 ;
                   end ;
          { Variance of signal }
          vVariance : Result := ACVariance^[Rec] ;
          { Skew of signal }
          vSkew : Result := ACSkew^[Rec] ;
          { Median frequency of fluctuations }
          vMedianFrequency : Result := MedFreq^[Rec] ;
          else Result := 0.0 ;
          end ;
     end ;


function TPwrSpecFrm.GetVariableUnits( ItemIndex : Integer ) : string ;
{ --------------------------------------------------------------------
  Get the units for the variable selected by the combo box "ItemIndex"
  --------------------------------------------------------------------}
begin
     case ItemIndex of
          vRecordNum : Result := '' ;
          vTime : Result := Settings.TUnits ;
          vMeanDC : Result := DCChannel.ADCUnits ;
          vStDev : Result := ACChannel.ADCUnits ;
          vVariance : Result := ACChannel.ADCUnits + '^2' ;
          vSkew : Result := ACChannel.ADCUnits + '^3' ;
          vMedianFrequency : Result := 'Hz' ;
          else Result := '' ;
          end ;
     end ;


procedure TPwrSpecFrm.pbVarDisplayPaint(Sender: TObject);
{ ----------------------------
  Update mean/variance display
  ----------------------------}
begin
     if Variance.Available then begin
        PlotGraph ;
        Main.CopyAndPrintMenus( True, True ) ;
        end
     else begin
          EraseDisplay( pbVarDisplay ) ;
          lbVarCursor0.Visible := False ;
          lbVarCursor1.Visible := False ;
          Main.CopyAndPrintMenus( False, False ) ;
          end ;
     end;

procedure TPwrSpecFrm.bVarSetAxesClick(Sender: TObject);
{ -----------------------------
  Customise mean/variance VarPlot
  -----------------------------}
begin
     SetAxesFrm.SetPlot := VarPlot ;
     SetAxesFrm.ShowModal ;
     if SetAxesFrm.ModalResult = mrOK then begin
        VarPlot := SetAxesFrm.SetPlot ;
        pbVarDisplay.Repaint ;
        end ;
     end;


procedure TPwrSpecFrm.bVarFitClick(Sender: TObject);
{ -------------------------------------------
  Fit a function to variance/mean VarPlot
  -------------------------------------------
    25/6/98 Clipboard buffer limit reduced to 31000}
var
   i,nFit,iStart,iEnd,Row,nPars,iSkip,Rec : Integer ;
   ParUnits,ParName : string ;
   KeepPars : Array[0..LastParameter] of Single ;
   x,xLoLimit,xHiLimit : single ;
   FitData : ^TXYData ;
begin

     Try
        New( FitData ) ;
        { Select type of equation to be fitted }
        case cbVarEquation.ItemIndex  of
          1 : Variance.Equation.EqnType := Linear ;
          2 : Variance.Equation.EqnType := Parabola ;
          3 : Variance.Equation.EqnType := Exponential ;
          else  Variance.Equation.EqnType := None ;
          end ;

        Variance.Equation.Available := False ;
        Variance.Equation.ParametersSet := False ;
        if Variance.Equation.EqnType <> None then begin

           { Copy data into fitting array
             (Note. records skipped if necessary to fit into buffer) }
           iSkip := ((Variance.EndAt - Variance.StartAt ) div High(FitData^.x)) + 1 ;
           nFit := 0 ;
           Rec := Variance.StartAt ;
           { Lower and upper x data limit set by display cursors }
           xLoLimit := MinFlt( [Variance.Cursor0.xVal,Variance.Cursor1.xVal] ) ;
           xHiLimit := MaxFlt( [Variance.Cursor0.xVal,Variance.Cursor1.xVal] ) ;
           repeat
                 if RecordStatus^[Rec].Valid and
                    (RecordStatus^[Rec].RecType = Test) then begin
                    x := GetVariable( cbVarXAxis.ItemIndex, Rec ) ;
                    if (xLoLimit <= x) and (x <= xHiLimit) then begin
                       FitData^.x[nFit] := x ;
                       FitData^.y[nFit] := GetVariable( cbVarYAxis.ItemIndex, Rec ) ;
                       Inc(nFit) ;
                       end ;
                    end ;
                 Rec := Rec + iSkip ;
                 until Rec >= Variance.EndAt ;

           { If an exponential function is being fitted, measure X relative
             to xLoLimit set by display cursor }
           if Variance.Equation.EqnType = Exponential then begin
              for i := 0 to nFit-1 do FitData^.x[i] := FitData^.x[i] - xLoLimit ;
              end ;


           { Get number of parameters in equation }
           nPars := GetNumEquationParameters( Variance.Equation ) ;
           { Do curve fit, if enough data points }
           if nFit >= nPars then begin

              { Keep orginal parameters }
              for i := 0 to nPars-1 do KeepPars[i] := Variance.Equation.Par[i] ;

              { Create an initial set of guesses for parameters
             (keeping constant any "fixed" parameters) }
             InitialiseParameters( FitData^, nFit, Variance.Equation ) ;
             for i := 0 to nPars-1 do if Variance.Equation.ParSD[i] < 0. then
                 Variance.Equation.Par[i] := KeepPars[i] ;

              { Let user modify initial parameter settings and/or
                fix parameters at constant values }
              SetFitParsFrm.Equation := Variance.Equation ;
              SetFitParsFrm.ShowModal ;
              if SetFitParsFrm.ModalResult = mrOK then
                 Variance.Equation := SetFitParsFrm.Equation ;
              { Prevent FitCurve from changing parameter settings }
              Variance.Equation.ParametersSet := True ;

              { Fit curve using non-linear regression }
              FitCurve( FitData^, nFit, Variance.Equation ) ;

              sgVarResults.RowCount := nPars + 3 ;

              Row := 0 ;
              sgVarResults.Cells[0,Row] := GetFormula( Variance.Equation ) ;
              Inc(Row) ;

              { Best fit parameters and standard error }
              for i := 0 to nPars-1 do begin
                  GetParameterInfo(Variance.Equation,ParName,ParUnits,
                                   GetVariableUnits(cbVarXAxis.ItemIndex),
                                   GetVariableUnits(cbVarYAxis.ItemIndex),i) ;
                  if Variance.Equation.ParSD[i] >= 0.0 then
                     sgVarResults.Cells[0,Row] := format(' %s = %.4g +/- %.4g (sd) %s',
                                                         [ParName,
                                                          Variance.Equation.Par[i],
                                                          Variance.Equation.ParSD[i],
                                                          ParUnits] ) 
                  else
                     { Negative s.d. indicates a fixed parameter }
                     sgVarResults.Cells[0,Row] := format(' %s = %.4g (fixed) %s',
                                                [ParName,
                                                 Variance.Equation.Par[i],
                                                 ParUnits] ) ;

                  Inc(Row) ;
                  end ;

              { Residual standard deviation }
              sgVarResults.Cells[0,Row] := format(' Residual S.D. = %.4g %s',
                                            [Variance.Equation.ResidualSD,
                                             GetVariableUnits(cbVarYAxis.ItemIndex)] ) ;
              Inc(Row) ;

              { Statistical degrees of freedom }
              sgVarResults.Cells[0,Row] := format(' Degrees of freedom = %d ',
                                            [Variance.Equation.DegreesFreedom]) ;
              Inc(Row) ;

              { No. of iterations }
              sgVarResults.Cells[0,Row] := format(' No. of iterations = %d ',
                                            [Variance.Equation.NumIterations]) ;
              Inc(Row) ;

              end
           else begin
              MessageDlg( format('%d points is insufficient for fit',[nFit]),
                           mtWarning, [mbOK], 0 ) ;
              end ;

           end ;

           { Request a re-VarPlot of variance/mean graph}
           pbVarDisplay.Repaint ;

     finally
            Dispose(FitData) ;
            end ;
     end ;


procedure TPwrSpecFrm.pbVarDisplayMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
{ ---------------------------------------------------
  Variance/mean display :
  If a cursor has been selected make it active when
  mouse button is depressed
  ---------------------------------------------------}
begin
     if Variance.Available then begin
        Variance.Cursor0.Active := Variance.Cursor0.Selected ;
        Variance.Cursor1.Active := Variance.Cursor1.Selected ;
        end ;
     end;

procedure TPwrSpecFrm.pbVarDisplayMouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);
{ --------------------------------------------------------------------
  Variance/mean display :  Update display cursors when mouse is moved
  --------------------------------------------------------------------}
Const
     Margin = 4 ;
var
   OldIndex,xPos0,xPos1 : Integer ;
   OK : Boolean ;
begin

     if Variance.Available then begin
        if (not Variance.Cursor0.Active)
           and (not Variance.Cursor1.Active) then begin

           OK := ScaleXToScreenCoord( VarPlot, Variance.Cursor0.xVal, xPos0 ) ;
           OK := ScaleXToScreenCoord( VarPlot, Variance.Cursor1.xVal, xPos1 ) ;
           if Abs(X - xPos0) < Margin then begin
                pbVarDisplay.Cursor := crSizeWE ;
                Variance.Cursor0.Selected := True ;
                end
           else if Abs(X - xPos1) < Margin  then begin
                pbVarDisplay.Cursor := crSizeWE ;
                Variance.Cursor1.Selected := True ;
                end
           else begin
                pbVarDisplay.Cursor := crDefault ;
                Variance.Cursor0.Selected := False ;
                Variance.Cursor1.Selected := False ;
                end ;
           end
        else begin

          { If a cursor is active, move it to new position }

          if Variance.Cursor0.Active then begin
              MoveVerticalCursor( pbVarDisplay, VarPlot, Variance.Cursor0, X ) ;
              end
          else if Variance.Cursor1.Active then begin
              MoveVerticalCursor( pbVarDisplay, VarPlot, Variance.Cursor1, X ) ;
              end ;
          end ;
        end ;
     end ;


procedure TPwrSpecFrm.pbVarDisplayMouseUp(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
{ ------------------------------------------------------------------------
  Variance/mean display : Make cursor inactive when mouse button released
  ------------------------------------------------------------------------}
begin
     if Variance.Available then begin
        Variance.Cursor0.Active := False ;
        Variance.Cursor1.Active := False ;
        end ;
     end;


procedure TPwrSpecFrm.bMEPCFrequencyClick(Sender: TObject);
{ ----------------------------------------------------
  Calculate MEPC frequency from Avg. Skew and Variance
  (See Segal et al, Biophys J. (1985) Vol. 47 pp183-202,)
  ----------------------------------------------------}
var
   MEPCFrequency,TauRise,TauDecay : double ;
   SumSkew,SumVar,AvgVariance,AvgSkew,I2,I3,Alpha,Beta,BetaMinusAlpha : double ;
   Rec,nAvg : LongInt ;
   Row : Integer ;
begin
     { Get MEPC rise/decay time constants from user }
     MEPCFreqFrm.ShowModal ;
     if MEPCFreqFrm.ModalResult = mrOK then begin

        { Calculate average variance and skew of records in selected range }

        SumVar := 0.0 ;
        SumSkew := 0.0 ;
        nAvg := 0 ;
        for Rec := Variance.StartAt to Variance.EndAt do
            if RecordStatus^[Rec].Valid
               and (RecordStatus^[Rec].RecType = Test) then begin
               SumVar := SumVar + ACVariance^[Rec] ;
               SumSkew := SumSkew + ACSkew^[Rec] ;
               Inc(nAvg)
               end ;
        if nAvg > 0 then begin
           AvgVariance := SumVar / nAvg ;
           AvgSkew := SumSkew / nAvg ;
           end ;

        TauRise := Settings.Variance.TauRise ;
        TauDecay := Settings.Variance.TauDecay ;

        I2 := ((TauRise - TauDecay)*(TauRise - TauDecay)) / (2.0*(TauRise + TauDecay)) ;
        Alpha := 1.0 / TauRise ;
        Beta := 1.0 / TauDecay ;
        BetaMinusAlpha := Beta - Alpha ;
        I3 := (2.0*BetaMinusAlpha*BetaMinusAlpha*BetaMinusAlpha) /
              (3.0*Alpha*Beta*(2.0*Alpha + Beta)*(2.0*Beta + Alpha)) ;

        MEPCFrequency := ( (AvgVariance*AvgVariance*AvgVariance)
                           / ( AvgSkew*AvgSkew) ) * ( (I3*I3) / (I2*I2*I2) ) ;

        sgVarResults.RowCount := 4 ;
        Row := 0 ;
        sgVarResults.Cells[0,Row] := format(' Avg. Variance = %.4g %s^2',
                                        [AvgVariance,ACChannel.ADCUnits] ) ;
        Inc(Row) ;
        sgVarResults.Cells[0,Row] := format(' Avg. Skew = %.4g %s^3',
                                        [AvgSkew,ACChannel.ADCUnits] ) ;
        Inc(Row) ;
        sgVarResults.Cells[0,Row] := format(' MEPC Frequency = %f /s',
                                        [MEPCFrequency] ) ;
        Inc(Row) ;
        sgVarResults.Cells[0,Row] := format(' No. records = %d',[nAvg]) ;

        end ;

     end;




{ ***********************************************************************
  Power spectrum routines
  ***********************************************************************}

procedure TPwrSpecFrm.bDoSpectrumClick(Sender: TObject);
{ -----------------------
  Calculate power spectra
  -----------------------}
var
   Rec,i,j : Integer ;
   StartAt,EndAt : LongInt ;
   Sum,MeanAC,xMin,xMax,Denom : single ;
begin

     if rbSpecAllRecords.Checked then begin
        { Use all records }
        StartAt := 1 ;
        EndAt := Data.MaxRecord ;
        end
     else begin
        { Use selected range of records }
        GetIntRangeFromEditBox( edSpecRange, StartAt, EndAt, 1,Data.MaxRecord) ;
        end ;

    ComputePowerSpectrum( Test, StartAt, EndAt, PowerSpectrum^ ) ;

    { Subtract background spectrum, if required }
    if ckSpecSubtractBackground.checked then begin
       ComputePowerSpectrum( Background, 1, Data.MaxRecord, BackgroundSpectrum^ ) ;
       for i := 0 to PowerSpectrum^.NumPoints-1 do
           PowerSpectrum^.Power[i] := PowerSpectrum^.Power[i] -
                                      BackgroundSpectrum^.Power[i] ;
       PowerSpectrum^.AvgDCMean := PowerSpectrum^.AvgDCMean
                                   - BackgroundSpectrum^.AvgDCMean ;
       PowerSpectrum^.Variance := PowerSpectrum^.Variance
                                   - BackgroundSpectrum^.Variance ;
       end ;


    { Standard settings for a new spectrum }
    SpecPlot.xAxis.AutoRange := True ;
    SpecPlot.yAxis.AutoRange := True ;

    if rbLogFreqAveraging.checked then begin
       { Make axis logarithmic, if log. averaging in use }
       SpecPlot.xAxis.Log := True ;
       SpecPlot.yAxis.Log := True ;
       end
    else begin
       { Make axes linear for all other averaging modes }
       SpecPlot.xAxis.Log := False ;
       SpecPlot.yAxis.Log := False ;
       end ;

    SpecPlot.xAxis.Lab := 'Hz' ;
    SpecPlot.yAxis.Lab := ACChannel.ADCUnits + '^2' ;
    { No equation }
    cbSpecEquation.ItemIndex := 0 ;
    { Request spectrum to be VarPlotted }
    pbSpecDisplay.RePaint ;

    end;


procedure TPwrSpecFrm.ComputePowerSpectrum( RecType : TRecType ;
                                            StartAt,EndAt : Integer ;
                                            var Spectrum : TSpectrum ) ;
{ ---------------------------------------------------
  Compute averaged power spectrum
  Using data record of type "RecType",
  starting at record "StartAt" and ending at "EndAt"
  Store spectrum in "Spectrum"
  ---------------------------------------------------}
var
   Rec,i,j,n,npFFT,Row : Integer ;
   Sum,MeanAC,xMin,xMax,Denom,YReal,YImag,dFreq : single ;
   FFT : ^TSingleArray ;
   VarianceCorrection : single ;
begin

     try
        New(FFT) ;
        Spectrum.StartAt := StartAt ;
        Spectrum.EndAt := EndAt ;
        Spectrum.Available := False ;
        Spectrum.NumAveraged := 0 ;
        Spectrum.RecordSize := Data.RecordSize ;
        Spectrum.AvgDCMean := 0.0 ;

        npFFT := Data.RecordSize div 2 ;
        dFreq := 1.0 / (Spectrum.RecordSize*CdrFH.dt) ;
        for i := 0 to npFFT-1 do begin
            Spectrum.Power[i] := 0.0 ;
            Spectrum.Frequency[i] := (i+1)*dFreq ;
            end ;

        for Rec := StartAt to EndAt do
            if  (RecordStatus^[Rec].Valid = True)
            and (RecordStatus^[Rec].RecType = RecType) then begin

            { Read record from file }
             ReadRecord(Rec, ADC^, xMin, xMax ) ;

            { Calculate mean signal level of DC Channel }
            Sum := 0.0 ;
            j := DCChannel.ChannelOffset ;
            for i := 0 to Spectrum.RecordSize-1 do begin
                Sum := Sum + (ADC^[j] - DCChannel.ADCZero) ;
                j := j + CdrFH.NumChannels ;
                end ;
            Spectrum.AvgDCMean := Spectrum.AvgDCMean +
                                  (Sum/Spectrum.RecordSize) * DCChannel.ADCScale ;

            { Calculate mean signal level of AC Channel }
            Sum := 0.0 ;
            j := ACChannel.ChannelOffset ;
            for i := 0 to Spectrum.RecordSize-1 do begin
                Sum := Sum + (ADC^[j]) ;
                j := j + CdrFH.NumChannels ;
                end ;
            MeanAC := Sum / Spectrum.RecordSize ;
            ACChannel.ADCZero := Trunc(MeanAC) ;

            { Copy signal to be transformed into FFT buffer,
              after subtracting DC level }
            j := ACChannel.ChannelOffset ;
            for i := 1 to Spectrum.RecordSize do begin
                FFT^[i] := (ADC^[j] - ACChannel.ADCZero)*ACChannel.ADCScale ;
                j := j + CdrFH.NumChannels ;
                end ;

            { Subtract linear trend from signal, if required }
            if ckSubtractTrends.checked then
               SubtractLinearTrend( FFT^,1, Data.RecordSize ) ;

            { Apply 10% cosine windows, if required }
            if rbCosineWindow.Checked then
               CosineWindow( 1, Data.RecordSize, FFT^, VarianceCorrection ) 
            else VarianceCorrection := 1.0 ;

            {Transform to frequency domain }
            RealFFT( FFT^, npFFT, 1 ) ;

            { Compute power spectrum }
            j := 3 ;
            n := 0 ;
            for i := 2 to npFFT do begin
               YReal := FFT^[j] ;
               YImag := FFT^[j+1] ;
               Spectrum.Power[n] := Spectrum.Power[n] + ((YReal*YReal) + (YImag*YImag)) ;
               Inc(n) ;
               j := j + 2 ;
               end ;
            Spectrum.Power[n] := FFT^[2]*FFT^[2] ;
            Spectrum.NumPoints := n + 1 ;

            if RecType = Test then
               edSpecStatus.text := format( ' Test %d/%d ', [Rec,Spectrum.EndAt] )
            else
               edSpecStatus.text := format( ' Back %d/%d ', [Rec,Spectrum.EndAt] ) ;

            { Allow other events to be processed }
            application.ProcessMessages ;

            Spectrum.Available := True ;
            Inc(Spectrum.NumAveraged) ;
            end ;

        { Average spectral time periods }
        if Spectrum.NumAveraged > 0 then begin
           Denom := (Spectrum.NumAveraged*Spectrum.RecordSize*VarianceCorrection)
                    /(2.0*CdrFH.DT) ;
           for i := 0 to Spectrum.NumPoints-1 do begin
               Spectrum.Power[i] := Spectrum.Power[i] / Denom ;
               end ;

           { Average adjacent frequencies within spectrum }
           AverageFrequencies( Spectrum ) ;

           { Compute total variance of spectrum and median power  frequency }
           SpectralVariance( Spectrum ) ;

           { Average DC channel signal level
             (used for unitary current calculation)}
           Spectrum.AvgDCMean := Spectrum.AvgDCMean / Spectrum.NumAveraged ;

           { Place display cursors at ends of data }
           Spectrum.Cursor0.xVal := Spectrum.Frequency[0] ;
           Spectrum.Cursor1.xVal := Spectrum.Frequency[Spectrum.NumPoints-1] ;

           { Display spectrum results }
           sgSpecResults.RowCount := 2 ;
           Row := 0 ;
           sgSpecResults.Cells[0,Row] := format(' Median power frequency = %.4g Hz',
                                                [PowerSpectrum^.MedianFrequency]) ;
           Inc(Row) ;
           sgSpecResults.Cells[0,Row] := format(' Total variance = %.4g %s^2',
                                               [PowerSpectrum^.Variance,
                                                ACChannel.ADCUnits]) ;
           end ;

     finally
            Dispose(FFT) ;
            end ;
     end ;

procedure TPwrSpecFrm.SubtractLinearTrend( var Y : Array of single ;
                                           iStart,iEnd : Integer ) ;
{ ------------------------------------------
  Subtract any linear trend from data in Buf
  (NOTE ... Data starts at 1 and ends at Data.RecordSize )
  ------------------------------------------}
var
   SumX,SumY,SumXX,SumXY,AvgX,AvgY,Slope,X,YIntercept : single ;
   i : Integer ;
begin

     {Calculate average of X and Y data points}
     SumX := 0.0 ;
     SumY := 0.0 ;
     for i := iStart to iEnd do begin
         SumX := SumX + i ;
         SumY := SumY + Y[i] ;
         end ;
     AvgX := SumX / (iEnd - iStart + 1) ;
     AvgY := SumY / (iEnd - iStart + 1) ;

     { Calculate best fit straight line }
     SumXY  := 0.0 ;
     SumXX := 0.0 ;
     for i := iStart to iEnd do begin
         X := i - AvgX ;
         SumXY := SumXY + (Y[i] - AvgY)*X ;
	 SumXX := SumXX + X*X ;
         end ;
     Slope := SumXY / SumXX ;
     YIntercept := AvgY - Slope*AvgX ;

      for i := iStart to iEnd do begin
          Y[i] := Y[i] - (Slope*i) - YIntercept ;
          end ;
      end ;


procedure TPwrSpecFrm.CosineWindow( iStart,iEnd : Integer ;
                                    var Y : Array of single ;
                                    var VarianceCorrection : single ) ;
{ -------------------------------------------------
  Apply 10% cosine taper to ends of data in array Y
  Starting at iStart and ending at iEnd.
  Return variance correction factor in VarCorrection
  -------------------------------------------------}
var
   i,nPoints,i10,i90 : Integer ;
   Pi10,YScale,AmplitudeCorrection : single ;
begin
    nPoints := iEnd - iStart + 1 ;
    i10 := nPoints div 10 ;
    i90 := npoints - i10 ;
    Pi10 := Pi / i10 ;
    AmplitudeCorrection := 0.0 ;
    VarianceCorrection := 0.0 ;
    for i := iStart to iEnd do begin
        { Scaling factors }
        if i <= i10 then YScale := 0.5*(1.0 - cos((i - iStart)*Pi10))
        else if i >= i90 then YScale := 0.5*(1.0 - cos((iEnd - i)*Pi10))
        else YScale := 1.0 ;

        Y[i] := Y[i] * YScale ;

        AmplitudeCorrection := AmplitudeCorrection + YScale ;
        VarianceCorrection := VarianceCorrection + YScale*Yscale
        end ;
    AmplitudeCorrection := AmplitudeCorrection / nPoints ;
    VarianceCorrection := VarianceCorrection / nPoints ;
    { Re-scale data to account for loss amplitude at ends }
   { for i := iStart to iEnd do Y[i] := Y[i] / AmplitudeCorrection ;}

    end ;


procedure TPwrSpecFrm.AverageFrequencies( var Spectrum : TSpectrum ) ;
{ ------------------------------------------------
  Average adjacent frequency bands within spectrum
  (Using linear or logarithmic rule)
  ------------------------------------------------}
var
   iFrom,iTo,NumAveraged,NumFrequenciesRequired,BlockSize,BlockCount : Integer ;
   Pwr,Freq : single ;
begin

     if rbLinFreqAveraging.Checked then begin
        { ** Linear averaging **
          Average adjacent frequencies in blocks of fixed size,
          set by user (edNumFreqAveraged) }
        iFrom := 0 ;
        iTo := 0 ;
        NumAveraged := 0 ;
        NumFrequenciesRequired := ExtractInt( edNumFreqAveraged.text ) ;
        Pwr := 0.0 ;
        Freq := 0.0 ;
        repeat
            { Summate power and frequency for average }
            Pwr := Pwr + Spectrum.Power[iFrom] ;
            Freq := Freq + Spectrum.Frequency[iFrom] ;
            Inc(iFrom) ;
            Inc(NumAveraged) ;
            { Calculate average when required }
            if NumAveraged = NumFrequenciesRequired then begin
               Spectrum.Power[iTo] := Pwr / NumAveraged ;
               Spectrum.Frequency[iTo] := Freq / NumAveraged ;
               Inc(iTo) ;
               NumAveraged := 0 ;
               Pwr := 0.0 ;
               Freq := 0.0 ;
               end ;
            until iFrom >= Spectrum.NumPoints ;
        Spectrum.NumPoints := iTo ;
        end
     else if rbLogFreqAveraging.Checked then begin
        { ** Logarithmic averaging **
          Double the size of the averaging block with increasing
          frequency, at intervals set by the user (edNumFreqAveraged) }
        iFrom := 0 ;
        iTo := 0 ;
        NumAveraged := 0 ;
        { Start with  no averaging }
        NumFrequenciesRequired := 1 ;
        { Set size of block using this average }
        BlockSize := ExtractInt( edNumFreqAveraged.text ) ;
        BlockCount := 0 ;
        Pwr := 0.0 ;
        Freq := 0.0 ;
        repeat
              { Summate power and frequency for average }
              Pwr := Pwr + Spectrum.Power[iFrom] ;
              Freq := Freq + Spectrum.Frequency[iFrom] ;
              Inc(iFrom) ;
              Inc(NumAveraged) ;
              { Calculate average when required }
              if NumAveraged = NumFrequenciesRequired then begin
                 Spectrum.Power[iTo] := Pwr / NumAveraged ;
                 Spectrum.Frequency[iTo] := Freq / NumAveraged ;
                 Inc(iTo) ;
                 NumAveraged := 0 ;
                 Pwr := 0.0 ;
                 Freq := 0.0 ;
                 Inc(BlockCount) ;
                 { Double size of averaging block, when required }
                 if BlockCount = BlockSize then begin
                    NumFrequenciesRequired := NumFrequenciesRequired*2 ;
                    BlockCount := 0 ;
                    end ;
                 end ;
              until iFrom >= Spectrum.NumPoints ;
        Spectrum.NumPoints := iTo ;
        end ;

     end ;


procedure TPwrSpecFrm.SpectralVariance( var Spectrum : TSpectrum ) ;
{ ---------------------------------------------------------------
  Calculate total variance of spectrum and median power frequency
  ---------------------------------------------------------------}
var
   Sum,BinWidth,dF,HalfPower,SumHi,SumLo : single ;
   i,iLo,iHi : Integer ;
begin
     { Calculate variance as integral of power spectrum }
     Spectrum.Variance := 0.0 ;
     for i := 0 to Spectrum.NumPoints-1 do begin
         if i < (Spectrum.NumPoints-2) then
            BinWidth := Spectrum.Frequency[i+1] - Spectrum.Frequency[i] ;
         Spectrum.Variance := Spectrum.Variance + Spectrum.Power[i]*BinWidth ;
         end ;

     { Calculate median power frequency }
     i := 0 ;
     Sum := 0.0 ;
     HalfPower := Spectrum.Variance/2.0 ;
     repeat
         if i < Spectrum.NumPoints-2 then
            BinWidth := Spectrum.Frequency[i+1] - Spectrum.Frequency[i] ;
         Sum := Sum + Spectrum.Power[i]*BinWidth ;
         Inc(i) ;
         until (Sum >= HalfPower) or (i >= Spectrum.NumPoints) ;

     iLo := MaxInt([MinInt([i-2,Spectrum.NumPoints-1]),0]) ;
     iHi := MinInt([i-1,Spectrum.NumPoints-1]) ;
     SumLo := Sum - (BinWidth*Spectrum.Power[iHi]) ;
     SumHi := Sum ;

     if iLo <> iHi then begin
        dF := ((Spectrum.Frequency[iHi] - Spectrum.Frequency[iLo])
               * (HalfPower - SumLo)) / (SumHi - SumLo ) ;
        end
     else dF := 0.0 ;

     Spectrum.MedianFrequency := Spectrum.Frequency[iLo] + dF ;

     end ;


procedure TPwrSpecFrm.PlotSpectrum ;
{ -------------------
  Plot power spectrum
  -------------------}
var
   i,Row : Integer ;
   xy : TxyBuf ;
begin
     try
        { Create x,y VarPlot data object }
        xy := TxyBuf.Create ;
        { Read selected X and Y axis data from arrays into xy }
        xy.NumPoints := PowerSpectrum^.NumPoints ;
        for i := 0 to PowerSpectrum^.NumPoints-1 do begin
               xy.x[i] := PowerSpectrum^.Frequency[i] ;
               xy.y[i] := PowerSpectrum^.Power[i] ;
               end ;

        { Set VarPlot size/fonts for screen }
        SpecPlot.Left :=   pbSpecDisplay.Width div 6 ;
        SpecPlot.Right :=  pbSpecDisplay.Width - (pbSpecDisplay.Width div 10) ;
        SpecPlot.Top :=    pbSpecDisplay.Height div 8 ;
        SpecPlot.Bottom := pbSpecDisplay.Height - SpecPlot.Top*2 ;
        SpecPlot.MarkerSize := Settings.Plot.MarkerSize ;
        xy.MarkerSize := SpecPlot.MarkerSize ;
        xy.Color := clRed ;
        SpecPlot.LineThickness := Settings.Plot.LineThickness ;
        SpecPlot.NumGraphs := 1 ;
        SpecPlot.Title := ' ' ;

        pbSpecDisplay.canvas.font.name := Settings.Plot.FontName ;
        pbSpecDisplay.canvas.font.size := Settings.Plot.FontSize ;
        { ** Plot graph on screen ** }
        pbSpecDisplay.Canvas.Brush.color := clWhite ;
        pbSpecDisplay.Canvas.Pen.Width := SpecPlot.LineThickness ;
        pbSpecDisplay.canvas.FillRect( pbSpecDisplay.canvas.ClipRect ) ;

        { Plot axes }
        DrawAxes( pbSpecDisplay.canvas, SpecPlot, xy ) ;

        { Join up power spectrum points with lines (if required)
          (Note. If a fitted curve available postpone line till later) }
        if Settings.Plot.ShowLines
           and (not PowerSpectrum^.Equation.Available) then begin
           DrawLine(pbSpecDisplay.canvas,SpecPlot,xy) ;
           end ;

        { Plot data points }
        if Settings.Plot.ShowMarkers then DrawMarkers(pbSpecDisplay.canvas,SpecPlot,xy) ;

        { Read selected X and Y axis data from arrays into xy }
        xy.NumPoints := PowerSpectrum^.NumPoints ;
        for i := 0 to PowerSpectrum^.NumPoints-1 do begin
               xy.x[i] := PowerSpectrum^.Frequency[i] ;
               xy.y[i] := PowerSpectrum^.Power[i] ;
               end ;

        { Display background spectrum }
        if ckSpecSubtractBackground.checked
           and BackgroundSpectrum^.Available then begin
           for i := 0 to PowerSpectrum^.NumPoints-1 do
               xy.y[i] := BackgroundSpectrum^.Power[i] ;
           if Settings.Plot.ShowLines then DrawLine(pbSpecDisplay.canvas,SpecPlot,xy ) ;
           if Settings.Plot.ShowMarkers then DrawMarkers( pbSpecDisplay.canvas, SpecPlot, xy ) ;
           end ;

        { Display fitted spectrum (if one available) }
        if PowerSpectrum^.Equation.Available then begin
           for i := 0 to PowerSpectrum^.NumPoints-1 do
               xy.y[i] := MathFunc( PowerSpectrum^.Equation,
                                    PowerSpectrum^.Frequency[i]) ;
           DrawLine(pbSpecDisplay.canvas,SpecPlot,xy ) ;
           end ;

        { Set position of cursor 0 to nearest data point }
        PowerSpectrum^.Cursor0.Index := FindNearest(PowerSpectrum^.Frequency,
                                                    PowerSpectrum^.NumPoints,
                                                    PowerSpectrum^.Cursor0.xVal) ;
        PowerSpectrum^.Cursor0.xVal := PowerSpectrum^.Frequency[
                                                   PowerSpectrum^.Cursor0.Index] ;
        DrawVerticalCursor( pbSpecDisplay, SpecPlot,
                            PowerSpectrum^.Cursor0.xVal,
                            PowerSpectrum^.Cursor0.Lab) ;

        { Set position of cursor 1 to nearest data point}
        PowerSpectrum^.Cursor1.Index := FindNearest(PowerSpectrum^.Frequency,
                                                    PowerSpectrum^.NumPoints,
                                                    PowerSpectrum^.Cursor1.xVal) ;
        PowerSpectrum^.Cursor1.xVal := PowerSpectrum^.Frequency[
                                                   PowerSpectrum^.Cursor1.Index] ;
        DrawVerticalCursor( pbSpecDisplay, SpecPlot,
                            PowerSpectrum^.Cursor1.xVal,
                            PowerSpectrum^.Cursor1.Lab) ;

     finally
        xy.Free ;
        end ;

     end ;


procedure TPwrSpecFrm.PrintSpectrum ;
{ ---------------------
  Print power spectrum
  ---------------------}
var
   i,Row,yEndOfText : Integer ;
   xy : TxyBuf ;
   Canvas : TCanvas ;
begin
     try

        Screen.Cursor := crHourglass ;
        Canvas := Printer.Canvas ;

        { Create x,y plotting data object }
        xy := TxyBuf.Create ;
        { Read selected X and Y axis data from arrays into xy }
        xy.NumPoints := PowerSpectrum^.NumPoints ;
        for i := 0 to PowerSpectrum^.NumPoints-1 do begin
               xy.x[i] := PowerSpectrum^.Frequency[i] ;
               xy.y[i] := PowerSpectrum^.Power[i] ;
               end ;

        { Set Plot size/fonts }
        SpecPlot.Left :=   PrinterCmToPixels('H',Settings.Plot.LeftMargin) ;
        SpecPlot.Right :=  Printer.PageWidth -
                          PrinterCmToPixels('H',Settings.Plot.RightMargin) ;
        SpecPlot.Top :=    PrinterCmToPixels('V',Settings.Plot.TopMargin) ;
        SpecPlot.Bottom := Printer.PageHeight -
                          PrinterCmToPixels('V',Settings.Plot.BottomMargin) ;

        Printer.BeginDoc ;

        Canvas.Pen.Color := clBlack ;
        SpecPlot.MarkerSize := Settings.Plot.MarkerSize ;
        xy.MarkerSize := PrinterPointsToPixels(SpecPlot.MarkerSize) ;
        xy.Color := clRed ;
        SpecPlot.LineThickness := PrinterPointsToPixels(Settings.Plot.LineThickness) ;
        SpecPlot.NumGraphs := 1 ;
        SpecPlot.Title := ' ' ;

        Canvas.font.name := Settings.Plot.FontName ;
        Canvas.font.size := PrinterPointsToPixels(Settings.Plot.FontSize) ;

        { ** Plot graph on printer ** }
        Canvas.Brush.color := clWhite ;
        Canvas.Pen.Width := SpecPlot.LineThickness ;

        { Print file/experiment identification information }
        PrintPageTitle( Canvas, PowerSpectrum^.Equation.Available,
                        sgSpecResults, YEndofText ) ;
        if SpecPlot.Top < YEndofText then SpecPlot.Top := YEndofText ;


        { Plot axes }
        DrawAxes( Canvas, SpecPlot, xy ) ;

        { Join up power spectrum points with lines (if required)
          (Note. If a fitted curve available postpone line till later) }
        if Settings.Plot.ShowLines
           and (not PowerSpectrum^.Equation.Available) then begin
           DrawLine(Canvas,SpecPlot,xy) ;
           end ;

        { Plot data points }
        if Settings.Plot.ShowMarkers then DrawMarkers(Canvas,SpecPlot,xy) ;

        { Read selected X and Y axis data from arrays into xy }
        xy.NumPoints := PowerSpectrum^.NumPoints ;
        for i := 0 to PowerSpectrum^.NumPoints-1 do begin
               xy.x[i] := PowerSpectrum^.Frequency[i] ;
               xy.y[i] := PowerSpectrum^.Power[i] ;
               end ;

        { Display background spectrum }
        if ckSpecSubtractBackground.checked
           and BackgroundSpectrum^.Available then begin
           for i := 0 to PowerSpectrum^.NumPoints-1 do
               xy.y[i] := BackgroundSpectrum^.Power[i] ;
           if Settings.Plot.ShowLines then DrawLine(Canvas,SpecPlot,xy ) ;
           if Settings.Plot.ShowMarkers then DrawMarkers(Canvas, SpecPlot, xy ) ;
           end ;

        { Display fitted spectrum (if one available) }
        if PowerSpectrum^.Equation.Available then begin
           for i := 0 to PowerSpectrum^.NumPoints-1 do
               xy.y[i] := MathFunc( PowerSpectrum^.Equation,
                                    PowerSpectrum^.Frequency[i]) ;
           DrawLine(Canvas,SpecPlot,xy ) ;
           end ;

     finally
        Printer.EndDoc ;
        xy.Free ;
        Screen.Cursor := crDefault ;
        end ;

     end ;


procedure TPwrSpecFrm.CopySpectrumDataToClipBoard ;
{ ----------------------------------------------------------------
  Copy the data points in the power spectrum graph to the clipboard
  ----------------------------------------------------------------
    25/6/98 Clipboard buffer limit reduced to 31000}
const
     BufSize = 31000 ;
     NumBytesPerNumber = 12 ;
var
   Rec,i,iSkip,NumBytesNeeded,NumColumns : LongInt ;
   xOffset : single ;
   Line : String ;
   CopyBuf0,Line0 : PChar ;
   OK : Boolean ;

begin
     try
        OK := OpenClipboard( Main.Handle ) ;
        OK := EmptyClipBoard ;
        { Allocate ASCII text buffer to hold graph }
        CopyBuf0 := StrAlloc( BufSize ) ;
        StrPCopy( CopyBuf0, '' ) ;
        Line0 := StrAlloc(  256 ) ;

        { Determine sample skip factor to ensure that the compete record
          fits into the buffer }
        NumColumns := 2 ;
        if PowerSpectrum^.Equation.Available then NumColumns := NumColumns + 1 ;
        NumBytesNeeded := PowerSpectrum^.NumPoints*NumBytesPerNumber*NumColumns ;
        iSkip := MaxInt( [(NumBytesNeeded+BufSize-1) div BufSize,1]) ;

        i := 0 ;
        screen.cursor := crHourglass ;
        repeat
              Line := format( '%8.5g', [PowerSpectrum^.Frequency[i]] ) + chr(9) ;
              Line := Line + format( '%8.5g', [PowerSpectrum^.Power[i]] ) ;
              if PowerSpectrum^.Equation.Available then begin
                 Line := Line + chr(9) + format( '%8.5g',
                 [MathFunc(PowerSpectrum^.Equation,PowerSpectrum^.Frequency[i])]) ;
                 end ;
             Line := Line + chr(13) + chr(10) ;
             StrPCopy( Line0, Line ) ;
             CopyBuf0 := StrCat( CopyBuf0, Line0 ) ;
             i := i + iSkip ;
             until i >= PowerSpectrum^.NumPoints ;

        { Copy text accumulated in copy buffer to clipboard }
         ClipBoard.SetTextBuf( CopyBuf0 ) ;

     finally
         screen.cursor := crDefault ;
         { Dispose of buffers }
         StrDispose( Line0 ) ;
         StrDispose( CopyBuf0 ) ;
         OK := CloseClipboard ;
         end ;
     end ;


procedure TPwrSpecFrm.pbSpecDisplayPaint(Sender: TObject);
{ ----------------------------
  Update mean/variance display
  ----------------------------}
begin
     if PowerSpectrum^.Available then begin
        PlotSpectrum ;
        Main.CopyAndPrintMenus( True, True ) ;
        end
     else begin
          EraseDisplay( pbSpecDisplay ) ;
          lbSpecCursor0.Visible := False ;
          lbSpecCursor1.Visible := False ;
          Main.CopyAndPrintMenus( False, False ) ;
          end ;
     end;


procedure TPwrSpecFrm.bSpecSetAxesClick(Sender: TObject);
{ -----------------------------
  Customise power spectrum Plot
  -----------------------------}
begin
     SetAxesFrm.SetPlot := SpecPlot ;
     SetAxesFrm.ShowModal ;
     if SetAxesFrm.ModalResult = mrOK then begin
        SpecPlot := SetAxesFrm.SetPlot ;
        pbSpecDisplay.Repaint ;
        end ;
     end;


procedure TPwrSpecFrm.bFitLorentzianClick(Sender: TObject);
{ -------------------------------------------
  Fit a Lorentzian function to power spectrum
  -------------------------------------------}
var
   i,nFit,iStart,iEnd,Row,nPars : Integer ;
   ParUnits,ParName : string ;
   KeepPars : Array[0..LastParameter] of Single ;
   FitData : ^TXYData ;
   Tau,IUnit : single ;
begin

     Try
        New( FitData ) ;
        { Select type of equation to be fitted }
        case cbSpecEquation.ItemIndex  of
          1 : PowerSpectrum^.Equation.EqnType := Lorentzian ;
          2 : PowerSpectrum^.Equation.EqnType := Lorentzian2 ;
          3 : PowerSpectrum^.Equation.EqnType := LorAndOneOverF ;
          4 : PowerSpectrum^.Equation.EqnType := MEPCNoise ;
          else  PowerSpectrum^.Equation.EqnType := None ;
          end ;

        PowerSpectrum^.Equation.Available := False ;
        PowerSpectrum^.Equation.ParametersSet := False ;
        if PowerSpectrum^.Equation.EqnType <> None then begin

           { Get range of points to be fitted }
           iStart := MinInt( [PowerSpectrum^.Cursor0.Index,
                              PowerSpectrum^.Cursor1.Index]) ;
           iEnd :=   MaxInt( [PowerSpectrum^.Cursor0.Index,
                              PowerSpectrum^.Cursor1.Index]) ;

           { Copy data to fitting array }
           nFit := 0 ;
           for i := iStart to iEnd do begin
               FitData^.x[i] := PowerSpectrum^.Frequency[i] ;
               FitData^.y[i] := PowerSpectrum^.Power[i] ;
               Inc(nFit) ;
               end ;

           { Create an initial set of guesses for parameters
             (keeping constant any "fixed" parameters) }
           nPars := GetNumEquationParameters( PowerSpectrum^.Equation ) ;
           for i := 0 to nPars-1 do KeepPars[i] := PowerSpectrum^.Equation.Par[i] ;
           InitialiseParameters( FitData^, nFit, PowerSpectrum^.Equation ) ;
           for i := 0 to nPars-1 do if PowerSpectrum^.Equation.ParSD[i] < 0. then
               PowerSpectrum^.Equation.Par[i] := KeepPars[i] ;

           { Let user modify initial parameter settings and/or
           fix parameters at constant values }
           SetFitParsFrm.Equation := PowerSpectrum^.Equation ;
           SetFitParsFrm.ShowModal ;
           if SetFitParsFrm.ModalResult = mrOK then
              PowerSpectrum^.Equation := SetFitParsFrm.Equation ;
           { Prevent FitCurve from changing parameter settings }
           PowerSpectrum^.Equation.ParametersSet := True ;

           { Fit curve using non-linear regression }
           if nFit > 0 then FitCurve( FitData^, nFit, PowerSpectrum^.Equation )
           else PowerSpectrum^.Equation.Available := False ;

           if PowerSpectrum^.Equation.Available then begin

              sgSpecResults.RowCount := nPars + 3 ;

              Row := 0 ;
              sgSpecResults.Cells[0,Row] := GetFormula(PowerSpectrum^.Equation) ;
              Inc(Row) ;

              { Best fit parameters and standard error }
              for i := 0 to nPars-1 do begin
                  GetParameterInfo(PowerSpectrum^.Equation,ParName,ParUnits,
                                   'Hz',ACChannel.ADCUnits,i) ;
                  if PowerSpectrum^.Equation.ParSD[i] >= 0.0 then
                     sgSpecResults.Cells[0,Row] := format(' %s = %.4g +/- %.4g (sd) %s',
                                                   [ParName,
                                                    PowerSpectrum^.Equation.Par[i],
                                                    PowerSpectrum^.Equation.ParSD[i],
                                                    ParUnits] )
                  else
                     { Negative s.d. indicates a fixed parameter }
                     sgSpecResults.Cells[0,Row] := format(' %s = %.4g (fixed) %s',
                                                   [ParName,
                                                    PowerSpectrum^.Equation.Par[i],
                                                    ParUnits] ) ;
                  Inc(Row) ;
                  end ;

              { If a Lorentzian has been fitted,
                calculate unitary current and time constant }
              if PowerSpectrum^.Equation.EqnType = Lorentzian then begin
                 sgSpecResults.RowCount := sgSpecResults.RowCount + 2 ;
                 Tau := SecsToMs / (2.0*Pi*PowerSpectrum^.Equation.Par[1]) ;
                 sgSpecResults.Cells[0,Row] := format(' Tau = %.4g ms',[Tau]) ;
                 Inc(Row) ;
                 IUnit := (Pi*PowerSpectrum^.Equation.Par[0]
                           *PowerSpectrum^.Equation.Par[1]) /
                           (2.0*PowerSpectrum^.AvgDCMean) ;
                 sgSpecResults.Cells[0,Row] := format(' Iu = %.4g %s',
                                               [IUnit,ACChannel.ADCUnits]) ;
                 Inc(Row) ;
                 end ;

              { If the MEPC Noise spectrum has been fitted
                Calculate rising and decay time constants }
              if PowerSpectrum^.Equation.EqnType = MEPCNoise then begin
                 sgSpecResults.RowCount := sgSpecResults.RowCount + 2 ;
                 Tau := SecsToMs / (2.0*Pi*PowerSpectrum^.Equation.Par[2]) ;
                 sgSpecResults.Cells[0,Row] := format(' TauR = %.4g ms',[Tau]) ;
                 Inc(Row) ;
                 Tau := SecsToMs / (2.0*Pi*PowerSpectrum^.Equation.Par[1]) ;
                 sgSpecResults.Cells[0,Row] := format(' TauD = %.4g ms',[Tau]) ;
                 Inc(Row) ;
                 end ;

              { Residual standard deviation }
              sgSpecResults.Cells[0,Row] := format(' Residual S.D. = %.4g %s',
                                            [PowerSpectrum^.Equation.ResidualSD,
                                             ACChannel.ADCUnits] ) ;
              Inc(Row) ;

              { Statistical degrees of freedom }
              sgSpecResults.Cells[0,Row] := format(' Degrees of freedom = %d ',
                                            [PowerSpectrum^.Equation.DegreesFreedom]) ;
              Inc(Row) ;

              { No. of iterations }
              sgSpecResults.Cells[0,Row] := format(' No. of iterations = %d ',
                                            [PowerSpectrum^.Equation.NumIterations]) ;
              Inc(Row) ;

              end
           else begin
              sgSpecResults.Cells[0,Row] := ' Fit Failed!' ;
              end ;
           end ;
     finally
            Dispose(FitData) ;
            pbSpecDisplay.Repaint ;
            end ;
     end ;


procedure TPwrSpecFrm.DrawVerticalCursor( var pb : TPaintBox ;
                              var Plot : TPlot ;
                              X : single ;
                              const Lab : TLabel ) ;
{ --------------------------------------------------------
  Draw/Remove a vertical cursor on the record display area
  --------------------------------------------------------}
var
   Diam : LongInt ;
   xPos : Integer ;
   OldColor : TColor ;
   OldStyle : TPenStyle ;
   OldMode : TPenMode ;
   OK : Boolean ;
begin

     with pb.canvas do begin
          OldColor := pen.color ;
          OldStyle := pen.Style ;
          OldMode := pen.mode ;
          pen.mode := pmXor ;
          pen.color := clRed ;
          end ;

     OK := ScaleXToScreenCoord( Plot, X, xPos ) ;

     if OK then begin
           pb.canvas.polyline([Point(xPos,Plot.Top),Point(xPos,Plot.Bottom)] ) ;

           if (xPos >= Plot.Left) and (xPos <= Plot.Right) then begin
              Lab.Left := pb.Left + xPos ;
              Lab.Top := pb.Top + pb.Height + 2;
              Lab.visible := True ;
              end
           else Lab.visible := False ;
           end
     else Lab.visible := False ;

     with pb.canvas do begin
          pen.style := OldStyle ;
          pen.color := OldColor ;
          pen.mode := OldMode ;
          end ;
     end ;


procedure TPwrSpecFrm.pbSpecDisplayMouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);
{ --------------------------------------------------------------------
  Power spectrum display :  Update display cursors when mouse is moved
  --------------------------------------------------------------------}
Const
     Margin = 4 ;
var
   OldIndex,xPos0,xPos1 : Integer ;
   OK : Boolean ;
begin

     if PowerSpectrum^.Available then begin
        if (not PowerSpectrum^.Cursor0.Active)
           and (not PowerSpectrum^.Cursor1.Active) then begin

           OK := ScaleXToScreenCoord( SpecPlot, PowerSpectrum^.Cursor0.xVal, xPos0 ) ;
           OK := ScaleXToScreenCoord( SpecPlot, PowerSpectrum^.Cursor1.xVal, xPos1 ) ;
           if Abs(X - xPos0) < Margin then begin
                pbSpecDisplay.Cursor := crSizeWE ;
                PowerSpectrum^.Cursor0.Selected := True ;
                end
           else if Abs(X - xPos1) < Margin  then begin
                pbSpecDisplay.Cursor := crSizeWE ;
                PowerSpectrum^.Cursor1.Selected := True ;
                end
           else begin
                pbSpecDisplay.Cursor := crDefault ;
                PowerSpectrum^.Cursor0.Selected := False ;
                PowerSpectrum^.Cursor1.Selected := False ;
                end ;
           end
        else begin

          { If a cursor is active, move it to new position }

          if PowerSpectrum^.Cursor0.Active then begin
              MoveVerticalCursor( pbSpecDisplay, SpecPlot, PowerSpectrum^.Cursor0, X ) ;
              end
          else if PowerSpectrum^.Cursor1.Active then begin
              MoveVerticalCursor( pbSpecDisplay, SpecPlot, PowerSpectrum^.Cursor1, X ) ;
              end ;
          end ;
        end ;
     end ;


procedure TPwrSpecFrm.MoveVerticalCursor( var pb : TPaintbox ;
                                          var Plot : TPlot ;
                                          var Cursor : TCursor ;
                                          XPix : Integer ) ;
{ ---------------------------------------
  Move vertical cursor on graph Plot
  --------------------------------------}
begin

     DrawVerticalCursor( pb, Plot, Cursor.xVal, Cursor.Lab ) ;
     ScaleScreenCoordToX( Plot, XPix,  Cursor.XVal ) ;
     DrawVerticalCursor( pb, Plot, Cursor.xVal, Cursor.Lab ) ;
     end ;


procedure TPwrSpecFrm.pbSpecDisplayMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
{ ---------------------------------------------------
  Power spectrum display :
  If a cursor has been selected make it active when
  mouse button is depressed
  ---------------------------------------------------}
begin
     if PowerSpectrum^.Available then begin
        PowerSpectrum^.Cursor0.Active := PowerSpectrum^.Cursor0.Selected ;
        PowerSpectrum^.Cursor1.Active := PowerSpectrum^.Cursor1.Selected ;
        end ;
     end;


procedure TPwrSpecFrm.pbSpecDisplayMouseUp(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
{ ------------------------------------------------------------------------
  Power spectrum display : Make cursor inactive when mouse button released
  ------------------------------------------------------------------------}
begin
     if PowerSpectrum^.Available then begin
        PowerSpectrum^.Cursor0.Active := False ;
        PowerSpectrum^.Cursor1.Active := False ;
        { Request a repaint to update cursor positions }
        pbSpecDisplay.RePaint ;
        end ;
     end;


procedure TPwrSpecFrm.cbSpecEquationChange(Sender: TObject);
var
   i : Integer ;
begin
     { Clear any fixed parameters }
     for i := 0 to High(PowerSpectrum^.Equation.ParSD) do
         PowerSpectrum^.Equation.ParSD[i] := 0.0 ;
     pbSpecDisplay.Repaint ;
     end;



procedure TPwrSpecFrm.FormResize(Sender: TObject);
{ -------------------------------------------
  Resize components from form size is changed
  -------------------------------------------}
begin
     TabbedNotebook.Height := ClientHeight - TabbedNotebook.Top - 5 ;
     TabbedNotebook.Width := ClientWidth - TabbedNotebook.Left - 5 ;

     pbACDisplay.Width := TabbedNotebook.Width - pbACDisplay.Left - 20 ;
     pbDCDisplay.Width := TabbedNotebook.Width - pbDCDisplay.Left - 20 ;
     pbVarDisplay.Width := TabbedNotebook.Width - pbVarDisplay.Left - 20 ;
     pbSpecDisplay.Width := TabbedNotebook.Width - pbSpecDisplay.Left - 20 ;

     { Extend variance page control group to bottom of page }
     VarGrp.Height := TabbedNotebook.Height - VarGrp.Top - 40 ;
     edVarStatus.Top := VarGrp.Height - edVarStatus.Height - 2 ;

     { Place variance curve fit group at bottom of page }
     VarFitGrp.Top := VarGrp.Top + VarGrp.Height - VarFitGrp.Height ;
     VarFitGrp.Width := pbVarDisplay.Width ;
     pbVarDisplay.Height := VarFitGrp.Top - pbVarDisplay.Top
                            - lbVarCursor0.Height - 2 ;

     { Extend spectrum page control group to bottom of page }
     SpecGrp.Height := TabbedNotebook.Height - SpecGrp.Top - 40 ;
     edSpecStatus.Top := SpecGrp.Height - edSpecStatus.Height - 2 ;

     { Place spectrum curve fit group at bottom of page }
     SpecFitGrp.Top := SpecGrp.Top + SpecGrp.Height - SpecFitGrp.Height ;
     SpecFitGrp.Width := pbSpecDisplay.Width ;
     pbSpecDisplay.Height := SpecFitGrp.Top - pbSpecDisplay.Top
                            - lbSpecCursor0.Height - 2 ;
     end;

procedure TPwrSpecFrm.bDataCloseClick(Sender: TObject);
begin
     Close ;
     end;

procedure TPwrSpecFrm.sbRecordOverlapDownClick(Sender: TObject);
{ -----------------------
  Decrease record overlap
  -----------------------}
var
   QuarterRecord : LongInt ;
begin
     Data.RecordOverlap := MaxInt([Data.RecordOverlap-1,0]) ;
     QuarterRecord := Data.RecordSize div 4 ;
     Data.RecordOffset := Data.RecordSize - (Data.RecordOverlap*QuarterRecord) ;
     SetRecordSize( Data.RecordSize ) ;

     GetRecord ;
     pbACDisplay.RePaint ;
     pbDCDisplay.RePaint ;

     end;

procedure TPwrSpecFrm.sbRecordOverlapUpClick(Sender: TObject);
{ -----------------------
  Increase record overlap
  -----------------------}
var
   QuarterRecord : LongInt ;
begin
     Data.RecordOverlap := MinInt([Data.RecordOverlap+1,3]) ;
     QuarterRecord := Data.RecordSize div 4 ;
     Data.RecordOffset := Data.RecordSize - (Data.RecordOverlap*QuarterRecord) ;
     SetRecordSize( Data.RecordSize ) ;

     GetRecord ;
     pbACDisplay.RePaint ;
     pbDCDisplay.RePaint ;

     end;


procedure TPwrSpecFrm.cbVarEquationChange(Sender: TObject);
var
   i : Integer ;
begin
     { Clear any fixed parameters }
     for i := 0 to High(Variance.Equation.ParSD) do
        Variance.Equation.ParSD[i] := 0.0 ;
     pbVarDisplay.Repaint ;
     end;


end.
