unit Detsigs;
{ ========================================================
  WinCDR (c) John Dempster, University of Strathclyde 1998
  SIGNAL DETECTION MODULE
  V0.99 0/3/98
  V1.0b 2/6/98 ... Changes to Record Size now correctly handled
                   Data Entry -> Parameter update now done by UpdateParameters ;
  ========================================================}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, ExtCtrls, global, maths, shared, fileio ;

type
  TDetSignalsFrm = class(TForm)
    pbDispCDR: TPaintBox;
    AnalysisGrp: TGroupBox;
    bDetect: TButton;
    bAbort: TButton;
    GroupBox8: TGroupBox;
    edRange: TEdit;
    rbAllRecords: TRadioButton;
    rbRange: TRadioButton;
    CriteriaGrp: TGroupBox;
    edYThreshold: TEdit;
    Label1: TLabel;
    Label2: TLabel;
    edDeadTime: TEdit;
    Label3: TLabel;
    cbChannel: TComboBox;
    Label4: TLabel;
    OutputGrp: TGroupBox;
    edOutputFileName: TEdit;
    Label5: TLabel;
    Label6: TLabel;
    bNewFileName: TButton;
    pbDispWCP: TPaintBox;
    lbTMax: TLabel;
    lbTMin: TLabel;
    edStatus: TEdit;
    edRecordSize: TEdit;
    edPreTriggerPercentage: TEdit;
    Label7: TLabel;
    Timer: TTimer;
    sbDispCDR: TScrollBar;
    bClose: TButton;
    EdtThreshold: TEdit;
    Label8: TLabel;
    lbTMinWCP: TLabel;
    lbTMaxWCP: TLabel;
    edBaseLineAverage: TEdit;
    SaveDialog: TSaveDialog;
    ckDisableDisplay: TCheckBox;
    procedure FormShow(Sender: TObject);
    procedure TimerTimer(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure bCloseClick(Sender: TObject);
    procedure sbDispCDRChange(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure bDetectClick(Sender: TObject);
    procedure edYThresholdKeyPress(Sender: TObject; var Key: Char);
    procedure EdtThresholdKeyPress(Sender: TObject; var Key: Char);
    procedure edDeadTimeKeyPress(Sender: TObject; var Key: Char);
    procedure edBaseLineAverageKeyPress(Sender: TObject; var Key: Char);
    procedure edPreTriggerPercentageKeyPress(Sender: TObject; var Key: Char);
    procedure edRecordSizeKeyPress(Sender: TObject; var Key: Char);
    procedure bAbortClick(Sender: TObject);
    procedure cbChannelChange(Sender: TObject);
    procedure edRangeKeyPress(Sender: TObject; var Key: Char);
    procedure bNewFileNameClick(Sender: TObject);
  private
    { Private declarations }
    procedure InitializeDisplay( const PB : TPaintBox ; var Chan : TChannel ) ;
    procedure HeapBuffers( Operation : THeapBufferOp ) ;
    procedure PlotChannel( PB : TPaintBox ; var Buf : Array of Integer ) ;
    procedure ScanForSignals ;
    procedure DrawHorizontalCursor( pb : TPaintBox ;
                                               Const Chan : TChannel ;
                                               Level : Integer ;
                                               Colour : TColor ) ;
   procedure UpdateParameters ;
  public
    { Public declarations }
  end;

var
  DetSignalsFrm: TDetSignalsFrm;

implementation

{$R *.DFM}

uses mdiform ;

type
    TState = (PlotRecord,Idle,DetectSignals,EndDetection) ;
    TPointArray = Array[0..4096] of TPoint ;
    TDetector = record
              yThreshold : Integer ;
              tCounter : Integer ;
              tThreshold : Integer ;
              yBaseline : single ;
              yOldBaseline : single ;
              RunningMean : single ;
              PreTriggerSamples : Integer ;
              DetectedAt : LongInt ;
              BlockStart : LongInt ;
              BlockSize : LongInt ;
              StartAt : LongInt ;
              EndAt : LongInt ;
              SkipSamples : LongInt ;
              JustStarted : Boolean ;
              SignalDetected : Boolean ;
              end ;
var
   ADC : ^TIntArray ;
   DetBuf : ^TIntArray ;
   xy : ^TPointArray ;
   rH : ^TWCPRecHeader ;
   BuffersAllocated : Boolean ;
   DetChannel : TChannel ;
   TimerBusy : Boolean ;
   State : TState ;
   NumBytesPerRecord : Integer ;
   MinHeight : Integer ;
   Detector : TDetector ;
   ScrollBarToBlockScale : LongInt ;

procedure TDetSignalsFrm.HeapBuffers( Operation : THeapBufferOp ) ;
{ -----------------------------------------------
  Allocate/deallocation dynamic buffers from heap
  -----------------------------------------------}
begin
     case Operation of
          Allocate : begin
             if not BuffersAllocated then begin
                New(ADC) ;
                New(xy) ;
                New(DetBuf) ;
                New(rH) ;
                BuffersAllocated := True ;
                end ;
             end ;
          Deallocate : begin
             if BuffersAllocated then begin
                Dispose(ADC) ;
                Dispose(xy) ;
                Dispose(DetBuf) ;
                Dispose( rH ) ;
                BuffersAllocated := False ;
                end ;
             end ;
          end ;
     end ;

procedure TDetSignalsFrm.FormShow(Sender: TObject);
{ --------------------------------------
  Initialisations when form is displayed
  --------------------------------------}
var
   ch,iSmall : Integer ;
begin

     Main.mnDetectSignals.Enabled := False ;
     MinHeight := AnalysisGrp.Top + AnalysisGrp.Height + 5 ;
     Resize ;

     HeapBuffers( Allocate ) ;

     { Set block of CDR file to be scanned }
     edRange.text := format(' 0.0 - %.1f s', [CdrFH.RecordDuration]);

     { Fill channel selection list }
     cbChannel.Clear ;
     for ch := 0 to CdrFH.NumChannels-1 do
          cbChannel.items.add( format('Ch.%d %s',[ch,Channel[ch].ADCName]) ) ;
     cbChannel.ItemIndex := Settings.EventDetector.Channel ;
     DetChannel := Channel[Settings.EventDetector.Channel] ;

     { Detection threhold amplitude }
     edYThreshold.text := format( ' %.3g %s', [ Settings.EventDetector.yThreshold,
                         Channel[Settings.EventDetector.Channel].ADCUnits]);
     { Detection threshold duration }
     edTThreshold.text := format( ' %.3g %s',
                          [ Settings.EventDetector.tThreshold*Settings.TScale,
                            Settings.TUnits]);
     { Dead time after an event has been detected }
     edDeadTime.text := format( ' %.3g %s',
                        [ Settings.EventDetector.DeadTime*Settings.TScale,
                          Settings.TUnits]);
     { Baseline averaging time }
     edBaselineAverage.text := format( ' %.3g %s',
                        [ Settings.EventDetector.BaselineAverage
                          *Settings.TScale*CdrFH.dt,Settings.TUnits]);
     { Size of record in which detected events are stored }
     edRecordSize.text := format( ' %d',[ Settings.EventDetector.RecordSize]);
     WCPfH.NumSamples := Settings.EventDetector.RecordSize ;
     NumBytesPerRecord := WCPfH.NumSamples*CdrFH.NumChannels*2 ;
     { % of record which is pre-trigger }
     edPreTriggerPercentage.text := format( ' %f%%',
                          [ Settings.EventDetector.PreTriggerPercentage]);

     { Update parameters to ensure consistent starting data in text boxes }
     UpdateParameters ;

     { Name of file which is to hold detected events }
     if CdrFH.WCPFileName = '' then begin
        WCPfH.FileName := ReplaceFileEnding( CdrFH.FileName, '.WCP' ) ;
        end
     else begin
        WCPfH.FileName := CdrFH.WCPFileName ;
        end ;
     edOutputFileName.text := ExtractFilename(WCPfH.FileName) ;

     { Scroll bar control }
     ScrollBarToBlockScale := MaxInt( [ ((CdrFH.NumBlocksInFile+High(iSmall))
                                                 div High(iSmall)),1]) ;
     sbDispCDR.Max := (CdrFH.NumBlocksInFile div ScrollBarToBlockScale) - 1 ;
     sbDispCDR.Position := 0 ;

     { Clear display area }
     EraseDisplay( pbDispCDR ) ;
     EraseDisplay( pbDispWCP ) ;
     Detector.SignalDetected := False ;

     State := PlotRecord ;
     TimerBusy := False ;
     end;


procedure TDetSignalsFrm.TimerTimer(Sender: TObject);
{ -----------------------------------------------------------
  Process scheduler ...
  -----------------------------------------------------------}
var
   i,j,n,ch : Integer ;
   x : single ;
   OK : Boolean ;
begin
     if not TimerBusy then  begin

        TimerBusy := True ;

        case State of
             { ** Plot a block from the source data file }
             PlotRecord : begin

                 WCPfH.NumSamples := Settings.EventDetector.RecordSize ;
                 NumBytesPerRecord := WCPfH.NumSamples*CdrFH.NumChannels*2 ;

                 CdrFH.FilePointer := sbDispCDR.Position ;
                 CdrFH.FilePointer := CdrFH.FilePointer*ScrollBarToBlockScale ;
                 CdrFH.FilePointer := CdrFH.FilePointer*CdrFH.NumBytesPerBlock
                                     + CdrFH.NumBytesInHeader ;
                 CdrFH.FilePointer := FileSeek( CdrFH.FileHandle,
                                      CdrFH.FilePointer, 0 ) ;

                 DetChannel.xMin := sbDispCDR.Position*ScrollBarToBlockScale ;
                 DetChannel.xMin := (DetChannel.xMin * CdrFH.NumSamplesPerBlock)
                                    / CdrFH.NumChannels ;
                 DetChannel.xMax := DetChannel.xMin + WCPFH.NumSamples ;

                 if FileRead(CdrFH.FileHandle,ADC^,NumBytesPerRecord)
                    = NumBytesPerRecord then PlotChannel( pbDispCDR, ADC^ ) ;
                 if Detector.SignalDetected then begin
                    PlotChannel( pbDispCDR, DetBuf^ ) ;
                    end
                 else EraseDisplay( pbDispWCP ) ;
                 State := Idle ;
                 end ;

             DetectSignals : Begin
                 ScanForSignals ;
                 end ;
             EndDetection : begin
                State := Idle ;
                close ;
                end ;
             end ;

        TimerBusy := False ;
        end;
     end ;


procedure TDetSignalsFrm.PlotChannel( PB : TPaintBox ; var Buf : Array of Integer ) ;
{ ------------------------------------------------
  Plot a block of the signal channel being scanned
  ------------------------------------------------}
var
   i,j,n,ch : Integer ;
   x : single ;
   OK : Boolean ;
begin
     { Clear display area }
     InitializeDisplay( PB, DetChannel ) ;
     EraseDisplay( PB ) ;

     n := 0 ;
     x := DetChannel.xMin ;
     for i := 0 to WCPfH.NumSamples-1 do begin
         j := (i*CdrFH.NumChannels) + DetChannel.ChannelOffset ;
         xy^[n].y := DetChannel.Bottom - Trunc(
                     DetChannel.yScale*(Buf[j] - DetChannel.yMin));
         xy^[n].x := Trunc(DetChannel.xScale*(x -
                     DetChannel.xMin) + DetChannel.Left) ;
         Inc(n) ;
         x := x + 1.0 ;
         end ;
     OK := Polyline( PB.Canvas.Handle, xy^, n ) ;


     Detector.yThreshold := Trunc ( Settings.EventDetector.yThreshold /
                                    DetChannel.ADCScale ) ;

     DrawHorizontalCursor( PB , DetChannel, Trunc(Detector.yBaseline), clRed ) ;
     DrawHorizontalCursor( PB , DetChannel,
                           Trunc(Detector.yBaseline) + Detector.yThreshold, clBlue ) ;
     end ;


procedure TDetSignalsFrm.ScanForSignals ;
{ -------------------------------------------
  Scan and extract signals from CDR data file
  -------------------------------------------}
var
   i,j,n,ch,y : Integer ;
   x : single ;
   OK,Done : Boolean ;
begin

     WCPfH.NumSamples := Settings.EventDetector.RecordSize ;
     NumBytesPerRecord := WCPfH.NumSamples*CdrFH.NumChannels*2 ;

     DetChannel.xMin := Detector.BlockStart ;
     DetChannel.xMax := DetChannel.xMin + Detector.BlockSize ;

     { Indicate position within data file with slider bar }
     sbDispCDR.Position := (Detector.BlockStart*2) div
                           (CdrFH.NumBytesPerBlock*ScrollBarToBlockScale) ;

     { Read block from source file into ADC buffer }
     CdrFH.FilePointer := ((Detector.BlockStart) *(CdrFH.NumChannels*2))
                          + CdrFH.NumBytesInHeader ;
     CdrFH.FilePointer := FileSeek( CdrFH.FileHandle, CdrFH.FilePointer, 0 ) ;
     if (FileRead(CdrFH.FileHandle,ADC^,NumBytesPerRecord) = NumBytesPerRecord)
        and (Detector.BlockStart < Detector.EndAt) then begin

         if Detector.JustStarted then begin
            Detector.yBaseline := ADC^[DetChannel.ChannelOffset] ;
            Detector.yOldBaseline := Detector.yBaseline ;
            { Create a new WCP file to hold detected signals }
            cdrFH.WCPFileName := WCPfH.FileName ;
            if WCPfH.FileHandle >= 0 then FileClose( WCPfH.FileHandle ) ;
            WCPfH.FileHandle := FileCreate( WCPfH.FileName ) ;
            WCPfH.NumRecords := 0 ;
            WCPfH.NumSamples := Settings.EventDetector.RecordSize ;
            WCPfH.NumChannels := CdrFH.NumChannels ;
            WCPfH.NumBytesInHeader := 512 ;
            WCPfH.NumDataBytesPerRecord := WCPfH.NumSamples*WCPfH.NumChannels*2 ;
            WCPfH.NumAnalysisBytesPerRecord := 512 ;
            WCPfH.NumBytesPerRecord := WCPfH.NumAnalysisBytesPerRecord +
                                        WCPfH.NumDataBytesPerRecord ;
            for ch := 0 to WCPfH.NumChannels-1 do Channel[ch].ADCZeroAt := 1 ;
            WCPfH.NumZeroAvg := 20 ;
            WCPfH.dt := CDRfH.dt ;
            { Save data to WCP file header }
            SaveWCPHeader( WCPfH ) ;
            Detector.SignalDetected := False ;
            Detector.JustStarted := False ;
            end ;
          if (not ckDisableDisplay.Checked) then PlotChannel( pbDispCDR, ADC^ ) ;
          Done := False ;
          end
     else begin
          State := EndDetection ;
          SaveWCPHeader( WCPfH ) ;
          { Close and re-open WCP file }
          FileClose( WCPfH.FileHandle ) ;
          WCPfH.FileHandle := FileOpen( WCPfH.FileName, fmOpenReadWrite ) ;
          if Main.ViewWCPChildExists then Main.ViewWCPChild.NewFile := True ;
          WriteToLogFile( format('%d signals detected in %s',
                          [WCPfH.NumRecords,CdrFH.FileName] )) ;
          WriteToLogFile( 'written to ' + WCPfH.FileName ) ;
          Done := True ;
          end ;

     { Scan through buffer looking for signal }
     i := 0 ;
     while not Done do begin

         j := (i*CdrFH.NumChannels) + DetChannel.ChannelOffset ;
         y := ADC^[j] - Trunc(Detector.yBaseline) ;

         { Update running mean baseline }
         Detector.yBaseline := ((Detector.yBaseline*Detector.RunningMean)+ ADC^[j])
                                /(Detector.RunningMean + 1.0 ) ;

         { Plot baseline and trigger level cursors }
         if ((i mod 10) = 0) and (not ckDisableDisplay.Checked) then begin
            DrawHorizontalCursor( pbDispCDR , DetChannel,
                                  Trunc(Detector.yOldBaseline), clRed ) ;
            DrawHorizontalCursor( PBDispCDR , DetChannel,
                                  Trunc(Detector.yOldBaseline) + Detector.yThreshold,
                                  clBlue ) ;
            DrawHorizontalCursor( pbDispCDR , DetChannel,
                                  Trunc(Detector.yBaseline), clRed ) ;
            DrawHorizontalCursor( PBDispCDR , DetChannel,
                                  Trunc(Detector.yBaseline) + Detector.yThreshold,
                                  clBlue ) ;
            Detector.yOldBaseline := Detector.yBaseline ;
            end ;

         { If signal exceeds detection threshold ... decrement
           super-threshold sample counter }
         if Detector.yThreshold > 0 then begin
            if y >= Detector.yThreshold then Inc(Detector.tCounter)
                                        else Detector.tCounter := 0 ;
            end
         else begin
            if y <= Detector.yThreshold then Inc(Detector.tCounter)
                                        else Detector.tCounter := 0 ;
            end ;

         if Detector.tCounter = 1 then Detector.DetectedAt :=
                                       Detector.BlockStart + i ;

         { If an event has been detected ... copy it to a WCP file }
         if Detector.tCounter > Detector.tThreshold then begin
            CdrFH.FilePointer := ((Detector.DetectedAt - Detector.PreTriggerSamples)
                                *(CdrFH.NumChannels*2)) + CdrFH.NumBytesInHeader ;
            CdrFH.FilePointer := MaxInt([CdrFH.FilePointer,CdrFH.NumBytesInHeader]) ;
            CdrFH.FilePointer := FileSeek( CdrFH.FileHandle,CdrFH.FilePointer,0) ;
            if FileRead(CdrFH.FileHandle,DetBuf^,NumBytesPerRecord)
               = NumBytesPerRecord then PlotChannel( pbDispWCP, DetBuf^ ) ;

            { Save record to WCP data file }
            Inc(WCPFH.NumRecords) ;
            edStatus.text := format('Event #%d detected at %.1f %s',
                             [WCPfH.NumRecords,
                              Detector.DetectedAt*CdrFH.dt*Settings.TScale,
                              Settings.TUnits]) ;

            rH^.Status := 'ACCEPTED' ;
            rH^.RecType := 'TEST' ;
            rH^.Number := WCPFH.NumRecords ;
            rH^.Time := Detector.DetectedAt*CdrFH.dt ;
            rH^.dt := CdrFH.dt ;
            rH^.Ident := ' ' ;
            for ch := 0 to CdrFH.NumChannels do rH^.ADCVoltageRange[ch] :=
                                                CdrFH.ADCVoltageRange ;
            rH^.Equation.Available := False ;
            rH^.Analysis.Available := False ;
            PutRecord( WCPfH, rH^, WCPfH.NumRecords, DetBuf^ ) ;

            Detector.BlockStart := Detector.DetectedAt + Detector.SkipSamples ;
            Detector.yBaseline := y + Detector.yBaseline;
            Detector.yOldBaseline := Detector.yBaseline ;

            lbTMinWCP.Caption := lbTMin.Caption ;
            lbTMaxWCP.Caption := lbTMax.Caption ;
            lbTMaxWCP.Left := lbTMax.Left ;
            lbTminWCP.Left := lbTMin.Left ;
            Done := True ;
            end
         else begin
            { Increment to next sample }
            Inc(i) ;
            if i >= Detector.BlockSize then begin
               Detector.BlockStart := Detector.BlockStart + Detector.BlockSize ;
               Done := True ;
               end ;
            end ;
         end ;
     end ;


procedure TDetSignalsFrm.InitializeDisplay( const PB : TPaintBox ;
                                            var Chan : TChannel ) ;
{ ----------------------------
  Initialise a display window
  ---------------------------}
var
   ch,cTop : Integer ;
begin

     { Set trace colour }

     PB.canvas.pen.color := Channel[0].color ;

     { Define display area for each channel in use }

     Chan.Left := 0 ;
     Chan.Right := PB.width ;
     Chan.Top := 0 ;
     Chan.Bottom := Chan.Top + PB.Height ;
     Chan.xScale := (Chan.Right - Chan.Left) / (Chan.xMax - Chan.xMin ) ;
     Chan.yScale := (Chan.Bottom - Chan.Top) / (Chan.yMax - Chan.yMin ) ;

     lbTMin.caption := Format( '%5.4g %s', [Chan.xMin*CdrFH.dt*Settings.TScale,
                                               Settings.TUnits] ) ;
     lbTMax.caption := Format( '%5.4g %s', [Chan.xMax*CdrFH.dt*Settings.TScale,
                                               Settings.TUnits] ) ;
     lbTMax.Left := PB.Left + PB.Width - lbTMax.Width ;

     { Display Channel Name(s) }

     for ch := 0 to CdrFH.NumChannels-1 do
           PB.Canvas.TextOut( Chan.Left,
           (Chan.Top + Chan.Bottom) div 2,' ' + Chan.ADCName ) ;

     end ;


procedure TDetSignalsFrm.FormClose(Sender: TObject;
  var Action: TCloseAction);
{ -------------------------
  Close and dispose of form
  -------------------------}
begin
     HeapBuffers( Deallocate ) ;
     Main.mnDetectSignals.Enabled := True ;
     if WCPFH.NumRecords > 0 then begin
        CdrFH.WCPFileName := WcpFH.FileName ;
        Main.mnViewWCP.Visible := True ;
        Main.mnViewWCP.Enabled := True ;
        Main.mnViewWCP.Click ;
        SaveCDRHeader( CDRfH ) ;
        end
     else begin
        Main.mnViewWCP.Enabled := False ;
        CdrFH.WCPFileName := '' ;
        SaveCDRHeader( CDRfH ) ;
        end ;

     Action := caFree ;
     end;


procedure TDetSignalsFrm.bCloseClick(Sender: TObject);
begin
     close ;
     end;


procedure TDetSignalsFrm.sbDispCDRChange(Sender: TObject);
begin
     if State = Idle then State := PlotRecord ;
     end;

procedure TDetSignalsFrm.FormResize(Sender: TObject);
begin

    { if ClientHeight < MinHeight then ClientHeight := MinHeight ;
     AnalysisGrp.Height := ClientHeight - AnalysisGrp.Top - 5 ;
     pbDispCDR.Height := (AnalysisGrp.Height div 2 )
                         - 2*lbTMin.Height - edStatus.Height - 2 ;
     pbDispCDR.Width := ClientWidth - pbDispCDR.Left - 5 ;
     sbDispCDR.Top := pbDispCDR.Top + pbDispCDR.Height ;
     sbDispCDR.Left := pbDispCDR.Left ;
     sbDispCDR.Width := pbDispCDR.Width ;
     lbTmin.Top := sbDispCDR.Top + sbDispCDR.Height + 1 ;
     lbTMax.Top := lbTMin.Top ;

     pbDispWCP.Height := pbDispCDR.Height ;
     pbDispWCP.Top := lbTMax.Top + 2*lbTMax.Height ;
     pbDispWCP.Width := pbDispCDR.Width ;
     lbTMin.Caption := ' ' ;
     lbTMax.Caption := ' ' ;
     lbTMinWCP.Top := pbDispWCP.Top + pbDispWCP.Height + 1 ;
     lbTMaxWCP.Top := lbTMinWCP.Top ;
     lbTMinWCP.Left := pbDispWCP.Left ;
     lbTMaxWCP.Left := pbDispWCP.Left + pbDispWCP.Width - lbTMaxWCP.Width ;
     edStatus.Top := pbDispWCP.Top + pbDispWCP.Height + 1 ;
     edStatus.Left := pbDispWCP.Left + (pbDispWCP.Width div 2)
                      - (edStatus.Width div 2) ;  }
     end;


procedure TDetSignalsFrm.FormPaint(Sender: TObject);
begin
     if State = Idle then State := PlotRecord ;
     end;


procedure TDetSignalsFrm.bDetectClick(Sender: TObject);
{ --------------------------------------
  Set up and start signal detection scan
  --------------------------------------}
var
   Val,TLo,THi : single ;
   iVal : Integer ;
begin

     { Get latest data from data entry boxes }
     UpdateParameters ;

     Detector.tCounter := 0 ;
     Detector.BlockStart := Detector.StartAt ;
     Detector.DetectedAt := 0 ;
     Detector.JustStarted := True ;

     bDetect.Enabled := false ;
     bAbort.Enabled := True ;

     State := DetectSignals ;

     end;

procedure TDetSignalsFrm.UpdateParameters ;
{ -------------------------------------------------------
  Update detection parameters settings & data entry boxes
  -------------------------------------------------------
  All data entry boxes are processed here}
var
   Val,TLo,THi : single ;
   iVal : Integer ;
begin

     { Amplitude threshold }
     Val := Settings.EventDetector.yThreshold ;
     Val := ExtractFloat( edYThreshold.text, Val ) ;
     edYThreshold.text := format( ' %.3g %s',[Val,
                          Channel[Settings.EventDetector.Channel].ADCUnits] ) ;
     Settings.EventDetector.yThreshold := Val ;
     Detector.yThreshold := Trunc ( Settings.EventDetector.yThreshold /
                                    DetChannel.ADCScale ) ;

     { Time threshold }
     Val := Settings.EventDetector.tThreshold*Settings.TScale ;
     Val := ExtractFloat( edtThreshold.text, Val ) ;
     edtThreshold.text := format( ' %.3g %s',[Val,Settings.TUnits]);
     Settings.EventDetector.tThreshold := Val/Settings.TScale ;
     Detector.tThreshold := Trunc(Settings.EventDetector.tThreshold/CdrFH.dt) ;

     { Dead time after event detection }
     Val := Settings.EventDetector.DeadTime*Settings.TScale ;
     Val := ExtractFloat( edDeadTime.text, Val ) ;
     edDeadTime.text := format( ' %.3g %s',[Val,Settings.TUnits]) ;
     Settings.EventDetector.DeadTime := Val/Settings.TScale ;
     Detector.SkipSamples := Trunc( Settings.EventDetector.DeadTime / CdrFH.dt ) ;

     { Baseline tracking running mean }
     Val := Settings.EventDetector.BaselineAverage*(Settings.TScale*CdrFH.dt) ;
     Val := ExtractFloat( edBaselineAverage.text, Val ) ;
     Settings.EventDetector.BaselineAverage := Val/(Settings.TScale*CdrFH.dt) ;
     edBaselineAverage.text := format( ' %f %s',[Val,Settings.TUnits]) ;
     Detector.RunningMean := Settings.EventDetector.BaselineAverage ;

     { Record size }
     iVal := ExtractInt( edRecordSize.text ) ;
     iVal := (iVal div 256) * 256 ;
     iVal := MinInt( [MaxInt([iVal,256]),2048]) ;
     edRecordSize.text := format( ' %d ', [iVal] ) ;
     Settings.EventDetector.RecordSize := iVal ;
     Detector.BlockSize := Settings.EventDetector.RecordSize ;
     WCPfH.NumSamples := Settings.EventDetector.RecordSize ;
     NumBytesPerRecord := WCPfH.NumSamples*CdrFH.NumChannels*2 ;

     { Pre-trigger samples }
     Val := Settings.EventDetector.PreTriggerPercentage ;
     Val := ExtractFloat( edPreTriggerPercentage.text, Val ) ;
     Val := MinFlt( [MaxFlt([Val,0.0]), 100.0 ] ) ;
     edPreTriggerPercentage.text := format( ' %.3g %%',[Val] ) ;
     Settings.EventDetector.PreTriggerPercentage := Val ;
     Detector.PreTriggerSamples := Trunc( (Settings.EventDetector.PreTriggerPercentage
                                   * Settings.EventDetector.RecordSize) / 100.0 ) ;

     { Get range of blocks to be analysed }
     GetRangeFromEditBox( edRange, TLo, THi, 0.0, CdrFH.RecordDuration,
                          ' %f - %f ', 's' ) ;
     Detector.StartAt := Trunc(TLo/CdrFH.dt) ;
     TLo := Detector.StartAt*CdrFH.dt ;
     Detector.EndAt := Trunc(THi/CdrFH.dt) ;
     THi := Detector.EndAt*CdrFH.dt ;
     edRange.text := format( ' %f - %f s', [TLo,THi] ) ;

     end ;



procedure TDetSignalsFrm.edYThresholdKeyPress(Sender: TObject;
  var Key: Char);
var
   Val : single ;
begin
     if key = chr(13) then begin
        UpdateParameters ;
        if State = Idle then State := PlotRecord ;
        end ;
     end ;


procedure TDetSignalsFrm.EdtThresholdKeyPress(Sender: TObject;
  var Key: Char);
begin
     if key = chr(13) then UpdateParameters ;
     end ;


procedure TDetSignalsFrm.edDeadTimeKeyPress(Sender: TObject;
  var Key: Char);
begin
     if key = chr(13) then UpdateParameters ;
     end ;


procedure TDetSignalsFrm.edBaseLineAverageKeyPress(Sender: TObject;
  var Key: Char);
begin
     if key = chr(13) then UpdateParameters ;
     end ;


procedure TDetSignalsFrm.edPreTriggerPercentageKeyPress(Sender: TObject;
  var Key: Char);
begin
     if key = chr(13) then UpdateParameters ;
     end ;

procedure TDetSignalsFrm.edRecordSizeKeyPress(Sender: TObject;
  var Key: Char);
begin
     if key = chr(13) then UpdateParameters ;
     end ;

procedure TDetSignalsFrm.DrawHorizontalCursor( pb : TPaintBox ;
                                               Const Chan : TChannel ;
                                               Level : Integer ;
                                               Colour : TColor ) ;
{ ----------------------------------------------------
  Draw dotted horizontal cursor at ADC level 'Level'
  in display area defined by  record data channel 'Gr'
  ----------------------------------------------------}
var
   yPix,xPix,TicSize : Integer ;
   OldColor : TColor ;
   OldStyle : TPenStyle ;
   OldMode : TPenMode ;
begin
     with pb.canvas do begin
          OldColor := pen.color ;
          OldStyle := pen.Style ;
          OldMode := pen.mode ;
          pen.mode := pmXor ;
          pen.color := Colour ;
          end ;

     yPix := Trunc( Chan.Bottom - (Level - Chan.yMin)*Chan.yScale ) ;
     pb.canvas.polyline([Point(Chan.Left,yPix),Point(Chan.Right,yPix)]);

     with pb.canvas do begin
          pen.style := OldStyle ;
          pen.color := OldColor ;
          pen.mode := OldMode ;
          end ;
     { Request that the cursor readout memo box be updated }
    { State := UpdateCursorReadout ;}
    end ;


procedure TDetSignalsFrm.bAbortClick(Sender: TObject);
begin
     State := Idle ;
     bDetect.Enabled := True ;
     bAbort.Enabled := False ;
     end;

procedure TDetSignalsFrm.cbChannelChange(Sender: TObject);
begin
     DetChannel.ChannelOffset := cbChannel.ItemIndex ;
     State := PlotRecord ;
     end;

procedure TDetSignalsFrm.edRangeKeyPress(Sender: TObject; var Key: Char);
begin
     if key = chr(13) then UpdateParameters ;
     end;


procedure TDetSignalsFrm.bNewFileNameClick(Sender: TObject);
{ --------------------------------------------
  Let user change name of detected events file
  --------------------------------------------}
begin
     { Present user with standard Save File dialog box
       Note. Prevent user from changing directory. WCP file MUST
       be in same directory as WCD file }
     SaveDialog.options := [ofOverwritePrompt,ofHideReadOnly,ofPathMustExist,
                            ofNoChangeDir ] ;

     SaveDialog.DefaultExt := 'WCP' ;
     SaveDialog.FileName := ExtractFileName( wcpFH.FileName ) ;
     SaveDialog.Filter := ' WCP Files (*.wcp)|*.wcp' ;
     SaveDialog.Title := 'Detected signals file' ;

     if SaveDialog.execute then begin
        wcpFH.FileName := SaveDialog.FileName ;
        edOutputFileName.text := ExtractFileName( wcpFH.FileName ) ;
        end ;

     end;

end.
