unit TritonUnit;
// ------------------------------------
// Tecella Triton 8 channel patch clamp
// ------------------------------------
// (c) J. Dempster, University of Strathclyde, 2008
// 9.07.08
// 04.09.08 Revised to work with TecellaAmp.dll library

interface

uses WinTypes,Dialogs, SysUtils, WinProcs,mmsystem, math ;

const

	TECELLA_HW_MODEL_AUTO_DETECT = 0 ;
	TECELLA_HW_MODEL_TRITON = 1 ;
	TECELLA_HW_MODEL_TRITON_PLUS = 2 ;
	TECELLA_HW_MODEL_JET = 3 ;
	TECELLA_HW_MODEL_RICHMOND = 4 ;

    TECELLA_REG_CFAST = 0 ;
    TECELLA_REG_CSLOW_A = 1 ;
    TECELLA_REG_CSLOW_B = 2 ;
    TECELLA_REG_CSLOW_C = 3 ;
    TECELLA_REG_CSLOW_D = 4 ;
    TECELLA_REG_RSERIES = 5 ;
    TECELLA_REG_CHANOFF = 6 ;
    TECELLA_REG_LEAK = 7 ;

  TECELLA_STIMULUS_MODE_VCMD = 0 ; // **< A voltage stimulus is applied and the current response is acquired. */
	TECELLA_STIMULUS_MODE_ICMD = 1 ; //< A current stimulus is applied and voltage response is acquired. */
	TECELLA_STIMULUS_MODE_OSCOPE = 2 ;

  TECELLA_STIMULUS_SEGMENT_SET = 0 ; 		    //**< A flat segment. */
	TECELLA_STIMULUS_SEGMENT_DELTA = 1 ;    //**< A flat segment that changes its amplitude and/or duration after each iteration. */
	TECELLA_STIMULUS_SEGMENT_RAMP = 2 ;

    TECELLA_ERR_OK = 0 ;
    TECELLA_ERR_BAD_HANDLE = 1 ;
    TECELLA_ERR_BAD_HANDLE_RETURN = 2 ;
    TECELLA_ERR_INT_HANDLE_ALLOC = 3 ;
    TECELLA_ERR_R1 = 4 ;
    TECELLA_ERR_HWCONN = 5 ;
    TECELLA_ERR_RETURN_POINTER_NULL = 6 ;
    TECELLA_ERR_MAXVCMDS = 7 ;
    TECELLA_ERR_BAD_VCMD_HANDLE = 8 ;
    TECELLA_ERR_BAD_SEG_PTR = 9 ;
    TECELLA_ERR_VCMD_BAD_SEG_INDEX = 10 ;
    TECELLA_ERR_OKLIB_NOT_FOUND = 11 ;
    TECELLA_ERR_DEVICE_INIT_FAILED = 12 ;
    TECELLA_ERR_R2 = 13 ;
    TECELLA_ERR_DEVICE_OPEN_FAILED = 14 ;
    TECELLA_ERR_MIN_CHAN_EXCEEDED = 15 ;
    TECELLA_ERR_MAX_CHAN_EXCEEDED = 16 ;
    TECELLA_ERR_CANNOT_GET_ALLCHAN = 17 ;                                                                  

    TECELLA_GAIN_NONE = 0 ;
    TECELLA_GAIN_LOW = 1 ;
    TECELLA_GAIN_MID = 2 ;
    TECELLA_GAIN_HI = 3 ;
    TECELLA_GAIN_VHI = 4 ;

    TECELLA_SOURCE_NONE = 0 ;
    TECELLA_SOURCE_HEAD = 1 ;
    TECELLA_SOURCE_MODEL1 = 2 ;
    TECELLA_SOURCE_MODEL2 = 3 ;

    TECELLA_ALLCHAN = $8000 ;

type

Ttecella_stimulus_segment = record
	SegmentType : Integer ; //**< Type of segment.  Assumed to be SET, until DELTA and RAMP is supported. */
	Acquire : ByteBool ;    //**< If true, samples for the duration will be acquired.  If false, the stimulus will play as specified but the samples for the duration will not be acquired.  Not currently supported, but set this to true if you would like consistent behavior in future releases. */
  Value : Double ;        //**< Amplitude of segment in Volts (if vcmd) or Amps (if icmd). */
	Value_delta : Double ;  //**< If type is TECELLA_STIMULUS_SEGMENT_DELTA, increment value by this delta after each iteration of the vcmd.  If type is TECELLA_STIMULUS_SEGMENT_RAMP, value_delta is used as the value to ramp to by the end of duration. */
  Duration : Double ;     //**< Duration of segment in seconds. */
	Duration_delta : Double ;  //**< Increment duration by this delta after each iteration of the vcmd. Valid only if type is TECELLA_STIMULUS_SEGMENT_DELTA. */
  end ;

Ttecella_lib_props = packed record
    v_maj : Integer ;
    v_min : Integer ;
    v_dot : Integer ;
    Description : Array[0..255] of WideChar ;
    end ;

Ttecella_hw_props = record
	hw_model : Integer ;
	device_name : Array[0..31] of WideChar ;		//**< Name of the device. */
	serial_number : Array[0..15] of WideChar ;	//**< The serial number, which is unique to each unit. */
	hwvers : Integer ;					//**< Contains the firmware version being used. */
	nchans : Integer ;					//**< Number of channels.  Note: Channel numbering goes from [0, nchans-1]. */
	nsources : Integer ;				//**< Number of sources selectable (see TECELLA_REG_SOURCE) */
	ngains : Integer ;					//**< Number of gains selectable (see TECELLA_REG_GAIN) */
	nstimuli : Integer ;				//**< Number of simultaneous stimuli supported (see TECELLA_REG_VCMD_SELECT) */
	max_stimulus_segments : Integer ;  //**< Maximum number of segments supported per stimulus. */
	supports_async_stimulus : ByteBool ; //**< If true, tecella_acquire_start_stimulus() can be used. */
	supports_oscope : ByteBool ; //**< Indicates if the amplifier can be used as an oscilloscope. */
	supports_vcmd : ByteBool ; //**< Indicates if the amplifier can produce a voltage stimulus. (See TECELLA_STIMULUS_MODE_VCMD) */
	vcmd_value_min : Double ;	//**< Minimum voltage value supported by a vcmd stimulus. */
	vcmd_value_max : Double ;	//**< Maximum voltage value supported by a vcmd stimulus. */
	vcmd_value_lsb : Double ;	//**< The vcmd stimulus can only take on voltage values in intervals of vcmd_value_lsb from vcmd_value_min to vcmd_value_max. */
	supports_icmd : ByteBool ;	//**< Indicates if the amplifier can produce a current stimulus. (See TECELLA_STIMULUS_MODE_ICMD) */
	icmd_value_min : Double ;	//**< Minimum amp value supported by a icmd stimulus. */
	icmd_value_max : Double ;	//**< Maximum amp value supported by a icmd stimulus. */
	icmd_value_lsb : Double ;	//**< The icmd stimulus can only take on current values in intervals of vcmd_value_lsb from vcmd_value_min to vcmd_value_max. */
	stimulus_segment_duration_max : Double ;	//**< The maximum duration a segment can have. */
	stimulus_segment_duration_lsb : Double ;	//**< Duration can only take on values in intervals of stimulus_segment_duration_lsb. */
	supports_zap : ByteBool ; //**< Indicates if the amplifier supports tecella_stimulus_zap(). */
	zap_value_min : Double ;	//**< Minimum zap value supported by tecella_stimulus_zap(). */
	zap_value_max : Double ;	//**< Maximum zap value supported by tecella_stimulus_zap(). */
	zap_value_lsb : Double ;	//**< The zap stimulus can only take on voltage values in intervals of zap_value_lsb from zap_value_min to zap_value_max. */
	supports_bessel : ByteBool ; //**< Indicates if a bessel filter is supported */
	bessel_value_min : Integer ;  //**< Minimum programmable bessel value. */
	bessel_value_max : Integer ;  //**< Maximum programmable bessel value. */
	sample_period_min : Double ;	//**< Minimum sample period supported in seconds. */
	sample_period_max : Double ;	//**< Maximum sample period supported in seconds. */
	sample_period_lsb : Double ;	//**< The sample period can only take on values in intervals of sample_period_lsb from sample_period_min to sample_period_max */
  end ;

Ttecella_reg_props = record
  rlabel : Array[0..31] of WideChar ;
  units : Array[0..31] of WideChar ;
  supported : ByteBool ;
  can_be_disabled : ByteBool ;
  v_min : Double ;
  v_max : Double ;
  v_lsb : Double ;
  v_divisions : Integer ;
  v_default : Double ;
  pct_lsb : Double ;
  end ;



Ttecella_initialize = function(
                    var Handle : Integer ;
                    HWModel : Integer ) : Integer ; cdecl ;
Ttecella_finalize = function(
                    Handle : Integer ) : Integer ; cdecl ;

Ttecella_set_stimulus_mode = function(
                             Handle : Integer ;
                             Mode : Integer ) : Integer ; cdecl ;

Ttecella_get_stimulus_mode = function(
                             Handle : Integer ;
                             var Mode : Integer ) : Integer ; cdecl ;

Ttecella_get_lib_props = function(
                         var Props : Ttecella_lib_PROPS ) : Integer ; cdecl ;

Ttecella_get_hw_props = function(
                            Handle : Integer ;
                            var hwprops : TTECELLA_HW_PROPS ) : Integer ; cdecl ;

Ttecella_get_reg_props = function(
                         Handle : Integer ;
                         Reg : Integer ;
                         var RegProps : TTECELLA_reg_PROPS ) : Integer ; cdecl ;

Ttecella_error_get_last_msg = function(
                              Handle : Integer ;
                              Msg : PChar) : Integer ; cdecl ;
Ttecella_error_set_callback  = function(
                               Handle : Integer ;
                               f : Pointer ) : Integer ; cdecl ;

//* Manipulate registers */
Ttecella_chan_set = function(
                   Handle : Integer ;
                   Reg : Integer ;
                   chan : Integer ;
                   value : Double ) : Integer ; cdecl ;
Ttecella_chan_get = function(
                   Handle : Integer ;
                   Reg : Integer ;
                   chan : Integer ;
                   var value : Double 
                   ) : Integer ; cdecl ;
Ttecella_chan_set_pct = function(
                       Handle : Integer ;
                       Reg : Integer ;
                       chan : Integer ;
                       pct : Double) : Integer ; cdecl ;

Ttecella_chan_get_pct = function(
                       Handle : Integer ;
                       Reg : Integer ;
                       chan : Integer ;
                       var pct : Double) : Integer ; cdecl ;

Ttecella_chan_to_string = function(
                         Handle : Integer ;
                         Reg : Integer ;
                         chan : Integer ;
                         s : PChar ) : Integer ; cdecl ;
Ttecella_chan_get_units = function(
                         Handle : Integer ;
                         Reg : Integer ;
                         s : PChar ) : Integer ; cdecl ;
Ttecella_chan_set_enable = function(
                          Handle : Integer ;
                          Reg : Integer ;
                          chan : Integer ;
                          onoff : Integer ) : Integer ; cdecl ;
Ttecella_chan_get_enable = function(
                          Handle : Integer ;
                          Reg : Integer ;
                          chan : Integer ;
                          Enabled : ByteBool ) : Integer ; cdecl ;
Ttecella_chan_set_gain = function(
                        Handle : Integer ;
                        chan : Integer ;
                        Gain : Integer) : Integer ; cdecl ;
Ttecella_chan_get_gain = function(
                        Handle : Integer ;
                        chan : Integer ;
                        var Gain : Integer ) : Integer ; cdecl ;
Ttecella_chan_set_source = function(
                          Handle : Integer ;
                          chan : Integer ;
                          src : Integer ) : Integer ; cdecl ;
Ttecella_chan_get_source = function(
                          Handle : Integer ;
                          chan : Integer ;
                          var src : Integer ) : Integer ; cdecl ;
Ttecella_chan_set_bessel = function(
                        Handle : Integer ;
                        chan : Integer ;
                        Value : Integer) : Integer ; cdecl ;
Ttecella_chan_get_bessel = function(
                        Handle : Integer ;
                        chan : Integer ;
                        var Value : Integer ) : Integer ; cdecl ;

Ttecella_bessel_value2freq = function(
                        Handle : Integer ;
                        Value : Integer ;
                        var Freq : double ) : Integer ; cdecl ;

Ttecella_bessel_freq2value = function(
                        Handle : Integer ;
                        Freq : double ;
                        var Value : Integer ) : Integer ; cdecl ;


//* Vcmd */

Ttecella_stimulus_set =  function(
                         Handle : Integer ;
                         //var Segments : Array of Ttecella_stimulus_segment ;
                         pSegments : Pointer ;
                         segment_count : Integer ;
                         Iterations : Integer ;
                         Loop : ByteBool ;
                         Index : Integer ) : Integer ; cdecl ;

Ttecella_stimulus_get =  function(
                         Handle : Integer ;
                         //var Segments : Array of Ttecella_stimulus_segment ;
                         pSegments : Pointer ;
                         segments_max : Integer ;
                         var segments_in_stimulus : Integer ;
                         var Iterations : Integer ;
                         var Loop : ByteBool ;
                         Index : Integer ) : Integer ; cdecl ;

Ttecella_stimulus_sample_count =  function(
                                  pSegments  : Pointer ;
                                  segment_count : Integer ;
                                  sample_period : Double ;
                                  Iteration : Integer ) : Integer ; cdecl ;


//* Acquire */

Ttecella_acquire_enable_channel =  function(
                                   Handle : Integer ;
                                   channel : Integer ;
                                   enable : ByteBool
                                   ) : Integer ; cdecl ;

Ttecella_acquire_set_buffer_size =  function(
                                    Handle : Integer ;
                                    samples_per_chan : Integer
                                    ) : Integer ; cdecl ;
Ttecella_acquire_start =  function(
                          Handle : Integer ;
                          sample_period : Double ;
                          continuous : ByteBool ;
                          synchronous_stimuli : ByteBool
                          ) : Integer ; cdecl ;

Ttecella_acquire_start_stimulus =  function(
                                   Handle : Integer ;
                                   stimulus_index : Integer ;
                                   continuous : ByteBool
                                   ) : Integer ; cdecl ;

Ttecella_acquire_stop_stimulus =  function(
                                  Handle : Integer ;
                                  stimulus_index : Integer
                                  ) : Integer ; cdecl ;

Ttecella_acquire_stop =  function(
                         Handle : Integer
                         ) : Integer ; cdecl ;

Ttecella_acquire_samples_available =  function(
                                      Handle : Integer ;
                                      chan : Integer ;
                                      var samples_available : Integer
                                      ) : Integer ; cdecl ;

Ttecella_acquire_read_d =  function(
                           Handle : Integer ;
                           chan : Integer ;
                           requested_samples : Integer ;
                           data : Pointer ;
                           var actual_samples : Integer
                           ) : Integer ; cdecl ;

Ttecella_acquire_read_i =  function(
                           Handle : Integer ;
                           chan : Integer ;
                           requested_samples : Integer ;
                           data : Pointer ;
                           var actual_samples : Integer
                           ) : Integer ; cdecl ;

Ttecella_acquire_i2d_scale =  function(
                              Handle : Integer ;
                              chan : Integer ;
                              var scale : Double
                              ) : Integer ; cdecl ;

Ttecella_auto_comp =  function(
                      Handle : Integer ;
                      pChanArray : Pointer ;
                      NumChannels : Integer
                      ) : Integer ; cdecl ;

  procedure Triton_InitialiseBoard ;
  procedure Triton_LoadLibrary  ;

function  Triton_LoadProcedure(
         Hnd : THandle ;       { Library DLL handle }
         Name : string         { Procedure name within DLL }
         ) : Pointer ;         { Return pointer to procedure }

  procedure Triton_ConfigureHardware(
            EmptyFlagIn : Integer ) ;

  function  Triton_ADCToMemory(
            var HostADCBuf : Array of SmallInt  ;
            nChannels : Integer ;
            nSamples : Integer ;
            var dt : Double ;
            ADCVoltageRange : Single ;
            TriggerMode : Integer ;
            CircularBuffer : ByteBool
            ) : ByteBool ;
  function Triton_StopADC : ByteBool ;
  procedure Triton_GetADCSamples (
            var OutBuf : Array of SmallInt ;
            var OutBufPointer : Integer
            ) ;
  procedure Triton_CheckSamplingInterval(
            var SamplingInterval : Double
            ) ;

function  Triton_MemoryToDAC(
          var DACValues : Array of SmallInt  ;
          nChannels : Integer ;
          nPoints : Integer ;
          DACUpdateInterval : Single
          ) : ByteBool ;

function  Triton_MemoryToDACAndDigitalOut(
          var DACValues : Array of SmallInt  ; // D/A output values
          NumDACChannels : Integer ;                // No. D/A channels
          NumDACPoints : Integer ;                  // No. points per channel
          var DigValues : Array of SmallInt  ; // Digital port values
          DigitalInUse : ByteBool ;             // Output to digital outs
          ExternalTrigger : ByteBool            // Wait for ext. trigger
          ) : ByteBool ;                        // before starting output


function Triton_GetDACUpdateInterval : double ;

  function Triton_StopDAC : ByteBool ;
  procedure Triton_WriteDACsAndDigitalPort(
            var DACVolts : array of Single ;
            nChannels : Integer ;
            DigValue : Integer
            ) ;

  function  Triton_GetLabInterfaceInfo(
            var Model : string ; { Laboratory interface model name/number }
            var ADCMinSamplingInterval : Double ; { Smallest sampling interval }
            var ADCMaxSamplingInterval : Double ; { Largest sampling interval }
            var ADCMinValue : Integer ; { Negative limit of binary ADC values }
            var ADCMaxValue : Integer ; { Positive limit of binary ADC values }
            var ADCVoltageRanges : Array of single ; { A/D voltage range option list }
            var NumADCVoltageRanges : Integer ; { No. of options in above list }
            var ADCBufferLimit : Integer ;      { Max. no. samples in A/D buffer }
            var DACMaxVolts : Single ; { Positive limit of bipolar D/A voltage range }
            var DACMinUpdateInterval : Double {Min. D/A update interval }
            ) : ByteBool ;

  function Triton_GetMaxDACVolts : single ;

  function Triton_ReadADC( Channel : Integer ) : SmallInt ;

  procedure Triton_GetChannelOffsets(
            var Offsets : Array of Integer ;
            NumChannels : Integer
            ) ;
  procedure Triton_CloseLaboratoryInterface ;

  procedure Triton_Wait( Delay : Single ) ;

  procedure Triton_CheckError(
          Location : String ;
          Err : Integer
          ) ;

  procedure TritonGetRegister(
          Reg : Integer ;
          Chan : Integer ;
          var Value : Double ;
          var PercentValue : Double ;
          var Enabled : Boolean ) ;

  procedure TritonSetRegisterPercent(
          Reg : Integer ;
          Chan : Integer ;
          var PercentValue : Double ) ;

  function TritonGetGain( Chan : Integer ) : Integer ;
  procedure TritonSetGain( Chan : Integer ; iGain : Integer ) ;
  function TritonGetSource( Chan : Integer ) : Integer ;
  procedure TritonSetSource( Chan : Integer ; iSource : Integer ) ;

  function TritonGetRegisterEnabled( Reg : Integer ; Chan : Integer ) : ByteBool ;
  procedure TritonSetRegisterEnabled( Reg : Integer ;
                                Chan : Integer ;
                                Enabled : ByteBool ) ;

function Triton_CurrentCalibration( Chan : Integer ) : Single  ;
function Triton_CurrentGain( Chan : Integer ) : Single  ;
function Triton_VoltageCalibration : Single  ;

procedure TritonSetBesselFilter(
          Chan : Integer ;
          Value : Integer ;
          var CutOffFrequency : Single ) ;

procedure TritonAutoCompensate(
          Chan : Integer ) ;

implementation

uses seslabio ;

var
    TecHandle : Integer ;
    LibraryHnd : Integer ;
    LibraryLoaded : ByteBool ;
    DeviceInitialised : ByteBool ;
    HardwareProps : Ttecella_hw_props ;
    FADCMaxValue : Integer ;
    FADCMinSamplingInterval : Double ;
    FADCMaxSamplingInterval : Double ;
    FGainFactors : Array[0..63] of Single ;
    FNumChannels : Integer ;       // No. of channels in sweep
    fNumSamples : Integer ;        // No. samples/channel in sweep
    FSamplingInterval : Double ;   // Sampling interval (s)
    FSweepDuration : Double ;  // Duration of recording sweep (s)
    FDACMaxVolts : Double ;    // Max. D/A output voltage
    VmBuf : PSmallIntArray ;   // Voltage waveform storage buffer
    FOutPointer : Integer ;
        tecella_initialize : Ttecella_initialize ;
        tecella_finalize : Ttecella_finalize ;
        tecella_get_lib_props : Ttecella_get_lib_props ;
        tecella_get_hw_props : Ttecella_get_hw_props ;
        tecella_get_reg_props : Ttecella_get_reg_props ;
        tecella_error_get_last_msg : Ttecella_error_get_last_msg ;
        tecella_error_set_callback : Ttecella_error_set_callback ;
        tecella_chan_set : Ttecella_chan_set ;
        tecella_chan_get : Ttecella_chan_get ;
        tecella_chan_set_pct : Ttecella_chan_set_pct ;
        tecella_chan_get_pct : Ttecella_chan_get_pct ;
        tecella_get_stimulus_mode : Ttecella_get_stimulus_mode ;
        tecella_set_stimulus_mode : Ttecella_set_stimulus_mode ;
        tecella_chan_to_string : Ttecella_chan_to_string ;
        tecella_chan_get_units : Ttecella_chan_get_units ;
        tecella_chan_set_enable : Ttecella_chan_set_enable ;
        tecella_chan_get_enable : Ttecella_chan_get_enable ;
        tecella_chan_set_gain : Ttecella_chan_set_gain ;
        tecella_chan_get_gain : Ttecella_chan_get_gain ;
        tecella_chan_set_source : Ttecella_chan_set_source ;
        tecella_chan_get_source : Ttecella_chan_get_source ;
        tecella_chan_set_bessel : Ttecella_chan_set_bessel ;
        tecella_chan_get_bessel : Ttecella_chan_get_bessel ;
        tecella_bessel_value2freq : Ttecella_bessel_value2freq ;
        tecella_bessel_freq2value : Ttecella_bessel_freq2value ;
        tecella_stimulus_set : Ttecella_stimulus_set ;
        tecella_stimulus_get : Ttecella_stimulus_get ;
        tecella_stimulus_sample_count : Ttecella_stimulus_sample_count ;
        tecella_acquire_enable_channel : Ttecella_acquire_enable_channel ;
        tecella_acquire_set_buffer_size : Ttecella_acquire_set_buffer_size ;
        tecella_acquire_start : Ttecella_acquire_start ;
        tecella_acquire_start_stimulus : Ttecella_acquire_start_stimulus ;
        tecella_acquire_stop_stimulus : Ttecella_acquire_stop_stimulus ;
        tecella_acquire_stop : Ttecella_acquire_stop ;
        tecella_acquire_samples_available : Ttecella_acquire_samples_available ;
        tecella_acquire_read_d : Ttecella_acquire_read_d ;
        tecella_acquire_read_i : Ttecella_acquire_read_i ;
        tecella_acquire_i2d_scale : Ttecella_acquire_i2d_scale ;
        tecella_auto_comp : Ttecella_auto_comp ;

procedure Triton_LoadLibrary  ;
{ -------------------------------------
  Load TecellaAmp.DLL library into memory
  -------------------------------------}
var
     TritonDLLPath : String ; // TritonDLL file path
begin

     // Support DLLs loaded from program folder
     TritonDLLPath := ExtractFilePath(ParamStr(0)) + 'TecellaAmp.DLL' ;

     // Load main library
     LibraryHnd := LoadLibrary(PChar(TritonDLLPath)) ;
     if LibraryHnd <= 0 then
        ShowMessage( format('%s library not found',[TritonDLLPath])) ;

     { Get addresses of procedures in library }
     if LibraryHnd > 0 then begin

        @tecella_initialize := Triton_LoadProcedure( LibraryHnd, 'tecella_initialize' ) ;
        @tecella_finalize  := Triton_LoadProcedure( LibraryHnd, 'tecella_finalize' ) ;
        @tecella_get_lib_props := Triton_LoadProcedure( LibraryHnd, 'tecella_get_lib_props' ) ;
        @tecella_get_hw_props := Triton_LoadProcedure( LibraryHnd, 'tecella_get_hw_props' ) ;
        @tecella_get_reg_props := Triton_LoadProcedure( LibraryHnd, 'tecella_get_reg_props' ) ;
        //@tecella_error_get_last_msg := Triton_LoadProcedure( LibraryHnd, 'tecella_error_get_last_msg' ) ;
        @tecella_error_set_callback := Triton_LoadProcedure( LibraryHnd, 'tecella_error_set_callback' ) ;
        @tecella_chan_set := Triton_LoadProcedure( LibraryHnd, 'tecella_chan_set' ) ;
        @tecella_chan_get := Triton_LoadProcedure( LibraryHnd, 'tecella_chan_get' ) ;
        @tecella_chan_set_pct := Triton_LoadProcedure( LibraryHnd, 'tecella_chan_set_pct' ) ;
        @tecella_chan_get_pct := Triton_LoadProcedure( LibraryHnd, 'tecella_chan_get_pct' ) ;
        @tecella_get_stimulus_mode := Triton_LoadProcedure( LibraryHnd, 'tecella_get_stimulus_mode' ) ;
        @tecella_set_stimulus_mode := Triton_LoadProcedure( LibraryHnd, 'tecella_set_stimulus_mode' ) ;
        //@tecella_chan_to_string := Triton_LoadProcedure( LibraryHnd, 'tecella_chan_to_string' ) ;
        //@tecella_chan_get_units := Triton_LoadProcedure( LibraryHnd, 'tecella_chan_get_units' ) ;
        @tecella_chan_set_enable := Triton_LoadProcedure( LibraryHnd, 'tecella_chan_set_enable' ) ;
        @tecella_chan_get_enable := Triton_LoadProcedure( LibraryHnd, 'tecella_chan_get_enable' ) ;
        @tecella_chan_set_gain := Triton_LoadProcedure( LibraryHnd, 'tecella_chan_set_gain' ) ;
        @tecella_chan_get_gain := Triton_LoadProcedure( LibraryHnd, 'tecella_chan_get_gain' ) ;
        @tecella_chan_set_source := Triton_LoadProcedure( LibraryHnd, 'tecella_chan_set_source' ) ;
        @tecella_chan_get_source := Triton_LoadProcedure( LibraryHnd, 'tecella_chan_get_source' ) ;
        @tecella_chan_set_bessel := Triton_LoadProcedure( LibraryHnd, 'tecella_chan_set_bessel' ) ;
        @tecella_chan_get_bessel := Triton_LoadProcedure( LibraryHnd, 'tecella_chan_get_bessel' ) ;
        @tecella_bessel_value2freq := Triton_LoadProcedure( LibraryHnd, 'tecella_bessel_value2freq' ) ;
        @tecella_bessel_freq2value := Triton_LoadProcedure( LibraryHnd, 'tecella_bessel_freq2value' ) ;

        @tecella_stimulus_set := Triton_LoadProcedure( LibraryHnd, 'tecella_stimulus_set' ) ;
        @tecella_stimulus_get := Triton_LoadProcedure( LibraryHnd, 'tecella_stimulus_get' ) ;
        @tecella_stimulus_sample_count := Triton_LoadProcedure( LibraryHnd, 'tecella_stimulus_sample_count' ) ;

        @tecella_acquire_enable_channel := Triton_LoadProcedure( LibraryHnd, 'tecella_acquire_enable_channel' ) ;
        @tecella_acquire_set_buffer_size := Triton_LoadProcedure( LibraryHnd, 'tecella_acquire_set_buffer_size' ) ;
        @tecella_acquire_start := Triton_LoadProcedure( LibraryHnd, 'tecella_acquire_start' ) ;
        @tecella_acquire_start_stimulus := Triton_LoadProcedure( LibraryHnd, 'tecella_acquire_start_stimulus' ) ;
        @tecella_acquire_stop_stimulus := Triton_LoadProcedure( LibraryHnd, 'tecella_acquire_stop_stimulus' ) ;
        @tecella_acquire_stop := Triton_LoadProcedure( LibraryHnd, 'tecella_acquire_stop' ) ;
        @tecella_acquire_samples_available := Triton_LoadProcedure( LibraryHnd, 'tecella_acquire_samples_available' ) ;
        @tecella_acquire_read_d := Triton_LoadProcedure( LibraryHnd, 'tecella_acquire_read_d' ) ;
        @tecella_acquire_read_i := Triton_LoadProcedure( LibraryHnd, 'tecella_acquire_read_i' ) ;
        @tecella_acquire_i2d_scale := Triton_LoadProcedure( LibraryHnd, 'tecella_acquire_i2d_scale' ) ;
        @tecella_auto_comp := Triton_LoadProcedure( LibraryHnd, 'tecella_auto_comp' ) ;

        LibraryLoaded := True ;
        end
     else begin
          ShowMessage( 'TritonDLL.DLL library not found' ) ;
          LibraryLoaded := False ;
          end ;
     end ;


function  Triton_LoadProcedure(
         Hnd : THandle ;       { Library DLL handle }
         Name : string         { Procedure name within DLL }
         ) : Pointer ;         { Return pointer to procedure }
{ ----------------------------
  Get address of DLL procedure
  ----------------------------}
var
   P : Pointer ;
begin
     P := GetProcAddress(Hnd,PChar(Name)) ;
     if P = Nil then begin
        ShowMessage(format('TecellaAmp.DLL- %s not found',[Name])) ;
        end ;
     Result := P ;
     end ;


procedure Triton_ConfigureHardware(
            EmptyFlagIn : Integer ) ;
begin
    end ;

procedure Triton_InitialiseBoard ;
// -----------------------
// Initial interface board
// -----------------------
var
    Err : Integer ;
begin

     DeviceInitialised := False ;

     // Load DLL library
     if not LibraryLoaded then Triton_LoadLibrary ;
     if not LibraryLoaded then Exit ;

     // Initialise interface
     Err := tecella_initialize( TecHandle, TECELLA_HW_MODEL_TRITON ) ;
     Triton_CheckError('tecella_initialize :',Err) ;
     if Err = 0 then DeviceInitialised := True
                else DeviceInitialised := False ;

     end ;


function  Triton_GetLabInterfaceInfo(
            var Model : string ; { Laboratory interface model name/number }
            var ADCMinSamplingInterval : Double ; { Smallest sampling interval }
            var ADCMaxSamplingInterval : Double ; { Largest sampling interval }
            var ADCMinValue : Integer ; { Negative limit of binary ADC values }
            var ADCMaxValue : Integer ; { Positive limit of binary ADC values }
            var ADCVoltageRanges : Array of single ; { A/D voltage range option list }
            var NumADCVoltageRanges : Integer ; { No. of options in above list }
            var ADCBufferLimit : Integer ;      { Max. no. samples in A/D buffer }
            var DACMaxVolts : Single ; { Positive limit of bipolar D/A voltage range }
            var DACMinUpdateInterval : Double {Min. D/A update interval }
            ) : ByteBool ;
// -------------------------
// Get interface information
// -------------------------
var
  LibraryProperties : Ttecella_lib_PROPS ;
  i : Integer ;

begin

     Result := False ;

     if not DeviceInitialised then Triton_InitialiseBoard ;
     if not DeviceInitialised then Exit ;

     // Library properties
     Triton_CheckError( 'tecella_get_lib_props : ',
                        tecella_get_lib_props( LibraryProperties)) ;

     // Hardware properties
     Triton_CheckError( 'tecella_get_hw_props : ',
                        tecella_get_hw_props( TecHandle,
                                              HardwareProps)) ;

     Model := '' ;

     // Name
     for i := 0 to High(HardwareProps.device_name) do begin
         if (HardwareProps.device_name[i] = '?') or
            (HardwareProps.device_name[i] = #0) then Break ;
            Model := Model + HardwareProps.device_name[i] ;
            end ;

     // Serial numner
     Model := Model + ' (s/n ' ;
     for i := 0 to High(HardwareProps.serial_number) do begin
         if (HardwareProps.serial_number[i] = '?') or
            (HardwareProps.serial_number[i] = #0) then Break ;
            Model := Model + HardwareProps.serial_number[i] ;
            end ;
     Model := Model + ')' ;
     
     Model := Model + format( ' (Lib. V%d.%d) (%d channels)',
                      [LibraryProperties.v_maj,LibraryProperties.v_min,
                      HardwareProps.nchans]) ;

     ADCMinSamplingInterval := HardwareProps.sample_period_min ;
     FADCMinSamplingInterval := ADCMinSamplingInterval ;
     ADCMaxSamplingInterval := HardwareProps.sample_period_max ;
     FADCMaxSamplingInterval := ADCMaxSamplingInterval ;
     ADCMinValue := -32768 ;
     ADCMaxValue := 32767 ;
     FADCMaxValue := ADCMaxValue ;
     ADCVoltageRanges[0] := 0.2 ;
     NumADCVoltageRanges := 1 ;
     ADCBufferLimit := 32768*2 ;
     DACMaxVolts := ADCVoltageRanges[0] ;
     FDACMaxVolts := DACMaxVolts ;
     DACMinUpdateInterval := HardwareProps.sample_period_min ;

     // Patch clamp gain factors
     FGainFactors[0] := 0.01 ;
     FGainFactors[1] := 0.1 ;
     FGainFactors[2] := 1.0 ;
     FGainFactors[3] := 10.0 ;

     Result := True ;

     end ;

function  Triton_ADCToMemory(
            var HostADCBuf : Array of SmallInt  ;
            nChannels : Integer ;
            nSamples : Integer ;
            var dt : Double ;
            ADCVoltageRange : Single ;
            TriggerMode : Integer ;
            CircularBuffer : ByteBool
            ) : ByteBool ;
var
    ch, i, NumSamplesAvailable,NumSamplesRead : Integer ;
    iBuf : PSmallIntArray ;
    Enabled : Boolean ;
begin

    if not DeviceInitialised then Exit ;

    FNumChannels := nChannels ;
    fNumSamples := nSamples ;
    FSamplingInterval := dt ;

    FSweepDuration := FSamplingInterval*fNumSamples ;
    FOutPointer := 0 ;

    // Set Triton internal buffer size
    Triton_CheckError( 'tecella_acquire_set_buffer_size',
                       tecella_acquire_set_buffer_size( TecHandle, nSamples*2 )) ;

    // Clear any samples in queue
    iBuf := Nil ;
    for ch := 1 to nChannels-1 do begin
        // Get no. of samples in queue
        Triton_CheckError( 'tecella_acquire_samples_available',
                           tecella_acquire_samples_available( TecHandle,
                                                              ch-1,
                                                              NumSamplesAvailable )) ;
        // Read them to clear
        if NumSamplesAvailable > 0 then begin
           if iBuf <> Nil then FreeMem( iBuf ) ;
           GetMem( iBuf, NumSamplesAvailable*2 ) ;
           tecella_acquire_read_i( TecHandle,
                                ch-1,
                                NumSamplesAvailable,
                                iBuf,
                                NumSamplesRead ) ;
           end ;
        end ;

    // Enable Triton channels
    for ch := 0 to HardwareProps.nchans-1 do begin
      if ch < (FNumChannels-1) then Enabled := True
                               else Enabled := False ;
      Triton_CheckError( 'tecella_acquire_enable_channel  : ',
                         tecella_acquire_enable_channel(
                         TecHandle,ch,Enabled )) ;
      end ;


    if iBuf <> Nil then FreeMem(iBuf) ;

    end ;


function Triton_StopADC : ByteBool ;
// -----------------
// Stop A/D sampling
// -----------------
begin

     if not DeviceInitialised then Exit ;

    // Clear A/D data buffer
    Triton_CheckError( 'tecella_acquire_stop',
                       tecella_acquire_stop(TecHandle)) ;

    end ;


procedure Triton_GetADCSamples (
            var OutBuf : Array of SmallInt ;
            var OutBufPointer : Integer
            ) ;
// ----------------------------------------------------------------
// Get latest A/D samples from device and transfer to output buffer
// ----------------------------------------------------------------
var
    ch, i, j,iVmStart,NumSamplesAvailable,NumSamplesRead : Integer ;
    iBuf : PSmallIntArray ;
begin

     if not DeviceInitialised then Exit ;

    // Get number of samples available
    Triton_CheckError( 'tecella_acquire_samples_available',
                        tecella_acquire_samples_available( TecHandle,
                                                           0,
                                                           NumSamplesAvailable )) ;

//   outputDebugString( PChar(format('%.5g',[IScale]))) ;

    if NumSamplesAvailable <= 0 then Exit ;

    // Allocate buffer
    GetMem( iBuf, NumSamplesAvailable*2 ) ;

    // Copy voltage stimulus
    iVmStart := FOutPointer div FNumChannels ;
    for i:= iVmStart to Min(iVmStart + NumSamplesAvailable,FNumSamples) -1 do begin
        OutBuf[i*FNumChannels] := VmBuf^[i] ;
        end ;

    // Copy current channels

    for ch := 1 to FNumChannels-1 do begin

        // Read channel
        tecella_acquire_read_i( TecHandle,
                                ch-1,
                                NumSamplesAvailable,
                                iBuf,
                                NumSamplesRead ) ;

        // Copy to output buffer
        for i := 0 to NumSamplesRead-1 do begin
            j := FOutPointer + (i*FNumChannels) + ch ;
            OutBuf[j] := iBuf^[i] ;
            end ;

        end ;

    FOutPointer := FOutPointer  + NumSamplesRead*FNumChannels ;
    OutBufPointer := FOutPointer ;

    outputDebugString( PChar(format('%d %d',[OutBufPointer,NumSamplesAvailable]))) ;

    FreeMem( iBuf ) ;

    end ;


procedure Triton_CheckSamplingInterval(
            var SamplingInterval : Double
            ) ;
// ---------------------------------
// Apply limits to sampling interval
// ---------------------------------
begin

     if not DeviceInitialised then Exit ;

    SamplingInterval := Round(SamplingInterval/HardwareProps.sample_period_lsb)
                        *HardwareProps.sample_period_lsb ;

    SamplingInterval := Min(Max( SamplingInterval,
                                 FADCMinSamplingInterval),
                                 FADCMaxSamplingInterval) ;

    end ;


function  Triton_MemoryToDAC(
          var DACValues : Array of SmallInt  ;
          nChannels : Integer ;
          nPoints : Integer ;
          DACUpdateInterval : Single
          ) : ByteBool ;
// ------------------------------------------
// Start D/A stimulus and A/D recording sweep
// ------------------------------------------
const
    StimSegmentsSize = 500 ;
var
  StimSegments : Array[0..StimSegmentsSize-1] of Ttecella_stimulus_segment ;
  NumStimSegments : Integer ;
  i,j,ch : Integer ;
  V,VScale : Double ;
  StimDuration,StimEndDuration : Double ;
  NumIterations,NumSamplesInStim : Integer ;
  Loop : ByteBool ;
  Index : Integer ;
  ContinuousRecording : ByteBool ;
  SynchronousStimulus : ByteBool ;
  Enabled  : ByteBool ;
  MaxStimSegments : Integer ;

begin

    if not DeviceInitialised then Exit ;

    // Allocate voltage trace buffer
    if VmBuf <> Nil then FreeMem(VmBuf) ;
    GetMem( VmBuf, FNumSamples*2 ) ;

    // Max. no. of segments allowed
    MaxStimSegments := Min( StimSegmentsSize,
                            HardwareProps.max_stimulus_segments ) ;

    // Copy command voltage into voltage storage buffer
    for i := 0 to FNumSamples-1 do begin
        j := Min(i,nPoints-1)*nChannels ;
        VmBuf[i] := DACValues[j] ;
        end ;

    // Create stimulus table from Ch.0 of DACValues waveform

    VScale := (FDACMaxVolts/FADCMaxValue) ;

    // Initialise segment table
    for i := 0 to MaxStimSegments-1 do begin
        StimSegments[i].SegmentType := TECELLA_STIMULUS_SEGMENT_SET;
        StimSegments[i].duration := 0.0 ;
        end ;

    // Add waveform segments
    j := 0 ;
    NumStimSegments := 0 ;
    StimSegments[NumStimSegments].Value := DACValues[0]*VScale ;
    StimDuration := 0.0 ;
    for i := 0 to nPoints-1 do begin
        V := DACValues[j]*VScale ;
        if V <> StimSegments[NumStimSegments].Value then begin
           if NumStimSegments <= (MaxStimSegments-3) then begin
              Inc(NumStimSegments) ;
              StimSegments[NumStimSegments].value := V ;
              end ;
           end ;
        StimSegments[NumStimSegments].Duration :=
              StimSegments[NumStimSegments].Duration
              + DACUpdateInterval ;
        StimDuration := StimDuration + DACUpdateInterval ;
        j := j + nChannels ;
        end ;
    Inc(NumStimSegments) ;

    // Pad end of voltage stimulus to end of recording sweep
    StimEndDuration := FSweepDuration - StimDuration ;
    if StimEndDuration > 0.0  then begin
       StimSegments[NumStimSegments].value := StimSegments[NumStimSegments-1].value ;
       StimSegments[NumStimSegments].duration := StimEndDuration ;
       Inc(NumStimSegments) ;
       end ;

    // Add 10ms to end of sweep
    StimSegments[NumStimSegments].SegmentType := TECELLA_STIMULUS_SEGMENT_SET ;
    StimSegments[NumStimSegments].value := StimSegments[NumStimSegments-1].value ;
    StimSegments[NumStimSegments].duration := 0.01 ;
    Inc(NumStimSegments) ;

  { NumSamplesInStim := tecella_stimulus_sample_count( @StimSegments,
                                                       NumStimSegments,
                                                       DACUpdateInterval,
                                                       0 ) ;}
     //outputDebugString( PChar(format('No. samples in stim %d',[NumSamplesInStim]))) ;

    // Load stimulus into amplifier
    NumIterations := 1 ;
    Loop := False ;
    Index := 0 ;
    Triton_CheckError( 'tecella_stimulus_set',
                       tecella_stimulus_set( TecHandle,
                                             @StimSegments,
                                             NumStimSegments,
                                             NumIterations,
                                             Loop,
                                             Index )) ;

{  Triton_CheckError( 'tecella_stimulus_get',
                       tecella_stimulus_get( TecHandle,
                                             @StimSegments,
                                             MaxStimSegments,
                                             NumStimSegments,
                                             NumIterations,
                                             Loop,
                                             Index )) ;}


      // Start recording sweep
      ContinuousRecording := False ;
      SynchronousStimulus := True ;
      Triton_CheckError( 'tecella_acquire_start  : ',
                          tecella_acquire_start ( TecHandle,
                                                  DACUpdateInterval,
                                                  ContinuousRecording,
                                                  SynchronousStimulus )) ;
      end ;


function  Triton_MemoryToDACAndDigitalOut(
          var DACValues : Array of SmallInt  ; // D/A output values
          NumDACChannels : Integer ;                // No. D/A channels
          NumDACPoints : Integer ;                  // No. points per channel
          var DigValues : Array of SmallInt  ; // Digital port values
          DigitalInUse : ByteBool ;             // Output to digital outs
          ExternalTrigger : ByteBool            // Wait for ext. trigger
          ) : ByteBool ;                        // before starting output
begin
    end ;


function Triton_GetDACUpdateInterval : double ;
begin
    Result := FSamplingInterval ;
    end ;



function Triton_StopDAC : ByteBool ;
//
// Stop D/A output
begin

    if not DeviceInitialised then Exit ;
    // Note. stops both D/A and A/D

    Triton_CheckError( 'tecella_acquire_stop',
                       tecella_acquire_stop(TecHandle)) ;

    end ;


procedure Triton_WriteDACsAndDigitalPort(
            var DACVolts : array of Single ;
            nChannels : Integer ;
            DigValue : Integer
            ) ;
begin
    end ;


function Triton_GetMaxDACVolts : single ;
begin
    Result := 0.2 ;
    end ;



  function Triton_ReadADC( Channel : Integer ) : SmallInt ;
begin
    Result := 0 ;
    end ;



procedure Triton_GetChannelOffsets(
            var Offsets : Array of Integer ;
            NumChannels : Integer
            ) ;
var
    i : Integer ;
begin
    for i := 0 to NumChannels-1 do Offsets[i] := i ;
    end ;


procedure Triton_CloseLaboratoryInterface ;
var
    Err : Integer ;
begin

    if not DeviceInitialised then Exit ;
    Err := tecella_finalize (TecHandle);

    if VmBuf <> Nil then FreeMem(VmBuf) ;
    VmBuf := Nil ;

    DeviceInitialised := False ;

    end ;

function Triton_CurrentCalibration(
         Chan : Integer
         ) : Single  ;
// ----------------------------------
// Returns current calibration factor
// ----------------------------------
var
    IScale : Double ;
begin

     if not DeviceInitialised then Exit ;

    // Get current scaling factor
    Triton_CheckError( 'tecella_acquire_i2d_scale',
                       tecella_acquire_i2d_scale(TecHandle,Chan,IScale)) ;

    Result := (FDACMaxVolts)/(IScale*32768*1E12) ;

    end ;


function Triton_CurrentGain(
         Chan : Integer
         ) : Single  ;
// ----------------------------------
// Returns current calibration factor
// ----------------------------------
var
    iGain : Integer ;
begin

     if not DeviceInitialised then Exit ;

    // Get channel gain factor
    Triton_CheckError( 'tecella_chan_get_gain',
                       tecella_chan_get_gain(TecHandle,Chan,iGain)) ;

    iGain := Min(Max(0,iGain),High(FGainFactors)) ;
    Result := FGainFactors[iGain] ;

    end ;



function Triton_VoltageCalibration : Single  ;
// ----------------------------------
// Returns voltage calibration factor
// ----------------------------------
begin
    Result := FDACMaxVolts/(FDACMaxVolts*1E3) ;
    end ;


procedure Triton_Wait( Delay : Single ) ;
begin
    end ;


procedure Triton_CheckError(
          Location : String ;
          Err : Integer
          ) ;
{ --------------------------------------------------------------
  Warn User if the Lab. interface library returns an error
  --------------------------------------------------------------}
var
   s : string ;
begin
     if Err <> 0 then begin
        case Err of
          1: s := 'Bad Handle';
          2 : s := 'Bad Handle Return (did you pass in &handle?)';
          3 : s := 'Internal error: handle alloc failed';
          4 : s := '';
          5 : s := 'Hardware connection failed';
          6 : s := 'Return pointer is null (did you pass in &value?)';
          7 : s := 'Maximum vcmd allocation exceeded';
          8 : s := 'Vcmd handle out of range';
          9 : s := 'Vcmd segment pointer has illegal value';
          10 : s := 'Vcmd segment index out of range';
          11 : s := 'OkLib not loaded';
          12 : s := 'Device init failed';
          13 : s := '';
          14 : s := 'Device open failed';
          15 : s := 'Channel # too smalx : s := ';
          16 : s := 'Channel # too large';
          17 : s := 'Get requires accessing data for a single channel; not ALLCHAN';
          else s := 'Unknown error' ;
          end ;

        ShowMessage( format('%s : %s (%d)',[Location,s,Err]) ) ;
        end ;
     end ;


procedure TritonSetRegisterPercent(
          Reg : Integer ;
          Chan : Integer ;
          var PercentValue : Double ) ;
// -------------------------------------------
// Set register to percentage of maximum value
// -------------------------------------------
var
    RegProps : Ttecella_reg_props ;
begin

    Triton_CheckError( 'tecella_get_reg_props : ',
                        tecella_get_reg_props( TecHandle,
                                               Reg,
                                               RegProps ));

    if not RegProps.supported then exit ;

    Triton_CheckError( 'tecella_chan_set_pct : ',
                       tecella_chan_set_pct( TecHandle,
                                             Reg,
                                             Chan,
                                             PercentValue ) ) ;

    end ;

procedure TritonGetRegister(
          Reg : Integer ;
          Chan : Integer ;
          var Value : Double ;
          var PercentValue : Double ;
          var Enabled : Boolean ) ;
// -------------------------------------------
// Get register to percentage of maximum value
// -------------------------------------------
var
    RegProps : Ttecella_reg_props ;
    byteEnabled : ByteBool ;
begin

    Enabled := False ;

    Triton_CheckError( 'tecella_get_reg_props : ',
                        tecella_get_reg_props( TecHandle,
                                               Reg,
                                               RegProps ));

    if not RegProps.supported then exit ;

    // Get current value
    Triton_CheckError( 'tecella_chan_get : ',
                       tecella_chan_get( TecHandle,
                                        Reg,
                                        Chan,
                                        Value )) ;

    // Get current percent value
    Triton_CheckError( 'tecella_chan_get_pct : ',
                       tecella_chan_get_pct( TecHandle,
                                             Reg,
                                             Chan,
                                             PercentValue ) ) ;

    if RegProps.can_be_disabled then begin
       Triton_CheckError( 'tecella_chan_get_enable : ',
                          tecella_chan_get_enable( TecHandle,
                                                   Reg,
                                                   Chan,
                                                  byteEnabled ) ) ;
       Enabled := byteEnabled ;
       end
    else  Enabled := True ;

    end ;


function TritonGetGain(
         Chan : Integer ) : Integer ;
// ---------------------------------------
// Get Triton patch clamp gain for channel
// ---------------------------------------
var
    iGain : Integer ;
begin

    Triton_CheckError( 'TritonGetGain : ',
                       tecella_chan_get_gain( TecHandle,
                                             Chan,
                                             iGain )) ;
    Result := iGain ;
    end ;


procedure TritonSetGain(
          Chan : Integer ;
          iGain : Integer ) ;
// ---------------------------------------
// Set Triton patch clamp gain for channel
// ---------------------------------------
begin

    iGain := Min(Max(iGain,0),HardwareProps.nGains-1) ;
    Triton_CheckError( 'TritonSetGain : ',
                       tecella_chan_set_gain( TecHandle,
                                             Chan,
                                             iGain )) ;
    end ;


function TritonGetSource(
         Chan : Integer ) : Integer ;
// ---------------------------------------
// Get Triton patch clamp input source for channel
// ---------------------------------------
var
    iSource : Integer ;
begin

    Triton_CheckError( 'TritonGetSource : ',
                       tecella_chan_get_source( TecHandle,
                                               Chan,
                                               iSource )) ;
    Result := iSource ;
    end ;


procedure TritonSetSource(
          Chan : Integer ;
          iSource : Integer ) ;
// ---------------------------------------
// Set Triton patch clamp input source for channel
// ---------------------------------------
begin

    Triton_CheckError( 'TritonSetSource : ',
                       tecella_chan_set_source( TecHandle,
                                               Chan,
                                               iSource )) ;
    end ;


function TritonGetRegisterEnabled(
         Reg : Integer ;
         Chan : Integer ) : ByteBool ;
// ---------------------------------------
// Get Triton register Enable state
// ---------------------------------------
var
    OnOff : Integer ;
    Enabled : ByteBool ;
begin

    Triton_CheckError( 'TritonGetRegisterEnable : ',
                        tecella_chan_get_enable( TecHandle,
                                                Reg,
                                                Chan,
                                                Enabled )) ;

    Result := Enabled ;

    end ;


procedure TritonSetRegisterEnabled(
         Reg : Integer ;
         Chan : Integer ;
         Enabled : ByteBool ) ;
// ---------------------------------------
// Get Triton register Enable state
// ---------------------------------------
var
    OnOff : Integer ;
begin

    if Enabled then OnOff := 1
               else OnOff := 0 ;
    Triton_CheckError( 'TritonSetRegisterEnable : ',
                        tecella_chan_set_enable( TecHandle,
                                                Reg,
                                                Chan,
                                                OnOff )) ;

    end ;


procedure TritonSetBesselFilter(
          Chan : Integer ;
          Value : Integer ;
          var CutOffFrequency : Single ) ;
//
// Set bessel filter cut-off frequency for selected channel and return cut-off frequency
// -------------------------------------------------------------------------------------
var
    Freq : Double ;
begin

    if (not HardwareProps.supports_bessel) then Exit ;

    Value := Round((Value*0.01)*HardwareProps.bessel_value_max) ;

    Value := Min(Max( Value,
                      HardwareProps.bessel_value_min),
                      HardwareProps.bessel_value_max) ;

    // Set bessel filter
    Triton_CheckError( 'tecella_chan_set_bessel  : ',
                       tecella_chan_set_bessel( TecHandle, Chan, Value )) ;

    // Get cut-off frequency (Hz)
    Triton_CheckError( 'tecella_bessel_value2freq  : ',
                       tecella_bessel_value2freq(TecHandle, Value, Freq )) ;

    CutOffFrequency := Freq ;

    end ;

procedure TritonAutoCompensate(
          Chan : Integer ) ;
// ------------------------------------
// Apply automatic capacity compensation
// -------------------------------------
var
    ChanList : Array[0..7] of Integer ;
    NumChans : Integer ;
begin

     ChanList[0] := Chan ;
     NumChans := 1 ;
     Triton_CheckError( 'tecella_auto_comp  : ',
                        tecella_auto_comp( TecHandle,
                                           @ChanList,
                                           NumChans )) ;

     end ;

initialization
    DeviceInitialised := False ;
    LibraryLoaded := False ;
    VmBuf := Nil ;

end.
