unit ScopeDisplay;
{ =====================================================
  Oscilloscope Display Component
  (c) J. Dempster, University of Strathclyde, 1999-2001
  =====================================================
  24/1/99 ... Started
  5/3/99 .... Trunc changed to Round
  15/7/99 ... Vertical cursors can now be associated with individual channels
  1/8/99  ... Additional lines no longer displayed in zoom mode
              Channel.CalBars initialised with -1 to indicate no data yet
              FTCalBar now in units of samples rather than time
  6/8/99 .... Multi-record storage mode added
  17/8/99 ... T calibration bar now correct size
  20/8/99 ... PlotRecord now o/p's partial lines to canvas when more than
              16000 points are required, to avoid limitations of Windows polyline function
  27/8/99 ... Grid option added to display
  16/9/99 ... Horizontal cursors now visible only when channel is visible
  26/10/99 ... AddHorizontalCursor now has UseAsZeroLevel flag
               Channel zero levels updated when baseline cursors changed
  10/2/00 .... Bug in Print and CopyImageToClipboard when some channels
               disabled fixed.
  28/2/00 .... Left margin now leaves space for vertical cal. bar. text.

               PrinterShowLabels & PrinterShowZeroLevels added to properties
               Printer top margin now taken into account when setting plot height
  6/3/00  .... CopyImageToClipboard now adjusts pen width correctly
  23/7/00 .... Bug where zoom box jumped to mouse cursor on entry to zoom mode fixed
  4/1/01 ... OnCursorChange now called when returning from zoom to normal display mode
             Traces now restricted to limits of channel area within display
  21/1/01 ... Printer font name now set correctly (DefaultFont object handled correctly)
              Space correctly set for title at top of page
  26/4/01 ... Print and CopyImageToClipboard Left margin now set correctly
              allowing space for calibration labels
  20/7/01 ... CopyDataToClipboard now handles large blocks of data correctly
  14/8/01 ... Array properties shifted to Public to make component compilable under Delphi V5
  5/12/01 ... StorageMode now superimposes traces captured in real-time
  18/2/02 ... FNumPoints now correctly set to zero
  19/3/02 ... Channels now spaced apart vertically, top/bottom grid lines now correctly plotted
              ???Units/Div calibrations added to channel name labels
  26/10/02 ... Trails left when moving zoom box fixed
               Cursors no longer disappear when superimposed
               An internal bitmap now used to hold image upon which cursors are superimoposed
  29/10/02 ... Display corruption caused by superimposed windows fixed
  8/12/02 .... OnCursorChange now only called in normal (not zoom) display mode
               Horizontal cursors displayed with pen in pmMASK mode
  12/12/02 ... Downward shift of top margin with repeated printing fixed.
  12.02.03 ... Zero baseline cursors now displayed in a different colour
               to indicate when zero level is not a true ADC=0 level.
  3.04.03 .... Error when no printers are defined now fixed
  18.05.03 ... Marker text can be added to bottom of display
  24.04.03 ... Number of horizontal and vertical grid lines can be changed
  04.11.03 ... Display calibration now displays 1000s
  06.11.03 ... Vertical grid spacing now correct when MaxPoints is small value
  20.11.03 ... Divide error when xMax = xMin fixed
  03.02.04 ... Vertical cursors can now be moved using keyboard arrows
               without a trail being left on the screen (MoveActiveCursor)
  21.11.04 ... Max. no.of points increased to 131072
  20.06.05 ... Colour of vertical cursor can now be defined  again
               Selected pair of vertical cursors can be linked with a hor. line
               at bottom of display area (LinkVerticalCursors())
  23.09.05 ... .ZoomIn and .ZoomOut added
  }

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Clipbrd, printers, mmsystem, math ;
const
     ScopeChannelLimit = 15 ;
     AllChannels = -1 ;
     NoRecord = -1 ;
     MaxStoredRecords = 200 ;
     MaxPoints = 131072 ;
type
    TPointArray = Array[0..MaxPoints-1] of TPoint ;
    TSinglePoint = record
        x : single ;
        y : single ;
        end ;
    TSinglePointArray = Array[0..MaxPoints-1] of TSinglePoint ;
    TSmallIntArray = Array[0..$FFFFFF] of SmallInt ;
    PSmallIntArrayPointer = ^TSmallIntArray ;
    TScopeChannel = record
         xMin : single ;
         xMax : single ;
         yMin : single ;
         yMax : single ;
         xScale : single ;
         yScale : single ;
         Left : Integer ;
         Right : Integer ;
         Top : Integer ;
         Bottom : Integer ;
         ADCUnits : string ;
         ADCName : string ;
         ADCScale : single ;
         ADCZero : integer ;
         ADCZeroAt : Integer ;
         CalBar : single ;
         InUse : Boolean ;
         ADCOffset : Integer ;
         color : TColor ;
         Position : Integer ;
         ChanNum : Integer ;
         ZeroLevel : Boolean ;
         end ;
    TMousePos = ( TopLeft,
              TopRight,
              BottomLeft,
              BottomRight,
              MLeft,
              MRight,
              MTop,
              MBottom,
              MDrag,
              MNone ) ;

  TScopeDisplay = class(TGraphicControl)
  private
    { Private declarations }
    FMinADCValue : Integer ;   // Minimum A/D sample value
    FMaxADCValue : Integer ;   // Maximum A/D sample value
    FNumChannels : Integer ;   // No. of channels in display
    FNumPoints : Integer ;     // No. of points displayed
    FMaxPoints : Integer ;     // Max. display points allowed
    FXMin : Integer ;          // Index of first sample in buffer on display
    FXMax : Integer ;          // Index of last sample in buffer on display
    FXOffset : Integer ;       // User-settable offset added to sample index numbers
                               // when computing sample times

    Channel : Array[0..ScopeChannelLimit] of TScopeChannel ;
    KeepChannel : Array[0..ScopeChannelLimit] of TScopeChannel ;
    HorCursors : Array[0..ScopeChannelLimit] of TScopeChannel ;
    VertCursors : Array[0..64] of TScopeChannel ;
    FLinkVerticalCursors : Array[0..1] of Integer ;
    FChanZeroAvg : Integer ;
    FTopOfDisplayArea : Integer ;
    FBottomOfDisplayArea : Integer ;
    FBuf : ^TSmallIntArray ;
    FBufIsWordData : Boolean ;
    FCursorsEnabled : Boolean ;
    FHorCursorActive : Boolean ;
    FHorCursorSelected : Integer ;
    FVertCursorActive : Boolean ;
    FVertCursorSelected : Integer ;
    FLastVertCursorSelected : Integer ;
    FZoomBox : TRect ;
    FZoomMode : Boolean ;
    FZoomCh : Integer ;
    FMouseOverChannel : Integer ;
    FMoveZoomBox : Boolean ;
    FZoomDisableHorizontal : Boolean ;
    FZoomDisableVertical : Boolean ;
    MousePos : TMousePos ;
    FXOld : Integer ;
    FYOld : Integer ;
    FTScale : single ;
    FTFormat : string ;
    FTCalBar : single ;
    FOnCursorChange : TNotifyEvent ;
    FCursorChangeInProgress : Boolean ;
    { Printer settings }
    FPrinterFontSize : Integer ;
    FPrinterPenWidth : Integer ;
    FPrinterFontName : string ;
    FPrinterLeftMargin : Integer ;
    FPrinterRightMargin : Integer ;
    FPrinterTopMargin : Integer ;
    FPrinterBottomMargin : Integer ;
    FPrinterDisableColor : Boolean ;
    FPrinterShowLabels : Boolean ;
    FPrinterShowZeroLevels : Boolean ;
    FMetafileWidth : Integer ;
    FMetafileHeight : Integer ;
    FTitle : TStringList ;
    { Additional line }
    FLine : ^TSinglePointArray ;
    FLineCount : Integer ;
    FLineChannel : Integer ;
    FLinePen : TPen ;
    { Display storage mode internal variables }
    FStorageMode : Boolean ;
    FStorageFileName : Array[0..255] of char ;
    FStorageFile : TFileStream ;

    FStorageList : Array[0..MaxStoredRecords-1] of Integer ;
    FRecordNum : Integer ;
    FDrawGrid : Boolean ;
    FNumHorizontalGridLines : Integer ;
    FNumVerticalGridLines : Integer ;

    //Display settings
    FGridColor : TColor ;         // Calibration grid colour
    FTraceColor : TColor ;        // Trace colour
    FBackgroundColor : TColor ;   // Background colour
    FCursorColor : TColor ;       // Cursors colour
    FNonZeroHorizontalCursorColor : TColor ; // Cursor colour
    BackBitmap : TBitMap ;        // Display background bitmap (traces/grid)
    ForeBitmap : TBitmap ;        // Display foreground bitmap (cursors)
    DisplayRect : TRect ;         // Rectangle defining size of display area

    FMarkerText : TStringList ;   // Marker text list

    { -- Property read/write methods -------------- }

    procedure SetNumChannels(Value : Integer ) ;
    procedure SetNumPoints( Value : Integer ) ;
    procedure SetMaxPoints( Value : Integer ) ;

    procedure SetChanName( Ch : Integer ; Value : string ) ;
    function  GetChanName( Ch : Integer ) : string ;

    procedure SetChanUnits( Ch : Integer ; Value : string ) ;
    function  GetChanUnits( Ch : Integer ) : string ;

    procedure SetChanScale( Ch : Integer ; Value : single ) ;
    function  GetChanScale( Ch : Integer ) : single ;

    procedure SetChanCalBar( Ch : Integer ; Value : single ) ;
    function GetChanCalBar( Ch : Integer ) : single ;

    procedure SetChanZero( Ch : Integer ; Value : Integer ) ;
    function GetChanZero( Ch : Integer ) : Integer ;

    procedure SetChanZeroAt( Ch : Integer ; Value : Integer ) ;
    function GetChanZeroAt( Ch : Integer ) : Integer ;

    procedure SetChanZeroAvg( Value : Integer ) ;

    procedure SetChanOffset( Ch : Integer ; Value : Integer ) ;
    function GetChanOffset( Ch : Integer ) : Integer ;

    procedure SetChanVisible( Ch : Integer ; Value : boolean ) ;
    function GetChanVisible( Ch : Integer ) : Boolean ;

    procedure SetChanColor( Ch : Integer ; Value : TColor ) ;
    function GetChanColor( Ch : Integer ) : TColor ;

    procedure SetXMin( Value : Integer ) ;
    procedure SetXMax( Value : Integer ) ;
    procedure SetYMin( Ch : Integer ; Value : single ) ;
    function GetYMin( Ch : Integer ) : single ;
    procedure SetYMax( Ch : Integer ; Value : single ) ;
    function GetYMax( Ch : Integer ) : single ;

    procedure SetHorCursor( iCursor : Integer ; Value : Integer ) ;
    function GetHorCursor( iCursor : Integer ) : Integer ;
    procedure SetVertCursor( iCursor : Integer ; Value : Integer ) ;
    function GetVertCursor( iCursor : Integer ) : Integer ;

    procedure SetTFormat( Value : string ) ;
    function GetTFormat : string ;

    procedure SetPrinterFontName( Value : string ) ;
    function GetPrinterFontName : string ;

    procedure SetPrinterLeftMargin( Value : Integer ) ;
    function GetPrinterLeftMargin : integer ;

    procedure SetPrinterRightMargin( Value : Integer ) ;
    function GetPrinterRightMargin : integer ;

    procedure SetPrinterTopMargin( Value : Integer ) ;
    function GetPrinterTopMargin : integer ;

    procedure SetPrinterBottomMargin( Value : Integer ) ;
    function GetPrinterBottomMargin : integer ;

    function GetXScreenCoord( Value : Integer ) : Integer ;

    procedure SetStorageMode( Value : Boolean ) ;

    procedure SetGrid( Value : Boolean ) ;

    function GetNumVerticalCursors : Integer ;
    function GetNumHorizontalCursors : Integer ;

    { -- End of property read/write methods -------------- }


    { -- Methods used internally by component ------------ }

    function IntLimitTo( Value : Integer ; Lo : Integer ;  Hi : Integer ) : Integer ;
    procedure DrawHorizontalCursor( Canv : TCanvas ; iCurs : Integer ) ;
    function ProcessHorizontalCursors( Y : Integer ) : Boolean ;
    procedure DrawVerticalCursor( Canv : TCanvas ; iCurs : Integer ) ;
    procedure DrawVerticalCursorLink( Canv : TCanvas ) ;

    function ProcessVerticalCursors( X : Integer ; Y : Integer )  : Boolean ;
    procedure ProcessZoomBox(X : Integer ; Y : Integer ) ;
    procedure SwitchToZoomMode( Chan : Integer ) ;
    procedure SwitchToNormalMode ;
    function XToCanvasCoord( var Chan : TScopeChannel ; Value : single ) : Integer  ;
    function CanvasToXCoord( var Chan : TScopeChannel ; xPix : Integer ) : Integer  ;
    function YToCanvasCoord( var Chan : TScopeChannel ; Value : single ) : Integer  ;
    function CanvasToYCoord( var Chan : TScopeChannel ; yPix : Integer ) : Integer  ;
    procedure PlotRecord( Canv : TCanvas ; var Channels : Array of TScopeChannel ;
                          var xy : Array of TPoint ) ;

    procedure CodedTextOut(
              Canvas : TCanvas ;
              var LineLeft : Integer ;
              var LineYPos : Integer ;
              List : TStringList
              ) ;


  protected
    { Protected declarations }
    procedure Paint ; override ;
    procedure MouseMove( Shift: TShiftState; X, Y: Integer ); override ;
    procedure MouseDown( Button: TMouseButton; Shift: TShiftState;X, Y: Integer ); override ;
    procedure MouseUp(Button: TMouseButton;Shift: TShiftState;X, Y: Integer ); override ;
    procedure DblClick ; override ;
  public
    { Public declarations }
    Constructor Create(AOwner : TComponent) ; override ;
    Destructor Destroy ; override ;
    procedure ClearHorizontalCursors ;
    function AddHorizontalCursor( iChannel : Integer ;Color : TColor ;
                                  UseAsZeroLevel : Boolean ) : Integer ;
    procedure ClearVerticalCursors ;
    function AddVerticalCursor( Chan : Integer ; Color : TColor) : Integer ;
    procedure MoveActiveVerticalCursor( Step : Integer ) ;
    procedure LinkVerticalCursors( C0 : Integer ; C1 : Integer ) ;

    procedure ZoomIn( Chan : Integer ) ;
    procedure ZoomOut ;

    procedure XZoom( PercentChange : Single ) ;
    procedure YZoom( Chan : Integer ; PercentChange : Single ) ;

    procedure SetDataBuf( {var Buf : TSmallIntArray } var Buf : Array of SmallInt  ) ;
    procedure CopyDataToClipBoard ;
    procedure CopyImageToClipBoard ;
    procedure Print ;
    procedure ClearPrinterTitle ;
    procedure AddPrinterTitleLine( Line : string);
    procedure CreateLine(Ch : Integer ;iColor : TColor ;iStyle : TPenStyle) ;

    procedure AddPointToLine(x : single ; y : single ) ;

    procedure DisplayNewPoints( NewPoints : Integer ) ;
    procedure ClearDisplay( Canv : TCanvas ) ;

    procedure AddMarker ( AtPoint : Integer ; Text : String ) ;
    procedure ClearMarkers ;

    function XToScreenCoord(Chan : Integer ;Value : single ) : Integer  ;
    function YToScreenCoord(Chan : Integer ;Value : single ) : Integer  ;
    function ScreenCoordToX(Chan : Integer ;Value : Integer ) : single ;
    function ScreenCoordToY(Chan : Integer ;Value : Integer ) : single ;

    property ChanName[ i : Integer ] : string read GetChanName write SetChanName ;
    property ChanUnits[ i : Integer ] : string read GetChanUnits write SetChanUnits ;
    property ChanScale[ i : Integer ] : single read GetChanScale write SetChanScale ;
    property ChanCalBar[ i : Integer ] : single read GetChanCalBar write SetChanCalBar ;
    property ChanZero[ i : Integer ] : Integer read GetChanZero write SetChanZero ;
    property ChanZeroAt[ i : Integer ] : Integer read GetChanZeroAt write SetChanZeroAt ;
    property ChanZeroAvg : Integer read FChanZeroAvg write SetChanZeroAvg ;
    property ChanOffsets[ i : Integer ] : Integer read GetChanOffset write SetChanOffset ;
    property ChanVisible[ i : Integer ] : boolean read GetChanVisible write SetChanVisible ;
    property ChanColor[ i : Integer ] : TColor read GetChanColor write SetChanColor ;
    property YMin[ i : Integer ] : single read GetYMin write SetYMin ;
    property YMax[ i : Integer ] : single read GetYMax write SetYMax ;
    property XScreenCoord[ Value : Integer ] : Integer read GetXScreenCoord ;
    property HorizontalCursors[ i : Integer ] : Integer
             read GetHorCursor write SetHorCursor ;
    property VerticalCursors[ i : Integer ] : Integer
             read GetVertCursor write SetVertCursor ;


  published
    { Published declarations }
    property DragCursor ;
    property DragMode ;
    property OnDragDrop ;
    property OnDragOver ;
    property OnEndDrag ;
    property OnMouseDown ;
    property OnMouseMove ;
    property OnMouseUp ;
    property OnCursorChange : TNotifyEvent
             read FOnCursorChange write FOnCursorChange ;
    property CursorChangeInProgress : Boolean
             read FCursorChangeInProgress write FCursorChangeInProgress ;
    property Height default 150 ;
    property Width default 200 ;
    Property NumChannels : Integer Read FNumChannels write SetNumChannels ;
    Property NumPoints : Integer Read FNumPoints write SetNumPoints ;
    Property MaxPoints : Integer Read FMaxPoints write SetMaxPoints ;
    property XMin : Integer read FXMin write SetXMin ;
    property XMax : Integer read FXMax write SetXMax ;
    property XOffset : Integer read FXOffset write FXOffset ;
    property CursorsEnabled : Boolean read FCursorsEnabled write FCursorsEnabled ;

    property ActiveHorizontalCursor : Integer read FHorCursorSelected ;
    property ActiveVerticalCursor : Integer read FHorCursorSelected ;
    property TScale : single read FTScale write FTScale ;
    property TFormat : string read GetTFormat write SetTFormat ;
    property TCalBar : single read FTCalBar write FTCalBar ;
    property ZoomMode : boolean read FZoomMode ;
    property ZoomDisableHorizontal : Boolean
             read FZoomDisableHorizontal write FZoomDisableHorizontal ;
    property ZoomDisableVertical : Boolean
             read FZoomDisableVertical write FZoomDisableVertical ;
    property PrinterFontSize : Integer read FPrinterFontSize write FPrinterFontSize ;
    property PrinterFontName : string
             read GetPrinterFontName write SetPrinterFontName ;
    property PrinterPenWidth : Integer
             read FPrinterPenWidth write FPrinterPenWidth ;
    property PrinterLeftMargin : Integer
             read GetPrinterLeftMargin write SetPrinterLeftMargin ;
    property PrinterRightMargin : Integer
             read GetPrinterRightMargin write SetPrinterRightMargin ;
    property PrinterTopMargin : Integer
             read GetPrinterTopMargin write SetPrinterTopMargin ;
    property PrinterBottomMargin : Integer
             read GetPrinterBottomMargin write SetPrinterBottomMargin ;
    property PrinterDisableColor : Boolean
             read FPrinterDisableColor write FPrinterDisableColor ;
    property PrinterShowLabels : Boolean
             read FPrinterShowLabels write FPrinterShowLabels ;
    property PrinterShowZeroLevels : Boolean
             read FPrinterShowZeroLevels write FPrinterShowZeroLevels ;

    property MetafileWidth : Integer
             read FMetafileWidth write FMetafileWidth ;
    property MetafileHeight : Integer
             read FMetafileHeight write FMetafileHeight ;
    property StorageMode : Boolean
             read FStorageMode write SetStorageMode ;
    property RecordNumber : Integer
             read FRecordNum write FRecordNum ;
    property DisplayGrid : Boolean
             Read FDrawGrid Write SetGrid ;
    property NumVerticalGridLines : Integer
             Read FNumVerticalGridLines Write FNumVerticalGridLines ;
    property NumHorizontalGridLines : Integer
             Read FNumHorizontalGridLines Write FNumHorizontalGridLines ;
    property MaxADCValue : Integer
             Read FMaxADCValue write FMaxADCValue ;
    property MinADCValue : Integer
             Read FMinADCValue write FMinADCValue ;
    property NumVerticalCursors : Integer read GetNumVerticalCursors ;
    property NumHorizontalCursors : Integer read GetNumHorizontalCursors ;
    property BufIsWordData : Boolean read FBufIsWordData write FBufIsWordData ;
  end;

procedure Register;
function MinInt( const Buf : array of Integer ) : Integer ;
function MaxInt( const Buf : array of Integer ) : Integer ;
function MinFlt( const Buf : array of Single ) : Single ;
function MaxFlt( const Buf : array of Single ) : Single ;


implementation

procedure Register;
begin
  RegisterComponents('Samples', [TScopeDisplay]);
end;


constructor TScopeDisplay.Create(AOwner : TComponent) ;
{ --------------------------------------------------
  Initialise component's internal objects and fields
  -------------------------------------------------- }
var
   i,ch : Integer ;
begin

     inherited Create(AOwner) ;

     { Set opaque background to minimise flicker when display updated }
     ControlStyle := ControlStyle + [csOpaque] ;

     BackBitmap := TBitMap.Create ;
     ForeBitmap := TBitMap.Create ;
     BackBitmap.Width := Width ;
     BackBitmap.Height := Height ;
     ForeBitmap.Width := Width ;
     ForeBitmap.Height := Height ;

     { Create a list to hold any printer title strings }
     FTitle := TStringList.Create ;

     { Create an empty line array }
     FLine := Nil ;
     FLineCount := 0 ;
     FLineChannel := 0 ;
     FLinePen := TPen.Create;
     FLinePen.Assign(Canvas.Pen) ;

     FGridColor := clLtGray ;
     FTraceColor := clBlue ;
     FBackgroundColor := clWhite ;
     FCursorColor := clNavy ;
     FNonZeroHorizontalCursorColor := clRed ;

     Width := 200 ;
     Height := 150 ;
     { Create internal objects used by control }

     FMinADCValue := -32768 ;
     FMaxADCValue := 32768 ;

     FNumChannels := 1 ;
     FNumPoints := 0 ;
     FMaxPoints := 1024 ;
     FXMin := 0 ;
     FXMax := FMaxPoints - 1 ;
     FXOffset := 0 ;
     FChanZeroAvg := 20 ;
     FBuf := Nil ;
     FBufIsWordData := False ;
     for ch := 0 to High(Channel) do begin
         Channel[ch].InUse := True ;
         Channel[ch].ADCName := format('Ch.%d',[ch]) ;
         Channel[ch].ADCUnits := '' ;
         Channel[ch].YMin := FMinADCValue ;
         Channel[ch].YMax := FMaxADCValue ;
         Channel[ch].XMin := FXMin ;
         Channel[ch].XMax := FXMax ;
         Channel[ch].ADCOffset := ch ;
         Channel[Ch].CalBar := -1.0 ;    { <0 indicates no value entered yet }
         Channel[ch].Color := FTraceColor ;
         Channel[ch].ADCZeroAt := -1 ;
         end ;

     for i := 0 to High(HorCursors) do HorCursors[i].InUse := False ;
     for i := 0 to High(VertCursors) do VertCursors[i].InUse := False ;
     FLinkVerticalCursors [0] := -1 ;
     FLinkVerticalCursors [1] := -1 ;

    FCursorsEnabled := True ;
    FHorCursorActive := False ;
    FHorCursorSelected := -1 ;
    FVertCursorActive := False ;
    FVertCursorSelected := -1 ;
    FLastVertCursorSelected := 0 ;
    FOnCursorChange := Nil ;
    FCursorChangeInProgress := False ;
    FTopOfDisplayArea := 0 ;
    FBottomOfDisplayArea := Height ;
    FZoomMode := False ;
    FZoomCh := 0 ;
    FMouseOverChannel := 0 ;
    FZoomDisableHorizontal := False ;
    FZoomDisableVertical := False ;
    FTFormat := '%.1f' ;
    FTScale := 1.0 ;
    FTCalBar := -1.0 ;
    FPrinterDisableColor := False ;
    FPrinterShowLabels := True ;
    FPrinterShowZeroLevels := True ;

    { Create file for holding records in stored display mode }
    FStorageMode := False ;
    FStorageFile := Nil ;
    for i := 0 to High(FStorageList) do FStorageList[i] := NoRecord ;
    FRecordNum := NoRecord ;

    FDrawGrid := True ;
    FNumHorizontalGridLines := 10 ;
    FNumVerticalGridLines := 10 ;

    // Create marker text list
    FMarkerText := TStringList.Create ;

    end ;


destructor TScopeDisplay.Destroy ;
{ ------------------------------------
   Tidy up when component is destroyed
   ----------------------------------- }
begin
     { Destroy internal objects created by TScopeDisplay.Create }
     FBuf := Nil ;

     BackBitmap.Free ;  // Free internal bitmap
     ForeBitmap.Free ;

     { Call inherited destructor }
     inherited Destroy ;

     FTitle.Free ;
     FLinePen.Free ;
     if FStorageFile <> Nil then begin
        FStorageFile.Destroy ;
        DeleteFile( FStorageFileName ) ;
        FStorageFile := Nil ;
        end ;
     if FLine <> Nil then Dispose(FLine) ;

     FMarkerText.Free ;

     end ;


procedure TScopeDisplay.Paint ;
{ ---------------------------
  Draw signal on display area
  ---------------------------}
const
     pFilePrefix : PChar = 'SCD' ;
var
   i,j,ch,Rec,NumBytesPerRecord,iNum : Integer ;
   OK : Boolean ;

   SaveColor : TColor ;
   ZoomScreen : TRect ;
   KeepPen : TPen ;
   TempPath : Array[0..255] of Char ;
   KeepColor : Array[0..ScopeChannelLimit] of TColor ;
   xy : ^TPointArray ;

begin
     { Create plotting points array }
     New(xy) ;

     try

        // Make bit map same size as control
        if (BackBitmap.Width <> Width) or
           (BackBitmap.Height <> Height) then begin
           BackBitmap.Width := Width ;
           BackBitmap.Height := Height ;
           ForeBitmap.Width := Width ;
           ForeBitmap.Height := Height ;
           end ;

        DisplayRect := Rect(0,0,Width-1,Height-1) ;
        SaveColor := Canvas.Pen.Color ;

        // Set colours
        BackBitmap.Canvas.Pen.Color := FTraceColor ;
        BackBitmap.Canvas.Brush.Color := FBackgroundColor ;

        // Clear display, add grid and labels
        ClearDisplay( BackBitmap.Canvas ) ;

        { Display records in storage list }
        if FStorageMode then begin

           { Create a temporary storage file, if one isn't already open }
           if FStorageFile = Nil then begin
              { Get path to temporary file directory }
              GetTempPath( High(TempPath), TempPath ) ;
              { Create a temp file name }
              iNum := GetTempFileName( TempPath, pFilePrefix, 0, FStorageFileName ) ;
              FStorageFile := TFileStream.Create( FStorageFileName, fmCreate ) ;
              end ;

           NumBytesPerRecord := FNumChannels*FNumPoints*2 ;

           { Save current record as first record in file }
           FStorageFile.Seek( 0, soFromBeginning ) ;
           FStorageFile.Write( FBuf^, NumBytesPerRecord ) ;

           { Change colour of stored records }
           for ch := 0 to FNumChannels-1 do begin
               KeepColor[Ch] := Channel[Ch].Color ;
               Channel[Ch].Color := clAqua ;
               end ;

           { Display old records stored in file }
           Rec := 1 ;
           while (FStorageList[Rec] <> NoRecord)
                 and (Rec <= High(FStorageList)) do begin
                 { Read buffer }
                 FStorageFile.Read( FBuf^, NumBytesPerRecord ) ;
                 { Plot record on display }
                 PlotRecord( BackBitmap.Canvas, Channel, xy^ ) ;
                 Inc(Rec) ;
                 end ;

           { Restore colour }
           for ch := 0 to FNumChannels-1 do Channel[Ch].Color := KeepColor[Ch] ;

           { Retrieve current record }
           FStorageFile.Seek( 0, soFromBeginning ) ;
           FStorageFile.Read( FBuf^, NumBytesPerRecord ) ;

           { Determine whether current record is already in list }
           Rec := 1 ;
           while (FStorageList[Rec] <> FRecordNum)
                 and (Rec <= High(FStorageList)) do Inc(Rec) ;

           { If record isn't in list add it to end }
           if Rec > High(FStorageList) then begin
              { Find next vacant storage slot }
              Rec := 1 ;
              while (FStorageList[Rec] <> NoRecord)
                    and (Rec <= High(FStorageList)) do Inc(Rec) ;
              { Add record number to list and store data in file }
              if Rec <= High(FStorageList) then begin
                 FStorageList[Rec] := FRecordNum ;
                 FStorageFile.Seek( Rec*NumBytesPerRecord, soFromBeginning ) ;
                 FStorageFile.Write( FBuf^, NumBytesPerRecord ) ;
                 end ;
              end ;
           end ;

        PlotRecord( BackBitmap.Canvas, Channel, xy^ ) ;

       { Plot external line on selected channel }
       if (FLine <> Nil) and (FLineCount > 1) and not FZoomMode then begin
           KeepPen := BackBitmap.Canvas.Pen ;
           BackBitmap.Canvas.Pen := FLinePen ;
           for i := 0 to FLineCount-1 do begin
               xy^[i].x := XToCanvasCoord( Channel[FLineChannel], FLine^[i].x ) ;
               xy^[i].y := YToCanvasCoord( Channel[FLineChannel], FLine^[i].y ) ;
               end ;
           OK := Polyline( BackBitmap.Canvas.Handle, xy^, FLineCount ) ;
           BackBitmap.Canvas.Pen := KeepPen ;
           end ;

        // Copy from internal bitmap to control
        Canvas.CopyRect( DisplayRect,
                         BackBitmap.Canvas,
                         DisplayRect) ;

        // Add cursors or zoom box
        if not FZoomMode then begin
           { Horizontal Cursors }
           for i := 0 to High(HorCursors) do if HorCursors[i].InUse
               and Channel[HorCursors[i].ChanNum].InUse then DrawHorizontalCursor(Canvas,i) ;
           { Vertical Cursors }
           for i := 0 to High(VertCursors) do if VertCursors[i].InUse then
               DrawVerticalCursor(Canvas,i) ;

           // Draw link between selected pair of vertical cursors
           DrawVerticalCursorLink(Canvas) ;

           end
        else begin
           { Zoom box }
           Canvas.TextOut( 0,0, 'Zoom In/Out' ) ;
           ZoomScreen.Left :=  XToCanvasCoord(  Channel[FZoomCh], FZoomBox.Left ) ;
           ZoomScreen.Right := XToCanvasCoord(  Channel[FZoomCh], FZoomBox.Right ) ;
           ZoomScreen.Top :=   YToCanvasCoord(  Channel[FZoomCh], FZoomBox.Top ) ;
           ZoomScreen.Bottom := YToCanvasCoord( Channel[FZoomCh], FZoomBox.Bottom ) ;
           Canvas.DrawFocusRect( ZoomScreen ) ;
           end ;

        { Notify a change in cursors }
        if Assigned(OnCursorChange) and
          (not FCursorChangeInProgress) and
          (not FZoomMode) then OnCursorChange(Self) ;

     finally
        { Get rid of array }
        Dispose(xy) ;
        Canvas.Pen.Color := SaveColor ;
        end ;

     end ;


procedure TScopeDisplay.PlotRecord(
          Canv : TCanvas ;                        { Canvas to be plotted on }
          var Channels : Array of TScopeChannel ; { Channel definition array }
          var xy : Array of TPoint                { Work array }
          ) ;
{ -----------------------------------
  Plot a signal record on to a canvas
  ----------------------------------- }
var
   ch,n,i,j : Integer ;
   y : Single ;
begin

     // Exit if no buffer
     if FBuf = Nil then Exit ;

     { Plot each active channel }
     for ch := 0 to FNumChannels-1 do if Channels[ch].InUse then begin
         Canv.Pen.Color := Channels[ch].Color ;
         n := 0 ;
         for i := Round(FXMin) to Min(Round(FXMax),FNumPoints-1) do begin

             j := (i*FNumChannels) + Channels[ch].ADCOffset ;
             if FBufIsWordData then y := Word(FBuf^[j])
                               else y := FBuf^[j] ;
             xy[n].y := YToCanvasCoord( Channels[ch], y ;
             xy[n].x := XToCanvasCoord( Channels[ch],i ) ;
             Inc(n) ;

             { If line exceeds 16000 output a partial line to canvas,
               since polyline function seems to be unable to handle more than 16000 points }
             if n > 16000 then begin
                Polyline( Canv.Handle, xy, n ) ;
                xy[0] := xy[n-1] ;
                n := 1 ;
                end ;

             end ;
         Polyline( Canv.Handle, xy, n ) ;

         // Display lines indicating area from which "From Record" zero level is derived
         if Channels[ch].ADCZeroAt >= 0 then begin
            Canv.Pen.Color := FCursorColor ;
            xy[0].x := XToCanvasCoord( Channels[ch],Channels[ch].ADCZeroAt ) ;
            xy[1].x := xy[0].x ;
            xy[0].y := YToCanvasCoord( Channels[ch], Channels[ch].ADCZero ) - 15 ;
            xy[1].y := xy[0].y + 30 ;
            Polyline( Canv.Handle, xy, 2 ) ;

            xy[0].x := XToCanvasCoord( Channels[ch],
                                       Channels[ch].ADCZeroAt + FChanZeroAvg -1) ;
            xy[1].x := xy[0].x ;
            xy[0].y := YToCanvasCoord( Channels[ch], Channels[ch].ADCZero ) - 15 ;
            xy[1].y := xy[0].y + 30 ;
            Polyline( Canv.Handle, xy, 2 ) ;

            end ;
         end ;
     end ;


procedure TScopeDisplay.ClearDisplay(
          Canv : TCanvas               // Canvas to be cleared
          ) ;
{ ---------------------------
  Clear signal display canvas
  ---------------------------}
var
   CTop,ch,i,NumInUse,AvailableHeight,ChannelHeight,ChannelSpacing,LastActiveChannel : Integer ;
   Lab : string ;
   x,dx,xPix,y,dy,yPix : Integer ;
   KeepColor : TColor ;
   XGrid : Single ;                // Vertical grid X coord
   dXGrid : Single ;               // Vertical grid X step
begin

     { Clear display area }
     Canv.fillrect(DisplayRect);

     { Determine number of channels in use and the height
       available for each channel }
     NumInUse := 0 ;
     for ch := 0 to FNumChannels-1 do if Channel[ch].InUse then Inc(NumInUse) ;
     if NumInUse < 1 then NumInUse := 1 ;

     Canv.Font.Size := 8 ;
     Canv.Font.Color := FTraceColor ;
     AvailableHeight := Height - Canv.TextHeight('X') - 4 ;
     ChannelSpacing :=  Canv.TextHeight('X') ;
     ChannelHeight := (AvailableHeight div NumInUse) - ChannelSpacing ;

     { Define display area for each channel in use }
     cTop := 4 ;
     FTopOfDisplayArea := cTop ;
     for ch := 0 to FNumChannels-1 do if Channel[ch].InUse then begin

         LastActiveChannel := Ch ;
         Channel[ch].Left := 4 ;
         Channel[ch].Right := Width - 4 ;
         if FXMax = FXMin then FXMax := FXMin + 1 ;
         Channel[ch].xMin := FXMin ;
         Channel[ch].xMax := FXMax ;
         Channel[ch].xScale := (Channel[ch].Right - Channel[ch].Left) /
                               (FXMax - FXMin ) ;

         Channel[ch].Top := cTop ;
         Channel[ch].Bottom := Channel[ch].Top + ChannelHeight ;
         if Channel[ch].yMax = Channel[ch].yMin then
            Channel[ch].yMax := Channel[ch].yMin + 1.0 ;
         Channel[ch].yScale := (Channel[ch].Bottom - Channel[ch].Top) /
                               (Channel[ch].yMax - Channel[ch].yMin ) ;
         cTop := cTop + ChannelHeight + ChannelSpacing ;
         FBottomOfDisplayArea := Channel[ch].Bottom ;
         end ;

     { Update horizontal cursor limits/scale factors to match channel settings }
     for i := 0 to High(HorCursors) do if HorCursors[i].InUse then begin
         HorCursors[i].Left := Channel[HorCursors[i].ChanNum].Left ;
         HorCursors[i].Right := Channel[HorCursors[i].ChanNum].Right ;
         HorCursors[i].Top := Channel[HorCursors[i].ChanNum].Top ;
         HorCursors[i].Bottom := Channel[HorCursors[i].ChanNum].Bottom ;
         HorCursors[i].xMin := Channel[HorCursors[i].ChanNum].xMin ;
         HorCursors[i].xMax := Channel[HorCursors[i].ChanNum].xMax ;
         HorCursors[i].xScale := Channel[HorCursors[i].ChanNum].xScale ;
         HorCursors[i].yMin := Channel[HorCursors[i].ChanNum].yMin ;
         HorCursors[i].yMax := Channel[HorCursors[i].ChanNum].yMax ;
         HorCursors[i].yScale := Channel[HorCursors[i].ChanNum].yScale ;
         end ;

     { Update vertical cursor limits/scale factors  to match channel settings}
     for i := 0 to High(VertCursors) do if VertCursors[i].InUse then begin
         if VertCursors[i].ChanNum >= 0 then begin
            { Vertical cursors linked to individual channels }
            VertCursors[i].Left := Channel[VertCursors[i].ChanNum].Left ;
            VertCursors[i].Right := Channel[VertCursors[i].ChanNum].Right ;
            VertCursors[i].Top := Channel[VertCursors[i].ChanNum].Top ;
            VertCursors[i].Bottom := Channel[VertCursors[i].ChanNum].Bottom ;
            VertCursors[i].xMin := Channel[LastActiveChannel].xMin ;
            VertCursors[i].xMax := Channel[LastActiveChannel].xMax ;
            VertCursors[i].xScale := Channel[VertCursors[i].ChanNum].xScale ;
            VertCursors[i].yMin := Channel[LastActiveChannel].yMin ;
            VertCursors[i].yMax := Channel[LastActiveChannel].yMax ;
            VertCursors[i].yScale := Channel[VertCursors[i].ChanNum].yScale ;
            end
         else begin
            { All channel cursors }
            VertCursors[i].Left := Channel[LastActiveChannel].Left ;
            VertCursors[i].Right := Channel[LastActiveChannel].Right ;
            VertCursors[i].Top := FTopOfDisplayArea ;
            VertCursors[i].Bottom := FBottomOfDisplayArea ;
            VertCursors[i].xMin := Channel[LastActiveChannel].xMin ;
            VertCursors[i].xMax := Channel[LastActiveChannel].xMax ;
            VertCursors[i].xScale := Channel[LastActiveChannel].xScale ;
            VertCursors[i].yScale := 1.0 ;
            end ;
         end ;

     if FDrawGrid then begin
         KeepColor := Canv.Pen.Color ;
         Canv.Pen.Color := FGridColor ;

         // Draw horizontal grid lines
         FNumHorizontalGridLines := MaxInt([1,FNumHorizontalGridLines]);
         for ch := 0 to FNumChannels-1 do if Channel[ch].InUse then begin
             dy := (FMaxADCValue - FMinADCValue) div FNumHorizontalGridLines ;
             y := FMinADCValue ;
             for i := 1 to FNumHorizontalGridLines+1 do begin
                 if (y >= Channel[ch].yMin) and
                    (y <= Channel[ch].yMax) then begin
                    yPix := YToCanvasCoord( Channel[ch], y ) ;
                    Canv.MoveTo( Channel[ch].Left, yPix )  ;
                    Canv.LineTo( Channel[ch].Right, yPix )  ;
                    end ;
                 y := y + dy ;
                 end ;
             end ;

         // Draw vertical grid lines
         FNumVerticalGridLines := MaxInt([1,FNumVerticalGridLines]) ;
         dXGrid := (FMaxPoints-1) / FNumVerticalGridLines ;
         XGrid := 0 ;
         for i := 1 to FNumVerticalGridLines-1 do begin
             xGrid := XGrid + dXGrid ;
             xPix := XToCanvasCoord( Channel[LastActiveChannel], xGrid ) ;
             if (xPix >= Channel[LastActiveChannel].Left) and
                (xPix <= Channel[LastActiveChannel].Right) then begin
                for ch := 0 to FNumChannels-1 do if Channel[ch].InUse then begin
                    Canv.MoveTo( xPix, Channel[ch].Bottom )  ;
                    Canv.LineTo( xPix, Channel[ch].Top )  ;
                    end ;
                 end ;
             end ;
         Canv.Pen.Color := KeepColor ;
         end ;

     // Display channel name(s)
     for ch := 0 to FNumChannels-1 do if Channel[ch].InUse then begin
         Canv.Pen.Color := Channel[ch].Color ;
         Lab := Channel[ch].ADCName ;
         if FDrawGrid then begin
            Lab := Lab + format(' %.4g%s/div',
                   [Channel[ch].ADCScale*(FMaxADCValue*2.0)/FNumHorizontalGridLines,
                    Channel[ch].ADCUnits]) ;
            end ;
         Canv.TextOut(Channel[ch].Left,Channel[ch].Top,Lab) ;
         end ;

     // Time labels
     Lab := format( FTFormat,[(FXMin+FXOffset)*FTScale] ) ;
     Canv.TextOut( 1, Height - Canv.TextHeight(Lab), Lab ) ;
     Lab := format( FTFormat, [(FXMax+FXOffset)*FTScale] ) ;
     Canv.TextOut( Width - Canv.TextWidth(Lab),
                     Height - Canv.TextHeight(Lab), Lab ) ;

     // Marker text
     for i := 0 to FMarkerText.Count-1 do begin
         x := Integer(FMarkerText.Objects[i]) ;
         xPix := XToCanvasCoord( Channel[LastActiveChannel], x ) ;
         yPix := Height - ((i Mod 2)+1)*Canv.TextHeight(Lab) ;
         Canv.TextOut( xPix, yPix, FMarkerText.Strings[i] );
         end ;

     end ;


procedure TScopeDisplay.DisplayNewPoints(
          NewPoints : Integer
          ) ;
{ -----------------------------------------
  Plot a new block of A/D samples of display
  -----------------------------------------}
var
   i,j,ch : Integer ;
   StartAt,EndAt : Integer ;
   y : single ;
begin
     { Start plot lines at last point in buffer }
     StartAt := MaxInt( [FNumPoints - 2,0] ) ;
     { End plot at newest point }
     FNumPoints := NewPoints ;
     EndAt := FNumPoints-1 ;
     for ch := 0 to FNumChannels-1 do
         if Channel[ch].InUse and (FBuf <> Nil) then begin
         Canvas.Pen.Color := Channel[ch].Color ;
         j := (StartAt*FNumChannels) + Channel[ch].ADCOffset ;
         if FBufIsWordData then y := Word(FBuf^[j])
                           else y := FBuf^[j] ;

         Canvas.MoveTo( XToCanvasCoord( Channel[ch], StartAt ),
                        YToCanvasCoord( Channel[ch], y) ) ;
         for i := StartAt to EndAt do begin
             j := (i*FNumChannels) + Channel[ch].ADCOffset ;
             if FBufIsWordData then y := Word(FBuf^[j])
                               else y := FBuf^[j] ;
             Canvas.LineTo( XToCanvasCoord( Channel[ch], i ),
                            YToCanvasCoord( Channel[ch], y ) ;
             end ;
         end ;
     end ;


procedure TScopeDisplay.AddMarker (
          AtPoint : Integer ;       // Marker display point
          Text : String             // Marker text
                    ) ;
// ------------------------------------
// Add marker text at bottom of display
// ------------------------------------
begin

    FMarkerText.AddObject( Text, TObject(AtPoint) ) ;
    Invalidate ;
    end ;


procedure TScopeDisplay.ClearMarkers ;
// ----------------------
// Clear marker text list
// ----------------------
begin
     FMarkerText.Clear ;
     Invalidate ;
     end ;


procedure TScopeDisplay.ClearHorizontalCursors ;
{ -----------------------------
  Remove all horizontal cursors
  -----------------------------}
var
   i : Integer ;
begin
     for i := 0 to High(HorCursors) do HorCursors[i].InUse := False ;
     end ;


function TScopeDisplay.AddHorizontalCursor(
         iChannel : Integer ;       { Signal channel associated with cursor }
         Color : TColor ;           { Colour of cursor }
         UseAsZeroLevel : Boolean   { If TRUE indicates this is a zero level cursor }
         ) : Integer ;
{ --------------------------------------------
  Add a new horizontal cursor to the display
  -------------------------------------------}
var
   iCursor : Integer ;
begin
     { Find an unused cursor }
     iCursor := 0 ;
     while HorCursors[iCursor].InUse
           and (iCursor < High(HorCursors)) do Inc(iCursor) ;

    { Attach the cursor to a channel }
    if iCursor <= High(HorCursors) then begin
       HorCursors[iCursor] := Channel[iChannel] ;
       HorCursors[iCursor].Position := 0 ;
       HorCursors[iCursor].InUse := True ;
       HorCursors[iCursor].Color := Color ;
       HorCursors[iCursor].ChanNum := iChannel ;
       HorCursors[iCursor].ZeroLevel := UseAsZeroLevel ;
       Result := iCursor ;
       end
    else begin
         { Return -1 if no cursors available }
         Result := -1 ;
         end ;
    end ;


procedure TScopeDisplay.DrawHorizontalCursor(
          Canv : TCanvas ;
          iCurs : Integer
          ) ;
{ -----------------------
  Draw horizontal cursor
 ------------------------}
var
   yPix : Integer ;
   OldPen : TPen ;
begin

     // Save pen settings
     OldPen := TPen.Create ;
     OldPen.Assign(Canv.Pen) ;

     // Settings for cursor
     Canv.Pen.Style := psDashDotDot ;
     Canv.Pen.Mode := pmMASK ;
     // Set line colour
     // (Note. If this cursor is being used as a zero level
     //  use a different colour when cursors is not at true zero)
     if HorCursors[iCurs].ZeroLevel and (HorCursors[iCurs].Position = 0) then
        Canv.Pen.Color := FCursorColor
     else Canv.Pen.Color := FNonZeroHorizontalCursorColor ;

     // Draw line
     yPix := YToCanvasCoord( HorCursors[iCurs],
                             HorCursors[iCurs].Position ) ;
     Canv.polyline( [Point(4,yPix),Point(Width-4,yPix)]);

     { If this cursor is being used as the zero baseline level for a signal
       channel, update the zero level for that channel }
     if (HorCursors[iCurs].ZeroLevel) then
        Channel[HorCursors[iCurs].ChanNum].ADCZero := HorCursors[iCurs].Position ;

     // Restore pen colour
     Canv.Pen.Assign(OldPen)  ;
     OldPen.Free ;

    end ;


procedure TScopeDisplay.ClearVerticalCursors ;
{ -----------------------------
  Remove all vertical cursors
  -----------------------------}
var
   i : Integer ;
begin
     for i := 0 to High(VertCursors) do VertCursors[i].InUse := False ;
     FLinkVerticalCursors [0] := -1 ;
     FLinkVerticalCursors [1] := -1 ;
     end ;


function TScopeDisplay.AddVerticalCursor(
         Chan : Integer ;                { Signal channel (-1=all channels) }
         Color : TColor                  { Cursor colour }
         ) : Integer ;
{ --------------------------------------------
  Add a new vertical cursor to the display
  -------------------------------------------}
var
   iCursor : Integer ;
begin
     { Find an unused cursor }
     iCursor := 0 ;
     while VertCursors[iCursor].InUse
           and (iCursor < High(VertCursors)) do Inc(iCursor) ;

    { Attach the cursor to a channel }
    if iCursor <= High(VertCursors) then begin
       VertCursors[iCursor] := Channel[0] ;
       VertCursors[iCursor].ChanNum := Chan ;
       VertCursors[iCursor].Position := FNumPoints div 2 ;
       VertCursors[iCursor].InUse := True ;
       VertCursors[iCursor].Color := Color ;
       Result := iCursor ;
       end
    else begin
         { Return -1 if no cursors available }
         Result := -1 ;
         end ;
    end ;


procedure TScopeDisplay.DrawVerticalCursor(
          Canv : TCanvas ;
          iCurs : Integer
          ) ;
{ -----------------------
  Draw vertical cursor
 ------------------------}
var
   xPix : Integer ;
   OldColor : TColor ;
begin

     // Set pen to cursor colour (saving old)
     OldColor := Canvas.Pen.Color ;
     Canv.Pen.Color := VertCursors[iCurs].Color ;

     // Plot cursor line
     xPix := XToCanvasCoord( VertCursors[iCurs], VertCursors[iCurs].Position ) ;
     Canv.polyline( [Point(xPix,VertCursors[iCurs].Top),
                     Point(xPix,VertCursors[iCurs].Bottom)] );

     // Restore pen colour
     Canv.Pen.Color := OldColor ;

     end ;


procedure TScopeDisplay.DrawVerticalCursorLink(
          Canv : TCanvas
          ) ;
{ ---------------------------------------------
  Draw horizontal link between vertical cursors
 ----------------------------------------------}
var
   iCurs0,iCurs1 : Integer ;
   xPix0,xPix1,yPix : Integer ;
   OldColor : TColor ;
begin

     iCurs0 := FLinkVerticalCursors[0] ;
     iCurs1 := FLinkVerticalCursors[1] ;

     if (not VertCursors[iCurs0].InUse) or (not VertCursors[iCurs1].InUse) then Exit ;

     // Set pen to cursor colour (saving old)
     OldColor := Canvas.Pen.Color ;
     Canv.Pen.Color := VertCursors[iCurs0].Color ;

     // Plot cursor line
     xPix0 := XToCanvasCoord( VertCursors[iCurs0], VertCursors[iCurs0].Position ) ;
     xPix1 := XToCanvasCoord( VertCursors[iCurs1], VertCursors[iCurs1].Position ) ;

     yPix := VertCursors[iCurs0].Bottom + (Canv.TextHeight('X') div 2) ;

     Canv.polyline( [Point(xPix0,yPix),
                     Point(xPix1,yPix)] );

     Canv.polyline( [Point(xPix0,yPix-3),
                     Point(xPix0,yPix+3)] );
     Canv.polyline( [Point(xPix1,yPix-3),
                     Point(xPix1,yPix+3)] );

     // Restore pen colour
     Canv.Pen.Color := OldColor ;

     end ;



{ ==========================================================
  PROPERTY READ / WRITE METHODS
  ==========================================================}


procedure TScopeDisplay.SetNumChannels(
          Value : Integer
          ) ;
{ ------------------------------------------
  Set the number of channels to be displayed
  ------------------------------------------ }
begin
     FNumChannels := IntLimitTo(Value,1,High(Channel)+1) ;
     end ;


procedure TScopeDisplay.SetNumPoints(
          Value : Integer
          ) ;
{ ------------------------------------
  Set the number of points per channel
  ------------------------------------ }
begin
     FNumPoints := IntLimitTo(Value,0,High(TSmallIntArray)) ;
     end ;


procedure TScopeDisplay.SetMaxPoints(
          Value : Integer
          ) ;
{ --------------------------------------------
  Set the maximum number of points per channel
  ------------------------------------------- }
begin
     FMaxPoints := IntLimitTo(Value,1,High(TSmallIntArray)) ;
     end ;


 procedure TScopeDisplay.SetXMin(
          Value : Integer
          ) ;
{ ----------------------
  Set the X axis minimum
  ---------------------- }
var
   ch : Integer ;
begin
     FXMin := MinInt([Value,FMaxPoints-1]) ;
     for ch := 0 to High(Channel) do Channel[ch].XMin := FXMin ;
     end ;


 procedure TScopeDisplay.SetXMax(
          Value : Integer
          ) ;
{ ----------------------
  Set the X axis maximum
  ---------------------- }
var
   ch : Integer ;
begin
     FXMax := MinInt([Value,FMaxPoints-1]) ;
     for ch := 0 to High(Channel) do Channel[ch].XMax := FXMax ;
     end ;


procedure TScopeDisplay.SetYMin(
          Ch : Integer ;
          Value : single
          ) ;
{ -------------------------
  Set the channel Y minimum
  ------------------------- }
begin
     if (ch < 0) or (ch > ScopeChannelLimit) then Exit ;
     Channel[Ch].YMin := MaxFlt([Value,FMinADCValue]) ;
     end ;


procedure TScopeDisplay.SetYMax(
          Ch : Integer ;
          Value : single
          ) ;
{ -------------------------
  Set the channel Y maximum
  ------------------------- }
begin
     if (ch < 0) or (ch > ScopeChannelLimit) then Exit ;
     Channel[Ch].YMax := MinFlt([Value,FMaxADCValue]) ;
     end ;



function TScopeDisplay.GetYMin(
         Ch : Integer
          ) : single ;
{ -------------------------
  Get the channel Y minimum
  ------------------------- }
begin
     Ch := IntLimitTo(Ch,0,ScopeChannelLimit) ;
     Result := Channel[Ch].YMin ;
     end ;


function TScopeDisplay.GetYMax(
         Ch : Integer
          ) : single ;
{ -------------------------
  Get the channel Y maximum
  ------------------------- }
begin
     Ch := IntLimitTo(Ch,0,ScopeChannelLimit) ;
     Result := Channel[Ch].YMax ;
     end ;


procedure TScopeDisplay.SetChanName(
          Ch : Integer ;
          Value : string
          ) ;
{ ------------------
  Set a channel name
  ------------------ }
begin
     if (ch < 0) or (ch > ScopeChannelLimit) then Exit ;
     Channel[Ch].ADCName := Value ;
     end ;


function TScopeDisplay.GetChanName(
          Ch : Integer
          ) : string ;
{ ------------------
  Get a channel name
  ------------------ }
begin
     Ch := IntLimitTo(Ch,0,ScopeChannelLimit) ;
     Result := Channel[Ch].ADCName ;
     end ;


procedure TScopeDisplay.SetChanUnits(
          Ch : Integer ;
          Value : string
          ) ;
{ ------------------
  Set a channel units
  ------------------ }

begin
     if (ch < 0) or (ch > ScopeChannelLimit) then Exit ;
     Channel[Ch].ADCUnits := Value ;
     end ;


function TScopeDisplay.GetChanUnits(
          Ch : Integer
          ) : string ;
{ ------------------
  Get a channel units
  ------------------ }
begin
     Ch := IntLimitTo(Ch,0,FNumChannels-1) ;
     Result := Channel[Ch].ADCUnits ;
     end ;


procedure TScopeDisplay.SetChanScale(
          Ch : Integer ;
          Value : single
          ) ;
{ ------------------------------------------------
  Set a channel A/D -> physical units scale factor
  ------------------------------------------------ }
begin
     if (ch < 0) or (ch > ScopeChannelLimit) then Exit ;
     Channel[Ch].ADCScale := Value ;
     end ;


function TScopeDisplay.GetChanScale(
          Ch : Integer
          ) : single ;
{ --------------------------------------------------
  Get a channel A/D -> physical units scaling factor
  -------------------------------------------------- }
begin
     Ch := IntLimitTo(Ch,0,ScopeChannelLimit) ;
     Result := Channel[Ch].ADCScale ;
     end ;


procedure TScopeDisplay.SetChanCalBar(
          Ch : Integer ;
          Value : single
          ) ;
{ -----------------------------
  Set a channel calibration bar
  ----------------------------- }
begin
     if (ch < 0) or (ch > ScopeChannelLimit) then Exit ;
     Channel[Ch].CalBar := Value ;
     end ;


function TScopeDisplay.GetChanCalBar(
          Ch : Integer
          ) : single ;
{ -----------------------------------
  Get a channel calibration bar value
  ----------------------------------- }
begin
     Ch := IntLimitTo(Ch,0,ScopeChannelLimit) ;
     Result := Channel[Ch].CalBar ;
     end ;


procedure TScopeDisplay.SetChanOffset(
          Ch : Integer ;
          Value : Integer
          ) ;
{ ---------------------------------------------
  Get data interleaving offset for this channel
  ---------------------------------------------}
begin
     if (ch < 0) or (ch > ScopeChannelLimit) then Exit ;
     Channel[Ch].ADCOffset := Value ;
     end ;


function TScopeDisplay.GetChanOffset(
          Ch : Integer
          ) : Integer ;
{ ---------------------------------------------
  Get data interleaving offset for this channel
  ---------------------------------------------}
begin
     Ch := IntLimitTo(Ch,0,ScopeChannelLimit) ;
     Result := Channel[Ch].ADCOffset ;
     end ;


procedure TScopeDisplay.SetChanZero(
          Ch : Integer ;
          Value : Integer
          ) ;
{ ------------------------
  Set a channel zero level
  ------------------------ }
begin
     if (ch < 0) or (ch > ScopeChannelLimit) then Exit ;
     Channel[Ch].ADCZero := Value ;
     end ;


function TScopeDisplay.GetChanZero(
          Ch : Integer
          ) : Integer ;
{ ----------------------------
  Get a channel A/D zero level
  ---------------------------- }
begin
     Ch := IntLimitTo(Ch,0,ScopeChannelLimit) ;
     Result := Channel[Ch].ADCZero ;
     end ;


procedure TScopeDisplay.SetChanZeroAt(
          Ch : Integer ;
          Value : Integer
          ) ;
{ ------------------------
  Set a channel zero level
  ------------------------ }
begin
     if (ch < 0) or (ch > ScopeChannelLimit) then Exit ;
     Channel[Ch].ADCZeroAt := Value ;
     end ;


function TScopeDisplay.GetChanZeroAt(
          Ch : Integer
          ) : Integer ;
{ ----------------------------
  Get a channel A/D zero level
  ---------------------------- }
begin
     Ch := IntLimitTo(Ch,0,ScopeChannelLimit) ;
     Result := Channel[Ch].ADCZeroAt ;
     end ;


procedure TScopeDisplay.SetChanZeroAvg(
          Value : Integer
          ) ;
{ ------------------------------------------------------------
  Set no. of points to average to get From Record channel zero
  ------------------------------------------------------------ }
begin
     FChanZeroAvg := IntLimitTo(Value,1,FNumPoints) ;
     end ;


procedure TScopeDisplay.SetChanVisible(
          Ch : Integer ;
          Value : boolean
          ) ;
{ ----------------------
  Set channel visibility
  ---------------------- }
begin
     if (ch < 0) or (ch > ScopeChannelLimit) then Exit ;
     Channel[Ch].InUse := Value ;
     end ;


function TScopeDisplay.GetChanVisible(
          Ch : Integer
          ) : boolean ;
{ ----------------------
  Get channel visibility
  ---------------------- }
begin
     Ch := IntLimitTo(Ch,0,ScopeChannelLimit) ;
     Result := Channel[Ch].InUse ;
     end ;


procedure TScopeDisplay.SetChanColor(
          Ch : Integer ;
          Value : TColor
          ) ;
{ ----------------------
  Set channel colour
  ---------------------- }
begin
     if (ch < 0) or (ch > ScopeChannelLimit) then Exit ;
     Channel[Ch].Color := Value ;
     end ;


function TScopeDisplay.GetChanColor(
          Ch : Integer
          ) : TColor ;
{ ----------------------
  Get channel colour
  ---------------------- }
begin
     Ch := IntLimitTo(Ch,0,ScopeChannelLimit) ;
     Result := Channel[Ch].Color ;
     end ;


function TScopeDisplay.GetHorCursor(
         iCursor : Integer
         ) : Integer ;
{ ---------------------------------
  Get position of horizontal cursor
  ---------------------------------}
begin
     iCursor := IntLimitTo(iCursor,0,High(HorCursors)) ;
     if HorCursors[iCursor].InUse then Result := HorCursors[iCursor].Position
                                   else Result := -1 ;
     end ;


procedure TScopeDisplay.SetHorCursor(
          iCursor : Integer ;           { Cursor # }
          Value : Integer               { New Cursor position }
          )  ;
{ ---------------------------------
  Set position of horizontal cursor
  ---------------------------------}
begin

     iCursor := IntLimitTo(iCursor,0,High(HorCursors)) ;
     HorCursors[iCursor].Position := Value ;

     { If this cursor is being used as the zero baseline level for a signal
       channel, update the zero level for that channel }
     if (HorCursors[iCursor].ZeroLevel) then
        Channel[HorCursors[iCursor].ChanNum].ADCZero := HorCursors[iCursor].Position ;

     Invalidate ;
     end ;


function TScopeDisplay.GetVertCursor(
         iCursor : Integer
         ) : Integer ;
{ -------------------------------
  Get position of vertical cursor
  -------------------------------}
begin
     iCursor := IntLimitTo(iCursor,0,High(VertCursors)) ;
     if VertCursors[iCursor].InUse then Result := VertCursors[iCursor].Position
                                   else Result := -1 ;
     end ;


procedure TScopeDisplay.SetVertCursor(
          iCursor : Integer ;           { Cursor # }
          Value : Integer               { New Cursor position }
          )  ;
{ -------------------------------
  Set position of Vertical cursor
  -------------------------------}
begin
     iCursor := IntLimitTo(iCursor,0,High(VertCursors)) ;
     VertCursors[iCursor].Position := Value ;
     Invalidate ;
     end ;


procedure TScopeDisplay.SetTFormat(
          Value : string
          ) ;
{ ---------------------
  Set time label format
  --------------------- }
begin
     FTFormat := Value ;
     end ;


function TScopeDisplay.GetTFormat : string ;
{ ----------------------------
  Get time label format string
  ---------------------------- }
begin
     Result := FTFormat ;
     end ;


function TScopeDisplay.GetXScreenCoord(
         Value : Integer               { Index into display data array (IN) }
         ) : Integer ;
{ --------------------------------------------------------------------------
  Get the screen coordinate within the paint box from the data array index
  -------------------------------------------------------------------------- }
begin
     Result := XToScreenCoord( 0, Value ) ;
     end ;


procedure TScopeDisplay.SetPrinterFontName(
          Value : string
          ) ;
{ -----------------------
  Set printer font name
  ----------------------- }
begin
     FPrinterFontName := Value ;
     end ;


function TScopeDisplay.GetPrinterFontName : string ;
{ -----------------------
  Get printer font name
  ----------------------- }
begin
     Result := FPrinterFontName ;
     end ;


procedure TScopeDisplay.SetPrinterLeftMargin(
          Value : Integer                    { Left margin (mm) }
          ) ;
{ -----------------------
  Set printer left margin
  ----------------------- }
begin
     { Printer pixel height (mm) }
     if Printer.Printers.Count > 0 then begin
        FPrinterLeftMargin := (Printer.PageWidth*Value)
                              div GetDeviceCaps( printer.handle, HORZSIZE ) ;
        end
     else FPrinterLeftMargin := 0 ;
     end ;


function TScopeDisplay.GetPrinterLeftMargin : integer ;
{ ----------------------------------------
  Get printer left margin (returned in mm)
  ---------------------------------------- }
begin
     if Printer.Printers.Count > 0 then begin
        Result := (FPrinterLeftMargin*GetDeviceCaps(Printer.Handle,HORZSIZE))
                     div Printer.PageWidth ;
        end
     else Result := 0 ;
     end ;


procedure TScopeDisplay.SetPrinterRightMargin(
          Value : Integer                    { Right margin (mm) }
          ) ;
{ -----------------------
  Set printer Right margin
  ----------------------- }
begin
     { Printe
     r pixel height (mm) }
     if Printer.Printers.Count > 0 then begin
        FPrinterRightMargin := (Printer.PageWidth*Value)
                                div GetDeviceCaps( printer.handle, HORZSIZE ) ;
        end
     else FPrinterRightMargin := 0 ;
     end ;


function TScopeDisplay.GetPrinterRightMargin : integer ;
{ ----------------------------------------
  Get printer Right margin (returned in mm)
  ---------------------------------------- }
begin
    if Printer.Printers.Count > 0 then begin
       Result := (FPrinterRightMargin*GetDeviceCaps(Printer.Handle,HORZSIZE))
                 div Printer.PageWidth ;
       end
    else Result := 0 ;
    end ;


procedure TScopeDisplay.SetPrinterTopMargin(
          Value : Integer                    { Top margin (mm) }
          ) ;
{ -----------------------
  Set printer Top margin
  ----------------------- }
begin
     { Printer pixel height (mm) }
     if Printer.Printers.Count > 0 then begin
        FPrinterTopMargin := (Printer.PageHeight*Value)
                           div GetDeviceCaps( printer.handle, VERTSIZE ) ;
        end
     else FPrinterTopMargin := 0 ;
     end ;


function TScopeDisplay.GetPrinterTopMargin : integer ;
{ ----------------------------------------
  Get printer Top margin (returned in mm)
  ---------------------------------------- }
begin
     if Printer.Printers.Count > 0 then begin
        Result := (FPrinterTopMargin*GetDeviceCaps(Printer.Handle,VERTSIZE))
                  div Printer.PageHeight ;
        end
     else Result := 0 ;
     end ;


procedure TScopeDisplay.SetPrinterBottomMargin(
          Value : Integer                    { Bottom margin (mm) }
          ) ;
{ -----------------------
  Set printer Bottom margin
  ----------------------- }
begin
     { Printer pixel height (mm) }
     if Printer.Printers.Count > 0 then begin
        FPrinterBottomMargin := (Printer.PageHeight*Value)
                                div GetDeviceCaps( printer.handle, VERTSIZE ) ;
        end
     else FPrinterBottomMargin := 0 ;
     end ;


function TScopeDisplay.GetPrinterBottomMargin : integer ;
{ ----------------------------------------
  Get printer Bottom margin (returned in mm)
  ---------------------------------------- }
begin
     if Printer.Printers.Count > 0 then begin
        Result := (FPrinterBottomMargin*GetDeviceCaps(Printer.Handle,VERTSIZE))
                  div Printer.PageHeight ;
        end
     else Result := 0 ;
     end ;


procedure TScopeDisplay.SetStorageMode(
          Value : Boolean                    { True=storage mode on }
          ) ;
{ ------------------------------------------------------
  Set display storage mode
  (in store mode records are superimposed on the screen
  ------------------------------------------------------ }
var
   i : Integer ;
begin
     FStorageMode := Value ;
     { Clear out list of stored records }
     for i := 0 to High(FStorageList) do FStorageList[i] := NoRecord ;
     Invalidate ;
     end ;


procedure TScopeDisplay.SetGrid(
          Value : Boolean                    { True=storage mode on }
          ) ;
{ ---------------------------
  Enable/disable display grid
  --------------------------- }
begin
     FDrawGrid := Value ;
     Invalidate ;
     end ;


function TScopeDisplay.GetNumVerticalCursors : Integer ;
// ---------------------------------------------------
// Get number of vertical cursors defined in displayed
// ---------------------------------------------------
var
    i,NumCursors : Integer ;
begin
    NumCursors := 0 ;
    for i := 0 to High(VertCursors) do if
        VertCursors[i].InUse then Inc(NumCursors) ;
    Result := NumCursors ;
    end ;


function TScopeDisplay.GetNumHorizontalCursors : Integer ;
// ---------------------------------------------------
// Get number of horizontal cursors defined in displayed
// ---------------------------------------------------
var
    i,NumCursors : Integer ;
begin
    NumCursors := 0 ;
    for i := 0 to High(HorCursors) do if
        HorCursors[i].InUse then Inc(NumCursors) ;
    Result := NumCursors ;
    end ;



{ =======================================================
  INTERNAL EVENT HANDLING METHODS
  ======================================================= }

procedure TScopeDisplay.MouseDown(
          Button: TMouseButton;
          Shift: TShiftState;
          X, Y: Integer
          ) ;
{ --------------------
  Mouse button is down
  -------------------- }
begin
     Inherited MouseDown( Button, Shift, X, Y ) ;

     if FHorCursorSelected > -1  then FHorCursorActive := True ;

     if (not FHorCursorActive)
        and (FVertCursorSelected > -1) then FVertCursorActive := True ;

     FMoveZoomBox := True ;

     end ;


procedure TScopeDisplay.MouseUp(
          Button: TMouseButton;
          Shift: TShiftState;
          X, Y: Integer
          ) ;
{ --------------------
  Mouse button is up
  -------------------- }
begin
     Inherited MouseUp( Button, Shift, X, Y ) ;

     FHorCursorActive := false ;
     FVertCursorActive := false ;
     FMoveZoomBox := False ;

     { Notify a change in cursors }
    { if Assigned(OnCursorChange) and
        (not FCursorChangeInProgress) and
        (not FZoomMode) then OnCursorChange(Self) ;}


     end ;



procedure TScopeDisplay.MouseMove(
          Shift: TShiftState;
          X, Y: Integer) ;
{ --------------------------------------------------------
  Select/deselect cursors as mouse is moved over display
  -------------------------------------------------------}
var
   ch,i : Integer ;
   HorizontalChanged : Boolean ; // Horizontal cursor changed flag
   VerticalChanged : Boolean ;   // Vertical cursor changed flag
begin
     Inherited MouseMove( Shift, X, Y ) ;

     { Find and store channel mouse is over }
     for ch := 0 to FNumChannels-1 do if Channel[ch].InUse then begin
         if (Channel[ch].Top <= Y) and (Y <= Channel[ch].Bottom) then
            FMouseOverChannel := ch ;
         end ;

     if FZoomMode then begin
        ProcessZoomBox( X, Y ) ;
        end
     else if FCursorsEnabled then begin

        { Find/move any active horizontal cursor }
        HorizontalChanged := ProcessHorizontalCursors( Y ) ;
        { Find/move any active vertical cursor }
        if not FHorCursorActive then VerticalChanged := ProcessVerticalCursors( X, Y ) ;

        if (HorizontalChanged or VerticalChanged) then begin

           // Copy image from internal bitmap
           ForeBitmap.Canvas.CopyRect( DisplayRect,
                                       BackBitmap.Canvas,
                                       DisplayRect ) ;

           // Re-draw cursors
           for i := 0 to High(HorCursors) do if HorCursors[i].InUse
               and Channel[HorCursors[i].ChanNum].InUse then DrawHorizontalCursor(ForeBitmap.Canvas,i) ;
           for i := 0 to High(VertCursors) do if VertCursors[i].InUse then DrawVerticalCursor(ForeBitmap.Canvas,i) ;
           // Draw link between selected pair of vertical cursors
           DrawVerticalCursorLink(ForeBitmap.Canvas) ;

          Canvas.CopyRect( DisplayRect,
                           ForeBitmap.Canvas,
                           DisplayRect) ;

           end ;

        { Set type of cursor icon }
        if FHorCursorSelected > -1 then Cursor := crSizeNS
        else if FVertCursorSelected > -1 then Cursor := crSizeWE
                                         else Cursor := crDefault ;
        end ;
     end ;


procedure TScopeDisplay.DblClick ;
{ -------------------------------------------------
  Switch between zoom control / normal display mode
  -------------------------------------------------}
begin
     if not FZoomMode then SwitchToZoomMode( FMouseOverChannel )
                      else SwitchToNormalMode ;
     Invalidate ;
     end ;


procedure TScopeDisplay.SwitchToZoomMode(
          Chan : Integer
          ) ;
{ ----------------------------------------------------
  Switch display to zoom in/out mode on channel "Chan"
  ---------------------------------------------------- }
var
   ch : Integer ;
begin
     { Get channel to zoom }
     FZoomCh := Chan ;
     FZoomMode := True ;
     for ch := 0 to FNumChannels-1 do begin
         KeepChannel[ch] := Channel[ch] ;
         if ch = FZoomCh then begin
            { Calculate zoom box position }
            FZoomBox.Bottom := Round(Channel[ch].yMin) ;
            FZoomBox.Top := Round(Channel[ch].yMax) ;
            FZoomBox.Left := FXMin ;
            FZoomBox.Right := FXMax ;
            Channel[ch].InUse := True ;
            Channel[ch].yMin := FMinADCValue ;
            Channel[ch].yMax := FMaxADCValue ;
            FXMin := 0 ;
            FXMax := FMaxPoints - 1 ;
            Channel[ch].xMin := FXMin ;
            Channel[ch].xMax := FXMax ;
            end
         else Channel[ch].InUse := False ;
         end ;

     { ** 23/7/00 Set to MNone to fix bug where zoom box jumped from
       set position to mouse cursor position }
     MousePos := MNone ;

     end ;


procedure TScopeDisplay.SwitchToNormalMode ;
{ -------------------------------------
  Switch to normal display mode
  ----------------------------- }
var
   ch : Integer ;
begin
    FZoomMode := False ;
    for ch := 0 to FNumChannels-1 do begin
        if ch = FZoomCh then begin
           Channel[ch].InUse := True ;
           Channel[ch].yMin := FZoomBox.Bottom ;
           Channel[ch].yMax := FZoomBox.Top ;
           FXMin := FZoomBox.Left ;
           FXMax := FZoomBox.Right ;
           Channel[ch].xMin := FXMin ;
           Channel[ch].xMax := FXMax ;
           end
        else Channel[ch] := KeepChannel[ch] ;
        end ;

     { Notify a change in cursors }
     if Assigned(OnCursorChange) and
        (not FCursorChangeInProgress) and
        (not FZoomMode) then OnCursorChange(Self) ;

     end ;


procedure TScopeDisplay.ZoomIn(
          Chan : Integer
          ) ;
{ -----------------------------------------------------------
  Switch to zoom in/out mode on selected chan (External call)
  ----------------------------------------------------------- }
begin
     SwitchToZoomMode( Chan ) ;
     Invalidate ;
     end ;


procedure TScopeDisplay.XZoom( PercentChange : Single ) ;
// ----------------------------------------------------------
// Change horizontal display magnification for selected channel
// ----------------------------------------------------------
const
    XLoLimit = 16 ;
var
    YShift : Integer ;
begin

     YShift := Round(Abs(FXMax - FXMin)*Min(Max(PercentChange*0.01,-1.0),10.0)) ;
     FXMin := Min( Max( FXMin + YShift, 0 ), FMaxPoints - 1 ) ;
     FXMax := Min( Max( FXMax + YShift, 0 ), FMaxPoints - 1 ) ;
     if Abs(FXMax - FXMin) < XLoLimit then FXMax := FXMin + XLoLimit ;

     end ;


procedure TScopeDisplay.YZoom(
          Chan : Integer ;       // Selected channel (-1 = all channels)
          PercentChange : Single // % change (-100..10000%) (Negative values zoom in)
          ) ;
// ----------------------------------------------------------
// Change vertical display magnification for selected channel
// ----------------------------------------------------------
const
    YLoLimit = 16 ;
var
    ch : Integer ;
    YShift,YMid : Integer ;
begin

     if Chan >= FNumChannels then Exit ;

     YShift := Round( Abs(Channel[Chan].YMax - Channel[Chan].YMin)
                      *Min(Max(PercentChange*0.01,-1.0),10.0)) div 2 ;

     if Chan < 0 then begin
        // Zoom all channels
        for ch := 0 to FNumChannels-1 do begin
            Channel[ch].YMax := Max(Min(Channel[ch].YMax + YShift, FMaxADCValue),FMinADCValue) ;
            Channel[ch].YMin := Max(Min(Channel[ch].YMin - YShift, FMaxADCValue),FMinADCValue) ;
            if Abs(Channel[ch].YMax - Channel[ch].YMin) < YLoLimit then begin
               YMid := Round((Channel[ch].YMax + Channel[ch].YMin)*0.5) ;
               Channel[ch].YMax := YMid + (YLoLimit div 2) ;
               Channel[ch].YMin := YMid - (YLoLimit div 2) ;
               end ;
            end ;
        end
     else begin
        // Zoom selected channel
        Channel[Chan].YMax := Max(Min(Channel[Chan].YMax + YShift, FMaxADCValue),FMinADCValue) ;
        Channel[Chan].YMin := Max(Min(Channel[Chan].YMin - YShift, FMaxADCValue),FMinADCValue) ;
        if Abs(Channel[Chan].YMax - Channel[Chan].YMin) < YLoLimit then begin
           YMid := Round((Channel[Chan].YMax + Channel[Chan].YMin)*0.5) ;
           Channel[Chan].YMax := YMid + (YLoLimit div 2) ;
           Channel[Chan].YMin := YMid - (YLoLimit div 2) ;
           end ;
        end ;

     Self.Invalidate ;

     end ;


procedure TScopeDisplay.ZoomOut ;
{ ---------------------------------
  Zoom out to minimum magnification
  ---------------------------------}
var
   ch : Integer ;
begin
     FZoomMode := False ;
     for ch := 0 to FNumChannels-1 do begin
         Channel[ch].yMin := FMinADCValue ;
         Channel[ch].yMax := FMaxADCValue ;
         FXMin := 0 ;
         FXMax := FMaxPoints - 1;
         Channel[ch].xMin := FXMin ;
         Channel[ch].xMax := FXMax ;
         end ;
     Invalidate ;
     end ;


function TScopeDisplay.ProcessHorizontalCursors(
         Y : Integer
         ) : Boolean ;                       // Returns TRUE if a cursor changed
{ ----------------------------------
  Find/move active horizontal cursor
  ----------------------------------}
const
     Margin = 4 ;
var
   YPosition,i : Integer ;
begin

     if FHorCursorActive and (FHorCursorSelected > -1) then begin
        { ** Move the currently activated cursor to a new position ** }
        { Keep within display limits }
        Y := IntLimitTo( Y,
                         HorCursors[FHorCursorSelected].Top,
                         HorCursors[FHorCursorSelected].Bottom ) ;
        HorCursors[FHorCursorSelected].Position := CanvasToYCoord(
                                                   HorCursors[FHorCursorSelected],Y) ;

        { Notify a change in cursors }
        if Assigned(OnCursorChange) and
           (not FCursorChangeInProgress) and
           (not FZoomMode) then OnCursorChange(Self) ;

        Result := True ;
        end
     else begin
        { Find the active horizontal cursor (if any) }
        FHorCursorSelected := -1 ;
        for i := 0 to High(HorCursors) do if HorCursors[i].InUse then begin
            YPosition := YToCanvasCoord(HorCursors[i],HorCursors[i].Position) ;
            if Abs(Y - YPosition) <= Margin then FHorCursorSelected := i ;
            end ;

        Result := False ;
        end ;

     end ;


function TScopeDisplay.ProcessVerticalCursors(
         X : Integer ;                        // X mouse coord (IN)
         Y : Integer                          // Y mouse coord (IN)
         ) : Boolean ;                       // Returns TRUE is cursor changed
{ --------------------------------
  Find/move active vertical cursor
  --------------------------------}
const
     Margin = 4 ;
var
   XPosition,i : Integer ;
begin

     if FVertCursorActive and (FVertCursorSelected > -1) then begin
        { ** Move the currently activated cursor to a new position ** }
        { Keep within channel display area }
        X := IntLimitTo( X,
                         VertCursors[FVertCursorSelected].Left,
                         VertCursors[FVertCursorSelected].Right ) ;
        { Calculate new X value }
        VertCursors[FVertCursorSelected].Position := CanvasToXCoord(
                               VertCursors[FVertCursorSelected], X ) ;

        { Notify a change in cursors }
        if Assigned(OnCursorChange) and
           (not FCursorChangeInProgress) and
           (not FZoomMode) then OnCursorChange(Self) ;

        Result := True ;
        end
     else begin
        { ** Find the active vertical cursor (if any) ** }
        FVertCursorSelected := -1 ;
        for i := 0 to High(VertCursors) do if VertCursors[i].InUse and
            (VertCursors[i].Bottom >= Y) and (Y >= VertCursors[i].Top) then begin
            XPosition := XToCanvasCoord( VertCursors[i], VertCursors[i].Position ) ;
            if Abs(X - XPosition) <= Margin then begin
               FVertCursorSelected := i ;
               FLastVertCursorSelected := FVertCursorSelected ;
               end ;
            end ;
        Result := False ;
        end ;
     end ;


procedure TScopeDisplay.MoveActiveVerticalCursor( Step : Integer ) ;
{ ----------------------------------------------------------------
  Move the currently selected vertical cursor by "Step" increments
  ---------------------------------------------------------------- }
begin

    VertCursors[FLastVertCursorSelected].Position := IntLimitTo(
        VertCursors[FLastVertCursorSelected].Position + Step,FXMin,FXMax );

     { Notify a change in cursors }
     if Assigned(OnCursorChange) and
        (not FCursorChangeInProgress) and
        (not FZoomMode) then OnCursorChange(Self) ;

    Invalidate ;

    end ;


procedure TScopeDisplay.LinkVerticalCursors(
          C0 : Integer ;                     // First cursor to link
          C1 : Integer                       // Second cursor to link
          ) ;
// -----------------------------------------------------
// Link a pair of cursors with line at bottom of display
// -----------------------------------------------------
begin

    FLinkVerticalCursors[0] := -1 ;
    FLinkVerticalCursors[1] := -1 ;

    if (C0 < 0) or (C0 > High(VertCursors)) then Exit ;
    if (C1 < 0) or (C1 > High(VertCursors)) then Exit ;
    if (not VertCursors[C0].InUse) or (not VertCursors[C1].InUse) then Exit ;

    FLinkVerticalCursors[0] := C0 ;
    FLinkVerticalCursors[1] := C1 ;

    end ;


procedure TScopeDisplay.ProcessZoomBox(
          X : Integer ;      { Mouse X Coord (IN) }
          Y : Integer ) ;    { Mouse Y Coord (IN) }
{ ------------------------------------------------------------
  Update size/location of display magnification adjustment box
  ------------------------------------------------------------}
const
     Margin = 4 ;
     ZoomMin = 0 ;
var
   ZoomScreen : TRect ;
   BoxWidth,BoxHeight : Integer ;
begin

     { Calculate zoom rectangle in screen coords. }
     ZoomScreen.Left :=   XToCanvasCoord( Channel[FZoomCh], FZoomBox.Left ) ;
     ZoomScreen.Right :=  XToCanvasCoord( Channel[FZoomCh], FZoomBox.Right ) ;
     ZoomScreen.Top :=    YToCanvasCoord( Channel[FZoomCh], FZoomBox.Top ) ;
     ZoomScreen.Bottom := YToCanvasCoord( Channel[FZoomCh], FZoomBox.Bottom ) ;

     if FMoveZoomBox then begin

        // Copy image from internal bitmap
        Canvas.CopyRect( DisplayRect,
                         BackBitmap.Canvas,
                         DisplayRect ) ;

        { Move the part of the zoom box which is under the mouse }

        case MousePos of
          MTop : Begin
              { Move top margin of zoom box }
              if (ZoomScreen.Bottom-Y) > ZoomMin then ZoomScreen.Top := Y ;
              end ;
          MBottom : Begin
              { Move bottom margin }
              if Abs(Y - ZoomScreen.Top) > ZoomMin then ZoomScreen.Bottom := Y ;
              end ;
          MLeft : Begin
              { Move left margin }
              if (ZoomScreen.Right-X) > ZoomMin then ZoomScreen.Left := X ;
              end ;
          MRight : Begin
              { Move right margin }
              if (X - ZoomScreen.Left) > ZoomMin then ZoomScreen.Right := X ;
              end ;
          MDrag : begin
              { Move whole box }
              BoxWidth := ZoomScreen.Right - ZoomScreen.Left ;
              BoxHeight := ZoomScreen.Bottom - ZoomScreen.Top ;
              ZoomScreen.Left := IntLimitTo( ZoomScreen.Left + (X - FXOld),
                                             Channel[FZoomCh].Left,
                                             Channel[FZoomCh].Right - BoxWidth ) ;
              ZoomScreen.Right := ZoomScreen.Left + BoxWidth ;
              ZoomScreen.Top := IntLimitTo( ZoomScreen.Top + (Y - FYOld),
                                            Channel[FZoomCh].Top,
                                            Channel[FZoomCh].Bottom - BoxHeight ) ;
              ZoomScreen.Bottom := ZoomScreen.Top + BoxHeight ;
              FXOld := X ;
              FYOld := Y ;
              end ;
          else
          end ;

          { Keep within bounds }

        ZoomScreen.Left :=      IntLimitTo(ZoomScreen.Left,
                                           Channel[FZoomCh].Left,
                                           Channel[FZoomCh].Right ) ;
        ZoomScreen.Right :=     IntLimitTo(ZoomScreen.Right,
                                           Channel[FZoomCh].Left,
                                           Channel[FZoomCh].Right ) ;
        ZoomScreen.Top :=       IntLimitTo(ZoomScreen.Top,
                                           Channel[FZoomCh].Top,
                                           Channel[FZoomCh].Bottom ) ;
        ZoomScreen.Bottom :=    IntLimitTo(ZoomScreen.Bottom,
                                           Channel[FZoomCh].Top,
                                           Channel[FZoomCh].Bottom ) ;

        Canvas.DrawFocusRect( ZoomScreen ) ;

             { Calculate zoom rectangle in screen coords. }
        FZoomBox.Left :=  CanvasToXCoord( Channel[FZoomCh], ZoomScreen.Left ) ;
        FZoomBox.Right := CanvasToXCoord( Channel[FZoomCh], ZoomScreen.Right ) ;
        FZoomBox.Top :=   CanvasToYCoord( Channel[FZoomCh], ZoomScreen.Top ) ;
        FZoomBox.Bottom := CanvasToYCoord( Channel[FZoomCh], ZoomScreen.Bottom ) ;

        end
     else begin

        { *** Determine if the mouse is over part of the zoom box *** }

        if (Abs(X - ZoomScreen.Left) < Margin ) and
           (not FZoomDisableHorizontal) and
           (Y <= ZoomScreen.Bottom) and (Y >= ZoomScreen.Top ) then begin
           { Left margin }
           Cursor := crSizeWE ;
           MousePos := MLeft ;
           end
        else if (Abs(X - ZoomScreen.Right) < Margin) and
                (not FZoomDisableHorizontal) and
                (Y <= ZoomScreen.Bottom) and (Y >= ZoomScreen.Top ) then begin
                { Right margin }
                Cursor := crSizeWE ;
                MousePos := MRight ;
                end
        else if (Abs(Y - ZoomScreen.Top) < Margin) and
                (not FZoomDisableVertical) and
                (X <= ZoomScreen.Right) and (X >= ZoomScreen.Left ) then begin
                { Top margin }
                Cursor := crSizeNS ;
                MousePos := MTop ;
                end
        else if (Abs(Y - ZoomScreen.Bottom) < Margin ) and
                (not FZoomDisableVertical) and
                (X <= ZoomScreen.Right) and (X >= ZoomScreen.Left ) then begin
                { Bottom margin }
                Cursor := crSizeNS ;
                MousePos := MBottom ;
                end
        else if (ZoomScreen.Bottom > Y) and (Y > ZoomScreen.Top) and
                (ZoomScreen.Right > X) and (X > ZoomScreen.Left) then begin
                { Cursor within zoom box }
                Cursor := CrSize ;
                MousePos := MDrag ;
                FXOld := X ;
                FYOld := Y ;
                end
        else
            Cursor := crDefault ;

        end ;
     end ;


function TScopeDisplay.IntLimitTo(
         Value : Integer ;          { Value to be checked }
         Lo : Integer ;             { Lower limit }
         Hi : Integer               { Upper limit }
         ) : Integer ;              { Return limited value }
{ --------------------------------
  Limit Value to the range Lo - Hi
  --------------------------------}
begin
     if Value < Lo then Value := Lo ;
     if Value > Hi then Value := Hi ;
     Result := Value ;
     end ;


function TScopeDisplay.XToCanvasCoord(
         var Chan : TScopeChannel ;
         Value : single
         ) : Integer  ;
var
   XScale : single ;
begin
     XScale := (Chan.Right - Chan.Left) / ( FXMax - FXMin ) ;
     Result := Round( (Value - FXMin)*XScale + Chan.Left ) ;
     end ;


function TScopeDisplay.XToScreenCoord(
         Chan : Integer ;
         Value : single
         ) : Integer  ;
{ ------------------------------------------------------------------------
  Public function which allows pixel coord to be obtained for X axis coord
  ------------------------------------------------------------------------}
var
   XScale : single ;
begin
     XScale := (Channel[Chan].Right - Channel[Chan].Left) / ( FXMax - FXMin ) ;
     Result := Round( (Value - FXMin)*XScale + Channel[Chan].Left ) ;
     end ;


function TScopeDisplay.ScreenCoordToX(
         Chan : Integer ;
         Value : Integer
         ) : Single  ;
{ ------------------------------------------------------------------------
  Public function which allows pixel coord to be obtained for X axis coord
  ------------------------------------------------------------------------}
var
   XScale : single ;
begin
     XScale := (Channel[Chan].Right - Channel[Chan].Left) / ( FXMax - FXMin ) ;
     Result := (Value - Channel[Chan].Left)/XSCale + FXMin ;
     end ;



function TScopeDisplay.CanvasToXCoord(
         var Chan : TScopeChannel ;
         xPix : Integer
         ) : Integer  ;
var
   XScale : single ;
begin
     XScale := (Chan.Right - Chan.Left) / ( FXMax - FXMin ) ;
     Result := Round((xPix - Chan.Left)/XScale + FXMin) ;
     end ;


function TScopeDisplay.YToCanvasCoord(
         var Chan : TScopeChannel ;
         Value : single
         ) : Integer  ;
begin

     Chan.yScale := (Chan.Bottom - Chan.Top)/(Chan.yMax - Chan.yMin ) ;
     Result := Round( Chan.Bottom - (Value - Chan.yMin)*Chan.yScale ) ;
     if Result > Chan.Bottom then Result := Chan.Bottom ;
     if Result < Chan.Top then Result := Chan.Top ;
     end ;


function TScopeDisplay.YToScreenCoord(
         Chan : Integer ;
         Value : single
         ) : Integer  ;
{ ------------------------------------------------------------------------
  Public function which allows pixel coord to be obtained from Y axis coord
  ------------------------------------------------------------------------}

begin
     Channel[Chan].yScale := (Channel[Chan].Bottom - Channel[Chan].Top)/
                             (Channel[Chan].yMax - Channel[Chan].yMin ) ;
     Result := Round( Channel[Chan].Bottom
               - (Value - Channel[Chan].yMin)*Channel[Chan].yScale ) ;

     end ;


function TScopeDisplay.ScreenCoordToY(
         Chan : Integer ;
         Value : Integer
         ) : single  ;
{ ------------------------------------------------------------------------
  Public function which allows pixel coord to be obtained from Y axis coord
  ------------------------------------------------------------------------}

begin
     Channel[Chan].yScale := (Channel[Chan].Bottom - Channel[Chan].Top)/
                             (Channel[Chan].yMax - Channel[Chan].yMin ) ;
     Result := (Channel[Chan].Bottom - Value)/Channel[Chan].yScale
               + Channel[Chan].yMin ;
     end ;



function TScopeDisplay.CanvasToYCoord(
         var Chan : TScopeChannel ;
         yPix : Integer
         ) : Integer  ;
begin
     Chan.yScale := (Chan.Bottom - Chan.Top)/(Chan.yMax - Chan.yMin ) ;
     Result := Round( (Chan.Bottom - yPix)/Chan.yScale + Chan.yMin ) ;
     end ;

procedure TScopeDisplay.SetDataBuf(
          var Buf : Array {of SmallInt} ) ;
// ----------------------------------------------------
// Supply address of data buffer containing digitised signals
// to be displayed
// ----------------------------------------------------
begin
     FBuf := @Buf ;
     //Invalidate ; Removed 5/12/01
     end ;


procedure TScopeDisplay.CopyDataToClipBoard ;
{ ------------------------------------------------
  Copy the data points on display to the clipboard
  ------------------------------------------------}
var
   i,j,ch,BufSize : Integer ;
   t : single ;
   CopyBuf : PChar ;
   y : single ;

begin

     Screen.Cursor := crHourGlass ;

     // Open clipboard preventing others acceessing it
     Clipboard.Open ;

     try

       // Determine size of and allocate string buffer
       BufSize := 1 ;
       for ch := 0 to FNumChannels-1 do if Channel[ch].InUse then BufSize := BufSize + 1 ;
       BufSize := BufSize*10*(FXMax - FXMin + 1) ;
       CopyBuf := StrAlloc( BufSize ) ;

       // Write table of data to buffer

       StrCopy(CopyBuf,PChar('')) ;
       t := 0.0 ;
       for i := FXMin to FXMax do begin
           // Time
           StrCat(CopyBuf,PChar(format( '%.4g', [t] ))) ;
           // Channel sample values
           for ch := 0 to FNumChannels-1 do if Channel[ch].InUse then begin
               j := i*FNumChannels + Channel[ch].ADCOffset ;
               if FBufIsWordData then y := Word(FBuf^[j])
                                 else y := FBuf^[j] ;
               StrCat(CopyBuf,
                      PChar(Format(#9'%.4g',
                      [(y - Channel[ch].ADCZero)*Channel[ch].ADCScale] ))) ;
               end ;
           // CR+LF at end of line
           StrCat(CopyBuf, PChar(#13#10)) ;
           t := t + FTScale ;
           end ;

       // Copy text accumulated in copy buffer to clipboard
       ClipBoard.SetTextBuf( CopyBuf ) ;

     finally
       // Free buffer
       StrDispose(CopyBuf) ;
       // Release clipboard
       Clipboard.Close ;
       Screen.Cursor := crDefault ;
       end ;

     end ;


procedure TScopeDisplay.Print ;
{ ---------------------------------
  Copy signal on display to printer
  ---------------------------------}
var
   i,j,n,ch,LastCh,YPix,xPix,Rec,NumBytesPerRecord : Integer ;
   x : single ;
   LeftMarginShift, TopMarginShift, XPos,YPos : Integer ;
   OK : Boolean ;
   ChannelHeight,cTop,NumInUse,AvailableHeight : Integer ;
   PrChan : Array[0..ScopeChannelLimit] of TScopeChannel ;
   xy : ^TPointArray ;
   Bar : TRect ;
   Lab : string ;
   DefaultPen : TPen ;
   TopSpaceNeeded : Integer ;
begin
     { Create plotting points array }
     New(xy) ;
     DefaultPen := TPen.Create ;
     Printer.BeginDoc ;
     Cursor := crHourglass ;

     try

        Printer.Canvas.Pen.Color := clBlack ;
        Printer.Canvas.font.Name := FPrinterFontName ;
        Printer.Canvas.font.size := FPrinterFontSize ;
        Printer.Canvas.Pen.Width := FPrinterPenWidth ;
        Printer.Canvas.Pen.Style := psSolid ;
        Printer.Canvas.Pen.Color := clBlack ;
        DefaultPen.Assign(Printer.Canvas.Pen) ;

        { Determine number of channels in use and the height
          available for each channel }
        NumInUse := 0 ;
        TopSpaceNeeded := (3 + FTitle.Count)*Printer.Canvas.TextHeight('X') ;

        AvailableHeight := Printer.PageHeight
                           - FPrinterBottomMargin
                           - FPrinterTopMargin
                           - TopSpaceNeeded ;

        for ch := 0 to FNumChannels-1 do if Channel[ch].InUse then Inc(NumInUse) ;
        if NumInUse < 1 then NumInUse := 1 ;
        ChannelHeight := AvailableHeight div NumInUse ;

        { Make space at left margin for channel names/cal. bars }
        LeftMarginShift := 0 ;
        for ch := 0 to FNumChannels-1 do if Channel[ch].InUse then begin
            Lab := Channel[ch].ADCName + ' ' ;
            if (LeftMarginShift < Printer.Canvas.TextWidth(Lab)) then
                LeftMarginShift := Printer.Canvas.TextWidth(Lab) ;
            Lab := format( ' %.3g %s ', [Channel[ch].CalBar,Channel[ch].ADCUnits] ) ;
            if (LeftMarginShift < Printer.Canvas.TextWidth(Lab)) then
                LeftMarginShift := Printer.Canvas.TextWidth(Lab) ;
            end ;

        { Define display area for each channel in use }
        cTop := FPrinterTopMargin + TopSpaceNeeded ;
                ;
        for ch := 0 to FNumChannels-1 do begin
             PrChan[ch] := Channel[ch] ;
             if Channel[ch].InUse then begin
                if FPrinterDisableColor then PrChan[ch].Color := clBlack ;
                PrChan[ch].Left := FPrinterLeftMargin + LeftMarginShift ;
                PrChan[ch].Right := Printer.PageWidth - FPrinterRightMargin ;
                PrChan[ch].Top := cTop ;
                PrChan[ch].Bottom := PrChan[ch].Top + ChannelHeight ;
                PrChan[ch].xMin := FXMin ;
                PrChan[ch].xMax := FXMax ;
                PrChan[ch].xScale := (PrChan[ch].Right - PrChan[ch].Left) /
                                     (PrChan[ch].xMax - PrChan[ch].xMin ) ;
                PrChan[ch].yScale := (PrChan[ch].Bottom - PrChan[ch].Top) /
                                     (PrChan[ch].yMax - PrChan[ch].yMin ) ;
                cTop := cTop + ChannelHeight ;
                end ;
             end ;

        { Plot channel }
        for ch := 0 to FNumChannels-1 do
           if PrChan[ch].InUse and (FBuf <> Nil) and FPrinterShowLabels then begin
           { Display channel name(s) }
           Lab := PrChan[ch].ADCName + ' ' ;
           Printer.Canvas.TextOut( PrChan[ch].Left - Printer.Canvas.TextWidth(Lab),
                                   (PrChan[ch].Top + PrChan[ch].Bottom) div 2,
                                   Lab ) ;
           end ;

        { Plot record(s) on screen }

        if FStorageMode then begin
           { Display all records stored on screen }
           NumBytesPerRecord := FNumChannels*FNumPoints*2 ;
           Rec := 1 ;
           while (FStorageList[Rec] <> NoRecord)
                 and (Rec <= High(FStorageList)) do begin
                 { Read buffer }
                 FStorageFile.Seek( Rec*NumBytesPerRecord, soFromBeginning ) ;
                 FStorageFile.Read( FBuf^, NumBytesPerRecord ) ;
                 { Plot record on display }
                 PlotRecord( Printer.Canvas, PrChan, xy^ ) ;
                 Inc(Rec) ;
                 end ;
           end
        else begin
           { Single-record mode }
           PlotRecord( Printer.Canvas, PrChan, xy^ ) ;
           end ;


       { Plot external line on selected channel }
       if (FLine <> Nil) and (FLineCount > 1) then begin
           Printer.Canvas.Pen.Assign(FLinePen) ;
           Printer.Canvas.Pen.Width := FPrinterPenWidth ;
           for i := 0 to FLineCount-1 do begin
               xy^[i].x := XToCanvasCoord( PrChan[FLineChannel], FLine^[i].x ) ;
               xy^[i].y := YToCanvasCoord( PrChan[FLineChannel], FLine^[i].y ) ;
               end ;
           OK := Polyline( Printer.Canvas.Handle, xy^, FLineCount ) ;
           Printer.Canvas.Pen.Assign(DefaultPen) ;
           end ;

       { Draw baseline levels }
       if FPrinterShowZeroLevels then begin
          Printer.Canvas.Pen.Style := psDot ;
          Printer.Canvas.Pen.Width := 1 ;
          for ch := 0 to FNumChannels-1 do if PrChan[ch].InUse then begin
              YPix := YToCanvasCoord( PrChan[ch], PrChan[ch].ADCZero ) ;
              Printer.Canvas.MoveTo( PrChan[ch].Left,  YPix ) ;
              Printer.Canvas.LineTo( PrChan[ch].Right, YPix ) ;
              end ;
          end ;

       { Restore pen to black and solid for cal. bars }
       Printer.Canvas.Pen.Assign(DefaultPen) ;

       if FPrinterShowLabels then begin
          { Draw vertical calibration bars }
          for ch := 0 to FNumChannels-1 do
              if PrChan[ch].InUse and (PrChan[ch].CalBar <> 0.0) then begin
              { Bar label }
              Lab := format( '%.3g %s ', [PrChan[ch].CalBar,PrChan[ch].ADCUnits] ) ;
              { Calculate position/size of bar }
              Bar.Left := PrChan[ch].Left - Printer.Canvas.TextWidth(Lab+' ') div 2 ;
              Bar.Right := Bar.Left + Printer.Canvas.TextWidth('X') ;
              Bar.Bottom := PrChan[ch].Bottom ;
              Bar.Top := Bar.Bottom
                         - Abs( Round((PrChan[ch].CalBar*PrChan[ch].yScale)
                                    /PrChan[ch].ADCScale) ) ;
              { Draw vertical bar with T's at each end }
              Printer.Canvas.MoveTo( Bar.Left ,  Bar.Bottom ) ;
              Printer.Canvas.LineTo( Bar.Right , Bar.Bottom ) ;
              Printer.Canvas.MoveTo( Bar.Left ,  Bar.Top ) ;
              Printer.Canvas.LineTo( Bar.Right , Bar.Top ) ;
              Printer.Canvas.MoveTo( (Bar.Left + Bar.Right) div 2,  Bar.Bottom ) ;
              Printer.Canvas.LineTo( (Bar.Left + Bar.Right) div 2,  Bar.Top ) ;
              { Draw bar label }
              Printer.Canvas.TextOut(PrChan[ch].Left - Printer.Canvas.TextWidth(Lab),
                                     prChan[ch].Bottom
                                     + Printer.Canvas.TextHeight(Lab) div 4,
                                     Lab ) ;
              end ;

          { Draw horizontal time calibration bar }
          Lab := format( FTFormat, [FTCalBar*FTScale] ) ;
          { Calculate position/size of bar }
          for ch := 0 to FNumChannels-1 do if Channel[ch].InUse then begin
              Bar.Top := PrChan[ch].Bottom + Printer.Canvas.TextHeight(Lab) ;
              LastCh := ch ;
              end ;
          Bar.Bottom := Bar.Top + (Printer.Canvas.TextHeight(Lab) div 2);
          Bar.Left := PrChan[LastCh].Left ;
          Bar.Right := Bar.Left + Abs(Round(FTCalBar*PrChan[LastCh].xScale)) ;
          { Draw vertical bar with T's at each end }
          Printer.Canvas.MoveTo( Bar.Left ,  Bar.Bottom ) ;
          Printer.Canvas.LineTo( Bar.Left ,  Bar.Top ) ;
          Printer.Canvas.MoveTo( Bar.Right , Bar.Bottom ) ;
          Printer.Canvas.LineTo( Bar.Right , Bar.Top ) ;
          Printer.Canvas.MoveTo( Bar.Left, (Bar.Top + Bar.Bottom) div 2 ) ;
          Printer.Canvas.LineTo( Bar.Right,(Bar.Top + Bar.Bottom) div 2 ) ;
          { Draw bar label }
          Printer.Canvas.TextOut(Bar.Left ,
                              Bar.Bottom + Printer.Canvas.TextHeight(Lab) div 4,
                              Lab ) ;

           // Marker text
          for i := 0 to FMarkerText.Count-1 do begin
              xPix := XToCanvasCoord( PrChan[LastCh],
                                      Integer(FMarkerText.Objects[i]) ) ;
              yPix := PrChan[LastCh].Bottom +
                      ((i Mod 2)+1)*Printer.Canvas.TextHeight(FMarkerText.Strings[i]) ;
              Printer.Canvas.TextOut( xPix, yPix, FMarkerText.Strings[i] );
              end ;


          { Draw printer title }
          XPos := FPrinterLeftMargin ;
          YPos := FPrinterTopMargin ;
          CodedTextOut( Printer.Canvas, XPos, YPos, FTitle ) ;

          end ;

     finally
           { Get rid of array }
           Dispose(xy) ;
           DefaultPen.Free ;
           { Close down printer }
           Printer.EndDoc ;
           Cursor := crDefault ;
           end ;

     end ;


procedure TScopeDisplay.CopyImageToClipboard ;
{ -----------------------------------------
  Copy signal image on display to clipboard
  -----------------------------------------}
var
   i,j,n,ch,LastCh,yPix,xPix,Rec,NumBytesPerRecord : Integer ;
   x : single ;
   LeftMarginShift, TopMarginShift : Integer ;
   OK : Boolean ;
   ChannelHeight,cTop,NumInUse,AvailableHeight : Integer ;
   MFChan : Array[0..ScopeChannelLimit] of TScopeChannel ;
   xy : ^TPointArray ;
   Bar : TRect ;
   Lab : string ;
   TMF : TMetafile ;
   TMFC : TMetafileCanvas ;
   DefaultPen : TPen ;
begin

     { Create plotting points array }
     New(xy) ;
     DefaultPen := TPen.Create ;
     Cursor := crHourglass ;

     { Create Windows metafile object }
     TMF := TMetafile.Create ;
     TMF.Width := FMetafileWidth ;
     TMF.Height := FMetafileHeight ;

     try
        { Create a metafile canvas to draw on }
        TMFC := TMetafileCanvas.Create( TMF, 0 ) ;

        try
            { Set type face }
            TMFC.Font.Name := FPrinterFontName ;
            TMFC.Font.Size := FPrinterFontSize ;
            TMFC.Pen.Width := FPrinterPenWidth ;
            DefaultPen.Assign(TMFC.Pen) ;

            { Make the size of the canvas the same as the displayed area
              AGAIN ... See above. Not sure why we need to do this again
              but clipboard image doesn't come out right if we don't}
            TMF.Width := FMetafileWidth ;
            TMF.Height := FMetafileHeight ;
            { ** NOTE ALSO The above two lines MUST come
              BEFORE the setting of the plot margins next }


            { Determine number of channels in use and the height
              available for each channel. NOTE This includes 3
              lines at bottom for time calibration bar }
            NumInUse := 0 ;
            AvailableHeight := TMF.Height - 4*TMFC.TextHeight('X') - 4 ;
            for ch := 0 to FNumChannels-1 do if Channel[ch].InUse then Inc(NumInUse) ;
            if NumInUse < 1 then NumInUse := 1 ;
            ChannelHeight := AvailableHeight div NumInUse ;

            { Make space at left margin for channel names/cal. bars }
            LeftMarginShift := 0 ;
            for ch := 0 to FNumChannels-1 do if Channel[ch].InUse then begin
                Lab := Channel[ch].ADCName + ' ' ;
                if (LeftMarginShift < TMFC.TextWidth(Lab)) then
                   LeftMarginShift := TMFC.TextWidth(Lab) ;
                Lab := format( ' %.3g %s ', [Channel[ch].CalBar,Channel[ch].ADCUnits] ) ;
                if (LeftMarginShift < TMFC.TextWidth(Lab)) then
                   LeftMarginShift := TMFC.TextWidth(Lab) ;
                end ;

            { Define display area for each channel in use }
            cTop := TMFC.TextHeight('X') ;
            for ch := 0 to FNumChannels-1 do begin
                MFChan[ch] := Channel[ch] ;
                if Channel[ch].InUse then begin
                   if FPrinterDisableColor then MFChan[ch].Color := clBlack ;
                   MFChan[ch].Left := TMFC.TextWidth('X') + LeftMarginShift ;
                   MFChan[ch].Right := TMF.Width - TMFC.TextWidth('X') ;
                   MFChan[ch].Top := cTop ;
                   MFChan[ch].Bottom := MFChan[ch].Top + ChannelHeight ;
                   MFChan[ch].xMin := FXMin ;
                   MFChan[ch].xMax := FXMax ;
                   MFChan[ch].xScale := (MFChan[ch].Right - MFChan[ch].Left) /
                                        (MFChan[ch].xMax - MFChan[ch].xMin ) ;
                   MFChan[ch].yScale := (MFChan[ch].Bottom - MFChan[ch].Top) /
                                        (MFChan[ch].yMax - MFChan[ch].yMin ) ;
                   cTop := cTop + ChannelHeight ;
                   End ;
                end ;

            { Plot channel names }
            if FPrinterShowLabels then begin
               for ch := 0 to FNumChannels-1 do
                   if MFChan[ch].InUse and (FBuf <> Nil) then begin
                   Lab := MFChan[ch].ADCName + ' ' ;
                   TMFC.TextOut( MFChan[ch].Left - TMFC.TextWidth(Lab),
                                (MFChan[ch].Top + MFChan[ch].Bottom) div 2,
                                 Lab ) ;
                   end ;
               end ;

            { Plot record(s) on metafile image canvas }

            if FStorageMode then begin
               { Display all records stored on screen }
               NumBytesPerRecord := FNumChannels*FNumPoints*2 ;
               Rec := 1 ;
               FStorageFile.Seek( Rec*NumBytesPerRecord, soFromBeginning ) ;
               while (FStorageList[Rec] <> NoRecord)
                     and (Rec <= High(FStorageList)) do begin
                     { Read buffer }
                     FStorageFile.Read( FBuf^, NumBytesPerRecord ) ;
                     { Plot record on display }
                     PlotRecord( TMFC, MFChan, xy^ ) ;
                     Inc(Rec) ;
                     end ;
               end
            else begin
               { Single-record mode }
               PlotRecord( TMFC, MFChan, xy^ ) ;
               end ;

            { Plot external line on selected channel }
            if (FLine <> Nil) and (FLineCount > 1) then begin
               TMFC.Pen.Assign(FLinePen) ;
               for i := 0 to FLineCount-1 do begin
                   xy^[i].x := XToCanvasCoord( MFChan[FLineChannel], FLine^[i].x ) ;
                   xy^[i].y := YToCanvasCoord( MFChan[FLineChannel], FLine^[i].y ) ;
                   end ;
               OK := Polyline( TMFC.Handle, xy^, FLineCount ) ;
               TMFC.Pen.Assign(DefaultPen) ;
               end ;

            { Draw baseline levels }
            if FPrinterShowZeroLevels then begin
               TMFC.Pen.Width := 1 ;
               TMFC.Pen.Style := psDot ;
               for ch := 0 to FNumChannels-1 do if MFChan[ch].InUse then begin
                   YPix := YToCanvasCoord( MFChan[ch], MFChan[ch].ADCZero ) ;
                   TMFC.MoveTo( MFChan[ch].Left,  YPix ) ;
                   TMFC.LineTo( MFChan[ch].Right, YPix ) ;
                   end ;
               end ;

            { Restore pen to black and solid for cal. bars }
            TMFC.Pen.Assign(DefaultPen) ;

            if FPrinterShowLabels then begin
               { Draw vertical calibration bars }
               for ch := 0 to FNumChannels-1 do
                   if MFChan[ch].InUse and (MFChan[ch].CalBar <> 0.0) then begin
                   { Bar label }
                   Lab := format( '%.3g %s ', [MFChan[ch].CalBar,MFChan[ch].ADCUnits] ) ;
                   { Calculate position/size of bar }
                   Bar.Left := MFChan[ch].Left - TMFC.TextWidth(Lab+' ') div 2 ;
                   Bar.Right := Bar.Left + TMFC.TextWidth('X') ;
                   Bar.Bottom := MFChan[ch].Bottom ;
                   Bar.Top := Bar.Bottom
                              - Abs( Round((MFChan[ch].CalBar*MFChan[ch].yScale)/
                                         MFChan[ch].ADCScale) ) ;
                   { Draw vertical bar with T's at each end }
                   TMFC.MoveTo( Bar.Left ,  Bar.Bottom ) ;
                   TMFC.LineTo( Bar.Right , Bar.Bottom ) ;
                   TMFC.MoveTo( Bar.Left ,  Bar.Top ) ;
                   TMFC.LineTo( Bar.Right , Bar.Top ) ;
                   TMFC.MoveTo( (Bar.Left + Bar.Right) div 2,  Bar.Bottom ) ;
                   TMFC.LineTo( (Bar.Left + Bar.Right) div 2,  Bar.Top ) ;
                   { Draw bar label }
                   TMFC.TextOut(MFChan[ch].Left - TMFC.TextWidth(Lab),
                                MFChan[ch].Bottom
                                + TMFC.TextHeight(Lab) div 4,
                                Lab ) ;
                   end ;

               { Draw horizontal time calibration bar }
               Lab := format( FTFormat, [FTCalBar*FTScale] ) ;
               { Calculate position/size of bar }
               for ch := 0 to FNumChannels-1 do if Channel[ch].InUse then begin
                   Bar.Top := MFChan[ch].Bottom + TMFC.TextHeight(Lab) ;
                   LastCh := ch ;
                   end ;
               Bar.Bottom := Bar.Top + (TMFC.TextHeight(Lab) div 2);
               Bar.Left := MFChan[LastCh].Left ;
               Bar.Right := Bar.Left + Round(FTCalBar*MFChan[LastCh].xScale) ;
               { Draw vertical bar with T's at each end }
               TMFC.MoveTo( Bar.Left ,  Bar.Bottom ) ;
               TMFC.LineTo( Bar.Left ,  Bar.Top ) ;
               TMFC.MoveTo( Bar.Right , Bar.Bottom ) ;
               TMFC.LineTo( Bar.Right , Bar.Top ) ;
               TMFC.MoveTo( Bar.Left, (Bar.Top + Bar.Bottom) div 2 ) ;
               TMFC.LineTo( Bar.Right,(Bar.Top + Bar.Bottom) div 2 ) ;
               { Draw bar label }
               TMFC.TextOut(Bar.Left ,
                         Bar.Bottom + TMFC.TextHeight(Lab) div 4,
                         Lab ) ;

               // Marker text
               for i := 0 to FMarkerText.Count-1 do begin
                   xPix := XToCanvasCoord( MFChan[LastCh],
                                           Integer(FMarkerText.Objects[i]) ) ;
                   yPix := MFChan[LastCh].Bottom +
                           ((i Mod 2)+1)*TMFC.TextHeight(FMarkerText.Strings[i]) ;
                   TMFC.TextOut( xPix, yPix, FMarkerText.Strings[i] );
                   end ;

               end ;
        finally
            { Free metafile canvas. Note this copies plot into metafile object }
            DefaultPen.Free ;
            TMFC.Free ;
            end ;

        { Copy metafile to clipboard }
        Clipboard.Assign(TMF) ;

     finally
           { Get rid of array }
           Dispose(xy) ;
           Cursor := crDefault ;
           end ;

     end ;


procedure TScopeDisplay.CodedTextOut(
          Canvas : TCanvas ;           // Output Canvas
          var LineLeft : Integer ;     // Position of left edge of line
          var LineYPos : Integer ;     // Vertical line position
          List : TStringList           // Strings to be displayed
          ) ;
//----------------------------------------------------------------
// Display lines of text with ^-coded super/subscripts and symbols
//----------------------------------------------------------------
// Added 17/7/01
var
   DefaultFont : TFont ;
   Line,LineSpacing,YSuperscriptShift,YSubscriptShift,i,X,Y : Integer ;
   Done : Boolean ;
   TextLine : string ;
begin

     try

     // Store default font settings
     DefaultFont := TFont.Create ;
     DefaultFont.Assign(Canvas.Font) ;

     // Inter-line spacing and offset used for super/subscripting
     LineSpacing := Canvas.TextHeight('X') ;
     YSuperscriptShift := LineSpacing div 4 ;
     YSubscriptShift := LineSpacing div 2 ;
     LineSpacing := LineSpacing + YSuperscriptShift ;

     // Move to start position for text output
     Canvas.MoveTo( LineLeft, LineYPos ) ;

     { Display coded lines of text on device }

     for Line := 0 to FTitle.Count-1 do begin

         // Get line of text
         TextLine := FTitle.Strings[Line] ;

         // Move to start of line
         X := LineLeft ;
         Y := LineYPos ;
         Canvas.MoveTo( X, Y ) ;

         // Decode and output line
         Done := False ;
         i := 1 ;
         while not Done do begin

             // Get current cursor position
             X := Canvas.PenPos.X ;
             Y := LineYPos ;
             // Restore default font setting
             Canvas.Font.Assign(DefaultFont) ;

             if i <= Length(TextLine) then begin
                if TextLine[i] = '^' then begin
                   Inc(i) ;
                   case TextLine[i] of
                        // Bold
                        'b' : begin
                           Canvas.Font.Style := [fsBold] ;
                           Canvas.TextOut( X, Y, TextLine[i+1] ) ;
                           Inc(i) ;
                           end ;
                        // Italic
                        'i' : begin
                           Canvas.Font.Style := [fsItalic] ;
                           Canvas.TextOut( X, Y, TextLine[i+1] ) ;
                           Inc(i) ;
                           end ;
                        // Subscript
                        '-' : begin
                           Y := Y + YSubscriptShift ;
                           Canvas.Font.Size := (3*Canvas.Font.Size) div 4 ;
                           Canvas.TextOut( X, Y, TextLine[i+1] ) ;
                           Inc(i) ;
                           end ;
                        // Superscript
                        '+' : begin
                           Y := Y - YSuperscriptShift ;
                           Canvas.Font.Size := (3*Canvas.Font.Size) div 4 ;
                           Canvas.TextOut( X, Y, TextLine[i+1] ) ;
                           Inc(i) ;
                           end ;
                        // Superscripted 2
                        '2' : begin
                           Y := Y - YSuperscriptShift ;
                           Canvas.Font.Size := (3*Canvas.Font.Size) div 4 ;
                           Canvas.TextOut( X, Y, '2' ) ;
                           end ;

                        // Greek letter from Symbol character set
                        's' : begin
                           Canvas.Font.Name := 'Symbol' ;
                           Canvas.TextOut( X, Y, TextLine[i+1] ) ;
                           Inc(i) ;
                           end ;
                        // Square root symbol
                        '!' : begin
                           Canvas.Font.Name := 'Symbol' ;
                           Canvas.TextOut( X, Y, chr(214) ) ;
                           end ;
                        // +/- symbol
                        '~' : begin
                           Canvas.Font.Name := 'Symbol' ;
                           Canvas.TextOut( X, Y, chr(177) ) ;
                           end ;

                        end ;
                   end
                else Canvas.TextOut( X, Y, TextLine[i] ) ;

                Inc(i) ;
                end
             else Done := True ;
             end ;

         // Increment position to next line
         LineYPos := LineYPos + LineSpacing ;

         end ;

     // Restore default font setting
     Canvas.Font.Assign(DefaultFont) ;

     finally

            DefaultFont.Free ;

            end ;

     end ;



procedure TScopeDisplay.ClearPrinterTitle ;
{ -------------------------
  Clear printer title lines
  -------------------------}
begin
     FTitle.Clear ;
     end ;


procedure TScopeDisplay.AddPrinterTitleLine(
          Line : string
          );
{ ---------------------------
  Add a line to printer title
  ---------------------------}
begin
     FTitle.Add( Line ) ;
     end ;


procedure TScopeDisplay.CreateLine(
          Ch : Integer ;                    { Display channel to be drawn on [IN] }
          iColor : TColor ;                 { Line colour [IN] }
          iStyle : TPenStyle                { Line style [IN] }
          ) ;
{ -----------------------------------------------
  Create a line to be superimposed on the display
  -----------------------------------------------}
begin
     { Create line data array }
     if FLine = Nil then New(FLine) ;
     FLineCount := 0 ;
     FLineChannel := IntLimitTo(Ch,0,FNumChannels-1) ;
     FLinePen.Color := iColor ;
     FLinePen.Style := iStyle ;
     end ;


procedure TScopeDisplay.AddPointToLine(
          x : single ;
          y : single
          ) ;
{ ---------------------------
  Add a point to end of line
  ---------------------------}
var
   xPix, yPix : Integer ;
   KeepPen : TPen ;
begin
     if FLine <> Nil then begin
        { Add x,y point to array }
        FLine^[FLineCount].x := x ;
        FLine^[FLineCount].y := y ;
        { Add line to end of plot }
        if FLineCount > 0 then begin
           KeepPen := Canvas.Pen ;
           Canvas.Pen := FLinePen ;
           xPix := XToCanvasCoord( Channel[FLineChannel], FLine^[FLineCount-1].x ) ;
           yPix := YToCanvasCoord( Channel[FLineChannel], FLine^[FLineCount-1].y ) ;
           Canvas.MoveTo( xPix, yPix ) ;
           xPix := XToCanvasCoord( Channel[FLineChannel], x ) ;
           yPix := YToCanvasCoord( Channel[FLineChannel], y ) ;
           Canvas.LineTo( xPix, yPix ) ;
           Canvas.Pen := KeepPen ;
           end ;
        { Increment counter }
        if FLineCount < High(TSinglePointArray) then Inc(FLineCount) ;
        end ;
     end ;


function MaxInt(
         const Buf : array of Integer  { List of numbers (IN) }
         ) : Integer ;                 { Returns maximum of Buf }
{ ---------------------------------------------------------
  Return the largest long integer value in the array 'Buf'
  ---------------------------------------------------------}
var
   Max : Integer ;
   i : Integer ;
begin
     Max:= -High(Max) ;
     for i := 0 to High(Buf) do
         if Buf[i] > Max then Max := Buf[i] ;
     Result := Max ;
     end ;

function MinInt(
         const Buf : array of Integer { List of numbers (IN) }
         ) : Integer ;                { Returns Minimum of Buf }
{ -------------------------------------------
  Return the smallest value in the array 'Buf'
  -------------------------------------------}
var
   i,Min : Integer ;
begin
     Min := High(Min) ;
     for i := 0 to High(Buf) do
         if Buf[i] < Min then Min := Buf[i] ;
     Result := Min ;
     end ;


function MaxFlt(
         const Buf : array of Single  { List of numbers (IN) }
         ) : Single ;                 { Returns maximum of Buf }
{ ---------------------------------------------------------
  Return the largest long integer value in the array 'Buf'
  ---------------------------------------------------------}
var
   Max : Single ;
   i : Integer ;
begin
     Max:= -1E30 ;
     for i := 0 to High(Buf) do
         if Buf[i] > Max then Max := Buf[i] ;
     Result := Max ;
     end ;


function MinFlt(
         const Buf : array of Single { List of numbers (IN) }
         ) : Single ;                { Returns Minimum of Buf }
{ -------------------------------------------
  Return the smallest value in the array 'Buf'
  -------------------------------------------}
var
   i : Integer ;
   Min : Single ;
begin
     Min := 1E30 ;
     for i := 0 to High(Buf) do
         if Buf[i] < Min then Min := Buf[i] ;
     Result := Min ;
     end ;


end.
