unit SESLabIO;
{ ================================================================================
  SESLabIO - Data Acquisition Component (c) J. Dempster, University of Strathclyde
  11/12/00 Digidata 132X interface support added
  19/02/01 ReadADC method now supported by all interfaces
  20/4/01  10V CED 1401 support removed
  11/10/01 Error which prevented selection of No Lab Interface (16 bit) option
           now fixed
  20/12/01 MinDACValue and MaxDACValue properties added to support cards such
           as the PCI-6035E which has 16 bit A/D but 12 bit D/A.
  17/3/02 .... Empty buffers now filled with pairs of positive/negative empty flags
               to avoid hang ups when buffer was filled with A/D values equal
               to empty flag.
  19/3/02 ... ITC16/18 support added
  15.7.02  Support for National Instruments E-Series digital outputs added (10 ms resolution)
           10 ms stimulus start timer now included within component
  17.7.02  DAC update interval now constrained to be 10ms when digitsl stimuli used
           with E-Series N.I. boards
  25.7.02  Support for ITC-16 and ITC-18 with old driver added
  25.11.02 Support from CED Micro1401 Mk2 added
           CED commands and DLL library now loaded from c:\1401 folder
  6/12/02 ... Changes to ITCMM support
  7/01/03 ... Support for each channel to have a different A/D voltage now added
              (currently for National Instruments interfaces only)
  30/1/03 ... Support for ITC-16 and ITC-18 using old (EPC-9 compatible) drivers added
  26/6/03 ... StartTimer now automatically stops/restarts existings timer
  03.02.04 .. Recording sweep trigger pulse can now be inverted (DACInvertTriggerLevel property)
  05.02.04 .. Stimulus output waveforms can now be triggered by a digital I/P pulse
              Bug in ReadADC (with National Instruments cards) corrected so that
              correct input voltage range is now used.
  16.02.04 .. Support for Biologic VP500 added
  25.03.04 .. StartTimer no longer waits for StimulusStarted to be cleared
              (fixes problem with WinEDR stimulus pulses)
  27.07.04 .. ADCExternalTriggerActiveHigh property added
  28.09.04  .. NI_ReadADC now works correctly when channel A/D voltages different
  25.10.04  .. CED 1401 ADC & DAC now synchronised internally
  11.11.04  .. NI A/D and D/A sweeps synchronised internally
  18.11.04  .. A/D & D/A Buffer sizes increased to 128 Kbyte
  07.03.05  .. DACRepeatWaveform property added
  14.04.05  .. National Instruments DAQmx support added
  21.07.05  .. A/D input mode (Differential, Single Ended) can now be set
  12.09.05  .. CED digital input read now works
  14.11.05  .. StartStimulus now works with all trigger modes (fixed problem with Chart)
  26.07.06 ... Original Win-RT driver now used again for Digidata 1200 support under Win 95/98/ME
               (unit = dd1200win98.pas)
  20.08.06 ... Stimulus timer tick counter now set to zero when stimulus repeat
               interval changed. This is ensures a full inter-record interval
               when stimulus protocol changes
  06.06.07 ... DACBufferLimit now varies depending on type of interface
               (set = ADCBufferLimit, except CED 1401)
  15.12.07 ... Support for Digidata 1440 added
  19.05.08 ... Support for Tecella Triton added
  02.0.08 .... .SetADCSamplingInterval bug with NIDAQ-MX cards which caused incorrect
               sampling interval to be set when large record sizes and short duration sweeps
               were selected fixed.
  19.08.08 ... .SetADCNumChannels & .SetADCSamplingInterval Error which lead to
               incorrect A/D sampling interval being reported (possibly introduced
               WinWCP V3.9.6-8) with NIDAQ-MX when sampling intervals shorter than
               board limits selected fixed.
  12.09.08 ... CED 1401 (10V added) interface option added             
  ================================================================================ }

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  mmsystem, math ;

const

     //MaxADCSamples = 131072 ;
     MaxADCSamples = 1048576 ;
     // A/D sampling sweep trigger modes
     tmFreeRun = 0 ;    // Start sampling immediately
     tmExtTrigger = 1 ; // Wait for external TTL trigger pulse
     tmWaveGen = 2 ;    // Simultaneous D/A waveform and A/D sampling

     imSingleEnded = 0  ; // Single ended A/D input mode
     imDifferential = 1 ; // Differential input mode
     imBNC2110 = 2 ;      // Standard mode for BNC-2110 panel (differential)
     imBNC2090 = 3 ;      // Standard mode for BNC 2090 panel (SE)
     imSingleEndedRSE = 4 ;          // Single Ended (grounded)

     { Interface cards supported }
     NoInterface12 = 0 ;
     NationalInstruments = 1 ;
     Digidata1200 = 2 ;
     CED1401_12 = 3 ;
     CED1401_16 = 4 ;
     NoInterface16 = 5 ;
     Digidata132X = 6 ;
     Instrutech = 7 ;
     ITC_16 = 8 ;
     ITC_18 = 9 ;
     VP500 = 10 ;
     NIDAQMX = 11 ;
     Digidata1440 = 12 ;
     Triton = 13 ;
     CED1401_10V = 14 ;
     NumLabInterfaceTypes = 15 ;
     StimulusExtTriggerFlag = -1 ;  // Stimulus program started by external trig pulse.



type

  TSmallIntArray = Array[0..MaxADCSamples-1] of SmallInt ;
  PSmallIntArray = ^TSmallIntArray ;

  TIntArray = Array[0..MaxADCSamples-1] of Integer ;
  PIntArray = ^TIntArray ;

  TDoubleArray = Array[0..MaxADCSamples-1] of Double ;
  PDoubleArray = ^TDoubleArray ;

  TSingleArray = Array[0..MaxADCSamples-1] of Single ;
  PSingleArray = ^TSingleArray ;

  TLongIntArray = Array[0..MaxADCSamples-1] of LongInt ;
  PLongIntArray = ^TLongIntArray ;

  TBooleanArray = Array[0..MaxADCSamples-1] of Boolean ;
  PBooleanArray = ^TBooleanArray ;

  TSESLabIO = class(TComponent)
  private
    { Private declarations }
    FLabInterfaceType : Integer ;      // Type of lab. interface hardware
    FLabInterfaceName : string ;       // Name of interface
    FLabInterfaceModel : string ;      // Model
    FLabInterfaceAvailable : Boolean ; // Interface hardware is available for use
    FADCInputMode : Integer ;             // A/D Input mode (Differential, SingleEnded)

    FADCMinValue : Integer ;           // Lower limit of A/D sample value
    FADCMaxValue : Integer ;           // upper limit of A/D sample value

    FADCNumChannels : Integer ;        // Number of A/D channels being sampled
    FADCMaxChannels : Integer ;        // Number of A/D channels available.

    FADCNumSamples : Integer ;         // Number of A/D samples (per channel) to be acquired

    FADCSamplingInterval : Double ;    // Inter-sample interval (s) (per channel)
    FADCMaxSamplingInterval : Double ; // Max. valid interval (s)
    FADCMinSamplingInterval : Double ; // Min. valid interval (s)

    FADCTriggerMode : Integer ;          // A/D sweep trigger mode
    FADCExternalTriggerActiveHigh : Boolean ; // TRUE= Active high TTL External trigger
    FADCCircularBuffer : Boolean ;       // Continuous sampling flag using circular buffer

    FADCVoltageRanges : Array[0..15] of single ;        // A/D input voltage ranges
    FADCNumVoltageRanges : Integer ;                    // Number of ranges available
    FADCVoltageRangeIndex : Integer ;                   // Current range in use
    FADCChannelVoltageRanges : Array[0..15] of Single ;  // Channel A/D voltage range

    FADCChannelOffsets : Array[0..15] of Integer ;  // Channel data offset in A/D buffer
    FADCBufferLimit : Integer ;                     // Max. number of samples in A/D buffer
    FADCActive : Boolean ;                          // A/D sampling in progress flag
    FADCEmptyFlag : Integer ;                       // Code indicating an unfilled data

    FDACMinValue : Integer ;           // Lower limit of D/A sample value
    FDACMaxValue : Integer ;           // upper limit of D/A sample value

    FDACNumChannels : Integer ;          // Number A/D channels being sampled
    FDACMaxChannels : Integer ;          // Max. number of channels supported

    FDACNumSamples : Integer ;           // No. D/A values (per channel) in waveform
    FDACBufferLimit : Integer ;          // D/A buffer limit

    FDACUpdateInterval : Double ;         // DAC update interval in use
    FDACMinUpdateInterval : Double ;      // Smallest valid DAC update interval
    FDACVoltageRange : Single ;           // DAC output voltage range (+/-V)
    FDACActive : Boolean ;                // DAC output in progress flag
    FDACTriggerOnLevel : Integer ;        // Trigger pulse on level
    FDACTriggerOffLevel : Integer ;       // Trigger pulse off level
    FDACInvertTriggerLevel : Boolean ;    // Inverted trigger level
    FDACInitialPoints : Integer ;         // Initial points in D/A waveforms
    FDACRepeatedWaveform : Boolean ;
    FDIGNumOutputs : Integer ;            // Number digital outputs available
    FDIGInterval : Integer ;
    FDIGActive : Boolean ;                // Digital output sweep active
    FDigMinUpdateInterval : Double ;      // Smallest valid digital update interval
    FDigMaxUpdateInterval : Double ;      // Smallest valid digital update interval

    { Default D/A and digital settings }
    FLastDACVolts : Array[0..3] of Single ;
    FLastDACNumChannels : Integer ;
    FLastDigValue : Integer ;

    FStimulusTimerID : Integer ;        // Interval time ID number
    FStimulusDigitalEnabled : Boolean ; // Enable output of digital pulse pattern
    FStimulusTimeStarted : Single ;     // Time (s) last stimulus pulse started
    FStimulusTimerActive : Boolean ;    // TRUE = timer is running
    FStimulusTimerTicks : Integer ;     // Current timer tick count
    FStimulusStartTime : Integer ;      // Tick count to start stimulus at
    FStimulusStartOffset : Integer ;    // No. of digital points to ignore
    FTimerProcInUse : Boolean ;         // TimerProc handle running

    FStimulusExtTrigger : Boolean ;     // Stimulus waveforms wait for ext. trigger

    ADCBuf : PSmallIntArray ;     // A/D sample storage buffer
    DACBuf : PSmallIntArray ;     // D/A waveform output buffer
    DigBuf : PSmallIntArray ;     // Digital output buffer
    DigWork : PSmallIntArray ;    // Working buffer for digital data

    procedure SetLabInterfaceType( InterfaceType : Integer ) ;

    procedure SetADCNumSamples( Value : Integer ) ;
    procedure SetADCNumChannels( Value : Integer ) ;
    procedure SetADCSamplingInterval( Value : Double ) ;
    procedure SetADCVoltageRangeIndex( Value : Integer ) ;
    procedure SetADCVoltageRange( Value : Single ) ;
    procedure SetADCChannelVoltageRange( Channel : Integer ; Value : Single) ;
    procedure SetADCInputMode( Value : Integer ) ;

    procedure SetDACNumChannels( Value : Integer ) ;
    procedure SetDACNumSamples( Value : Integer ) ;
    procedure SetDACUpdateInterval( Value : Double ) ;
    function GetADCVoltageRange : Single ;
    function GetADCChannelOffsets( Channel : Integer) : Integer ;
    function GetADCChannelVoltageRange( Channel : Integer) : Single ;
    function GetStimulusTimerPeriod : Single ;
    procedure SetStimulusTimerPeriod( Value : Single ) ;

    function GetTritonSource( Chan : Integer) : Integer ;
    procedure SetTritonSource( Chan : Integer ; Value : Integer) ;
    function GetTritonGain( Chan : Integer) : Integer ;
    procedure SetTritonGain( Chan : Integer ; Value : Integer) ;


    procedure TimerTickOperations ;
    procedure SetADCExternalTriggerActiveHigh( Value : Boolean ) ;

    function  IntLimit( Value : Integer ; LoLimit : Integer ; HiLimit : Integer
              ) : Integer ;
    function  FloatLimit( Value : Single ; LoLimit : Single ; HiLimit : Single
              ) : Single ;
    function GetLabInterfaceName( Num : Integer ) : String ;
    function GetDIGInputs : Integer ;

  protected
    { Protected declarations }
  public
    { Public declarations }
    StimulusStartFlag : Boolean ;
    FOutPointer : Integer ;

    Constructor Create(AOwner : TComponent) ; override ;
    Destructor Destroy ; override ;
    procedure GetLabInterfaceTypes( List : TStrings ) ;
    procedure GetADCInputModes( List : TStrings ) ;
    procedure ADCStart ;
    procedure ADCStop ;
    procedure DACStart ;
    procedure DACDIGStart( StartAt : Integer ) ;
    procedure DACStop ;
    function ReadADC( Channel : Integer ) : Integer ;
    procedure WriteDACs( DACVolts : Array of single ; NumChannels : Integer ) ;
    procedure GetADCBuffer( var BufPointer : PSmallIntArray ) ;
    procedure GetDACBuffer( var BufPointer : PSmallIntArray ) ;
    procedure GetDIGBuffer( var BufPointer : PSmallIntArray ) ;
    procedure ADCBufferRefresh ;
    procedure WriteDig( DigByte : Integer ) ;
    procedure StartTimer ;
    procedure StopTimer ;
    function GetDACTriggerOnLevel : Integer ;
    function GetDACTriggerOffLevel : Integer ;
    function ExternalStimulusTrigger : Boolean ;
    procedure StartStimulus ;
    procedure DisableDMA_LabPC ;
    procedure TritonGetReg(
              Reg : Integer ;
              Chan : Integer ;
              var Value : Single ;
              var PercentValue : Single ;
              var Enabled : Boolean ) ;

    procedure TritonSetRegPercent(
              Reg : Integer ;
              Chan : Integer ;
              var PercentValue : single ) ;

    function TritonGetRegEnabled(
              Reg : Integer ;
              Chan : Integer ) : Boolean ;
    procedure TritonSetRegEnabled(
              Reg : Integer ;
              Chan : Integer ;
              Enabled : Boolean ) ;
    procedure SetTritonBesselFilter( Chan : Integer ;
                                     Value : Integer ;
                                     var CutOffFrequency : Single) ;

    procedure TritonAutoCompensation(
              Chan : Integer                    // Channel to be compensated
              ) ;


    Property ADCChannelOffsets[ i : Integer ] : Integer
                                            Read GetADCChannelOffsets ;

    Property ADCChannelVoltageRange[ i : Integer ] : Single
                                            Read GetADCChannelVoltageRange
                                            Write SetADCChannelVoltageRange ;

    Property TritonSource[ Chan : Integer ] : Integer
                                            Read GetTritonSource
                                            write SetTritonSource ;

    Property TritonGain[ Chan : Integer ] : Integer
                                            Read GetTritonGain
                                            write SetTritonGain ;

  published
    { Published declarations }
    Property LabInterfaceType : Integer Read FLabInterfaceType
                                        Write SetLabInterfaceType ;
    Property LabInterfaceAvailable : Boolean Read FLabInterfaceAvailable ;
    Property LabInterfaceName : string read FLabInterfaceName ;
    Property LabInterfaceModel : string read FLabInterfaceModel ;

    Property ADCNumChannels : Integer Read FADCNumChannels Write SetADCNumChannels
                                      Default 1 ;
    Property ADCNumSamples : Integer Read FADCNumSamples Write SetADCNumSamples
                                     Default 512 ;
    Property ADCMinValue : Integer Read FADCMinValue ;
    Property ADCMaxValue : Integer Read FADCMaxValue ;
    Property ADCSamplingInterval : Double Read FADCSamplingInterval
                                          Write SetADCSamplingInterval ;
    Property ADCMinSamplingInterval : Double Read FADCMinSamplingInterval ;
    Property ADCMaxSamplingInterval : Double Read FADCMaxSamplingInterval ;
    Property ADCVoltageRange : Single Read GetADCVoltageRange
                                      Write SetADCVoltageRange ;
    Property ADCNumVoltageRanges : Integer Read FADCNumVoltageRanges ;
    Property ADCVoltageRangeIndex : Integer Read FADCVoltageRangeIndex
                                            Write SetADCVoltageRangeIndex ;
    Property ADCTriggerMode : Integer Read FADCTriggerMode
                                      Write FADCTriggerMode
                                      Default tmFreeRun ;
    Property ADCExternalTriggerActiveHigh : Boolean
             read FADCExternalTriggerActiveHigh
             write SetADCExternalTriggerActiveHigh ;
    Property ADCCircularBuffer : Boolean Read FADCCircularBuffer
                                         Write FADCCircularBuffer
                                         Default False ;
    Property ADCBufferLimit : Integer Read FADCBufferLimit ;
    Property ADCActive : Boolean Read FADCActive ;
    Property ADCEmptyFlag : Integer Read FADCEmptyFlag ;
    Property ADCInputMode : Integer Read FADCInputMode Write SetADCInputMode ;

    Property DACActive : Boolean Read FDACActive ;
    Property DACMinValue : Integer Read FDACMinValue ;
    Property DACMaxValue : Integer Read FDACMaxValue ;

    Property DACNumChannels : Integer Read FDACNumChannels Write SetDACNumChannels
                                      Default 2 ;
    Property DACNumSamples : Integer Read FDACNumSamples Write SetDACNumSamples
                                     Default 512 ;
    Property DACBufferLimit : Integer Read FDACBufferLimit ;
    Property DACMinUpdateInterval : Double Read FDACMinUpdateInterval ;

    Property DACUpdateInterval : Double Read FDACUpdateInterval
                                        Write SetDACUpdateInterval ;
    Property DACVoltageRange : Single Read FDACVoltageRange ;
    Property DACTriggerOnLevel : Integer Read GetDACTriggerOnLevel ;
    Property DACTriggerOffLevel : Integer Read GetDACTriggerOffLevel ;
    Property DACInvertTriggerLevel : Boolean Read FDACInvertTriggerLevel
                                             Write  FDACInvertTriggerLevel ;
    Property DACInitialPoints : Integer Read FDACInitialPoints ;
    Property DACRepeatWaveform : Boolean Read FDACRepeatedWaveform
                                           Write FDACRepeatedWaveform ;
    Property DIGNumOutputs : Integer Read FDIGNumOutputs ;
    Property DIGInterval : Integer Read FDIGInterval ;
    Property DigMinUpdateInterval : Double Read FDigMinUpdateInterval ;
    Property DIGInputs : Integer Read GetDIGInputs ;

    Property TimerPeriod : Single Read GetStimulusTimerPeriod Write SetStimulusTimerPeriod ;
    Property DigitalStimulusEnabled : Boolean Read FStimulusDigitalEnabled Write FStimulusDigitalEnabled ;
    Property TimerActive : Boolean Read FStimulusTimerActive ;
    Property StimulusTime : Single Read FStimulusTimeStarted ;
    Property DigitalStimulusStart : Integer Read FStimulusStartOffset Write FStimulusStartOffset;

  end;

procedure Register;

procedure TimerProc(
          uID,uMsg : SmallInt ; User : TSESLabIO ; dw1,dw2 : LongInt ) ; stdcall ;


implementation

uses NatInst, CED1401, dd1200, dd1200win98, dd1320,
     itcmm, itclib, vp500Unit, nidaqmxunit, dd1440, tritonunit ;


const
     { ------------------------- }
     EmptyFlag = 32767 ;
     ChannelLimit = 15 ;
     StimulusTimerTickInterval = 10 ; // Timer tick resolution (ms)


procedure Register;
begin
  RegisterComponents('Samples', [TSESLabIO]);
end;


constructor TSESLabIO.Create(AOwner : TComponent) ;
{ --------------------------------------------------
  Initialise component's internal objects and fields
  -------------------------------------------------- }
var
   i : Integer ;
begin
     inherited Create(AOwner) ;

     New(ADCBuf) ;
     New(DACBuf) ;
     New(DigBuf) ;
     New(DigWork) ;

     FLabInterfaceType := NoInterface12 ; { No Interface }
     LabInterfaceType := FLabInterfaceType  ;
     FLabInterfaceAvailable := False ;
     FADCMinValue := -2047 ;
     FADCMaxValue := 2048 ;

     FADCMaxChannels := 16 ;
     FADCNumChannels := 1 ;
     for i := 0 to High(FADCChannelOffsets) do FADCChannelOffsets[i] := i ;
     FADCNumSamples := 512 ;
     FADCActive := False ;
     FADCTriggerMode := tmFreeRun ;
     ADCExternalTriggerActiveHigh := False ;
     FADCInputMode := imSingleEnded ;

     { A/D converter voltage ranges }
     for i := 0 to High(FADCVoltageRanges) do FADCVoltageRanges[i] := 1.0 ;
     FADCNumVoltageRanges := 0 ;
     FADCVoltageRangeIndex := 0 ;
     for i := 0 to High(FADCChannelVoltageRanges) do FADCChannelVoltageRanges[i] := 1.0 ;

     FDACMaxChannels := 2 ;
     FDACMinValue := -2047 ;
     FDACMaxValue := 2048 ;
     FDACNumChannels := 2 ;
     FDACNumSamples := 512 ;
     FDACActive := False ;
     FADCEmptyFlag := EmptyFlag ;
     FDACInvertTriggerLevel := False ;
     FDACRepeatedWaveform := False ;

     for i := 0 to High(FLastDACVolts) do FLastDACVolts[i] := 0.0 ;
     FLastDACNumChannels := 1 ;
     FLastDigValue := 0 ;

     FDigMaxUpdateInterval := 1000.0 ;
     FDIGActive := False ;
     FOutPointer := 0 ;
     // Initialise stimulus timer
     FStimulusTimerActive := False ;
     FStimulusStartOffset := 0 ;
     FTimerProcInUse := False ;

     FStimulusExtTrigger := False ;
     StimulusStartFlag := False ;

     end ;


destructor TSESLabIO.Destroy ;
{ ------------------------------------
   Tidy up when component is destroyed
   ----------------------------------- }
begin

     // Stop stimulus timer
     StopTimer ;

     { Shut down lab. interface }
     case FLabInterfaceType of
          NationalInstruments : begin
             NI_CloseLaboratoryInterface ;
             end ;
          Digidata1200 : begin
             if Win32Platform = VER_PLATFORM_WIN32_NT then DD_CloseLaboratoryInterface
                                                      else DD98_CloseLaboratoryInterface ;
             end ;
          CED1401_12, CED1401_16, CED1401_10V : begin
             CED_CloseLaboratoryInterface ;
             end ;
          Digidata132X : begin
             DD132X_CloseLaboratoryInterface ;
             end ;
          Instrutech : begin
             ITCMM_CloseLaboratoryInterface ;
             end ;
          ITC_16, ITC_18 : begin
             ITC_CloseLaboratoryInterface ;
             end ;
          VP500 : begin
             VP500_CloseLaboratoryInterface ;
             end ;
          NIDAQMX : begin
             NIMX_CloseLaboratoryInterface ;
             end ;
          Digidata1440 : begin
             DD1440_CloseLaboratoryInterface ;
             end ;
          Triton : begin
             Triton_CloseLaboratoryInterface ;
             end ;

          end ;

     { Destroy internal objects created by TSESLabIO.Create }
     Dispose(ADCBuf) ;
     Dispose(DACBuf) ;
     Dispose(DigBuf) ;
     Dispose(DigWork) ;
     { Call inherited destructor }
     inherited Destroy ;
     end ;


function TSESLabIO.GetLabInterfaceName( Num : Integer ) : String ;
{ -------------------------------------
  Get name of laboratory interface unit
  ------------------------------------- }
begin
     case Num of
       NoInterface12 : Result := 'No Lab. Interface (12 bit)' ;
       NoInterface16 : Result := 'No Lab. Interface (16 bit)' ;
       NationalInstruments : Result := 'National Instruments (NIDAQ)' ;
       Digidata1200 : Result := 'Axon Instruments (Digidata 1200)' ;
       CED1401_12 : Result := 'CED 1401 (12 bit)(5V)' ;
       CED1401_16 : Result := 'CED 1401 (16 bit)(5V)' ;
       Digidata132X : Result := 'Axon Instruments (Digidata 132X)' ;
       Instrutech : Result := 'Instrutech ITC-16/18 (New drivers)' ;
       ITC_16 : Result := 'Instrutech ITC-16 (Old drivers)' ;
       ITC_18 : Result := 'Instrutech ITC-18 (Old drivers)' ;
       VP500 : Result := 'Biologic VP500' ;
       NIDAQMX : Result := 'National Instruments (NIDAQ-MX)' ;
       Digidata1440 : Result := 'Molecular Devices Digidata 1440' ;
       Triton : Result := 'Tecella Triton' ;
       CED1401_10V : Result := 'CED1401 (16 bit)(10V)' ;
       end ;
     end ;


procedure TSESLabIO.SetLabInterfaceType(
          InterfaceType : Integer
          ) ;
{ -------------------------------------------
  Set type of lab. interface hardware in use
  ------------------------------------------- }
var
     i : Integer ;
     fn : string ;
begin

     { Initialise lab. interface hardware }
     FLabInterfaceType := InterfaceType ;
     FLabInterfaceModel := 'Unknown' ;
     FADCBufferLimit := High(TSmallIntArray)+1 ;
     FDACBufferLimit := High(TSmallIntArray)+1 ;

     case FLabInterfaceType of

       { No interface 12 bit A/D data}
       NoInterface12 : begin
          FLabInterfaceName := 'No Lab. Interface' ;
          FADCMinValue := -2048 ;
          FADCMaxValue := 2047 ;
          FDACMinValue := FADCMinValue ;
          FDACMaxValue := FADCMaxValue ;
          FADCVoltageRanges[0] := 5.0 ;
          FADCNumVoltageRanges := 1 ;
          FADCMinSamplingInterval := 1E-5 ;
          FADCMaxSamplingInterval := 1000 ;
          FDACMinUpdateInterval := 1E-3 ;
          FDACVoltageRange := 5.0 ;
          FDACMinUpdateInterval := 1E-3 ;
          FDIGNumOutputs := 8 ; { No. of digital outputs }
          FDIGInterval := 1 ;
          FDigMinUpdateInterval := 1E-3 ;
          FDigMinUpdateInterval := 1000.0 ;
          FDACTriggerOnLevel := FDACMaxValue ;
          FDACTriggerOffLevel := 0 ;
          FDACInitialPoints := 5 ;
          FLabInterfaceAvailable := True ;
          end ;

       { No interface 16 bit A/D data}
       NoInterface16 : begin
          FLabInterfaceName := 'No Lab. Interface (16 bit)' ;
          FADCMinValue := -32768 ;
          FADCMaxValue := 32767 ;
          FDACMinValue := FADCMinValue ;
          FDACMaxValue := FADCMaxValue ;
          FADCVoltageRanges[0] := 5.0 ;
          FADCNumVoltageRanges := 1 ;
          FADCMinSamplingInterval := 1E-5 ;
          FADCMaxSamplingInterval := 1000 ;
          FDACMinUpdateInterval := 1E-3 ;
          FDACVoltageRange := 5.0 ;
          FDACMinUpdateInterval := 1E-3 ;
          FDIGNumOutputs := 8 ; { No. of digital outputs }
          FDIGInterval := 1 ;
          FDigMinUpdateInterval := 1E-3 ;
          FDigMinUpdateInterval := 1000.0 ;
          FDACTriggerOnLevel := FDACMaxValue ;
          FDACTriggerOffLevel := 0 ;
          FDACInitialPoints := 5 ;
          FLabInterfaceAvailable := True ;
          end ;

       NationalInstruments : begin
          FLabInterfaceName := 'National Instruments (NIDAQ)' ;
          FLabInterfaceAvailable := NI_GetLabInterfaceInfo( FLabInterfaceModel,
                                    FADCMinSamplingInterval,FADCMaxSamplingInterval,
                                    FADCMinValue, FADCMaxValue,
                                    FDACMinValue, FDACMaxValue,
                                    FADCVoltageRanges,FADCNumVoltageRanges,
                                    FDACVoltageRange,
                                    FDACMinUpdateInterval,
                                    FDigMinUpdateInterval,
                                    FDigMaxUpdateInterval,
                                    FDIGInterval,
                                    FADCBufferLimit ) ;

          if FLabInterfaceAvailable then begin
             FDACBufferLimit := FADCBufferLimit ;
             FDIGNumOutputs := 8 ; { No. of digital outputs }
             //FDIGInterval := 2 ;
             FDACTriggerOnLevel := 0 ;
             FDACTriggerOffLevel := IntLimit( Round((FDACMaxValue*4.99)/FDACVoltageRange),
                                              FDACMinValue,FDACMaxValue) ;
             FDACInitialPoints := 5 ;
             end ;
          end ;

       Digidata1200 : begin
          FLabInterfaceName := 'Axon Instruments (Digidata 1200)' ;
          if Win32Platform = VER_PLATFORM_WIN32_NT  then begin
             DD_ConfigureHardware( FADCEmptyFlag ) ;
             FLabInterfaceAvailable := DD_GetLabInterfaceInfo(
                                    FLabInterfaceModel,
                                    FADCMinSamplingInterval,
                                    FADCMaxSamplingInterval,
                                    FADCMinValue,
                                    FADCMaxValue,
                                    FADCVoltageRanges,
                                    FADCNumVoltageRanges,
                                    FADCBufferLimit,
                                    FDACVoltageRange,
                                    FDACMinUpdateInterval ) ;
             end
          else begin
             DD98_ConfigureHardware( FADCEmptyFlag ) ;
             FLabInterfaceAvailable := DD98_GetLabInterfaceInfo(
                                    FLabInterfaceModel,
                                    FADCMinSamplingInterval,
                                    FADCMaxSamplingInterval,
                                    FADCMinValue,
                                    FADCMaxValue,
                                    FADCVoltageRanges,
                                    FADCNumVoltageRanges,
                                    FADCBufferLimit,
                                    FDACVoltageRange,
                                    FDACMinUpdateInterval ) ;
             end ;
          FDACMinValue := FADCMinValue ;
          FDACMaxValue := FADCMaxValue ;
          if FLabInterfaceAvailable then begin
             FDACBufferLimit := FADCBufferLimit ;
             FDIGNumOutputs := 4 ; { No. of digital outputs }
             FDIGInterval := FDACBufferLimit ;
             FDigMinUpdateInterval := FDACMinUpdateInterval ;
             FDigMaxUpdateInterval := 1000.0 ;
             FDACTriggerOnLevel := IntLimit(Round((FDACMaxValue*4.99)/FDACVoltageRange),
                                   FDACMinValue,FDACMaxValue) ;
             FDACTriggerOffLevel := 0 ;
             FDACInitialPoints := 5 ;

             end ;
          end ;

       CED1401_12 : begin
          FLabInterfaceName := 'CED 1401 (12 bit)(5V)' ;
          CED_ConfigureHardware( 12, FADCEmptyFlag, 5.0 ) ;
          FLabInterfaceAvailable := CED_GetLabInterfaceInfo(
                                    FLabInterfaceModel,
                                    FADCMinSamplingInterval,
                                    FADCMaxSamplingInterval,
                                    FADCMinValue,
                                    FADCMaxValue,
                                    FADCVoltageRanges,
                                    FADCNumVoltageRanges,
                                    FADCBufferLimit,
                                    FDACBufferLimit,
                                    FDACVoltageRange,
                                    FDACMinUpdateInterval ) ;
          FDACMinValue := FADCMinValue ;
          FDACMaxValue := FADCMaxValue ;
          if FLabInterfaceAvailable then begin
             FDIGNumOutputs := 8 ; { No. of digital outputs }
             FDIGInterval := 1 ;
             FDigMinUpdateInterval := FDACMinUpdateInterval ;
             FDigMaxUpdateInterval := 1000.0 ;
             FDACTriggerOnLevel := 0 ;
             FDACTriggerOffLevel := IntLimit(Round((FDACMaxValue*4.99)/FDACVoltageRange),
                                    FDACMinValue,FDACMaxValue) ;

             FDACInitialPoints := 0 ;

             end ;
          end ;

       CED1401_16, CED1401_10V : begin
          if FLabInterfaceType = CED1401_16 then begin
             FLabInterfaceName := 'CED1401 (5V ADC/DAC range)' ;
             CED_ConfigureHardware( 16, FADCEmptyFlag, 5.0 ) ;
             end
          else begin
             FLabInterfaceName := 'CED1401 (10V ADC/DAC range)' ;
             CED_ConfigureHardware( 16, FADCEmptyFlag, 10.0 ) ;
             end ;
          FLabInterfaceAvailable := CED_GetLabInterfaceInfo(
                                    FLabInterfaceModel,
                                    FADCMinSamplingInterval,
                                    FADCMaxSamplingInterval,
                                    FADCMinValue,
                                    FADCMaxValue,
                                    FADCVoltageRanges,
                                    FADCNumVoltageRanges,
                                    FADCBufferLimit,
                                    FDACBufferLimit,
                                    FDACVoltageRange,
                                    FDACMinUpdateInterval ) ;
          FDACMinValue := FADCMinValue ;
          FDACMaxValue := FADCMaxValue ;
          if FLabInterfaceAvailable then begin
             // DAC buffer set to 1/3 of ADCBufferLimit because 256 KB CED1401-plus
             // does not have sufficient memory
//             FDACBufferLimit := FADCBufferLimit div 3 ;
             FDIGNumOutputs := 8 ; { No. of digital outputs }
             FDIGInterval := 1 ;
             FDigMinUpdateInterval := FDACMinUpdateInterval ;
             FDigMaxUpdateInterval := 1000.0 ;
             FDACTriggerOnLevel := 0 ;
             FDACTriggerOffLevel := IntLimit(Round((FDACMaxValue*4.99)/FDACVoltageRange),
                                    FDACMinValue,FDACMaxValue) ;
             FDACInitialPoints := 0 ;

             end ;
          end ;

       Digidata132X : begin
          FLabInterfaceName := 'Axon Instruments (Digidata 132X)' ;
          DD132X_ConfigureHardware( FADCEmptyFlag ) ;
          FLabInterfaceAvailable := DD132X_GetLabInterfaceInfo(
                                    FLabInterfaceModel,
                                    FADCMinSamplingInterval,
                                    FADCMaxSamplingInterval,
                                    FADCMinValue,
                                    FADCMaxValue,
                                    FADCVoltageRanges,
                                    FADCNumVoltageRanges,
                                    FADCBufferLimit,
                                    FDACVoltageRange,
                                    FDACMinUpdateInterval ) ;
          FDACMinValue := FADCMinValue ;
          FDACMaxValue := FADCMaxValue ;
          if FLabInterfaceAvailable then begin
             FDACBufferLimit := FADCBufferLimit ;
             FDIGNumOutputs := 4 ; { No. of digital outputs }
             FDIGInterval := FDACBufferLimit ;
             FDigMinUpdateInterval := FDACMinUpdateInterval ;
             FDigMaxUpdateInterval := 1000.0 ;
             FDACTriggerOnLevel := IntLimit(Round((FDACMaxValue*4.99)/FDACVoltageRange),
                                   FDACMinValue,FDACMaxValue) ;
             FDACTriggerOffLevel := 0 ;
             { Note. no initial points in D/A waveforms because A/D and D/A
               runs synchronously }
             FDACInitialPoints := 0 ;
             end ;
          end ;

       Instrutech : begin
          FLabInterfaceName := 'Instrutech ITC-16/18 (New Driver)' ;
          ITCMM_ConfigureHardware( FADCEmptyFlag ) ;
          FLabInterfaceAvailable := ITCMM_GetLabInterfaceInfo(
                                    FLabInterfaceModel,
                                    FADCMinSamplingInterval,
                                    FADCMaxSamplingInterval,
                                    FADCMinValue,
                                    FADCMaxValue,
                                    FADCVoltageRanges,
                                    FADCNumVoltageRanges,
                                    FADCBufferLimit,
                                    FDACVoltageRange,
                                    FDACMinUpdateInterval ) ;
          FDACMinValue := FADCMinValue ;
          FDACMaxValue := FADCMaxValue ;
          if FLabInterfaceAvailable then begin
             FDACBufferLimit := FADCBufferLimit ;
             FDIGNumOutputs := 4 ; { No. of digital outputs }
             FDIGInterval := FDACBufferLimit ;
             FDigMinUpdateInterval := FDACMinUpdateInterval ;
             FDigMaxUpdateInterval := 1000.0 ;
             FDACTriggerOnLevel := IntLimit(Round((FDACMaxValue*4.99)/FDACVoltageRange),
                                   FDACMinValue,FDACMaxValue) ;
             FDACTriggerOffLevel := 0 ;
             { Note. no initial points in D/A waveforms because A/D and D/A
               runs synchronously }
             FDACInitialPoints := 0 ;
             end ;
          end ;

       ITC_16, ITC_18 : begin

          if FLabInterfaceType = ITC_16 then begin
             // Configure library for ITC-16
             FLabInterfaceName := 'Instrutech ITC-16 (Old Driver)' ;
             ITC_ConfigureHardware( 0, FADCEmptyFlag ) ;
             end
          else begin
             // Configure library for ITC-18
             FLabInterfaceName := 'Instrutech ITC-18 (Old Driver)' ;
             ITC_ConfigureHardware( 1, FADCEmptyFlag ) ;
             end ;

          FLabInterfaceAvailable := ITC_GetLabInterfaceInfo(
                                    FLabInterfaceModel,
                                    FADCMinSamplingInterval,
                                    FADCMaxSamplingInterval,
                                    FADCMinValue,
                                    FADCMaxValue,
                                    FADCVoltageRanges,
                                    FADCNumVoltageRanges,
                                    FADCBufferLimit,
                                    FDACVoltageRange,
                                    FDACMinUpdateInterval ) ;
          FDACMinValue := FADCMinValue ;
          FDACMaxValue := FADCMaxValue ;
          if FLabInterfaceAvailable then begin
             FDACBufferLimit := FADCBufferLimit ;
             FDIGNumOutputs := 4 ; { No. of digital outputs }
             FDIGInterval := FDACBufferLimit ;
             FDigMinUpdateInterval := FDACMinUpdateInterval ;
             FDigMaxUpdateInterval := 1000.0 ;
             FDACTriggerOnLevel := IntLimit(Round((FDACMaxValue*4.99)/FDACVoltageRange),
                                   FDACMinValue,FDACMaxValue) ;
             FDACTriggerOffLevel := 0 ;
             { Note. no initial points in D/A waveforms because A/D and D/A
               runs synchronously }
             FDACInitialPoints := 0 ;
             end ;
          end ;

       VP500 : begin
          FLabInterfaceName := 'Biologic VP500' ;
          VP500_ConfigureHardware( FADCEmptyFlag ) ;
          FLabInterfaceAvailable := VP500_GetLabInterfaceInfo(
                                    FLabInterfaceModel,
                                    FADCMinSamplingInterval,
                                    FADCMaxSamplingInterval,
                                    FADCMinValue,
                                    FADCMaxValue,
                                    FADCVoltageRanges,
                                    FADCNumVoltageRanges,
                                    FADCBufferLimit,
                                    FDACVoltageRange,
                                    FDACMinUpdateInterval ) ;
          FDACMinValue := FADCMinValue ;
          FDACMaxValue := FADCMaxValue ;
          if FLabInterfaceAvailable then begin
             FDACBufferLimit := FADCBufferLimit ;
             FDIGNumOutputs := 4 ; { No. of digital outputs }
             FDIGInterval := FDACBufferLimit ;
             FDigMinUpdateInterval := FDACMinUpdateInterval ;
             FDigMaxUpdateInterval := 1000.0 ;
             FDACTriggerOnLevel := FDACMaxValue ;
             FDACTriggerOffLevel := 0 ;
             { Note. no initial points in D/A waveforms because A/D and D/A
               runs synchronously }
             FDACInitialPoints := 0 ;
             end ;
          end ;

        NIDAQMX : begin
          FLabInterfaceName := 'National Instruments (NIDAQ-MX)' ;
          FLabInterfaceAvailable := NIMX_GetLabInterfaceInfo(
                                    FLabInterfaceModel,
                                    FADCMinSamplingInterval,
                                    FADCMaxSamplingInterval,
                                    FADCMinValue, FADCMaxValue,
                                    FDACMinValue, FDACMaxValue,
                                    FADCVoltageRanges,FADCNumVoltageRanges,
                                    FDACVoltageRange,
                                    FDACMinUpdateInterval,
                                    FDigMinUpdateInterval,
                                    FDigMaxUpdateInterval,
                                    FDIGInterval,
                                    FADCBufferLimit ) ;

          if FLabInterfaceAvailable then begin
             FDACBufferLimit := FADCBufferLimit ;
             FDIGNumOutputs := 8 ; { No. of digital outputs }
             //FDIGInterval := 2 ;
             FDACTriggerOnLevel := 0 ;
             FDACTriggerOffLevel := IntLimit( Round((FDACMaxValue*4.99)/FDACVoltageRange),
                                              FDACMinValue,FDACMaxValue) ;
             FDACInitialPoints := 0 ;
             end ;
          end ;

       Digidata1440 : begin
          FLabInterfaceName := 'Molecular Devices (Digidata 1440)' ;
          DD1440_ConfigureHardware( FADCEmptyFlag ) ;
          FLabInterfaceAvailable := DD1440_GetLabInterfaceInfo(
                                    FLabInterfaceModel,
                                    FADCMinSamplingInterval,
                                    FADCMaxSamplingInterval,
                                    FADCMinValue,
                                    FADCMaxValue,
                                    FADCVoltageRanges,
                                    FADCNumVoltageRanges,
                                    FADCBufferLimit,
                                    FDACVoltageRange,
                                    FDACMinUpdateInterval ) ;
          FDACMinValue := FADCMinValue ;
          FDACMaxValue := FADCMaxValue ;
          if FLabInterfaceAvailable then begin
             FDACBufferLimit := FADCBufferLimit ;
             FDIGNumOutputs := 4 ; { No. of digital outputs }
             FDIGInterval := FDACBufferLimit ;
             FDigMinUpdateInterval := FDACMinUpdateInterval ;
             FDigMaxUpdateInterval := 1000.0 ;
             FDACTriggerOnLevel := IntLimit(Round((FDACMaxValue*4.99)/FDACVoltageRange),
                                   FDACMinValue,FDACMaxValue) ;
             FDACTriggerOffLevel := 0 ;
             { Note. no initial points in D/A waveforms because A/D and D/A
               runs synchronously }
             FDACInitialPoints := 0 ;
             end ;
          end ;

       Triton : begin
          FLabInterfaceName := 'Tecella Triton' ;
          Triton_ConfigureHardware( FADCEmptyFlag ) ;
          FLabInterfaceAvailable := Triton_GetLabInterfaceInfo(
                                    FLabInterfaceModel,
                                    FADCMinSamplingInterval,
                                    FADCMaxSamplingInterval,
                                    FADCMinValue,
                                    FADCMaxValue,
                                    FADCVoltageRanges,
                                    FADCNumVoltageRanges,
                                    FADCBufferLimit,
                                    FDACVoltageRange,
                                    FDACMinUpdateInterval ) ;
          FDACMinValue := FADCMinValue ;
          FDACMaxValue := FADCMaxValue ;
          if FLabInterfaceAvailable then begin
             FDACBufferLimit := FADCBufferLimit ;
             FDIGNumOutputs := 4 ; { No. of digital outputs }
             FDIGInterval := FDACBufferLimit ;
             FDigMinUpdateInterval := FDACMinUpdateInterval ;
             FDigMaxUpdateInterval := 1000.0 ;
             FDACTriggerOnLevel := IntLimit(Round((FDACMaxValue*4.99)/FDACVoltageRange),
                                   FDACMinValue,FDACMaxValue) ;
             FDACTriggerOffLevel := 0 ;
             { Note. no initial points in D/A waveforms because A/D and D/A
               runs synchronously }
             FDACInitialPoints := 0 ;
             end ;
          end ;


       end ;

     { Set type to no interface if selected one is not available }
     if not FLabInterfaceAvailable then begin
          FLabInterfaceType := NoInterface12 ;
          FLabInterfaceName := 'No Lab. Interface (12 bit)' ;
          FADCMinValue := -2048 ;
          FADCMaxValue := 2047 ;
          FDACMinValue := FADCMinValue ;
          FDACMaxValue := FADCMaxValue ;
          FADCVoltageRanges[0] := 5.0 ;
          FADCNumVoltageRanges := 1 ;
          FADCMinSamplingInterval := 1E-5 ;
          FADCMaxSamplingInterval := 1000 ;
          FDACMinUpdateInterval := 1E-3 ;
          FDACVoltageRange := 5.0 ;
          FDACMinUpdateInterval := 1E-3 ;
          FDIGNumOutputs := 8 ; { No. of digital outputs }
          FDIGInterval := 1 ;
          FDigMinUpdateInterval := FDACMinUpdateInterval ;
          FDigMaxUpdateInterval := 1000.0 ;
          FDACTriggerOnLevel := FDACMaxValue ;
          FDACTriggerOffLevel := 0 ;
          end ;

     // Set all channel A/D voltage ranges to default value
     for i := 0 to High(FADCChannelVoltageRanges) do
         FADCChannelVoltageRanges[i] := FADCVoltageRanges[0] ;

     end ;



procedure TSESLabIO.ADCStart ;
{ ----------------------
  Start A/D sampling
  ---------------------- }

var
   i,j : Integer ;
   WaitForExtTrigger : Boolean ;
begin

     { Fill buffer with empty flag }
     j := 0 ;
     for i := 1 to (FADCNumChannels*FADCNumSamples) div 2 do begin
        ADCBuf^[j] := EmptyFlag ;
        Inc(j) ;
        ADCBuf^[j] := -EmptyFlag ;
        Inc(j) ;
        end ;

     // Set wait for external trigger flag
     case FADCTriggerMode of
          tmFreeRun : WaitForExtTrigger := False ;
          tmWaveGen,tmExtTrigger : WaitForExtTrigger := True ;
          else WaitForExtTrigger := False ;
          end ;

     case FLabInterfaceType of
       NationalInstruments : begin
          NI_ADCToMemory( ADCBuf^, FADCNumChannels, FADCNumSamples,
                          FADCSamplingInterval,
                          FADCChannelVoltageRanges,
                          FADCTriggerMode,
                          FADCExternalTriggerActiveHigh,
                          FADCCircularBuffer,
                          FADCInputMode ) ;
          end ;

       Digidata1200 : begin
          if Win32Platform = VER_PLATFORM_WIN32_NT  then begin
             DD_ADCToMemory( ADCBuf^,
                          FADCNumChannels,
                          FADCNumSamples,
                          FADCSamplingInterval,
                          FADCVoltageRanges[FADCVoltageRangeIndex],
                          FADCTriggerMode,
                          FADCCircularBuffer ) ;
             end
          else begin
             DD98_ADCToMemory( ADCBuf^,
                          FADCNumChannels,
                          FADCNumSamples,
                          FADCSamplingInterval,
                          FADCVoltageRanges[FADCVoltageRangeIndex],
                          FADCTriggerMode,
                          FADCCircularBuffer ) ;
             end ;
          end ;

       CED1401_12, CED1401_16, CED1401_10V : begin
          CED_ADCToMemory( ADCBuf^,
                           FADCNumChannels,
                           FADCNumSamples,
                           FADCSamplingInterval,
                           FADCVoltageRanges[FADCVoltageRangeIndex],
                           FADCTriggerMode,
                           FADCExternalTriggerActiveHigh,
                           FADCCircularBuffer ) ;

          end ;

       Digidata132X : begin
          DD132X_ADCToMemory( ADCBuf^,
                              FADCNumChannels,
                              FADCNumSamples,
                              FADCSamplingInterval,
                              FADCVoltageRanges[FADCVoltageRangeIndex],
                              FADCTriggerMode,
                              FADCCircularBuffer ) ;
          end ;

       Instrutech : begin
          ITCMM_ADCToMemory( ADCBuf^,
                             FADCNumChannels,
                             FADCNumSamples,
                             FADCSamplingInterval,
                             FADCVoltageRanges[FADCVoltageRangeIndex],
                             FADCTriggerMode,
                             FADCExternalTriggerActiveHigh,
                             FADCCircularBuffer ) ;
          end ;

       ITC_16, ITC_18 : begin
          ITC_ADCToMemory( ADCBuf^,
                           FADCNumChannels,
                           FADCNumSamples,
                           FADCSamplingInterval,
                           FADCVoltageRanges[FADCVoltageRangeIndex],
                           FADCTriggerMode,
                           FADCCircularBuffer ) ;
          end ;

       VP500 : begin
          VP500_ADCToMemory( ADCBuf^, FADCNumChannels, FADCNumSamples,
                           FADCSamplingInterval,
                           FADCVoltageRanges[FADCVoltageRangeIndex],
                           FADCTriggerMode, FADCCircularBuffer ) ;
          end ;

       NIDAQMX : begin
          NIMX_ADCToMemory( ADCBuf^,
                            FADCNumChannels,
                            FADCNumSamples,
                            FADCSamplingInterval,
                            FADCChannelVoltageRanges,
                            FADCTriggerMode,
                            FADCExternalTriggerActiveHigh,
                            FADCCircularBuffer,
                            FADCInputMode ) ;
          end ;

       Digidata1440 : begin
          DD1440_ADCToMemory( ADCBuf,
                              FADCNumChannels,
                              FADCNumSamples,
                              FADCSamplingInterval,
                              FADCVoltageRanges[FADCVoltageRangeIndex],
                              FADCTriggerMode,
                              FADCCircularBuffer ) ;
          end ;

       Triton : begin
          Triton_ADCToMemory( ADCBuf^,
                              FADCNumChannels,
                              FADCNumSamples,
                              FADCSamplingInterval,
                              FADCVoltageRanges[FADCVoltageRangeIndex],
                              FADCTriggerMode,
                              FADCCircularBuffer ) ;
          end ;

       end ;
     FADCActive := True ;
     end ;


procedure TSESLabIO.ADCStop ;
{ ----------------------
  Terminate A/D sampling
  ---------------------- }
begin

     if not FADCActive then Exit ;
     
     case FLabInterfaceType of
       NationalInstruments : begin
          NI_StopADC ;
          end ;

       Digidata1200 : begin
          if Win32Platform = VER_PLATFORM_WIN32_NT  then DD_StopADC
                                                    else DD98_StopADC ;
          end ;

       CED1401_12, CED1401_16, CED1401_10V : begin
          CED_StopADC ;
          end ;

       Digidata132X : begin
          DD132X_StopADC ;
          end ;

       Instrutech : begin
          ITCMM_StopADC ;
          end ;

       VP500 : begin
          VP500_StopADC ;
          end ;

       NIDAQMX : begin
          NIMX_StopADC ;
          end ;

       Digidata1440 : begin
          DD1440_StopADC ;
          end ;

       Triton : begin
          Triton_StopADC ;
          end ;

       end ;
     FADCActive := False ;
     end ;


procedure TSESLabIO.DACStart ;
{ -------------------------
  Start D/A waveform output
  ------------------------- }
begin

     // Repeated waveform mode not available when trigger mode is tmWaveGen
     if FADCTriggerMode = tmWaveGen then FDACRepeatedWaveform := False ;

     case FLabInterfaceType of
          NationalInstruments : begin
             NI_MemoryToDAC( DACBuf^,
                             FDACNumChannels,
                             FDACNumSamples,
                             FDACUpdateInterval,
                             FDACRepeatedWaveform) ;
             end ;

          Digidata1200 : begin
             if Win32Platform = VER_PLATFORM_WIN32_NT  then begin
                DD_MemoryToDAC( DACBuf^,
                             FDACNumChannels,
                             FDACNumSamples,
                             FStimulusExtTrigger,
                             FADCSamplingInterval,
                             FADCNumChannels,
                             FADCNumSamples )  ;
                end
             else begin
                DD98_MemoryToDAC( DACBuf^,
                             FDACNumChannels,
                             FDACNumSamples,
                             FStimulusExtTrigger,
                             FADCSamplingInterval,
                             FADCNumChannels,
                             FADCNumSamples )  ;
                end ;
             end ;

          CED1401_12, CED1401_16, CED1401_10V : begin
             CED_MemoryToDAC( DACBuf^,
                              FDACNumChannels,
                              FDACNumSamples,
                              FDACUpdateInterval,
                              FADCTriggerMode,
                              FStimulusExtTrigger,
                              FDACRepeatedWaveform
                              ) ;
             end ;

          Digidata132X : begin
             DD132X_MemoryToDACAndDigitalOut( DACBuf^,
                                              FDACNumChannels,
                                              FDACNumSamples,
                                              DigBuf^,
                                              False,
                                              FStimulusExtTrigger ) ;
             end ;

          Instrutech : begin
             ITCMM_MemoryToDACAndDigitalOut( DACBuf^,
                                             FDACNumChannels,
                                             FDACNumSamples,
                                             DigBuf^,
                                             False,
                                             FStimulusExtTrigger ) ;
             end ;

       ITC_16, ITC_18 : begin
             ITC_MemoryToDACAndDigitalOut( DACBuf^,
                                           FDACNumChannels,
                                           FDACNumSamples,
                                           DigBuf^,
                                           False,
                                           FStimulusExtTrigger ) ;
             end ;

       VP500 : begin
             VP500_MemoryToDAC( DACBuf^,
                                FDACNumChannels,
                                FDACNumSamples ) ;
             end ;

       NIDAQMX : begin
             NIMX_MemoryToDAC( DACBuf^,
                               FDACNumChannels,
                               FDACNumSamples,
                               FDACUpdateInterval,
                               FStimulusExtTrigger,
                               FDACRepeatedWaveform) ;
             end ;

       Digidata1440 : begin
             DD1440_MemoryToDACAndDigitalOut( DACBuf^,
                                              FDACNumChannels,
                                              FDACNumSamples,
                                              DigBuf^,
                                              False,
                                              FStimulusExtTrigger ) ;
             end ;

       Triton : begin
             Triton_MemoryToDAC( DACBuf^,
                                 FDACNumChannels,
                                 FDACNumSamples,
                                 FDACUpdateInterval ) ;
             end ;
          end ;
     FDACActive := True ;
     end ;


procedure TSESLabIO.DACDIGStart(
          StartAt : Integer     { Start Digital O/P at sample }
          ) ;
{ -------------------------
  Start D/A waveform output
  ------------------------- }
begin

     case FLabInterfaceType of
          NationalInstruments : begin
             NI_MemoryToDig ( DIGBuf,
                              FDACNumSamples,
                              FDACUpdateInterval,
                              DigWork ) ;
             NI_MemoryToDAC( DACBuf^,
                             FDACNumChannels,
                             FDACNumSamples,
                             FDACUpdateInterval,
                             False) ;
             end ;

          Digidata1200 : begin
             if Win32Platform = VER_PLATFORM_WIN32_NT  then begin
                DD_AddDigitalWaveformToDACBuf( DIGBuf^,
                                               FDACNumChannels,
                                               FDACNumSamples) ;
                DD_MemoryToDAC( DACBuf^,
                                FDACNumChannels,
                                FDACNumSamples,
                                FStimulusExtTrigger,
                                FADCSamplingInterval,
                                FADCNumChannels,
                                FADCNumSamples )  ;
                end
             else begin
                DD98_AddDigitalWaveformToDACBuf( DIGBuf^,
                                               FDACNumChannels,
                                               FDACNumSamples) ;
                DD98_MemoryToDAC( DACBuf^,
                                FDACNumChannels,
                                FDACNumSamples,
                                FStimulusExtTrigger,
                                FADCSamplingInterval,
                                FADCNumChannels,
                                FADCNumSamples )  ;
                end ;
             end ;

          CED1401_12, CED1401_16, CED1401_10V : begin
             CED_MemoryToDigitalPort(DIGBuf^,FDACNumSamples,FDACUpdateInterval,StartAt) ;
             CED_MemoryToDAC(DACBuf^,
                             FDACNumChannels,
                             FDACNumSamples,
                             FDACUpdateInterval,
                             FADCTriggerMode,
                             FStimulusExtTrigger,
                             False) ;
             end ;

          Digidata132X : begin
             DD132X_MemoryToDACAndDigitalOut( DACBuf^,
                                              FDACNumChannels,
                                              FDACNumSamples,
                                              DigBuf^,
                                              True,
                                              FStimulusExtTrigger ) ;
             end ;

          Instrutech : begin
             ITCMM_MemoryToDACAndDigitalOut( DACBuf^,
                                             FDACNumChannels,
                                             FDACNumSamples,
                                             DigBuf^,
                                             True,
                                             FStimulusExtTrigger ) ;
             end ;

          ITC_16, ITC_18 : begin
             ITC_MemoryToDACAndDigitalOut( DACBuf^,
                                           FDACNumChannels,
                                           FDACNumSamples,
                                           DigBuf^,
                                           True,
                                           FStimulusExtTrigger ) ;
             end ;

          VP500 : begin
             VP500_MemoryToDAC( DACBuf^,
                                FDACNumChannels,
                                FDACNumSamples ) ;
             end ;

          NIDAQMX : begin
             NIMX_MemoryToDig ( DIGBuf^,
                                FDACNumSamples,
                                FDACUpdateInterval,
                                FDACRepeatedWaveform ) ;
             NIMX_MemoryToDAC( DACBuf^,
                               FDACNumChannels,
                               FDACNumSamples,
                               FDACUpdateInterval,
                               FStimulusExtTrigger,
                               FDACRepeatedWaveform ) ;
             end ;

          Digidata1440 : begin
             DD1440_MemoryToDACAndDigitalOut( DACBuf^,
                                              FDACNumChannels,
                                              FDACNumSamples,
                                              DigBuf^,
                                              True,
                                              FStimulusExtTrigger ) ;
             end ;

          Triton : begin
             Triton_MemoryToDAC( DACBuf^,
                                 FDACNumChannels,
                                 FDACNumSamples,
                                 FDACUpdateInterval ) ;
             end ;


          end ;
     FDACActive := True ;
     FDIGActive := True ;
     end ;



procedure TSESLabIO.DACStop ;
{ ----------------------
  Terminate D/A update
  ---------------------- }
begin

     if not FDACActive then Exit ;

     case FLabInterfaceType of
       NationalInstruments : begin
          NI_StopDAC ;
          if FDIGActive then NI_StopDig ;
          end ;

       Digidata1200 : begin
          if Win32Platform = VER_PLATFORM_WIN32_NT  then DD_StopDAC
                                                    else DD98_StopDAC ;
          end ;

       CED1401_12, CED1401_16, CED1401_10V : begin
          CED_StopDAC ;
          if FDIGActive then CED_StopDig ;
          end ;

       Digidata132X : begin
          DD132X_StopDAC ;
          end ;

       Instrutech : begin
          ITCMM_StopDAC ;
          end ;

       ITC_16, ITC_18 : begin
          ITC_StopDAC ;
          end ;

       VP500 : begin
          VP500_StopDAC ;
          end ;

       NIDAQMX : begin
          NIMX_StopDAC ;
          if FDIGActive then NIMX_StopDig ;
          end ;

       Digidata1440 : begin
          DD1440_StopDAC ;
          end ;

       Triton : begin
          Triton_StopDAC ;
          end ;

       end ;
     FDACActive := False ;
     FDIGActive := False ;
     end ;


function TSESLabIO.ReadADC(
         Channel : Integer
         ) : Integer ;
{ ----------------------------
  Read A/D on selected channel
  ---------------------------- }
begin
     case FLabInterfaceType of
       NationalInstruments : begin
          Result := NI_ReadADC( Channel,
                                FADCChannelVoltageRanges[Channel],
                                FADCInputMode ) ;
          end ;
       Digidata1200 : begin
          if Win32Platform = VER_PLATFORM_WIN32_NT  then begin
             Result := DD_ReadADC( Channel, FADCVoltageRanges[FADCVoltageRangeIndex]) ;
             end
          else begin
             Result := DD98_ReadADC( Channel, FADCVoltageRanges[FADCVoltageRangeIndex]) ;
             end ;
          end ;
       CED1401_12, CED1401_16, CED1401_10V : begin
          Result := CED_ReadADC( Channel ) ;
          end ;
       Digidata132X : begin
          Result := DD132X_ReadADC( Channel ) ;
          end ;

       Instrutech : begin
          Result := ITCMM_ReadADC( Channel ) ;
          end ;

       ITC_16, ITC_18 : begin
          Result := ITC_ReadADC( Channel,
                                 FADCVoltageRanges[FADCVoltageRangeIndex]) ;
          end ;
       VP500 : begin
          Result := VP500_ReadADC( Channel ) ;
          end ;
       NIDAQMX : begin
          Result := NIMX_ReadADC( Channel,
                                  FADCChannelVoltageRanges[Channel],
                                  FADCInputMode ) ;
          end ;

       Digidata1440 : begin
          Result := DD1440_ReadADC( Channel ) ;
          end ;

       Triton : begin
          Result := Triton_ReadADC( Channel ) ;
          end ;

       else Result := 0 ;
       end ;
     end ;


procedure TSESLabIO.WriteDACs(
          DACVolts : Array of single ;
          NumChannels : Integer ) ;
{ -----------------------
  Write to D/A converters
  ----------------------- }
var
   ch : Integer ;
begin

     { Retain DAC values }
     for ch := 0 to NumChannels-1 do FLastDACVolts[ch] := DACVolts[ch] ;
     FLastDACNumChannels := NumChannels ;

     case FLabInterfaceType of
       NationalInstruments : begin
          NI_WriteDACs( DACVolts, NumChannels ) ;
          end ;
       Digidata1200 : begin
          if Win32Platform = VER_PLATFORM_WIN32_NT  then begin
             DD_WriteDACsAndDigitalPort(FLastDACVolts,FLastDACNumChannels,FLastDigValue) ;
             end
          else begin
             DD98_WriteDACsAndDigitalPort(FLastDACVolts,FLastDACNumChannels,FLastDigValue) ;
             end ;
          end ;
       CED1401_12, CED1401_16, CED1401_10V : begin
          CED_WriteDACs( DACVolts, NumChannels ) ;
          end ;
       Digidata132X : begin
          DD132X_WriteDACsAndDigitalPort(FLastDACVolts,FLastDACNumChannels,FLastDigValue) ;
          end ;
       Instrutech : begin
          ITCMM_WriteDACsAndDigitalPort(FLastDACVolts,FLastDACNumChannels,FLastDigValue) ;
          end ;
       ITC_16, ITC_18 : begin
          ITC_WriteDACsAndDigitalPort(FLastDACVolts,FLastDACNumChannels,FLastDigValue) ;
          end ;

       VP500 : begin
          VP500_WriteDACsAndDigitalPort(FLastDACVolts,FLastDACNumChannels,FLastDigValue) ;
          end ;
       NIDAQMX : begin
          NIMX_WriteDACs( DACVolts, NumChannels ) ;
          end ;

       Digidata1440 : begin
          DD1440_WriteDACsAndDigitalPort(FLastDACVolts,FLastDACNumChannels,FLastDigValue) ;
          end ;

       Triton: begin
          Triton_WriteDACsAndDigitalPort(FLastDACVolts,FLastDACNumChannels,FLastDigValue) ;
          end ;

       end ;
     end ;


procedure TSESLabIO.GetADCBuffer(
          var BufPointer : PSmallIntArray  { Pointer to A/D data buffer }
          ) ;
{ ---------------------------------------------------
  Return pointer to start of internal A/D data buffer
  --------------------------------------------------- }
begin
     BufPointer := ADCBuf ;
     end ;


procedure TSESLabIO.GetDACBuffer(
          var BufPointer : PSmallIntArray
          ) ;
{ ---------------------------------------------------
  Return pointer to start of internal D/A data buffer
  --------------------------------------------------- }
begin
     BufPointer := DACBuf ;
     end ;


procedure TSESLabIO.GetDIGBuffer(
          var BufPointer : PSmallIntArray
          ) ;
{ -------------------------------------------------------
  Return pointer to start of internal digital data buffer
  ------------------------------------------------------- }
begin
     BufPointer := DigBuf ;
     end ;


procedure TSESLabIO.ADCBufferRefresh ;
{ -------------------------------------------------------
  Return pointer to start of internal digital data buffer
  ------------------------------------------------------- }
begin


   if FTimerProcInUse then Exit ;

   if FStimulusStartTime = StimulusExtTriggerFlag then begin
      // Stimulus triggered by 5V/10ms trigger pulse
      StimulusStartFlag := ExternalStimulusTrigger ;
      end ;

   case FLabInterfaceType of
       NationalInstruments : begin
          end ;
       Digidata1200 : begin
          if Win32Platform = VER_PLATFORM_WIN32_NT  then begin
             DD_GetADCSamples( ADCBuf^, FOutPointer ) ;
             end
          else begin
             DD98_GetADCSamples( ADCBuf^, FOutPointer ) ;
             end ;
          end ;
       CED1401_12, CED1401_16, CED1401_10V : begin
          if FADCActive then CED_GetADCSamples( ADCBuf^, FOutPointer ) ;
          end ;
       Digidata132X : begin
          DD132X_GetADCSamples( ADCBuf^, FOutPointer ) ;
          end ;
       Instrutech : begin
          ITCMM_GetADCSamples( ADCBuf, FOutPointer ) ;
          end ;
       ITC_16, ITC_18 : begin
          ITC_GetADCSamples( ADCBuf, FOutPointer ) ;
          end ;
       VP500 : begin
          VP500_GetADCSamples( ADCBuf, FOutPointer ) ;
          end ;
       NIDAQMX : begin
          NIMX_GetADCSamples( ADCBuf^, FOutPointer ) ;
          end ;
       Digidata1440 : begin
          DD1440_GetADCSamples( ADCBuf^, FOutPointer ) ;
          end ;

       Triton : begin
          Triton_GetADCSamples( ADCBuf^, FOutPointer ) ;
          end ;
       end ;

     end ;


procedure TSESLabIO.GetLabInterfaceTypes(
          List : TStrings
          ) ;
{ ---------------------------------------------
  Return list of names of laboratory interfaces
  --------------------------------------------- }
var
   i : Integer ;
begin
     List.Clear ;
     for i := 0 to NumLabInterfaceTypes-1 do begin
         List.Add(GetLabInterfaceName(i)) ;
         end ;
     end ;


procedure TSESLabIO.GetADCInputModes(
          List : TStrings
          ) ;
{ ---------------------------------------------
  Return list of names of A/D input modes
  --------------------------------------------- }
var
   i : Integer ;
begin
     List.Clear ;
     case FLabInterfaceType of
       NationalInstruments,NIDAQMX : begin
         List.Add(' Single Ended (NRSE)' ) ;
         List.Add(' Differential' ) ;
         List.Add(' BNC-2110 (Diff)' ) ;
         List.Add(' BNC-2090 (SE)' ) ;
         List.Add(' Single Ended (RSE)') ;
         end ;
       else begin
         List.Add(' Single Ended' ) ;
         end ;
       end ;
     end ;



procedure TSESLabIO.WriteDig(
          DigByte : Integer
          ) ;
{ -------------------------
  Write to digital O/P port
  ------------------------- }
begin
     { Keep digital O/P bits }
     FLastDigValue := DigByte ;

     case FLabInterfaceType of
       NationalInstruments : begin
          NI_WriteToDigitalOutPutPort( DigByte ) ;
          end ;
       Digidata1200 : begin
          if Win32Platform = VER_PLATFORM_WIN32_NT  then begin
             DD_WriteDACsAndDigitalPort( FLastDACVolts, FLastDACNumChannels, DigByte ) ;
             end
          else begin
             DD98_WriteDACsAndDigitalPort( FLastDACVolts, FLastDACNumChannels, DigByte ) ;
             end ;
          end ;
       CED1401_12, CED1401_16, CED1401_10V : begin
          CED_WriteToDigitalOutPutPort( DigByte ) ;
          end ;
       Digidata132X : begin
          DD132X_WriteDACsAndDigitalPort( FLastDACVolts, FLastDACNumChannels, DigByte ) ;
          end ;
       Instrutech : begin
          ITCMM_WriteDACsAndDigitalPort( FLastDACVolts, FLastDACNumChannels, DigByte ) ;
          end ;
       ITC_16, ITC_18 : begin
          ITC_WriteDACsAndDigitalPort( FLastDACVolts, FLastDACNumChannels, DigByte ) ;
          end ;
       VP500 : begin
          VP500_WriteDACsAndDigitalPort( FLastDACVolts, FLastDACNumChannels, DigByte ) ;
          end ;
       NIDAQMX : begin
          NIMX_WriteToDigitalOutPutPort( DigByte ) ;
          end ;
       Digidata1440 : begin
          DD1440_WriteDACsAndDigitalPort( FLastDACVolts, FLastDACNumChannels, DigByte ) ;
          end ;
       Triton : begin
          Triton_WriteDACsAndDigitalPort( FLastDACVolts, FLastDACNumChannels, DigByte ) ;
          end ;

       end ;
     end ;


procedure TSESLabIO.SetADCNumChannels( Value : Integer ) ;
// ---------------------------------
// Set the number A/D input channels
// ---------------------------------
begin
     FADCNumChannels := IntLimit(Value,1,FADCMaxChannels) ;

     // Ensure sampling interval remains valid
     SetADCSamplingInterval( FADCSamplingInterval ) ;

     { Create array of integer offsets for location of each A/D channel
       within sample data block }
     case FLabInterfaceType of
       NationalInstruments : begin
          NI_GetChannelOffsets( FADCChannelOffsets, FADCNumChannels ) ;
          end ;
       Digidata1200 : begin
          if Win32Platform = VER_PLATFORM_WIN32_NT  then begin
             DD_GetChannelOffsets( FADCChannelOffsets, FADCNumChannels ) ;
             end
          else begin
             DD98_GetChannelOffsets( FADCChannelOffsets, FADCNumChannels ) ;
             end ;
          end ;
       CED1401_12, CED1401_16, CED1401_10V : begin
          CED_GetChannelOffsets( FADCChannelOffsets, FADCNumChannels ) ;
          end ;
       Digidata132X : begin
          DD132X_GetChannelOffsets( FADCChannelOffsets, FADCNumChannels ) ;
          end ;
       Instrutech : begin
          ITCMM_GetChannelOffsets( FADCChannelOffsets, FADCNumChannels ) ;
          end ;
       ITC_16, ITC_18 : begin
          ITC_GetChannelOffsets( FADCChannelOffsets, FADCNumChannels ) ;
          end ;
       VP500 : begin
          VP500_GetChannelOffsets( FADCChannelOffsets, FADCNumChannels ) ;
          end ;
       NIDAQMX : begin
          NIMX_GetChannelOffsets( FADCChannelOffsets, FADCNumChannels ) ;
          end ;
       Digidata1440 : begin
          DD1440_GetChannelOffsets( FADCChannelOffsets, FADCNumChannels ) ;
          end ;
       Triton : begin
          Triton_GetChannelOffsets( FADCChannelOffsets, FADCNumChannels ) ;
          end ;

       end ;

     end ;


procedure TSESLabIO.SetADCNumSamples( Value : Integer ) ;
// -------------------------------------------------
// Set no. of A/D samples per channel to be acquired
// -------------------------------------------------
begin
     FADCNumSamples := IntLimit(Value,1,FADCBufferLimit div FADCNumChannels) ;
     end ;


procedure TSESLabIO.SetADCSamplingInterval( Value : Double ) ;
// -------------------------
// Set A/D sampling interval
// -------------------------
var
   TimeBase : SmallInt ;
   PreScale,ClockTicks,FrequencySource : Word ;
   DD_Ticks : Cardinal ;
begin

     FADCSamplingInterval := Value ;

     // Keep within valid hardware limits
 //    FADCSamplingInterval := FADCNumChannels*FloatLimit( FADCSamplingInterval/FADCNumChannels,
 //                                                        FADCMinSamplingInterval,
//                                                         FADCMaxSamplingInterval) ;

     case FLabInterfaceType of
       NationalInstruments : begin
          FADCSamplingInterval := FADCSamplingInterval / FADCNumChannels ;
          FADCSamplingInterval := Min( Max( FADCSamplingInterval,
                                            FADCMinSamplingInterval),
                                            FADCMaxSamplingInterval) ;
          NI_CheckSamplingInterval(FADCSamplingInterval,TimeBase,ClockTicks);
          FADCSamplingInterval := FADCSamplingInterval * FADCNumChannels ;
          end ;
       Digidata1200 : begin
          FADCSamplingInterval := FADCSamplingInterval / FADCNumChannels ;
          FADCSamplingInterval := Min( Max( FADCSamplingInterval,
                                            FADCMinSamplingInterval),
                                            FADCMaxSamplingInterval) ;
          if Win32Platform = VER_PLATFORM_WIN32_NT  then begin
             DD_CheckSamplingInterval(FADCSamplingInterval,DD_Ticks,FrequencySource);
             end
          else begin
             DD98_CheckSamplingInterval(FADCSamplingInterval,DD_Ticks,FrequencySource);
             end ;
          FADCSamplingInterval := FADCSamplingInterval * FADCNumChannels ;
          end ;
       CED1401_12, CED1401_16, CED1401_10V : begin
          FADCSamplingInterval := FADCSamplingInterval / FADCNumChannels ;
          FADCSamplingInterval := Min( Max( FADCSamplingInterval,
                                            FADCMinSamplingInterval),
                                            FADCMaxSamplingInterval) ;
          CED_CheckSamplingInterval(FADCSamplingInterval,PreScale,ClockTicks);
          FADCSamplingInterval := FADCSamplingInterval * FADCNumChannels ;
          end ;
       Digidata132X : begin
          FADCSamplingInterval := FADCSamplingInterval / FADCNumChannels ;
          FADCSamplingInterval := Min( Max( FADCSamplingInterval,
                                            FADCMinSamplingInterval),
                                            FADCMaxSamplingInterval) ;
          DD132X_CheckSamplingInterval(FADCSamplingInterval,DD_Ticks);
          FADCSamplingInterval := FADCSamplingInterval * FADCNumChannels ;
          end ;
       Instrutech : begin
          ITCMM_CheckSamplingInterval(FADCSamplingInterval,DD_Ticks);
          end ;
       ITC_16, ITC_18 : begin
          FADCSamplingInterval := FADCSamplingInterval / FADCNumChannels ;
          FADCSamplingInterval := Min( Max( FADCSamplingInterval,
                                            FADCMinSamplingInterval),
                                            FADCMaxSamplingInterval) ;
          ITC_CheckSamplingInterval(FADCSamplingInterval,DD_Ticks);
          FADCSamplingInterval := FADCSamplingInterval * FADCNumChannels ;
          end ;
       VP500 : begin
          FADCSamplingInterval := Min( Max( FADCSamplingInterval,
                                            FADCMinSamplingInterval),
                                            FADCMaxSamplingInterval) ;
          VP500_CheckSamplingInterval(FADCSamplingInterval);
          end ;
       NIDAQMX : begin
          FADCSamplingInterval := Min( Max( FADCSamplingInterval,
                                            FADCMinSamplingInterval*FADCNumChannels),
                                            FADCMaxSamplingInterval*FADCNumChannels) ;
          NIMX_CheckADCSamplingInterval(FADCSamplingInterval,FADCNumChannels,FADCInputMode);
          end ;
       Digidata1440 : begin
          FADCSamplingInterval := Min( Max( FADCSamplingInterval,
                                            FADCMinSamplingInterval),
                                            FADCMaxSamplingInterval) ;
          DD1440_CheckSamplingInterval(FADCSamplingInterval);
          end ;
       Triton : begin
          FADCSamplingInterval := Min( Max( FADCSamplingInterval,
                                            FADCMinSamplingInterval),
                                            FADCMaxSamplingInterval) ;
          Triton_CheckSamplingInterval(FADCSamplingInterval);
          end ;

       end ;

     end ;


procedure TSESLabIO.SetADCVoltageRangeIndex( Value : Integer ) ;
// ------------------------------
// Set A/D voltage range by index
// ------------------------------
var
   i : Integer ;
begin

     FADCVoltageRangeIndex := IntLimit(Value,0,FADCNumVoltageRanges-1) ;

     // Set all channels to this value
     for i := 0 to High(FADCChannelVoltageRanges) do
         FADCChannelVoltageRanges[i] := FADCVoltageRanges[FADCVoltageRangeIndex] ;

     end ;


procedure TSESLabIO.SetADCVoltageRange( Value : Single ) ;
// ---------------------------------------------------------
// Set A/D input voltage range of all channels to same value
// ---------------------------------------------------------
var
   i : Integer ;
begin

     // Get nearest valid range
     for i := 0 to FADCNumVoltageRanges-1 do begin
         if Abs(FADCVoltageRanges[i] - Value) <  0.01 then FADCVoltageRangeIndex := i ;
         end ;

     // Set all channels to this value
     for i := 0 to High(FADCChannelVoltageRanges) do
         FADCChannelVoltageRanges[i] := FADCVoltageRanges[FADCVoltageRangeIndex] ;

     end ;


function TSESLabIO.GetADCVoltageRange : Single ;
// ----------------------------
// Read A/D input voltage range
// ----------------------------
begin
     Result := FADCVoltageRanges[FADCVoltageRangeIndex] ;
     end ;


function TSESLabIO.GetADCChannelOffsets(
         Channel : Integer
         ) : Integer ;
{ --------------------------------------------------------------------------
  Get offset within multi-channel sample data block for selected A/D channel
  -------------------------------------------------------------------------- }
begin
     Channel := IntLimit( Channel, 0, High(FADCChannelOffsets) ) ;
     Result := FADCChannelOffsets[Channel] ;
     end ;


function TSESLabIO.GetADCChannelVoltageRange(
         Channel : Integer
         ) : Single ;
{ -------------------------------------------------
  Get A/D input voltage range for selected channel
  ------------------------------------------------- }
begin
     Channel := IntLimit( Channel, 0, High(FADCChannelVoltageRanges) ) ;
     Result := FADCChannelVoltageRanges[Channel] ;
     end ;


procedure TSESLabIO.SetADCChannelVoltageRange(
         Channel : Integer ;                   // Channel to be updated
         Value : Single                        // New A/D voltage range
         ) ;
{ -------------------------------------------------
  Set A/D input voltage range for selected channel
  ------------------------------------------------- }
var
    i : Integer ;
begin

     // Get nearest valid range
     for i := 0 to FADCNumVoltageRanges-1 do begin
         if Abs(FADCVoltageRanges[i] - Value) <  0.01 then FADCVoltageRangeIndex := i ;
         end ;

     // Set selected A/D channel
     Channel := IntLimit( Channel, 0, High(FADCChannelVoltageRanges) ) ;
     FADCChannelVoltageRanges[Channel] := FADCVoltageRanges[FADCVoltageRangeIndex] ;

     end ;


procedure TSESLabIO.SetADCInputMode( Value : Integer ) ;
// ------------------
// Set A/D input mode
// ------------------
begin
     case FLabInterfaceType of
        NationalInstruments,NIDAQMX : FADCInputMode := Value ;
        else FADCInputMode := imSingleEnded ;
        end ;
     end ;


procedure TSESLabIO.SetDACNumChannels( Value : Integer ) ;
// -----------------------------------------------------
// Set the number of D/A output channels in the waveform
// -----------------------------------------------------
begin
     FDACNumChannels := IntLimit(Value,1,FDACMaxChannels) ;
     end ;


procedure TSESLabIO.SetDACNumSamples( Value : Integer ) ;
// ------------------------------------------------------------------
// Set the number of D/A output samples (per channel) in the waveform
// ------------------------------------------------------------------
begin
     FDACNumSamples := IntLimit(Value,1,FDACBufferLimit div FDACNumChannels) ;
     end ;


procedure TSESLabIO.SetDACUpdateInterval( Value : Double ) ;
// ------------------------------------
// Set the D/A waveform update interval
// ------------------------------------
var
   TimeBase : SmallInt ;
   PreScale,ClockTicks,FrequencySource : Word ;
   dd_Ticks : Cardinal ;
begin

     FDACUpdateInterval := Value ;

     case FLabInterfaceType of
       NationalInstruments : begin
          // NOTE. This code ensures that D/A update interval is fixed at 10 ms
          // when a digital stimulus pattern is being outputed using an E-series board 17.7.02
          if FStimulusDigitalEnabled then begin
             FDACUpdateInterval := FloatLimit( FDACUpdateInterval,
                                               FDigMinUpdateInterval,
                                               FDigMaxUpdateInterval) ;
             end
          else
             NI_CheckSamplingInterval(FDACUpdateInterval,TimeBase,ClockTicks) ;
          end ;
       Digidata1200 : begin
          FDACUpdateInterval := (FADCSamplingInterval*FDACNumChannels)/FADCNumChannels ;
          //DD_CheckSamplingInterval(FDACUpdateInterval,DD_Ticks,FrequencySource);
          end ;
       CED1401_12, CED1401_16, CED1401_10V : begin
          CED_CheckSamplingInterval(FDACUpdateInterval,PreScale,ClockTicks);
          end ;
       Digidata132X : begin
          FDACUpdateInterval := FADCSamplingInterval ;
          if FStimulusDigitalEnabled then begin
             FDACUpdateInterval := (FADCSamplingInterval*(FDACNumChannels+1))/FADCNumChannels ;
             end
          else begin
             FDACUpdateInterval := (FADCSamplingInterval*FDACNumChannels)/FADCNumChannels ;
             end ;
          end ;
       Instrutech : begin
          FDACUpdateInterval := FADCSamplingInterval ;
          end ;
       ITC_16, ITC_18 : begin
          FDACUpdateInterval := FADCSamplingInterval ;
          end ;
       VP500 : begin
          FDACUpdateInterval := FADCSamplingInterval ;
          end ;
       NIDAQMX : begin
          NIMX_CheckDACSamplingInterval(FDACUpdateInterval, FDACNumChannels ) ;
          end ;
       Digidata1440 : begin
          FDACUpdateInterval := FADCSamplingInterval ;
     {     if FStimulusDigitalEnabled then begin
             FDACUpdateInterval := (FADCSamplingInterval*(FDACNumChannels+1))/FADCNumChannels ;
             end
          else begin
             FDACUpdateInterval := (FADCSamplingInterval*FDACNumChannels)/FADCNumChannels ;
             end ;      }
          end ;

       Triton : begin
          FDACUpdateInterval := FADCSamplingInterval ;
          end ;

       end ;
     end ;


function TSESLabIO.GetDIGInputs : Integer ;
// ---------------------------
// Get digital input port bits
// ---------------------------
var
     Bits : Integer ;
begin

     Bits := 0 ;

     case FLabInterfaceType of
       NationalInstruments : begin
          Bits := NI_ReadDigitalInputPort ;
          end ;
       Digidata1200 : begin
          if Win32Platform = VER_PLATFORM_WIN32_NT  then Bits := DD_ReadDigitalInputPort
                                                    else Bits := DD98_ReadDigitalInputPort ;
          end ;
       CED1401_12, CED1401_16, CED1401_10V : begin
          Bits := CED_ReadDigitalInputPort ;
          end ;
       Digidata132X : begin
          end ;
       Instrutech : begin
          end ;
       ITC_16, ITC_18 : begin
          end ;
       VP500 : begin
          end ;
       NIDAQMX : begin
          Bits := NIMX_ReadDigitalInputPort ;
          end ;
       Digidata1440 : begin
          end ;

       Triton : begin
          end ;
          
       end ;
     Result := Bits ;
     end ;


procedure TSESLabIO.SetStimulusTimerPeriod(
          Value : Single // Timer interval (s)
          ) ;
// -----------------------------------------
// Set time interval between stimulus pulses
// -----------------------------------------
var
    OldStimulusStartTime : Integer ;
begin

     // Note. Value = StimulusExtTriggerFlag indicates that the start of the
     // stimulus waveform is triggered by a digital input pulse
     if Value = StimulusExtTriggerFlag then begin
        FStimulusStartTime := Round(Value) ;
        FStimulusExtTrigger := True ;
        case FLabInterfaceType of
             NationalInstruments : NI_ArmStimulusTriggerInput ;
             CED1401_12, CED1401_16, CED1401_10V : CED_ArmStimulusTriggerInput ;
             //NIDAQMX : NIMX_ArmStimulusTriggerInput ;
             end ;
        end
     else begin
        // Timed stimulus start
        OldStimulusStartTime := FStimulusStartTime ;
        FStimulusStartTime := Round((Value*1E3)/StimulusTimerTickInterval) ;
        // Clear timer tick counter if timer period has changed
        // (This ensures that a full inter-pulse period occurs
        //  when the inter record interval is changed 20.08.06)
        if FStimulusStartTime <> OldStimulusStartTime then begin
           FStimulusTimerTicks := 0 ;
           end ;

        FStimulusExtTrigger := False ;
        end ;
     StimulusStartFlag := False ;
     end ;


function TSESLabIO.GetStimulusTimerPeriod : Single ;
// -----------------------------------------
// Get time interval between stimulus pulses
// -----------------------------------------
begin
     Result := FStimulusStartTime*StimulusTimerTickInterval*1E-3 ;
     end ;


procedure TSESLabIO.StartTimer ;
// ------------------------------
// Start periodic stimulus timer
// -----------------------------
begin

     if FStimulusTimerActive then begin
        timeKillEvent( FStimulusTimerID ) ;
        FStimulusTimerActive := False ;
        end ;

     // Set to maximum value to force immediate stimulus pulse
     FStimulusTimerTicks := High(FStimulusTimerTicks) ;

     // Start clock
     FStimulusTimerID := TimeSetEvent( StimulusTimerTickInterval,
                                       StimulusTimerTickInterval,
                                       @TimerProc,
                                       Cardinal(Self),
                                       TIME_PERIODIC ) ;

     FStimulusTimerActive := True ;
     end ;


procedure TSESLabIO.StopTimer ;
// ----------------------------
// Stop periodic stimulus timer
// ----------------------------
begin

     if FStimulusTimerActive then begin
        timeKillEvent( FStimulusTimerID ) ;
        FStimulusTimerActive := False ;
        end ;

     // Ensure external stimulus trigger flag is cleared
     FStimulusExtTrigger := False ;

     end ;


procedure TSESLabIO.TimerTickOperations ;
// ---------------------------------------------
// Operations to be carried out every timer tick
// ---------------------------------------------
begin
     case FLabInterfaceType of
       NationalInstruments : begin
          NI_UpdateDigOutput ;
          end ;
       end ;

     end ;


function TSESLabIO.ExternalStimulusTrigger : Boolean ;
// -----------------------------------------
// Read external stimulus trigger line state
// -----------------------------------------
var
     Triggered : Boolean ;
begin

     Triggered := False ;
     case FLabInterfaceType of
       NationalInstruments : begin
          Triggered := NI_StimulusTriggerInputState ;
          end ;
       Digidata1200 : begin
          // Bit 0 of digital I/P port used as trigger
          // (Note bit 0 must be high for more than 10 ms)
          //if (DD_ReadDigitalInputPort and 1) <> 0 then Triggered := True ;
          Triggered := True ;
          end ;
       CED1401_12, CED1401_16, CED1401_10V : begin
          Triggered := CED_StimulusTriggerInputState ;
          end ;
       Digidata132X : begin
          Triggered := True ;         // Triggering handled by DD132X_MemoryToDAC
          end ;
       Instrutech : begin
          Triggered := True ;         // Triggering handled by ITCMM_MemoryToDAC
          end ;
       ITC_16, ITC_18 : begin
          Triggered := True ;         // Triggering handled by ITC16_MemoryToDAC
          end ;

       VP500 : begin
          Triggered := True ;         // Not implemented
          end ;

       NIDAQMX : begin
          //Triggered := NIMX_StimulusTriggerInputState ;
          Triggered := True ;
          end ;

       Digidata1440 : begin
          Triggered := True ;         // Triggering handled by DD132X_MemoryToDAC
          end ;

       Triton : begin
          Triggered := True ;
          end ;
          
       end ;

     Result := Triggered ;
     end ;



procedure TimerProc(
          uID,uMsg : SmallInt ; User : TSESLabIO ; dw1,dw2 : LongInt ) ; stdcall ;
{ ----------------------------------------------
  Timer scheduled events, called a 10ms intervals
  ---------------------------------------------- }
begin

   if User.FTimerProcInUse then Exit ;
   User.FTimerProcInUse := True ;


   if User.FStimulusStartTime <> StimulusExtTriggerFlag then begin
      // Stimulus output at timed intervals
      // ----------------------------------
      if (User.FStimulusTimerTicks >= User.FStimulusStartTime) then begin
         // Save time that this stimulus pulse started
         User.FStimulusTimeStarted := GetTickCount*0.001 ;
         { Reset counter for next sweep }
         User.FStimulusTimerTicks := 0 ;
         User.StimulusStartFlag := True ;
         end
      else begin
         { Increment voltage program inter-pulse timer (10ms ticks) }
         Inc(User.FStimulusTimerTicks) ;
         end ;
      end ;

   // Call operations to be run on each timer tick
   User.TimerTickOperations ;

   User.FTimerProcInUse := False ;

   end ;


procedure TSESLabIO.StartStimulus ;
// -----------------------------------------
// Start D/A and digital waveform generation
// -----------------------------------------
begin

      if StimulusStartFlag = False then Exit ;

      if FStimulusDigitalEnabled then DACDIGStart( FStimulusStartOffset )
      else DACStart ;

      StimulusStartFlag := False ;

      end ;


function  TSESLabIO.GetDACTriggerOnLevel : Integer ;
// ------------------------------------------------------
// Returns DAC ON level for recording sweep trigger pulse
// ------------------------------------------------------
begin
     if FDACInvertTriggerLevel then Result := FDACTriggerOffLevel
                               else Result := FDACTriggerOnLevel ;
     end ;


function  TSESLabIO.GetDACTriggerOffLevel : Integer ;
// ------------------------------------------------------
// Returns DAC OFF level for recording sweep trigger pulse
// ------------------------------------------------------
begin
     if FDACInvertTriggerLevel then Result := FDACTriggerOnLevel
                               else Result := FDACTriggerOffLevel ;
     end ;


procedure TSESLabIO.SetADCExternalTriggerActiveHigh(
          Value : Boolean ) ;
// ------------------------------------------------------------------------
// Set External Trigger input Active High (Value=TRUE) or Low (Value=False)
// ------------------------------------------------------------------------
begin

     case FLabInterfaceType of
       NationalInstruments : begin
          FADCExternalTriggerActiveHigh := NI_GetValidExternalTriggerPolarity(Value) ;
          end ;
       Digidata1200 : begin
          FADCExternalTriggerActiveHigh := True ;
          end ;
       CED1401_12, CED1401_16, CED1401_10V : begin
          FADCExternalTriggerActiveHigh := Value ;
          end ;
       Digidata132X : begin
          FADCExternalTriggerActiveHigh := True ;
          end ;
       Instrutech : begin
          FADCExternalTriggerActiveHigh := Value ;
          end ;
       ITC_16, ITC_18 : begin
          FADCExternalTriggerActiveHigh := True ;
          end ;

       VP500 : begin
          FADCExternalTriggerActiveHigh := True ;
          end ;

       NIDAQMX : begin
          FADCExternalTriggerActiveHigh := NIMX_GetValidExternalTriggerPolarity(Value) ;
          end ;
       Digidata1440 : begin
          FADCExternalTriggerActiveHigh := True ;
          end ;

       Triton : begin
          FADCExternalTriggerActiveHigh := True ;
          end ;

       end ;
     end ;


function  TSESLabIO.IntLimit(
          Value : Integer ;
          LoLimit : Integer ;
          HiLimit : Integer
          ) : Integer ;
{ -------------------------------------------------------------
  Return integer Value constrained within range LoLimit-HiLimit
  ------------------------------------------------------------- }
begin
     if Value > HiLimit then Value := HiLimit ;
     if Value < LoLimit then Value := LoLimit ;
     Result := Value ;
     end ;


function  TSESLabIO.FloatLimit(
          Value : Single ;
          LoLimit : Single ;
          HiLimit : Single
          ) : Single ;
{ --------------------------------------------------------------------
  Return floating point Value constrained within range LoLimit-HiLimit
  -------------------------------------------------------------------- }
begin
     if Value > HiLimit then Value := HiLimit ;
     if Value < LoLimit then Value := LoLimit ;
     Result := Value ;
     end ;

procedure TSESLabIO.DisableDMA_LabPC ;
begin
     case FLabInterfaceType of
       NationalInstruments : NI_DisableDMA_LabPC ;
       end ;
     end ;

procedure TSESLabIO.TritonGetReg(
          Reg : Integer ;
          Chan : Integer ;
          var Value : Single ;
          var PercentValue : Single ;
          var Enabled : Boolean ) ;
var
    dbVal,dbPercentVal : Double ;
begin
     dbVal := 0.0 ;
     dbPercentVal := 0.0 ;
     case FLabInterfaceType of
       Triton : TritonGetRegister( Reg, Chan, dbVal, dbPercentVal, Enabled ) ;
       end ;
     Value := dbVal ;
     PercentValue := dbPercentVal ;

     end ;


procedure TSESLabIO.TritonSetRegPercent(
          Reg : Integer ;
          Chan : Integer ;
          var PercentValue : Single ) ;
var
    DbValue : Double ;
begin
     DbValue := PercentValue ;
     case FLabInterfaceType of
       Triton : TritonSetRegisterPercent( Reg, Chan, DbValue ) ;
       end ;
     end ;


function TSESLabIO.TritonGetRegEnabled(
          Reg : Integer ;
          Chan : Integer ) : Boolean ;
// ---------------------------------------------
// Get enabled/disabled state of Triton register
// ---------------------------------------------
begin
     case FLabInterfaceType of
       Triton : Result := TritonGetRegisterEnabled( Reg, Chan ) ;
       else Result := False ;
       end ;
     end ;


procedure TSESLabIO.TritonSetRegEnabled(
              Reg : Integer ;
              Chan : Integer ;
              Enabled : Boolean ) ;
// ---------------------------------------------
// Set enabled/disabled state of Triton register
// ---------------------------------------------
begin
     case FLabInterfaceType of
       Triton : TritonSetRegisterEnabled( Reg, Chan, Enabled ) ;
       end ;
     end ;

procedure TSESLabIO.SetTritonSource(
         Chan : Integer ;                   // Channel to be updated
         Value : Integer                       // New source
         ) ;
// -------------------------------------------------
//  Set Triton patch clamp input source for selected channel
// -------------------------------------------------
begin
     case FLabInterfaceType of
       Triton : TritonSetSource( Chan, Value ) ;
       end ;
     end ;


function TSESLabIO.GetTritonSource(
         Chan : Integer                        // New source
         ) : Integer ;
// -------------------------------------------------
//  Get Triton patch clamp input source for selected channel
// -------------------------------------------------
begin
     case FLabInterfaceType of
       Triton : Result := TritonGetSource( Chan ) ;
       else Result := 0 ;
       end ;
     end ;


procedure TSESLabIO.SetTritonGain(
         Chan : Integer ;                   // Channel to be updated
         Value : Integer                       // New source
         ) ;
// -------------------------------------------------
//  Set Triton patch clamp gain for selected channel
// -------------------------------------------------
begin
     case FLabInterfaceType of
       Triton : TritonSetGain( Chan, Value ) ;
       end ;
     end ;


function TSESLabIO.GetTritonGain(
         Chan : Integer                        // New gain
         ) : Integer ;
// -------------------------------------------------
//  Get Triton patch clamp gain for selected channel
// -------------------------------------------------
begin
     case FLabInterfaceType of
       Triton : Result := TritonGetGain( Chan ) ;
       else Result := 0 ;
       end ;
     end ;


procedure TSESLabIO.SetTritonBesselFilter(
         Chan : Integer ;                   // Channel to be updated
         Value : Integer ;                   // New source
         var CutOffFrequency : Single
         ) ;
// -------------------------------------------------
//  Set Triton patch clamp low pass filter for selected channel
// -------------------------------------------------
begin
     case FLabInterfaceType of
       Triton : TritonSetBesselFilter( Chan, Value, CutOffFrequency ) ;
       end ;
     end ;


procedure TSESLabIO.TritonAutoCompensation(
         Chan : Integer                    // Channel to be compensated
         ) ;
// -------------------------------------------------
//  Set Triton patch clamp compensation
// -------------------------------------------------
begin
     case FLabInterfaceType of
       Triton : TritonAutoCompensate( Chan ) ;
       end ;
     end ;



end.
