unit QCAMUnit;
// ----------------
// QImaging cameras
// ----------------
// 1/4/2006
// 21/01/09 AdditionalReadoutTime added to StartCapture

interface

uses WinTypes,sysutils, classes, dialogs, mmsystem, messages, controls, math ;

const

    QCAMMaxFrames = 64 ;

// Public Enum QCam_qcCameraType
  qcCameraTypes : Array[0..16] of String = (
  'Unknown',
  'Mi2',
  'Pmi',
  'Retiga 1350',
  'QICam',
  'Retiga 1300B',
  'Retiga 1350B',
  'QICam B',
  'Micropublisher',
  'Retiga Intensified',
  'QCAM IR',
  'Rochester',
  'Retiga 4000R',
  'Retiga 2000R',
  'Rolera XR',
  'Retiga SRV',
  'Unknown'
  ) ;

  qcCCDPixelWidths : Array[0..16] of Single = (
  1.0, {'Unknown',}
  1.0, {'Mi2',}
  9.0, {'Pmi',}
  6.45,{'Retiga 1350'}
  4.65, {'QICam',}
  6.7, {'Retiga 1300B'}
  6.7, {'Retiga 1350B'}
  4.65,{'QICam B',}
  3.4, {'Micropublisher',}
  6.45,{'Retiga Intensified}
  13.7, {'QCAM IR',}
  1.0, {'Rochester',}
  7.4, {'Retiga 4000R',}
  7.4, {''Retiga 2000R',}
  13.7, {Rolera XR}
  6.45, {Retiga SRV}
  1.0 {'Unknown'}
  ) ;

	qcCameraUnknown			= 0;
	qcCameraMi2				= 1;		// MicroImager II and Retiga 1300
	qcCameraPmi				= 2;
	qcCameraRet1350			= 3;		// Retiga EX
	qcCameraQICam			= 4;

	// Model-B cameras (new camera hardware):
	qcCameraRet1300B		= 5;
	qcCameraRet1350B		= 6;
	qcCameraQICamB			= 7;
	qcCameraMicroPub		= 8;		// First versions of MicroPub were model A;
										// but both A and B return the same type.
	qcCameraRetIT		 	= 9;		// Intensified Retiga
	qcCameraQICamIR			= 10;
	qcCameraRochester		= 11;

	qcCameraRet4000R		= 12;
	qcCameraRet2000R		= 13;

	qcCameraRoleraXR		= 14;

	qcCameraRetigaSRV		= 15;



//Public Enum QCam_qcCcdType
    qcCcdMonochrome = 0 ;
    qcCcdColorBayer = 1 ;

// Public Enum QCam_qcCcd
    qcCCDNames : Array[0..6] of string = ('KAF1400','KAF1600','KAF1600L',
                                         'KAF4200','ICX085AL','ICX0*%AK', 'Unknown') ;
    qcCcdKAF1400 = 0 ;
    qcCcdKAF1600 = 1 ;
    qcCcdKAF1600L = 2 ;
    qcCcdKAF4200 = 3 ;
    qcCcdICX085AL = 4 ;
    qcCcdICX085AK = 5 ;

//Public Enum QCam_qcBayerPattern
    qcBayerRGGB =  0 ;
    qcBayerGRBG = 1 ;
    qcBayerGBRG = 2 ;
    qcBayerBGGR = 3 ;

//Public Enum QCam_qcTriggerType
    qcTriggerFreerun = 0 ;
    qcTriggerNone = 0 ;
    qcTriggerEdgeHi = 1 ;
    qcTriggerEdgeLow = 2 ;
    qcTriggerPulseHi = 3 ;
    qcTriggerPulseLow = 4 ;
  	qcTriggerSoftware	= 5 ;
   	qcTriggerStrobeHi	= 6 ;		// Integrate over pulse without masking
	  qcTriggerStrobeLow = 7 ;


//Public Enum QCam_qcWheelColor
    qcWheelRed = 0 ;
    qcWheelGreen = 1 ;
    qcWheelBlack = 2 ;
    qcWheelBlue = 3 ;

//Public Enum QCam_qcReadoutSpeed
    qcReadout20M = 0 ;
    qcReadout10M = 1 ;
    qcReadout5M = 2 ;
    qcReadout2M5 = 3  ;
    qcReadoutSpeeds : array[0..3] of Single = ( 20.0, 10.0, 5.0, 2.5 ) ;

//Public Enum QCam_qcShutterControl
    qcShutterAuto = 0 ;
    qcShutterClose = 1 ;
    qcShutterOpen = 2 ;
    qcShutter_last = 3 ;

	qcCallbackDone			= 1 ;	// Callback when QueueFrame (or QueueSettings) is done
	qcCallbackExposeDone	= 2 ;		// Callback when exposure done (readout starts);


//Public Enum QCam_Err
    qerrSuccess = 0 ;
    qerrNotSupported = 1 ;
    qerrInvalidValue = 2 ;
    qerrBadSettings = 3 ;
    qerrNoUserDriver = 4 ;
    qerrNoFirewireDriver = 5 ;
    qerrDriverConnection = 6 ;
    qerrDriverAlreadyLoaded = 7 ;
    qerrDriverNotLoaded = 8 ;
    qerrInvalidHandle = 9 ;
    qerrUnknownCamera = 10 ;
    qerrInvalidCameraId = 11 ;
    qerrNoMoreConnections = 12 ;
    qerrHardwareFault = 13 ;
    qerrFirewireFault = 14 ;
    qerrCameraFault = 15 ;
    qerrDriverFault = 16 ;
    qerrInvalidFrameIndex = 17 ;
    qerrBufferTooSmall = 18 ;
    qerrOutOfMemory = 19 ;
    qerrOutOfSharedMemory = 20 ;
    qerrBusy				= 21;
	  qerrQueueFull			= 22;	// Cannot queue more items; queue is full
	  qerrCancelled			= 23;
	  qerrNotStreaming		= 24;	// Streaming must be on before calling this command
	  qerrLostSync			= 25;	// This frame is trash; frame sync was lost
	  qerrBlackFill			= 26;	// This frame is damanged; some data is missing
	  qerrFirewireOverflow	= 27;	// Firewire overflow - restart streaming
	  qerrUnplugged			= 28;	// Camera has been unplugged or turned off
	  qerrAccessDenied		= 29;	// The camera is already open
	  qerrStreamFault			= 30;	// Stream Allocation Failed.  Is there enough Bandwidth
	  qerrQCamUpdateNeeded	= 31;	// QCam driver software is insufficient for the camera
	  qerrRoiTooSmall			= 32;	// Used to return that the ROI is too small


//Public Enum QCam_ImageFormat
    qfmtRaw8 = 0 ;
    qfmtRaw16 = 1 ;
    qfmtMono8 = 2 ;
    qfmtMono16 = 3 ;
    qfmtBayer8 = 4 ;
    qfmtBayer16 = 5 ;
    qfmtRgbPlane8 = 6 ;
    qfmtRgbPlane16 = 7 ;
    qfmtBgr24 = 8 ;
    qfmtXrgb32 = 9 ;
    qfmtRgb48 = 10 ;
    qcImageFormats : Array[0..10] of String = (
                     'Raw (8 bit)',
                     'Raw (16 bit)',
                     'Monochrome (8 bit)',
                     'Monochrome (16 bit)',
                     'Bayer (8 bit)',
                     'Bayer (16 bit)',
                     'RGB plane (8 bit)',
                     'RGB plane (16 bit)',
                     'BGR (24 bit)',
                     'XRGB (32 bit)',
                     'RGB (48 bit)' ) ;

//Public Enum QCam_Param
    qprmGain = 0 ;
    qprmOffset = 1 ;
    qprmExposure = 2 ;
    qprmBinning = 3 ;
    qprmHorizontalBinning = 4 ;
    qprmVerticalBinning = 5 ;
    qprmReadoutSpeed = 6 ;
    qprmTriggerType = 7 ;
    qprmColorWheel = 8 ;
    qprmCoolerActive = 9 ;
    qprmExposureRed = 10 ;
    qprmExposureBlue = 11 ;
    qprmImageFormat = 12 ;
    qprmRoiX = 13 ;
    qprmRoiY = 14 ;
    qprmRoiWidth = 15 ;
    qprmRoiHeight = 16 ;
    qprmShutterState = 18 ;

  	qprmReserved2			= 19;
	  qprmSyncb				= 20;	// SyncB output on some model-B cameras
	  qprmReserved3			= 21;
	  qprmIntensifierGain		= 22;	// Gain value for the intensifier (Intensified cameras only)
	  qprmTriggerDelay		= 23;	// Trigger delay in nanoseconds.
	  qprmCameraMode			= 24;	// Camera mode
	  qprmNormalizedGain		= 25;	// Normalized camera gain (micro units)
	  qprmNormIntensGaindB	= 26;	// Normalized intensifier gain dB (micro units)
	  qprmDoPostProcessing	= 27;   // Turns post processing on and off; 1 = On 0 = Off
	  qprmPostProcessGainRed	= 28;	// parameter to set bayer gain
	  qprmPostProcessGainGreen= 29;	// parameter to set bayer gain
	  qprmPostProcessGainBlue	= 30;	// parameter to set bayer gain
	  qprmPostProcessBayerAlgorithm	= 31;	// specify the bayer interpolation. QCam_qcBayerInterp enum
                                            // with the possible algorithms is located in QCamImgfnc.h
	  qprmPostProcessImageFormat		= 32;	// image format for post processed images
	  qprmFan					= 33;	// use QCam_qcFanSpeed to modify speed
	  qprmBlackoutMode		= 34;	// 1 turns all lights off; 0 turns them back on
	  qprmHighSensitivityMode		= 35;	// 1 turns high sensitivity mode on; 0 turn it off

	  qprmS32NormalizedGaindB = 0 ;	// Normalized camera gain in dB (micro units)
	  qprmS32AbsoluteOffset	= 1 ;	// Absolute camera offset (offset in CCD ADC)
	  qprmS32RegulatedCoolingTemp = 2 ;
	  qprmS32_last			= 3 ;

  	qprm64Exposure		= 0 ;		// Exposure in nanoseconds
	  qprm64ExposureRed	= 1 ;		// For LCD filter mode: exposure (ns) of red shot
	  qprm64ExposureBlue	= 2 ;		// For LCD filter mode: exposure (ns) of green shot
	  qprm64NormIntensGain= 3 ;		// Normalized intensifier gain (micro units)
 	  qprm64_last			= 4 ;


//Public Enum QCam_Info
    qinfCameraType = 0 ;
    qinfSerialNumber = 1 ;
    qinfHardwareVersion = 2 ;
    qinfFirmwareVersion = 3 ;
    qinfCcd = 4 ;
    qinfBitDepth = 5 ;
    qinfCooled = 6 ;
    qinfReserved1 = 7 ;
    qinfImageWidth = 8 ;
    qinfImageHeight = 9 ;
    qinfImageSize = 10 ;
    qinfCcdType = 11 ;
    qinfCcdWidth = 12 ;
    qinfCcdHeight = 13 ;

type

  TQCam_Settings = packed record
    size : Integer ;
    private_data : Array[0..39] of Integer ;
    end ;

   TQCam_Frame = packed record
     pBuffer : Pointer ;
     bufferSize : Integer ;
     format : Integer ;
     width : Integer ;
     height : Integer ;
     size : Integer ;
     bits : Integer ;
     bayerPattern : Integer ;
     reserved : Array[0..9] of Integer ;
     end ;

  TQCAMSession = record
     CamHandle : Integer ;
     CameraType : Integer ;
     NumBytesPerFrame : Integer ;     // No. of bytes in image
     NumPixelsPerFrame : Integer ;    // No. of pixels in image
     NumFrames : Integer ;            // No. of images in circular transfer buffer
     FrameNum : Integer ;             // Current frame no.
     PFrameBuffer : Pointer ;         // Frame buffer pointer
     NumFramesAcquired : Integer ;
     NumFramesCopied : Integer ;
     CapturingImages : Boolean ;     // Image capture in progress
     CameraOpen : Boolean ;          // Camera open for use
     TimeStart : single ;
     Temperature : Integer ;
     FrameTransferTime : Single ;    // Frame transfer time (s)
     QCamSettings : TQCam_Settings ;
     FrameList : Array[0..QCAMMaxFrames-1] of TQCam_Frame ;
     Counter : Integer ;
     Gains : Array[0..99] of Double ;
     NumGains : Integer ;
     BinFactors : Array[0..99] of Integer ;
     NumBinFactors : Integer ;
     ReadoutSpeeds : Array[0..99] of Integer ;
     NumReadoutSpeeds : Integer ;
     end ;


   TQCam_CamListItem = packed record
      cameraId : Integer ;
      cameraType : Integer ;
      uniqueId : Integer ;
      isOpen : LongBool ;
      reserved : Array[0..9] of Integer ;
      end ;

TQCam_Abort = function(
              handle: Integer
              )  : Integer ; stdcall ;

TQCam_LoadDriver = function : Integer ; stdcall ;

TQCam_ReleaseDriver = procedure; stdcall ;

TQCam_LibVersion = function(
                    var verMajor : Word ;
                    var verMinor : Word ;
                    var BuildNum : Word
                    ) : Integer ; stdcall ;

TQCam_ListCameras = function(
                    pList : Pointer ;
                    pnumberInList : Pointer
                    )  : Integer ; stdcall ;

TQCam_OpenCamera = function(
                   cameraId: Integer ;
                   var handle: Integer
                   ) : Integer ; stdcall ;

TQCam_CloseCamera = function(
                    handle: Integer
                    )  : Integer ; stdcall ;

TQCam_GetInfo = function(
                handle: Integer ;
                infoKey : Integer ;
                var value: Integer
                )  : Integer ; stdcall ;

TQCam_ReadDefaultSettings = function(
      Handle: Integer ;
      var settings : TQCam_Settings
      )  : Integer ; stdcall ;

TQCam_ReadSettingsFromCam = function(
      handle: Integer ;
      var settings : TQCam_Settings
      )  : Integer ; stdcall ;

TQCam_SendSettingsToCam = function(
      handle: Integer ;
      var settings : TQCam_Settings
      ) : Integer ; stdcall ;

TQCam_TranslateSettings = function(
      handle: Integer ;
      var settings : TQCam_Settings
      )  : Integer ; stdcall ;

TQCam_PreflightSettings = function(
      handle: Integer ;
      pSettings : Pointer
      )  : Integer ; stdcall ;

TQCam_GetParam = function(
      var settings : TQCam_Settings ;
      paramKey  : Integer ;
      var value: Integer
      ) : Integer ; stdcall ;

TQCam_SetParam = function(
      var settings : TQCam_Settings ;
      paramKey  : Integer ;
      value: Integer
      )  : Integer ; stdcall ;

TQCam_SetParam64 = function(
      var settings : TQCam_Settings ;
      paramKey  : Integer ;
      value: Int64
      )  : Integer ; stdcall ;


TQCam_GetParamMin = function(
                   var settings : TQCam_Settings ;
                   paramKey  : Integer ;
                   var Value: Integer
                   )  : Integer ; stdcall ;

TQCam_GetParamMax = function(
                    var settings : TQCam_Settings ;
                    paramKey  : Integer ;
                    var value: Integer
                    )  : Integer ; stdcall ;

TQCam_GetParam64Min = function(
                   var settings : TQCam_Settings ;
                   paramKey  : Integer ;
                   var Value: Int64
                   )  : Integer ; stdcall ;

TQCam_GetParam64Max = function(
                    var settings : TQCam_Settings ;
                    paramKey  : Integer ;
                    var value: Int64
                    )  : Integer ; stdcall ;

                    TQCam_GetParamSparseTable = function(
                            var settings : TQCam_Settings ;
                            paramKey  : Integer ;
                            //var Table: Array of Integer ;
                            pTable : Pointer ;
                            var NumItems : Integer
                            )  : Integer ; stdcall ;


TQCam_GrabFrame = function(
                  handle: Integer ;
                  var frame : TQCam_Frame
                  ) : Integer ; stdcall ;

TQCam_SetFrameBuffers = function(
                        handle: Integer ;
                        var frames : Array of TQCam_Frame ;
                        number: Integer
                        )  : Integer ; stdcall ;

TQCam_SetStreaming = function(
                     handle: Integer ;
                     enable: Integer
                     )  : Integer ; stdcall ;

TQCam_GrabFrameNow = function(
                    handle: Integer ;
                    frameIndex : Integer
                    )  : Integer ; stdcall ;

TQCam_GrabRawNow = function(
                   handle: Integer ;
                   frameIndex: Integer
                   )  : Integer ; stdcall ;

TQCam_ProcessRaw = function(
                   handle: Integer ;
                   frameIndex: Integer
                   )  : Integer ; stdcall ;

TQCam_QueueFrame = function(
                   handle: Integer ;      // IN: handle to camera
                   FramePtr : Pointer ;   // IN: frame
                   CallBackPtr : Pointer ;  // IN: completion callback; can be NULL
                   cbFlags : Cardinal ;     // IN: qcCallbackFlags
                   UserPtr : Pointer ;      // IN: user specified value for callback
                   UserData : Cardinal      // IN: user specified value for callback
                   )  : Integer ; stdcall ;

TQCam_QueueSettings = function(
                   handle: Integer ;      // IN: handle to camera
                   FramePtr : Pointer ;   // IN: frame
                   CallBackPtr : Pointer ;  // IN: completion callback; can be NULL
                   cbFlags : Cardinal ;     // IN: qcCallbackFlags
                   UserPtr : Pointer ;      // IN: user specified value for callback
                   UserData : Cardinal      // IN: user specified value for callback
                   )  : Integer ; stdcall ;

function QCAMAPI_LoadLibrary : Boolean ;
function QCAMAPI_GetDLLAddress(
         Handle : Integer ;
         const ProcName : string ) : Pointer ;

function QCAMAPI_OpenCamera(
          var Session : TQCAMSession ;   // Camera session record
          var FrameWidthMax : Integer ;      // Returns camera frame width
          var FrameHeightMax : Integer ;     // Returns camera frame width
          var NumBytesPerPixel : Integer ;   // Returns bytes/pixel
          var PixelDepth : Integer ;         // Returns no. bits/pixel
          var PixelWidth : Single ;          // Returns pixel size (um)
          CameraInfo : TStringList         // Returns Camera details
          ) : Boolean ;

procedure QCAMAPI_CloseCamera(
          var Session : TQCAMSession // Session record
          ) ;

procedure QCAMAPI_GetCameraGainList(
          var Session : TQCAMSession ;   // Camera session record
          CameraGainList : TStringList
          ) ;

procedure QCAMAPI_GetCameraReadoutSpeedList(
          var Session : TQCAMSession ;   // Camera session record
          CameraReadoutSpeedList : TStringList
          ) ;

function QCAMAPI_StartCapture(
         var Session : TQCAMSession ;   // Camera session record
         var InterFrameTimeInterval : Double ;      // Frame exposure time
         AdditionalReadoutTime : Double ; // Additional readout time (s)
         AmpGain : Integer ;              // Camera amplifier gain index
         ReadoutSpeed : Integer ;         // Camera Read speed index number
         ExternalTrigger : Integer ;      // Trigger mode
         FrameLeft : Integer ;            // Left pixel in CCD readout area
         FrameRight : Integer ;           // Right pixel in CCD eadout area
         FrameTop : Integer ;             // Top of CCD readout area
         FrameBottom : Integer ;          // Bottom of CCD readout area
         BinFactor : Integer ;             // Binning factor (1,2,4,8,16)
         PFrameBuffer : Pointer ;         // Pointer to start of ring buffer
         NumFramesInBuffer : Integer ;    // No. of frames in ring buffer
         NumBytesPerFrame : Integer       // No. of bytes/frame
         ) : Boolean ;

procedure QCAMAPI_CheckROIBoundaries(
          var Session : TQCAMSession ;   // Camera session record
          var ReadoutSpeed : Integer ;
         var FrameLeft : Integer ;            // Left pixel in CCD readout area
         var FrameRight : Integer ;           // Right pixel in CCD eadout area
         var FrameTop : Integer ;             // Top of CCD readout area
         var FrameBottom : Integer ;          // Bottom of CCD readout area
         var  BinFactor : Integer ;   // Pixel binning factor (In)
         var FrameWidth : Integer ;
         var FrameHeight : Integer ;
         var FrameInterval : Double ;
         var ReadoutTime : Double ) ;

procedure QCAMAPI_Wait( Delay : Single ) ;


procedure QCAMAPI_GetImage(
          var Session : TQCAMSession  // Camera session record
          ) ;

procedure QCAMAPI_StopCapture(
          var Session : TQCAMSession   // Camera session record
          ) ;

procedure QCAMAPI_CheckError(
          FuncName : String ;   // Name of function called
          ErrNum : Integer      // Error # returned by function
          ) ;

procedure FrameDoneCallBack(
         UsrPtr : Pointer ;
         UserData : Cardinal ;
         ErrCode : Integer ;
         Flags : Cardinal ) ; stdcall ;



implementation

uses sescam ;

var

  QCam_Abort : TQCam_Abort ;
  QCam_LoadDriver : TQCam_LoadDriver ;
  QCam_ReleaseDriver : TQCam_ReleaseDriver ;
  QCam_LibVersion : TQCam_LibVersion ;
  QCam_ListCameras : TQCam_ListCameras ;
  QCam_OpenCamera : TQCam_OpenCamera ;
  QCam_CloseCamera : TQCam_CloseCamera ;
  QCam_GetInfo : TQCam_GetInfo ;
  QCam_ReadDefaultSettings : TQCam_ReadDefaultSettings ;
  QCam_ReadSettingsFromCam : TQCam_ReadSettingsFromCam ;
  QCam_SendSettingsToCam : TQCam_SendSettingsToCam ;
  QCam_TranslateSettings : TQCam_TranslateSettings ;
  QCam_PreflightSettings : TQCam_PreflightSettings ;
  QCam_GetParam : TQCam_GetParam ;
  QCam_SetParam : TQCam_SetParam ;
  QCam_SetParam64 : TQCam_SetParam64 ;
  QCam_GetParamMin : TQCam_GetParamMin ;
  QCam_GetParamMax : TQCam_GetParamMax ;
  QCam_GetParam64Min : TQCam_GetParam64Min ;
  QCam_GetParam64Max : TQCam_GetParam64Max ;
  QCam_GetParamSparseTable : TQCam_GetParamSparseTable ;
  QCam_GrabFrame : TQCam_GrabFrame ;
  QCam_SetFrameBuffers : TQCam_SetFrameBuffers ;
  QCam_SetStreaming : TQCam_SetStreaming ;
  QCam_GrabFrameNow : TQCam_GrabFrameNow ;
  QCam_GrabRawNow : TQCam_GrabRawNow ;
  QCam_ProcessRaw : TQCam_ProcessRaw ;
  QCam_QueueFrame : TQCam_QueueFrame ;
  QCam_QueueSettings : TQCam_QueueSettings ;

   LibraryLoaded : Boolean ;
   LibraryHnd : THandle ;

function QCAMAPI_LoadLibrary  : Boolean ;
{ ---------------------------------------------
  Load camera interface DLL library into memory
  ---------------------------------------------}
var
    LibFileName : string ;
begin

     Result := LibraryLoaded ;

     if LibraryLoaded then Exit ;

     { Load DLL camera interface library }
     LibFileName := 'QCamDriver.dll' ;
     LibraryHnd := LoadLibrary( PChar(LibFileName));
     if LibraryHnd > 0 then LibraryLoaded := True ;

     if not LibraryLoaded then begin
        ShowMessage( 'QCAM: ' + LibFileName + ' not found!' ) ;
        Result := LibraryLoaded ;
        Exit ;
        end ;

     @QCam_ProcessRaw := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_ProcessRaw') ;
     @QCam_GrabRawNow := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_GrabRawNow') ;
     @QCam_GrabFrameNow := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_GrabFrameNow') ;
     @QCam_SetStreaming := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_SetStreaming') ;
     @QCam_SetFrameBuffers := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_SetFrameBuffers') ;
     @QCam_GrabFrame := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_GrabFrame') ;
     @QCam_GetParamMax := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_GetParamMax') ;
     @QCam_GetParamMin := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_GetParamMin') ;
     @QCam_GetParam64Max := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_GetParam64Max') ;
     @QCam_GetParam64Min := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_GetParam64Min') ;
     @QCam_GetParamSparseTable := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_GetParamSparseTable') ;
     @QCam_SetParam := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_SetParam') ;
     @QCam_GetParam := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_GetParam') ;
     @QCam_SetParam64 := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_SetParam64') ;
     @QCam_TranslateSettings := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_TranslateSettings') ;
     @QCam_PreflightSettings := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_PreflightSettings') ;
     @QCam_SendSettingsToCam := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_SendSettingsToCam') ;
     @QCam_ReadSettingsFromCam := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_ReadSettingsFromCam') ;
     @QCam_ReadDefaultSettings := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_ReadDefaultSettings') ;
     @QCam_GetInfo := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_GetInfo') ;
     @QCam_CloseCamera:= QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_CloseCamera') ;
     @QCam_OpenCamera := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_OpenCamera') ;
     @QCam_ListCameras := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_ListCameras') ;
     @QCam_LibVersion := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_LibVersion') ;
     @QCam_ReleaseDriver := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_ReleaseDriver') ;
     @QCam_LoadDriver := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_LoadDriver') ;
     @QCam_Abort := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_Abort') ;
     @QCam_QueueFrame := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_QueueFrame') ;
     @QCam_QueueSettings := QCAMAPI_GetDLLAddress(LibraryHnd,'QCam_QueueSettings');

     Result := LibraryLoaded ;

     end ;


function QCAMAPI_GetDLLAddress(
         Handle : Integer ;
         const ProcName : string ) : Pointer ;
// -----------------------------------------
// Get address of procedure within DLL
// -----------------------------------------
begin
    Result := GetProcAddress(Handle,PChar(ProcName)) ;
    if Result = Nil then ShowMessage('Qcamdriver.dll: ' + ProcName + ' not found') ;
    end ;


function QCAMAPI_OpenCamera(
         var Session : TQCAMSession ;   // Camera session record
         var FrameWidthMax : Integer ;      // Returns camera frame width
         var FrameHeightMax : Integer ;     // Returns camera frame width
         var NumBytesPerPixel : Integer ;   // Returns bytes/pixel
         var PixelDepth : Integer ;         // Returns no. bits/pixel
         var PixelWidth : Single ;          // Returns pixel size (um)
         CameraInfo : TStringList         // Returns Camera details
         ) : Boolean ;
// -------------------
// Open camera for use
// -------------------
const
    BufSize = 200 ;
var
    cBuf : Array[0..BufSize] of Char ;
    iBuf : Array[0..BufSize] of Integer ;
    i,iValue,iMin,iMax : Integer ;
    iMin64,iMax64 : Int64 ;
    s : String ;
    CameraList : Array[0..9] of TQCam_CamListItem ;
    NumCameras : Integer ;
    VersionMajor, VersionMinor, BuildNum : Word ;
    NumItems : Integer ;
    iTable : Array[0..99] of Integer ;
    Mono16Available : Boolean ;
    Mono8Available : Boolean ;

begin

     // Load DLL camera control library
     if not QCAMAPI_LoadLibrary then Exit ;

     if QCAM_LoadDriver <> 0 then begin
        ShowMessage('QCAM: Unable to initialise driver!') ;
        Exit ;
        End ;

     QCam_LibVersion( VersionMajor, VersionMinor, BuildNum ) ;
     CameraInfo.Add( format( 'QCAM Library V%d.%d (%d)',[VersionMajor, VersionMinor,BuildNum] )) ;
     NumCameras := High(CameraList) ;
     QCAM_ListCameras( @CameraList, @NumCameras ) ;
     if NumCameras <= 0 then begin
        ShowMessage('QCAM: No cameras detected!') ;
        QCAM_ReleaseDriver ;
        Exit ;
        End ;

     if not CameraList[0].IsOpen then begin
        // Open first camera in list
        QCAMAPI_CheckError( 'QCAM_OpenCamera',
           QCAM_OpenCamera( CameraList[0].CameraID, Session.CamHandle )) ;
        end
     else ShowMessage('QCAM: Camera already open!') ;

     // Read settings from camera
     Session.QCAMSettings.size := Sizeof(Session.QCAMSettings) ;
     QCAMAPI_CheckError( 'QQCAM_ReadDefaultSettings',
       QCAM_ReadDefaultSettings( Session.CamHandle,
                                 Session.QCAMSettings ) ) ;

     // Get type of camera
     QCAM_GetInfo( Session.CamHandle, qinfCameraType, Session.CameraType ) ;
     CameraInfo.Add( 'Camera: ' + qcCameraTypes[Min(Max(Session.CameraType,0),High(qcCameraTypes))]);

     // Get max. size of camera image
     QCAM_GetInfo( Session.CamHandle, qinfCCDWidth, FrameWidthMax ) ;
     QCAM_GetInfo( Session.CamHandle, qinfCCDHeight, FrameHeightMax ) ;

     // Get size of CCD pixel
     PixelWidth := qcCCDPixelWidths[Min(Max(Session.CameraType,0),High(qcCCDPixelWidths))] ;

     // Get type of CCD
     QCAM_GetInfo( Session.CamHandle, qinfCCDType, iValue ) ;

     // Report CCD properties
     CameraInfo.Add( format('CCD: %s (%d x %d pixels) %.2f um pixels',
                     [qcCCDNames[Min(Max(iValue,0),High(qcCCDNames))],
                      FrameWidthMax,
                      FrameHeightMax,
                      PixelWidth])) ;

     // Get number of bits per pixel
     QCAM_GetInfo( Session.CamHandle, qinfBitDepth, PixelDepth ) ;
     CameraInfo.Add(format( 'Pixel depth: %d bits',[PixelDepth] )) ;

{    qinfCameraType = 0 ;
    qinfSerialNumber = 1 ;
    qinfHardwareVersion = 2 ;
    qinfFirmwareVersion = 3 ;
    qinfCcd = 4 ;
    qinfBitDepth = 5 ;
   qinfCooled = 6 ;
}
     // Cooling availab;e
     QCAM_GetInfo( Session.CamHandle, qinfCooled, iValue ) ;
     if iValue <> 0 then CameraInfo.Add( 'CCD cooling available' ) ;

{    qinfReserved1 = 7 ;
    qinfImageWidth = 8 ;
    qinfImageHeight = 9 ;
    qinfImageSize = 10 ;
    qinfCcdType = 11 ;
    qinfCcdWidth = 12 ;
    qinfCcdHeight = 13 ;   }

   { qprmGain = 0 ;
    qprmOffset = 1 ;
    qprmExposure = 2 ;
    qprmBinning = 3 ;
    qprmHorizontalBinning = 4 ;
    qprmVerticalBinning = 5 ;
    qprmReadoutSpeed = 6 ;
    qprmTriggerType = 7 ;
    qprmColorWheel = 8 ;
    qprmCoolerActive = 9 ;
    qprmExposureRed = 10 ;
    qprmExposureBlue = 11 ;
    qprmImageFormat = 12 ;
    qprmRoiX = 13 ;
    qprmRoiY = 14 ;
    qprmRoiWidth = 15 ;
    qprmRoiHeight = 16 ;
    qprmShutterState = 18 ;}

     // Gain range
     if Session.CameraType = qcCameraRetIT then begin
        // Intensified cameras
        QCam_GetParam64Min( Session.QCAMSettings, qprm64NormIntensGain, iMin64 ) ;
        QCam_GetParam64Max( Session.QCAMSettings, qprm64NormIntensGain, iMax64 ) ;
        CameraInfo.Add(format( 'Intensifier Gains: %.0f - %.0f',[iMin64*1E-6,iMax64*1E-6] )) ;
        end
     else begin
        QCam_GetParamMin( Session.QCAMSettings, qprmNormalizedGain, iMin ) ;
        QCam_GetParamMax( Session.QCAMSettings, qprmNormalizedGain, iMax ) ;
        CameraInfo.Add(format( 'Gains: %.0f - %.0f',[iMin*1E-6,iMax*1E-6] )) ;
        end ;

     // Readout speed range
     Session.NumReadoutSpeeds :=  High(Session.ReadoutSpeeds) ;
     QCam_GetParamSparseTable( Session.QCAMSettings,
                               qprmReadoutSpeed,
                               @Session.ReadoutSpeeds,
                               Session.NumReadoutSpeeds ) ;
     s := 'Readout speeds: ' ;
     for i := 0 to Session.NumReadoutSpeeds-1 do begin
         s := s + format( '%.1f MHz',[qcReadoutSpeeds[Session.ReadoutSpeeds[i]]] ) ;
         if i < (Session.NumReadoutSpeeds-1) then s := s + ', ' ;
         end ;
     CameraInfo.Add( s ) ;

     // Binning factors
     Session.NumBinFactors :=  High(Session.BinFactors) ;
     QCam_GetParamSparseTable( Session.QCAMSettings, qprmBinning,
                               @Session.BinFactors, Session.NumBinFactors ) ;
     s := 'Binning factors: ' ;
     for i := 0 to Session.NumBinFactors-1 do begin
         s := s + format( '%d',[Session.BinFactors[i]] ) ;
         if i < (NumItems-1) then s := s + ', ' ;
         end ;
     CameraInfo.Add( s ) ;

     // Available image formats
     NumItems :=  High(iTable) ;
     QCam_GetParamSparseTable( Session.QCAMSettings, qprmImageFormat, @iTable, NumItems ) ;
     s := 'Image formats: ' ;
     for i := 0 to NumItems-1 do begin
         s := s + format( '%s',[qcImageFormats[iTable[i]]] ) ;
         if i < (NumItems-1) then s := s + ', ' ;
         end ;
     CameraInfo.Add( s ) ;

     // Select 16 bit monochrome format if it is available
     Mono16Available := False ;
     Mono8Available := False ;
     for i := 0 to NumItems-1 do begin
         if iTable[i] = qfmtMono16 then Mono16Available := True
         else if iTable[i] = qfmtMono16 then Mono8Available := True ;
         end ;

     if Mono16Available then QCam_SetParam( Session.QCAMSettings, qprmImageFormat, qfmtMono16 )
     else if Mono8Available then QCam_SetParam( Session.QCAMSettings, qprmImageFormat, qfmtMono8 ) ;

     // Image format
     QCam_GetParam( Session.QCAMSettings, qprmImageFormat, iValue ) ;
     CameraInfo.Add( 'Image format: ' + qcImageFormats[iValue] ) ;

     NumBytesPerPixel := 2 ;

     Session.CameraOpen := True ;
     Result := Session.CameraOpen ;

     PixelWidth := 10.0 ;

     end ;


procedure QCAMAPI_CloseCamera(
          var Session : TQCAMSession // Session record
          ) ;
// ------------
// Close camera
// ------------
begin

    if not Session.CameraOpen then Exit ;

    // Stop image capture
    QCAMAPI_StopCapture( Session ) ;

    // Close camera
    QCAM_CloseCamera( Session.CamHandle ) ;

    // Close QCAM driver
    QCAM_ReleaseDriver ;

    // Free DLL library
    FreeLibrary( LibraryHnd ) ;

    Session.CameraOpen := False ;

    end ;


procedure QCAMAPI_GetCameraGainList(
          var Session : TQCAMSession ;   // Camera session record
          CameraGainList : TStringList
          ) ;
// -------------------------------------
// Return list of available camera gains
// -------------------------------------
const
    GainList : Array[0..9] of Double = (1.0,2.0,5.0,10.,20.,50.0,100.0,200.0,500.0,1000.0) ;
var
    i : Integer ;
    iMin,iMax : Integer ;
    iMin64,iMax64 : Int64 ;
    GMin,GMax : Double ;
begin

    CameraGainList.Clear ;

    if Session.CameraType = qcCameraRetIT then begin
        // Intensified cameras
        QCam_GetParam64Min( Session.QCAMSettings, qprm64NormIntensGain, iMin64 ) ;
        QCam_GetParam64Max( Session.QCAMSettings, qprm64NormIntensGain, iMax64 ) ;
        GMin := iMin64*1E-6 ;
        GMax := IMax64*1E-6 ;
        end
    else begin
        // Get camera normalised gain range
        QCam_GetParamMin( Session.QCAMSettings, qprmNormalizedGain, iMin ) ;
        QCam_GetParamMax( Session.QCAMSettings, qprmNormalizedGain, iMax ) ; // S
        GMin := iMin*1E-6 ;
        GMax := IMax*1E-6 ;
        end ;

    // Set to gains available from list (and max)
    Session.NumGains := 0 ;
    for i := 0 to High(GainList) do begin
        if (GainList[i] >= GMin) and (GainList[i] <= GMax) then begin
           Session.Gains[Session.NumGains] := GainList[i] ;
           CameraGainList.Add( format( 'X%.1f',[Session.Gains[Session.NumGains]]));
          Inc( Session.NumGains ) ;
          end ;
        end ;
    if Session.Gains[Session.NumGains-1] <> GMax then begin
       Session.Gains[Session.NumGains] := GMax ;
       CameraGainList.Add( format( 'X%.1f',[Session.Gains[Session.NumGains]]));
       Inc( Session.NumGains ) ;
       end ;

    end ;


procedure QCAMAPI_GetCameraReadoutSpeedList(
          var Session : TQCAMSession ;   // Camera session record
          CameraReadoutSpeedList : TStringList
          ) ;
// -------------------------------------
// Return list of camera readout speeds
// -------------------------------------
var
    i : Integer ;
    NumItems : Integer ;
    iTable : Array[0..99] of Integer ;
begin

    CameraReadoutSpeedList.Clear ;

    // Readout speed range
     NumItems :=  High(iTable) ;
     QCam_GetParamSparseTable( Session.QCAMSettings, qprmReadoutSpeed, @iTable, NumItems ) ;
     for i := 0 to NumItems-1 do begin
         CameraReadoutSpeedList.Add( format( '%.1f Mhz', [qcReadoutSpeeds[iTable[i]]]) ) ;
         end ;

    end ;


function QCAMAPI_StartCapture(
         var Session : TQCAMSession ;   // Camera session record
         var InterFrameTimeInterval : Double ;      // Frame exposure time
         AdditionalReadoutTime : Double ; // Additional readout time (s)
         AmpGain : Integer ;              // Camera amplifier gain index
         ReadoutSpeed : Integer ;         // Camera Read speed index number
         ExternalTrigger : Integer ;      // Trigger mode
         FrameLeft : Integer ;            // Left pixel in CCD readout area
         FrameRight : Integer ;           // Right pixel in CCD eadout area
         FrameTop : Integer ;             // Top of CCD readout area
         FrameBottom : Integer ;          // Bottom of CCD readout area
         BinFactor : Integer ;             // Binning factor (1,2,4,8,16)
         PFrameBuffer : Pointer ;         // Pointer to start of ring buffer
         NumFramesInBuffer : Integer ;    // No. of frames in ring buffer
         NumBytesPerFrame : Integer       // No. of bytes/frame
         ) : Boolean ;
var
    Err : Integer ;
    i,iGain : Integer ;
    iGain64 : Int64 ;
    ExposureTime_ns : Int64 ;
    ReadoutTime : Double ;
    qcTriggerMode : Integer ;
begin

    if not Session.CameraOpen then Exit ;

     // Abort any existing queued frames
     QCAM_Abort( Session.CamHandle ) ;

     // Set camera readout speed
     ReadoutSpeed := Min(Max(ReadoutSpeed,0),Session.NumReadoutSpeeds-1) ;
     QCam_SetParam( Session.QCAMSettings,
                    qprmReadoutSpeed,
                    Session.ReadoutSpeeds[ReadoutSpeed] ) ;

     // Set camera normalised gain range
    if Session.CameraType = qcCameraRetIT then begin
        // Intensified cameras
        iGain64 := Round(Session.Gains[Min(Max(AmpGain,0),Session.NumGains-1)]*1E6) ;
        QCam_SetParam64( Session.QCAMSettings, qprm64NormIntensGain, iGain64 ) ;
        QCam_SetParam( Session.QCAMSettings, qprmNormalizedGain, 1000000 ) ;
        end
     else begin
         iGain := Round(Session.Gains[Min(Max(AmpGain,0),Session.NumGains-1)]*1E6) ;
         QCam_SetParam( Session.QCAMSettings, qprmNormalizedGain, iGain ) ;
         end ;

     // Set exposure time
     InterFrameTimeInterval := Max( ReadoutTime, InterFrameTimeInterval ) ;
     ExposureTime_ns := Int64(Round(InterFrameTimeInterval*1E9)) ;
     QCAM_SetParam64( Session.QCAMSettings, qprm64Exposure, ExposureTime_ns ) ;
     InterFrameTimeInterval := ExposureTime_ns / 1E9 ;

     // Set trigger mode
     if ExternalTrigger = CamExtTrigger then begin
        qcTriggerMode := qcTriggerEdgeHi ;
        // Note. Exposure set to 95% of inter-frame interval to ensure frame exposure
        // has finished before next trigger pulse
        ExposureTime_ns := Int64(Round(((InterFrameTimeInterval*0.95) - AdditionalReadoutTime)*1E9)) ;
        QCAM_SetParam64( Session.QCAMSettings, qprm64Exposure, ExposureTime_ns ) ;
        end
     else qcTriggerMode := qcTriggerFreerun ;
     QCAM_SetParam( Session.QCAMSettings, qprmTriggerType, qcTriggerMode ) ;

     // Update camera
     QCAMAPI_CheckError( 'QCam_SendSettingsToCam',
                         QCam_SendSettingsToCam( Session.CamHandle, Session.QCAMSettings)) ;

     // Turn streaming on
     QCAMAPI_CheckError( 'QCam_SetStreaming',
                         QCam_SetStreaming( Session.CamHandle, 1 )) ;

     // Create initial list of queued frames
     Session.NumFrames := NumFramesInBuffer ;
     Session.NumBytesPerFrame := NumBytesPerFrame ;
     for i := 0 to NumFramesInBuffer-1 do begin
         Session.FrameList[i].pBuffer := Pointer(Cardinal(PFrameBuffer) + i*NumBytesPerFrame) ;
         Session.FrameList[i].bufferSize := NumBytesPerFrame ;
         Err := QCAM_QueueFrame( Session.CamHandle,
                                 @Session.FrameList[i],
                                 @FrameDoneCallBack,
                                 qcCallbackDone,
                                 @Session,
                                 i) ;
         if Err <> 0 then Break ;
         end ;

     Session.FrameNum := 0 ;

     end ;


procedure QCAMAPI_CheckROIBoundaries(
          var Session : TQCAMSession ;   // Camera session record
         var ReadoutSpeed : Integer ;         // Readout rate (index no.)
         var FrameLeft : Integer ;            // Left pixel in CCD readout area
         var FrameRight : Integer ;           // Right pixel in CCD eadout area
         var FrameTop : Integer ;             // Top of CCD readout area
         var FrameBottom : Integer ;          // Bottom of CCD readout area
         var BinFactor : Integer ;           // Pixel binning factor (In)
         var FrameWidth : Integer ;          // Image width
         var FrameHeight : Integer ;         // Image height
         var FrameInterval : Double ;        // Time interval between frames (s)
         var ReadoutTime : Double ) ;        // Frame readout time (s)
// ----------------------------------------------------------
// Check and set CCD ROI boundaries and return valid settings
// (Also calculates minimum readout time)
// -----------------------------------------------------------
var
    iMin64 : Int64 ;
    i : Integer ;
    RScale : Double ;
begin

    // Set binning factor
    i := 0 ;
    while (BinFactor > Session.BinFactors[i]) and
           (i < Session.NumBinFactors) do Inc(i) ;
    BinFactor := Max( Session.BinFactors[i],1) ;

    QCAM_SetParam( Session.QCAMSettings, qprmBinning, BinFactor ) ;

    // Set CCD ROI region
    QCAM_SetParam( Session.QCAMSettings, qprmRoiX, FrameLeft div BinFactor ) ;
    QCAM_SetParam( Session.QCAMSettings, qprmRoiY, FrameTop  div BinFactor ) ;

    // Set CCD ROI width
    FrameWidth := (FrameRight - FrameLeft + 1) div BinFactor ;
    QCAM_SetParam( Session.QCAMSettings, qprmRoiWidth, FrameWidth ) ;

    // Set CCD ROI height
    FrameHeight := (FrameBottom - FrameTop + 1 ) div BinFactor ;
    QCAM_SetParam( Session.QCAMSettings, qprmRoiHeight, FrameHeight ) ;

     // Update camera
     if QCam_PreflightSettings( Session.CamHandle, @Session.QCAMSettings) <> 0 then Exit ;

    QCAM_GetParam( Session.QCAMSettings, qprmRoiX, FrameLeft ) ;
    FrameLeft := FrameLeft*BinFactor ;

    QCAM_GetParam( Session.QCAMSettings, qprmRoiY, FrameTop ) ;
    FrameTop := FrameTop*BinFactor ;

    // Set CCD ROI width

    QCAM_GetParam( Session.QCAMSettings, qprmRoiWidth, FrameWidth ) ;
    FrameRight := FrameWidth*BinFactor + FrameLeft - 1 ;

    // Set CCD ROI height
    QCAM_GetParam( Session.QCAMSettings, qprmRoiHeight, FrameHeight ) ;
    FrameBottom := FrameHeight*BinFactor + FrameTop - 1 ;

    // Calculate frame readout time
    // Note. Relationship between CCD readout time and no. of lines
    // calculated at 20 MHz and scaled for other readout rates
    // Minimum readout rate appears to be 20 ms
    ReadoutSpeed := Min(Max(ReadoutSpeed,0),Session.NumReadoutSpeeds-1) ;
    RScale := 20.0 / qcReadOutSpeeds[Session.ReadoutSpeeds[ReadoutSpeed]] ;
    ReadoutTime := RScale * Max( (FrameHeight*9E-5 + 0.0037)+ 1E-3, 0.02 ) ;
    FrameInterval := Max( FrameInterval, ReadoutTime ) ;
    end ;


procedure QCAMAPI_Wait( Delay : Single ) ;
begin
    end ;


procedure QCAMAPI_GetImage(
          var Session : TQCAMSession  // Camera session record
          ) ;
//
//
//
var
    i,Err : Integer ;
begin

    //outputdebugString(PChar(format('%d %d',[Session.FrameNum,Session.Counter]))) ;
    end ;


procedure QCAMAPI_StopCapture(
          var Session : TQCAMSession   // Camera session record
          ) ;
// -------------------------
// Stop camera frame capture
// -------------------------
begin

    if (not Session.CameraOpen) then Exit ;
    if (not Session.CapturingImages) then Exit ;

    // Abort any existing queued frames
    QCAMAPI_CheckError( 'QCAM_Abort', QCAM_Abort( Session.CamHandle )) ;

    // Turn streaming off
    QCAMAPI_CheckError( 'QCam_SetStreaming',
                        QCam_SetStreaming( Session.CamHandle, 0 )) ;


    Session.CapturingImages := False ;

    end ;


procedure QCAMAPI_CheckError(
          FuncName : String ;   // Name of function called
          ErrNum : Integer      // Error # returned by function
          ) ;
var
    Err : String ;
begin

    if ErrNum = 0 then Exit ;
    case ErrNum of
      qerrNotSupported : Err := ' Not supported' ;
      qerrInvalidValue  : Err :=  ' Invalid value' ;
      qerrBadSettings  : Err :=  'Bad settings' ;
      qerrNoUserDriver  : Err :=  'No user driver' ;
      qerrNoFirewireDriver  : Err :=  'No firewire driver' ;
      qerrDriverConnection  : Err :=  'Driver connection' ;
      qerrDriverAlreadyLoaded  : Err :=  'Driver already loaded' ;
      qerrDriverNotLoaded  : Err :=  'Driver not loaded' ;
      qerrInvalidHandle  : Err :=  'Invalid handle' ;
      qerrUnknownCamera  : Err :=  'Unknown camera' ;
      qerrInvalidCameraId  : Err :=  'Invalid camera ID' ;
      qerrNoMoreConnections  : Err := 'No more connections' ;
      qerrHardwareFault  : Err :=  'Hardware fault' ;
      qerrFirewireFault  : Err :=  'Firewire fault' ;
      qerrCameraFault  : Err :=  'Camera fault' ;
      qerrDriverFault  : Err :=  'Driver fault' ;
      qerrInvalidFrameIndex  : Err := 'Invalid frame index' ;
      qerrBufferTooSmall  : Err :=  'Buffer too small' ;
      qerrOutOfMemory  : Err :=  'Out of memory' ;
      qerrOutOfSharedMemory  : Err :=  'Out of shared memory' ;
      qerrBusy : Err := 'Camera is busy' ;
  	  qerrQueueFull : Err := 'Cannot queue more items; queue is full';
	    qerrCancelled : Err := 'Cancelled' ;
	    qerrNotStreaming : Err := 'Streaming must be on before calling this command ' ;
	    qerrLostSync : Err := 'Frame sync was lost';
	    qerrBlackFill : Err := 'This frame is damanged; some data is missing';
	    qerrFirewireOverflow : Err := 'Firewire overflow - restart streaming';
	    qerrUnplugged : Err := 'Camera has been unplugged or turned off';
	    qerrAccessDenied : Err := 'The camera is already open';
	    qerrStreamFault : Err := 'Stream Allocation Failed.  Is there enough Bandwidth';
	    qerrQCamUpdateNeeded : Err := 'QCam driver software is insufficient for the camera';
	    qerrRoiTooSmall : Err := 'ROI is too small';

      else Err := 'Unknown error' ;
      end ;

    ShowMessage( FuncName + ': Err= ' + Err ) ;

    end ;


procedure FrameDoneCallBack(
         UsrPtr : Pointer ;        // Pointer to use data
         UserData : Cardinal ;     // Supplied user data value
         ErrCode : Integer ;
         Flags : Cardinal ) ; stdcall ;
// --------------------------------
// Frame acquired callback function
// --------------------------------
begin

      Inc(TQCAMSession(UsrPtr^).Counter) ;

      // Re-queue this frame
      QCAM_QueueFrame( TQCAMSession(UsrPtr^).CamHandle,
                       @TQCAMSession(UsrPtr^).FrameList[UserData],
                       @FrameDoneCallBack,
                       qccallBackDone,
                       UsrPtr,
                       UserData) ;
      TQCAMSession(UsrPtr^).FrameNum := UserData ;
      end ;

      initialization
    LibraryLoaded := False ;


end.
