unit HamDCAMUnit;
// ---------------------------------------
// Hamamatsu cameras supported by DCAM-API
// ---------------------------------------
// 01/05/08 Support for C4880 added
// 21/01/09 AdditionalReadoutTime added to StartCapture
// 18/3/09  JD FramePointer buffer increased 10,000 frames
// 03/4/09  JD Image-EM (C9100) now uses DCAM_TRIGMODE_SYNCREADOUT

interface

uses WinTypes,sysutils, classes, dialogs, mmsystem, messages,
     controls, math, forms, strutils ;

const

	DCAMERR_SUCCESS          = 1;            // P: */
	DCAMERR_NONE             = 0;            // no error             */
	DCAMERR_BUSY             = $80000101;   // busy; cannot process */
	DCAMERR_ABORT            = $80000102;   // abort process        */
	DCAMERR_NOTREADY         = $80000103;   // not ready state      */
	DCAMERR_NOTSTABLE        = $80000104;   // not stable state     */
	DCAMERR_UNSTABLE         = $80000105;   // O: now unstable state */
	DCAMERR_TIMEOUT          = $80000106;   // timeout              */
	DCAMERR_NOTBUSY          = $80000107;   // O: not busy state    */

	DCAMERR_NORESOURCE       = $80000201;   // O: not enough resource except memory */
	DCAMERR_NOMEMORY         = $80000203;   // not enough memory    */
	DCAMERR_NOMODULE         = $80000204;   // no sub module        */
	DCAMERR_NODRIVER         = $80000205;   // P: no driver            */
	DCAMERR_NOCAMERA         = $80000206;   // no camera            */
	DCAMERR_NOGRABBER		 = $80000207;	 // no grabber				*/
//	DCAMERR_NOCOMBINATION    = $80000208;*/ // 2.2: no combination on registry */

	DCAMERR_INVALIDMODULE	 = $80000211;   // 2.2: dcam_init() found invalid module */
	DCAMERR_INVALIDCOMMPORT	 = $80000212;   // invalid serial port */

	DCAMERR_LOSTFRAME        = $80000301;   // frame data is lost   */
	DCAMERR_COOLINGTROUBLE   = $80000302;   // something happens near cooler */

	DCAMERR_INVALIDCAMERA    = $80000806;   // invalid camera		 */
	DCAMERR_INVALIDHANDLE    = $80000807;   // invalid camera handle*/
	DCAMERR_INVALIDPARAM     = $80000808;   // invalid parameter    */

	DCAMERR_INVALIDVALUE	 = $80000821;   // invalid property value  */

// backward compatibility */
	DCAMERR_UNKNOWNMSGID     = $80000801;   // P: unknown message id   */
	DCAMERR_UNKNOWNSTRID     = $80000802;   // unknown string id    */
	DCAMERR_UNKNOWNPARAMID   = $80000803;   // unkown parameter id  */
	DCAMERR_UNKNOWNBITSTYPE  = $80000804;   // O: unknown bitmap bits type */
	DCAMERR_UNKNOWNDATATYPE  = $80000805;   // unknown frame data type  */
	DCAMERR_FAILOPENBUS      = $81001001;   // O:                   */
	DCAMERR_FAILOPENCAMERA   = $82001001;   //                      */

// internal error */
	DCAMERR_UNREACH          = $80000f01;   // internal error       */
	DCAMERR_NOTIMPLEMENT     = $80000f02;   // P: not yet implementation */
	DCAMERR_NOTSUPPORT       = $80000f03;   // function is not supported */
	DCAMERR_UNLOADED         = $80000f04;	 //	calling after process terminated */
	DCAMERR_THRUADAPTER		 = $80000f05;	 //	calling after process terminated */

	DCAMERR_FAILREADCAMERA   = $83001002;   // P: */
	DCAMERR_FAILWRITECAMERA  = $83001003;   // P: */


// ---------------------------------------------------------------- */

  DCAM_DATATYPE_NONE		=	0;
  DCAM_DATATYPE_UINT8		=	$00000001;	// bit 0 */
  DCAM_DATATYPE_UINT16	=	$00000002;	// bit 1 */
  DCAM_DATATYPE_UINT32	=	$00000008;	// bit 2 */
  DCAM_DATATYPE_INT8		=	$00000010;	// bit 4 */
  DCAM_DATATYPE_INT16		=	$00000020;	// bit 5 */
  DCAM_DATATYPE_INT32		=	$00000080;	// bit 7 */
  DCAM_DATATYPE_REAL32	=	$00000100;	//* bit 8 */
  DCAM_DATATYPE_REAL64	=	$00000200;	//-* bit 9 */
  DCAM_DATATYPE_INDEX8	=	$00010000;	//-* bit 16 */
  DCAM_DATATYPE_RGB16		=	$00020000;	//-* bit 17 */
  DCAM_DATATYPE_RGB32		=	$00080000;	//-* bit 19; */

  DCAM_DATATYPE_BGR24		=	$00000400;	// bit 10;  8bit*3; [ b0; g0; r0]; [b1; g1; r1] */
  DCAM_DATATYPE_BGR48		=	$00001000;	// bit 12; 16bit*3; [ b0; g0; r0]; [b1; g1; r1] */
  DCAM_DATATYPE_RGB24		=	$00040000;	// bit 18;  8bit*3; [ r0; g0; b0]; [r1; g1; b1] */
  DCAM_DATATYPE_RGB48		=	$00100000;	// bit 20; 16bit*3; [ r0; g0; b0]; [r1; g1; b1] */

	// just like 1394 format; Y is unsigned; U and V are signed. */
	// about U and V; signal level is from -128 to 128; data value is from $00 to $FF */
  DCAM_DATATYPE_YUV411	=	$01000000;	// 8bit; [ u0; y0; y1; v0; y2; y3 ]; [u4; y4; y5; v4; v6; y7]; */
  DCAM_DATATYPE_YUV422	=	$02000000;	// 8bit; [ u0; y0; v0; y1 ]; [u2; y2; v2; v3 ]; */
  DCAM_DATATYPE_YUV444	=	$04000000; // 8bit; [ u0; y0; v0 ]; [ u1; y1; v1 ]; */

  DCAM_BITSTYPE_NONE		=	$00000000;
  DCAM_BITSTYPE_INDEX8	=	$00000001;
  DCAM_BITSTYPE_RGB16		=	$00000002;
  DCAM_BITSTYPE_RGB24		=	$00000004;	// 8bit; [ b0; g0; r0] */
  DCAM_BITSTYPE_RGB32		=	$00000008;

//** --- camera capability	--- ***/


	DCAM_QUERYCAPABILITY_FUNCTIONS		= 0;
	DCAM_QUERYCAPABILITY_DATATYPE		= 1;
	DCAM_QUERYCAPABILITY_BITSTYPE		= 2;
	DCAM_QUERYCAPABILITY_EVENTS			= 3;

	DCAM_QUERYCAPABILITY_AREA			= 4 ;

	DCAM_CAPABILITY_BINNING2					= $00000002;
	DCAM_CAPABILITY_BINNING4					= $00000004;
	DCAM_CAPABILITY_BINNING8					= $00000008;
	DCAM_CAPABILITY_BINNING16					= $00000010;
	DCAM_CAPABILITY_BINNING32					= $00000020;
	DCAM_CAPABILITY_TRIGGER_EDGE				= $00000100;
	DCAM_CAPABILITY_TRIGGER_LEVEL				= $00000200;
	DCAM_CAPABILITY_TRIGGER_MULTISHOT_SENSITIVE = $00000400;
	DCAM_CAPABILITY_TRIGGER_CYCLE_DELAY			= $00000800;
	DCAM_CAPABILITY_TRIGGER_SOFTWARE			= $00001000;
	DCAM_CAPABILITY_TRIGGER_FASTREPETITION		= $00002000;
	DCAM_CAPABILITY_TRIGGER_TDI					= $00004000;
	DCAM_CAPABILITY_TRIGGER_TDIINTERNAL			= $00008000;
	DCAM_CAPABILITY_TRIGGER_POSI				= $00010000;
	DCAM_CAPABILITY_TRIGGER_NEGA				= $00020000;
	DCAM_CAPABILITY_TRIGGER_START				= $00040000;
									// reserved = $00080000; */
									// reserved = $00400000; */
	DCAM_CAPABILITY_TRIGGER_SYNCREADOUT			= $00800000;

	//** --- from 2.1.2 --- ***/
	DCAM_CAPABILITY_USERMEMORY					= $00100000;
	DCAM_CAPABILITY_RAWDATA						= $00200000;

	DCAM_CAPABILITY_ALL							= $00b7FF3E ;


//** --- status  --- ***/
	DCAM_STATUS_ERROR					= $0000;
	DCAM_STATUS_BUSY					= $0001;
	DCAM_STATUS_READY					= $0002;
	DCAM_STATUS_STABLE					= $0003;
	DCAM_STATUS_UNSTABLE				= $0004 ;

	DCAM_EVENT_FRAMESTART				= $0001;
	DCAM_EVENT_FRAMEEND					= $0002;	// all modules support	*/
	DCAM_EVENT_CYCLEEND					= $0004;	// all modules support	*/
	DCAM_EVENT_VVALIDBEGIN				= $0008 ;

	DCAM_UPDATE_RESOLUTION				= $0001;
	DCAM_UPDATE_AREA					= $0002;
	DCAM_UPDATE_DATATYPE				= $0004;
	DCAM_UPDATE_BITSTYPE				= $0008;
	DCAM_UPDATE_EXPOSURE				= $0010;
	DCAM_UPDATE_TRIGGER					= $0020;
	DCAM_UPDATE_DATARANGE				= $0040;
	DCAM_UPDATE_DATAFRAMEBYTES			= $0080;
	DCAM_UPDATE_PROPERTY				= $0100;
	DCAM_UPDATE_ALL						= $01ff ;

	DCAM_TRIGMODE_INTERNAL				= $0001;
	DCAM_TRIGMODE_EDGE					= $0002;
	DCAM_TRIGMODE_LEVEL					= $0004;
	DCAM_TRIGMODE_MULTISHOT_SENSITIVE	= $0008;
	DCAM_TRIGMODE_CYCLE_DELAY			= $0010;
	DCAM_TRIGMODE_SOFTWARE				= $0020;
	DCAM_TRIGMODE_FASTREPETITION		= $0040;
	DCAM_TRIGMODE_TDI					= $0080;
	DCAM_TRIGMODE_TDIINTERNAL			= $0100;
	DCAM_TRIGMODE_START					= $0200;
	DCAM_TRIGMODE_SYNCREADOUT			= $0400 ;

//** --- trigger polarity --- ***/
	DCAM_TRIGPOL_NEGATIVE				= $0000;
	DCAM_TRIGPOL_POSITIVE				= $0001 ;

//** --- string id --- ***/

   	DCAM_IDSTR_BUS =				$04000101 ;
   	DCAM_IDSTR_CAMERAID =		$04000102 ;
   	DCAM_IDSTR_VENDOR =		$04000103 ;
   	DCAM_IDSTR_MODEL =			$04000104 ;
   	DCAM_IDSTR_CAMERAVERSION =	$04000105 ;
   	DCAM_IDSTR_DRIVERVERSION =	$04000106 ;
   	DCAM_IDSTR_MODULEVERSION =	$04000107 ;
   	DCAM_IDSTR_DCAMAPIVERSION	=$04000108 ;

//*** -- iCmd parameter of dcam_extended = function() -- ***/
  	DCAM_IDMSG_QUERYPARAMCOUNT	= $0200 ;	//*		 DWORD* 		 param,   bytesize = sizeof = function( DWORD );	  */
	  DCAM_IDMSG_SETPARAM 		= $0201 ;	    //* const DCAM_HDR_PARAM* param,   bytesize = sizeof = function( parameters); */
	  DCAM_IDMSG_GETPARAM 		= $0202 ;	    //*		 DCAM_HDR_PARAM* param,   bytesize = sizeof = function( parameters); */
	  DCAM_IDMSG_QUERYPARAMID 	= $0204 ;	  //*		 DWORD			 param[], bytesize = sizeof = function( param);	  */

//*** -- parameter header -- ***/

    dcamparam_feature_featureid = $1 ;
    dcamparam_feature_flags = $2 ;
    dcamparam_feature_featurevalue = $4 ;

    dcamparam_featureinq_featureid = $1 ;
    dcamparam_featureinq_capflags = $2 ;
    dcamparam_featureinq_min = $4;
    dcamparam_featureinq_max = $8 ;
    dcamparam_featureinq_step = $10 ;
    dcamparam_featureinq_defaultvalue = $20 ;
    dcamparam_featureinq_units = $40 ;

//'featureid
    DCAM_IDFEATURE_INITIALIZE = $0 ;
    DCAM_IDFEATURE_BRIGHTNESS = $1 ;
    DCAM_IDFEATURE_GAIN = $2;
    DCAM_IDFEATURE_CONTRAST = $2 ;
    DCAM_IDFEATURE_HUE = $3 ;
    DCAM_IDFEATURE_SATURATION = $4 ;
    DCAM_IDFEATURE_SHARPNESS = $5 ;
    DCAM_IDFEATURE_GAMMA = $6 ;
    DCAM_IDFEATURE_WHITEBALANCE = $7 ;
    DCAM_IDFEATURE_PAN = $8 ;
    DCAM_IDFEATURE_TILT = $9 ;
    DCAM_IDFEATURE_ZOOM = $A ;
    DCAM_IDFEATURE_IRIS = $B ;
    DCAM_IDFEATURE_FOCUS = $C ;
    DCAM_IDFEATURE_AUTOEXPOSURE = $D ;
    DCAM_IDFEATURE_SHUTTER = $E ;
    DCAM_IDFEATURE_EXPOSURETIME = $E ;
    DCAM_IDFEATURE_TEMPERATURE = $F;
    DCAM_IDFEATURE_OPTICALFILTER = $10 ;
    DCAM_IDFEATURE_MECHANICALSHUTTER = $10 ;
    DCAM_IDFEATURE_LIGHTMODE = $11 ;
    DCAM_IDFEATURE_OFFSET = $12 ;
    //DCAM_IDFEATURE_CONTRASTOFFSET   = 0x00000012,
    DCAM_IDFEATURE_CONTRASTGAIN = $13 ;     //'12.03.2003
    DCAM_IDFEATURE_AMPLIFIERGAIN = $14 ;    //'12.03.2003
    DCAM_IDFEATURE_TEMPERATURETARGET = $15 ;//'12.03.2003
    DCAM_IDFEATURE_SENSITIVITY = $16 ;      //'12.03.2003

//' capflags only
    DCAM_FEATURE_FLAGS_READ_OUT = $10000 ;                   //' Allows the feature values to be read out.
    DCAM_FEATURE_FLAGS_DEFAULT = $20000 ;                    //' Allows DEFAULT function. If supported, when a feature's DEFAULT is turned ON, then
                                                             //   ' the values are ignored and the default setting is used.
    DCAM_FEATURE_FLAGS_STEPPING_INCONSISTENT = $40000 ;      //' step value of DCAM_PARAM_FEATURE_INQ function is not consistent across the
                                                             //   ' entire range of values. For example, if this flag is set, and:
                                                             //   '      min = 0
                                                             //   '      max = 3
                                                             //   '      step = 1
                                                             //   ' Valid values you can set may be 0,1,3 only. 2 is invalid. Therefore,
                                                             //   ' if you implement a scroll bar, Step is the minimum stepping within
                                                             //   ' the range, but a value within the range may be invalid and produce
                                                             //   ' an error. The application should be aware of this case.

//' capflags, flags get, and flags set
   DCAM_FEATURE_FLAGS_AUTO = $1 ;                           //' Auto mode (Controlled automatically by camera).
   DCAM_FEATURE_FLAGS_MANUAL = $2 ;                         //' Manual mode (Controlled by user).
   DCAM_FEATURE_FLAGS_ONE_PUSH = $100000 ;                  //' Capability allows One Push operation. Getting means One Push mode is in progress.
                                                            //    ' Setting One Push flag processes feature values once, then
                                                            //    ' turns off the feature and returns to default settings.

//' flags get and flags set
   DCAM_FEATURE_FLAGS_DEFAULT_OFF = $1000000 ;              //' Enable feature control by turning off DEFAULT. (See DCAM_FEATURE_FLAGS_DEFAULT)
   DCAM_FEATURE_FLAGS_DEFAULT_ON = $2000000 ;               //' Disable feature control and use default. (See DCAM_FEATURE_FLAGS_DEFAULT)

//'From 24.03.2003                                                                ' ** Note: If DEFAULT is ON or you turn DEFAULT ON, you must turn it OFF before
   DCAM_FEATURE_FLAGS_COOLING_ONOFF = $20000 ;                   //' capflags with DCAM_IDFEATURE_TEMPERATURE */
   DCAM_FEATURE_FLAGS_COOLING_ON = $1000000 ;                    //' flags with DCAM_IDFEATURE_TEMPERATURE */
   DCAM_FEATURE_FLAGS_COOLING_OFF = $2000000 ;                   //' flags with DCAM_IDFEATURE_TEMPERATURE */

   DCAM_FEATURE_FLAGS_MECHANICALSHUTTER_OPEN = $2000000 ;        //' flags with DCAM_IDFEATURE_MECHANICALSHUTTER */
   DCAM_FEATURE_FLAGS_MECHANICALSHUTTER_AUTO = $1000001 ;        //' flags with DCAM_IDFEATURE_MECHANICALSHUTTER */
   DCAM_FEATURE_FLAGS_MECHANICALSHUTTER_CLOSE = $1000002 ;       //' flags with DCAM_IDFEATURE_MECHANICALSHUTTER */

   DCAM_FEATURE_FLAGS_OFF = $2000000 ;
   DCAM_FEATURE_FLAGS_ONOFF = $20000 ;
   DCAM_FEATURE_FLAGS_ON = $1000000 ;
                                                                //'          trying to update a new feature value or mode.

// flags set only
   DCAM_FEATURE_FLAGS_IMMEDIATE = $4000000 ;                //' When setting a feature, you request for an immediate change.

   dcamparam_subarray_hpos = $1 ;
   dcamparam_subarray_vpos = $2 ;
   dcamparam_subarray_hsize = $4 ;
   dcamparam_subarray_vsize = $8 ;

   dcamparam_subarrayinq_binning = $1 ;
   dcamparam_subarrayinq_hmax = $2 ;
   dcamparam_subarrayinq_vmax = $4 ;
   dcamparam_subarrayinq_hposunit = $8 ;
   dcamparam_subarrayinq_vposunit = $10 ;
   dcamparam_subarrayinq_hunit = $20 ;
   dcamparam_subarrayinq_vunit = $40 ;

   dcamparam_framereadouttimeinq_framereadouttime = $1 ;

   dcamparam_scanmode_speed_slowest = $1 ;
   dcamparam_scanmode_speed_fastest = $FF ;         //  ' user specified this value, module may round down

    dcamparam_scanmodeinq_speedmax = $1 ;

    DCAM_IDMSG_SETGETPARAM = $203 ;        //  DCAM_HDR_PARAM* param, bytesize = sizeof( parameters);

    DCAM_IDMSG_SOFTWARE_TRIGGER = $400 ;   //'

//    parameter IDs

    DCAM_IDPARAM_C4742_95 = $C00181E1 ;
    DCAM_IDPARAM_C4742_95_INQ = $800181A1 ;
    DCAM_IDPARAM_C4742_95ER = $C00181E2 ;
    DCAM_IDPARAM_C4742_95ER_INQ = $800181A2 ;
    DCAM_IDPARAM_C7300 = $C00181E3;
    DCAM_IDPARAM_C7300_INQ = $800181A3 ;
    DCAM_IDPARAM_C4880 = $C00181E5 ;
    DCAM_IDPARAM_C4880_INQ = $800181A5 ;
    DCAM_IDPARAM_C8000 = $C00181E6 ;
    DCAM_IDPARAM_C8000_INQ = $800181A6 ;
    DCAM_IDPARAM_C8484 = $C00181E7 ;
    DCAM_IDPARAM_C8484_INQ = $800181A7 ;
    DCAM_IDPARAM_C4742_98BT = $C00181E8 ;
    DCAM_IDPARAM_C4742_98BT_INQ = $800181A8 ;
    DCAM_IDPARAM_C4742_95HR = $C00181E9 ;
    DCAM_IDPARAM_C4742_95HR_INQ = $800181A9 ;
    DCAM_IDPARAM_C7190_2X = $C00181EA ;
    DCAM_IDPARAM_C7190_2X_INQ = $800181AA ;
    DCAM_IDPARAM_C8000_20 = $C00181EB ;
    DCAM_IDPARAM_C8000_20_INQ = $800181AB ;
    DCAM_IDPARAM_C7780 = $C00181EC ;
    DCAM_IDPARAM_C7780_INQ = $800181AC ;
    DCAM_IDPARAM_C4742_98 = $C00281ED ;
    DCAM_IDPARAM_C4742_98_INQ = $800281AD ;
    DCAM_IDPARAM_C4742_98ER = $C00181EE ;
    DCAM_IDPARAM_C4742_98ER_INQ = $800181AE ;
    DCAM_IDPARAM_C7390 = $C00181EF ;
    DCAM_IDPARAM_C7390_INQ = $C00181AF ;
    DCAM_IDPARAM_C8190 = $C00281E1 ;
    DCAM_IDPARAM_C8190_INQ = $C00281A1 ;
    DCAM_IDPARAM_C7190_10 = $C00281E2 ;
    DCAM_IDPARAM_C7190_10_INQ = $800281A2 ;
    DCAM_IDPARAM_C8000_10 = $C00281E3 ;
    DCAM_IDPARAM_C8000_10_INQ = $800281A3 ;
    DCAM_IDPARAM_C4742_95NRK = $C00281E4 ;
    DCAM_IDPARAM_C4742_95NRK_INQ = $800281A4 ;
    DCAM_IDPARAM_C4880_80 = $C00281E5 ;
    DCAM_IDPARAM_C4880_80_INQ = $800281A5 ;
    DCAM_IDPARAM_PCDIG = $C00381E4 ;
    DCAM_IDPARAM_PCDIG_INQ = $800381A4 ;
    DCAM_IDPARAM_ICPCI = $C00381E5 ;
    DCAM_IDPARAM_ICPCI_INQ = $800381A5 ;
    DCAM_IDPARAM_IQV50 = $C00381E6 ;
    DCAM_IDPARAM_IQV50_LUT = $C00381E7 ;
    DCAM_IDPARAM_IQV50_STAT = $800381A8 ;
    DCAM_IDPARAM_MULTI = $C00481E1 ;
    DCAM_IDPARAM_MULTI_INQ = $800481A1 ;
    DCAM_IDPARAM_RGBRATIO = $C00481E2 ;

    DCAM_IDPARAM_FEATURE = $C00001E1 ;
    DCAM_IDPARAM_FEATURE_INQ = $800001A1 ;
    DCAM_IDPARAM_SUBARRAY = $C00001E2 ;
    DCAM_IDPARAM_SUBARRAY_INQ = $800001A2 ;
    DCAM_IDPARAM_FRAME_READOUT_TIME_INQ = $800001A3 ;

    DCAM_IDPARAM_SCANMODE_INQ = $800001A4 ;
    DCAM_IDPARAM_SCANMODE = $C00001E4 ;

     ccDatatype_none = 0 ;
     ccDatatype_uint8 = $1 ;
     ccDatatype_uint16 = $2 ;
     ccDatatype_uint32 = $8 ;
     ccDatatype_int8 = $10 ;
     ccDatatype_int16 = $20 ;
     ccDatatype_int32 = $80 ;
     ccDatatype_real32 = $100 ;
     ccDatatype_real64 = $200 ;
     ccDatatype_rgb8 = $10000 ;
     ccDatatype_rgb16 = $20000 ;
     ccDatatype_rgb24 = $40000 ;       // '? 8bit, [ r0, g0, b0], [r1, g1, b1]
     ccDatatype_rgb32 = $80000 ;
     ccDatatype_rgb48 = $100000 ;

//Bits Type
     ccBitstype_none = $0 ;
     ccBitstype_index8 = $1 ;
     ccBitstype_rgb16 = $2 ;
     ccBitstype_rgb24 = $4 ;               // 8bit, [ b0, g0, r0]
     ccBitstype_rgb32 = $8 ;
     ccBitstype_all = $F ;

//Precapture mode
     ccCapture_Snap = 0 ;
     ccCapture_Sequence = 1 ;

//Error values
     ccErr_none = 0 ;                       // no error
     ccErr_busy = $80000101 ;               // busy, cannot process
     ccErr_abort = $80000102 ;             // abort process
     ccErr_notready = $80000103 ;          // not ready state
     ccErr_notstable = $80000104 ;         // not stable state
     ccErr_timeout = $80000106 ;           // timeout

     ccErr_nomemory = $80000203 ;          // not enough memory

     ccErr_unknownmsgid = $80000801 ;      // unknown message id
     ccErr_unknownstrid = $80000802 ;      // unknown string id
     ccErr_unknownparamid = $80000803 ;    // unkown parameter id
     ccErr_unknownbitstype = $80000804 ;   // unknown transfer type for setbits()
     ccErr_unknowndatatype = $80000805 ;   // unknown transfer type for setdata
     ccErr_invalidhandle = $80000807 ;     // invalid camera handle
     ccErr_invalidparam = $80000808 ;      // invalid parameter

     ccErr_unreach = $80000F01 ;           // reach any point must not to run
     ccErr_notimplement = $80000F02 ;      // not yet implementation
     ccErr_notsupport = $80000F03 ;        // this driver or camera is not support

     ccErr_unstable = $80000105 ;          // now unstable state
     ccErr_noresource = $80000201 ;        // not enough resource without memory and freespace of disk
     ccErr_diskfull = $80000202 ;          // not enough freespace of disk
     ccErr_nomodule = $80000204 ;          // no sub module
     ccErr_nodriver = $80000205 ;          // no driver
     ccErr_nocamera = $80000206 ;          // no camera
     ccErr_unknowncamera = $80000806 ;     // unknown camera
     ccErr_failopen = $80001001 ;
     ccErr_failopenbus = $81001001 ;
     ccErr_failopencamera = $82001001 ;
     ccErr_failreadcamera = $83001002 ;
     ccErr_failwritecamera = $83001003 ;


//* **************************************************************** *
//	functions
// * **************************************************************** */

//*** --- error function --- ***/
type

TDCAMAPISession = record
     CamHandle : Integer ;
     NumCameras : Integer ;
     CameraModel : String ;
     NumBytesPerFrame : Integer ;     // No. of bytes in image
     NumPixelsPerFrame : Integer ;    // No. of pixels in image
     FramePointers : Array[0..9999] of Pointer ;
     NumFrames : Integer ;            // No. of images in circular transfer buffer
     FrameNum : Integer ;             // Current frame no.
     PFrameBuffer : Pointer ;         // Frame buffer pointer
     ImageBufferSize : Integer ;           // No. images in Andor image buffer
     PImageBuffer : PIntegerArray ;        // Local Andor image buffer
     NumFramesAcquired : Integer ;
     NumFramesCopied : Integer ;
     GetImageInUse : Boolean ;       // GetImage procedure running
     CapturingImages : Boolean ;     // Image capture in progress
     CameraOpen : Boolean ;          // Camera open for use
     TimeStart : single ;
     Temperature : Integer ;
     FrameTransferTime : Single ;    // Frame transfer time (s)
     NumFeatures : Cardinal ;
     FeatureID : Array[0..31] of Cardinal ;
     BinFactors : Array[0..9] of Integer ;
     NumBinFactors : Integer ;
     Gains : Array[0..999] of Single ;
     NumGains : Integer ;
     GainID : Integer ;

     // Current settings
     ReadoutSpeed : Integer ;         // Readout rate (index no.)
     FrameLeft : Integer ;            // Left pixel in CCD readout area
     FrameRight : Integer ;           // Right pixel in CCD eadout area
     FrameTop : Integer ;             // Top of CCD readout area
     FrameBottom : Integer ;          // Bottom of CCD readout area
     BinFactor : Integer ;           // Pixel binning factor (In)
     FrameWidth : Integer ;          // Image width
     FrameHeight : Integer ;         // Image height
     FrameInterval : Double ;        // Time interval between frames (s)
     ReadoutTime : Double  ;        // Frame readout time (s)

     end ;


TDCAM_HDR_PARAM = packed record
	Size : DWord ;						//* size of whole structure */
	id : DWord ;							//* specify the kind of this structure */
	iFlag : DWord ;						//* specify the member to be set or requested by application */
	oFlag : DWord ;						//* specify the member to be set or gotten by module */
  end ;

TDCAM_PARAM_FEATURE = packed record
    HDR : TDCAM_HDR_PARAM ;
    featureid : Integer ;   //' [in]
    Flags : Integer ;       //' [in/out]
    featurevalue : Single ; //     ' [in/out]
    LastElement : Integer ; //  'For size measurement purposes
    end ;

TDCAM_PARAM_FEATURE_INQ = packed record
    HDR : TDCAM_HDR_PARAM ;        //        ' id == DCAM_IDPARAM_FEATURE_INQ */
    featureid : Integer ;//' [in]
    capflags : Integer ; //' [out]    /
    min : Single ;       //       ' [out]
    max : Single ;       //       ' [out]
    step : Single ;      //       ' [out]
    defaultvalue : Single ; //    ' [out]
    units : Array[0..15] of char ; //' [out]
    LastElement : Integer ;  //'For size measurement purposes
    end ;

TDCAM_PARAM_SUBARRAY = packed record
    HDR : TDCAM_HDR_PARAM ;      // id == DCAM_IDPARAM_SUBARRAY */
    hpos : Integer ;             // ' [in/out]
    vpos : Integer ;             //  ' [in/out]
    hsize : Integer ;            //  ' [in/out]
    vsize : Integer ;            //  ' [in/out]
    LastElement : Integer ;      //'For size measurement purposes
    end ;

TDCAM_PARAM_SUBARRAY_INQ = packed record
    HDR : TDCAM_HDR_PARAM ;  //              ' id == DCAM_IDPARAM_SUBARRAY_INQ */
    binning : Integer ; //            ' [in]
    hmax : Integer ; //               ' [out]
    vmax : Integer ; //               ' [out]
    hposunit : Integer ; //           ' [out]
    vposunit : Integer ; //           ' [out]
    hunit : Integer ; //             ' [out]
    vunit : Integer ; //              ' [out]
    LastElement : Integer ; // 'For size measurement purposes
    end ;

TDCAM_PARAM_FRAME_READOUT_TIME_INQ = packed record
    HDR : TDCAM_HDR_PARAM ;  //                ' id == DCAM_IDPARAM_FRAME_READOUT_TIME_INQ */
    framereadouttime : Double ; //  ' [out]
    LastElement : Integer ; // 'For size measurement purposes
    end ;

TDCAM_PARAM_SCANMODE = packed record
    HDR : TDCAM_HDR_PARAM ;  //               ' id == DCAM_IDPARAM_SCANMODE */
    speed : Integer ; //            ' [in/out]
    LastElement : Integer ; // 'For size measurement purposes
    end ;

TDCAM_PARAM_SCANMODE_INQ = packed record
    HDR : TDCAM_HDR_PARAM ;  //                ' id == DCAM_IDPARAM_SCANMODE_INQ */
    speedmax : Integer ; //           ' [out]
    LastElement : Integer ; //  'For size measurement purposes
    end ;


Tdcam_getlasterror = function(
                     HDCAM : THandle ;
                     Buf : PChar ;
                     bytesize : Integer
                     ) : Integer ; stdcall ;

//*** --- initialize and finalize --- ***/

Tdcam_init = function(
             hInst : Integer ;
             var Count : Integer ;
             Reserved : Pointer
             ) : Integer ; stdcall ;

Tdcam_uninit = function(
               hInst : Integer ;
               Reserved : Pointer
               ) : Integer ; stdcall ;

Tdcam_getmodelinfo = function(
                     Index : Integer ;
                     StringID : Integer ;
                     Buf : PChar ;
                     bytesize : Integer
                     ) : Integer ; stdcall ;

Tdcam_open = function(
             HDCam : Pointer ;
             Index : Integer ;
             Reserved : Pointer
               ) : Integer ; stdcall ;

Tdcam_close	= function(
              HDCam : THandle
              ) : Integer ; stdcall ;

//*** --- camera infomation --- ***/

Tdcam_getstring	= function(
                  HDCam : THandle ;
                  StringID : PChar ;
                  Buf : PChar ;
                  ByteSize : Integer
                  ) : Integer ; stdcall ;

Tdcam_getcapability	= function(
                      HDCam : THandle ;
                      var Capability : Cardinal ;
                      CapTypeID : Cardinal
                      ) : Integer ; stdcall ;

Tdcam_getdatatype	= function(
                    HDCam : THandle ;
                    var Datatype : DWord
                    ) : Integer ; stdcall ;

Tdcam_getbitstype = function(
                    HDCam : THandle ;
                    var Bitstype : Cardinal
                    ) : Integer ; stdcall ;

Tdcam_setdatatype = function(
                    HDCam : THandle ;
                    Datatype : Cardinal
                    ) : Integer ; stdcall ;

Tdcam_setbitstype = function(
                    HDCam : THandle ;
                    Bitstype : Cardinal
                    ) : Integer ; stdcall ;

Tdcam_getdatasize = function(
                    HDCam : THandle ;
                    pBuf : Pointer
                    ) : Integer ; stdcall ;

Tdcam_getbitssize = function(
                    HDCam : THandle ;
                    var Size : Cardinal
                    ) : Integer ; stdcall ;

//*** --- parameters --- ***/

Tdcam_queryupdate = function(
                    HDCam : THandle ;
                    var Flag : Cardinal ;
                    Reserved  : Cardinal
                    ) : Integer ; stdcall ;

Tdcam_getbinning = function(
                   HDCam : THandle ;
                   var Binning : Cardinal
                   ) : Integer ; stdcall ;

Tdcam_getexposuretime	= function(
                        HDCam : THandle ;
                        var Sec : Double
                        ) : Integer ; stdcall ;

Tdcam_gettriggermode = function(
                       HDCam : THandle ;
                       var Mode : Cardinal
                       ) : Integer ; stdcall ;

Tdcam_gettriggerpolarity = function(
                           HDCam : THandle ;
                           var Polarity : Cardinal
                           ) : Integer ; stdcall ;

Tdcam_setbinning = function(
                   HDCam : THandle ;
                   binning : Cardinal
                   ) : Integer ; stdcall ;

Tdcam_setexposuretime	= function(
                        HDCam : THandle ;
                        Sec : Double
                        ) : Integer ; stdcall ;

Tdcam_settriggermode = function(
                       HDCam : THandle ;
                       Mode : Cardinal
                       ) : Integer ; stdcall ;

Tdcam_settriggerpolarity = function(
                           HDCam : THandle ;
                           Polarity  : Cardinal
                           ) : Integer ; stdcall ;

//*** --- capturing --- ***/

Tdcam_precapture = function(
                   HDCam : THandle ;
                   CaptureMode : Cardinal
                   ) : Integer ; stdcall ;

Tdcam_getdatarange = function(
                     HDCam : THandle ;
                     var Max : Integer ;
                     var Min : Integer
                     ) : Integer ; stdcall ;

Tdcam_getdataframebytes	= function(
                          HDCam : THandle ;
                          var Size: Cardinal
                          ) : Integer ; stdcall ;

Tdcam_allocframe = function(
                   HDCam : THandle ;
                   Frame : Cardinal
                   ) : Integer ; stdcall ;

Tdcam_getframecount = function(
                      HDCam : THandle ;
                      var Frame  : Cardinal
                      ) : Integer ; stdcall ;

Tdcam_capture	 = function(
                 HDCam : THandle
                 ) : Integer ; stdcall ;

Tdcam_idle = function(
             HDCam : THandle
             ) : Integer ; stdcall ;

Tdcam_wait = function(
             HDCam : THandle ;
             var Code : Cardinal ;
             timeout : Cardinal ;
             Event : THandle
             ) : Integer ; stdcall ;

Tdcam_getstatus	= function(
                  HDCam : THandle ;
                  var Status : Integer
                  ) : Integer ; stdcall ;

Tdcam_gettransferinfo	= function(
                        HDCam : THandle ;
                        var NewestFrameIndex : Cardinal ;
                        var FrameCount : Cardinal
                        ) : Integer ; stdcall ;

Tdcam_freeframe	 = function(
                   HDCam : THandle
                   ) : Integer ; stdcall ;

//*** --- user memory support --- ***/

Tdcam_attachbuffer = function(
                     HDCam : THandle ;
                     pTop : Pointer ;
                     size : Cardinal
                     ) : Integer ; stdcall ;

Tdcam_releasebuffer = function(
                      HDCam : THandle
                      ) : Integer ; stdcall ;

//*** --- data transfer --- ***/

Tdcam_lockdata = function(
                 HDCam : THandle ;
                 pTop : Pointer ;
                 var Rowbytes : Cardinal ;
                 frame : Cardinal
                 ) : Integer ; stdcall ;

Tdcam_lockbits = function(
                 HDCam : THandle ;
                 Top : Pointer ;
                 var Rowbytes : Cardinal ;
                 frame : Cardinal
                 ) : Integer ; stdcall ;

Tdcam_unlockdata = function(
                   HDCam : THandle
                   ) : Integer ; stdcall ;

Tdcam_unlockbits = function(
                   HDCam : THandle
                   ) : Integer ; stdcall ;

//*** --- LUT --- ***/

Tdcam_setbitsinputlutrange = function(
                             HDCam : THandle ;
                             inMax : Cardinal ;
                             inMin : Cardinal
                             ) : Integer ; stdcall ;

Tdcam_setbitsoutputlutrange	= function(
                              HDCam : THandle ;
                              outMax : Byte ;
                              outMin : Byte
                              ) : Integer ; stdcall ;

//*** --- Control Panel --- ***/

Tdcam_showpanel	 = function(
                   HDCam : THandle ;
                   hWnd : THandle ;
                   reserved : Cardinal
                   ) : Integer ; stdcall ;

//*** --- extended --- ***/

Tdcam_extended	 = function(
                   HDCam : THandle ;
                   iCmd : Cardinal ;
                   param : Pointer ;
                   size : Cardinal
                   ) : Integer ; stdcall ;

//*** --- software trigger --- ***/
Tdcam_firetrigger = function(
                    HDCam : THandle
                    ) : Integer ; stdcall ;


// Public procedures

function DCAMAPI_LoadLibrary : Boolean ;

function DCAMAPI_GetDLLAddress(
         Handle : Integer ;
         const ProcName : string ) : Pointer ;

function DCAMAPI_OpenCamera(
          var Session : TDCAMAPISession ;   // Camera session record
          var FrameWidthMax : Integer ;      // Returns camera frame width
          var FrameHeightMax : Integer ;     // Returns camera frame width
          var NumBytesPerPixel : Integer ;   // Returns bytes/pixel
          var PixelDepth : Integer ;         // Returns no. bits/pixel
          var PixelWidth : Single ;          // Returns pixel size (um)
          var BinFactorMax : Integer ;        // Max. bin factor
          CameraInfo : TStringList         // Returns Camera details
          ) : Boolean ;

procedure DCAMAPI_CloseCamera(
          var Session : TDCAMAPISession // Session record
          ) ;

procedure DCAMAPI_GetCameraGainList(
          var Session : TDCAMAPISession ;
          CameraGainList : TStringList
          ) ;

procedure DCAMAPI_GetCameraReadoutSpeedList(
          var Session : TDCAMAPISession ;
          CameraReadoutSpeedList : TStringList
          ) ;

function DCAMAPI_StartCapture(
         var Session : TDCAMAPISession ;   // Camera session record
         var InterFrameTimeInterval : Double ;      // Frame exposure time
         var AdditionalReadoutTime : Double ;
         AmpGain : Integer ;              // Camera amplifier gain index
         ReadoutSpeed : Integer ;         // Camera Read speed index number
         ExternalTrigger : Integer ;      // Trigger mode
         FrameLeft : Integer ;            // Left pixel in CCD readout area
         FrameRight : Integer ;           // Right pixel in CCD readout area
         FrameTop : Integer ;             // Top pixel in CCD eadout area
         FrameBottom : Integer ;          // Bottom pixel in CCD readout area
         BinFactor : Integer ;             // Binning factor (1,2,4,8,16)
         PFrameBuffer : Pointer ;         // Pointer to start of ring buffer
         NumFramesInBuffer : Integer ;    // No. of frames in ring buffer
         NumBytesPerFrame : Integer       // No. of bytes/frame
         ) : Boolean ;

function DCAMAPI_CheckFrameInterval(
          var Session : TDCAMAPISession ;   // Camera session record
          FrameLeft : Integer ;   // Left edge of capture region (In)
          FrameRight : Integer ;  // Right edge of capture region( In)
          FrameTop : Integer ;    // Top edge of capture region( In)
          FrameBottom : Integer ; // Bottom edge of capture region (In)
          BinFactor : Integer ;   // Pixel binning factor (In)
          Var FrameInterval : Double ;
          Var ReadoutTime : Double) : Boolean ;


procedure DCAMAPI_Wait( Delay : Single ) ;


procedure DCAMAPI_GetImage(
          var Session : TDCAMAPISession  // Camera session record
          ) ;

procedure DCAMAPI_StopCapture(
          var Session : TDCAMAPISession   // Camera session record
          ) ;

procedure DCAMAPI_CheckError(
          FuncName : String ;   // Name of function called
          ErrNum : Integer      // Error # returned by function
          ) ;

procedure DCAMAPI_CheckROIBoundaries(
          var Session : TDCAMAPISession ;   // Camera session record
         var ReadoutSpeed : Integer ;         // Readout rate (index no.)
         var FrameLeft : Integer ;            // Left pixel in CCD readout area
         var FrameRight : Integer ;           // Right pixel in CCD eadout area
         var FrameTop : Integer ;             // Top of CCD readout area
         var FrameBottom : Integer ;          // Bottom of CCD readout area
         var BinFactor : Integer ;           // Pixel binning factor (In)
         var FrameWidth : Integer ;          // Image width
         var FrameHeight : Integer ;         // Image height
         var FrameInterval : Double ;        // Time interval between frames (s)
         var ReadoutTime : Double ) ;        // Frame readout time (s)

function DCAMAPI_CheckStepSize( Value : Integer ;
                                StepSize : Integer ) : Integer ;


implementation

uses SESCam ;

var

   dcam_getlasterror : Tdcam_getlasterror ;
   dcam_init : Tdcam_init ;
   dcam_uninit : Tdcam_uninit ;
   dcam_getmodelinfo : Tdcam_getmodelinfo ;
   dcam_open : Tdcam_open ;
   dcam_close : Tdcam_close ;
   dcam_getstring : Tdcam_getstring ;
   dcam_getcapability : Tdcam_getcapability ;
   dcam_getdatatype : Tdcam_getdatatype ;
   dcam_getbitstype : Tdcam_getbitstype ;
   dcam_setdatatype : Tdcam_setdatatype ;
   dcam_setbitstype : Tdcam_setbitstype ;
   dcam_getdatasize : Tdcam_getdatasize ;
   dcam_getbitssize : Tdcam_getbitssize ;
   dcam_queryupdate : Tdcam_queryupdate ;
   dcam_getbinning : Tdcam_getbinning ;
   dcam_getexposuretime : Tdcam_getexposuretime ;
   dcam_gettriggermode : Tdcam_gettriggermode ;
   dcam_gettriggerpolarity : Tdcam_gettriggerpolarity ;
   dcam_setbinning : Tdcam_setbinning ;
   dcam_setexposuretime : Tdcam_setexposuretime ;
   dcam_settriggermode : Tdcam_settriggermode ;
   dcam_settriggerpolarity : Tdcam_settriggerpolarity ;
   dcam_precapture : Tdcam_precapture ;
   dcam_getdatarange : Tdcam_getdatarange ;
   dcam_getdataframebytes	: Tdcam_getdataframebytes	;
   dcam_allocframe : Tdcam_allocframe ;
   dcam_getframecount : Tdcam_getframecount ;
   dcam_capture : Tdcam_capture ;
   dcam_idle : Tdcam_idle ;
   dcam_wait : Tdcam_wait ;
   dcam_getstatus : Tdcam_getstatus ;
   dcam_gettransferinfo : Tdcam_gettransferinfo ;
   dcam_freeframe : Tdcam_freeframe ;
   dcam_attachbuffer : Tdcam_attachbuffer ;
   dcam_releasebuffer  : Tdcam_releasebuffer ;
   dcam_lockdata : Tdcam_lockdata ;
   dcam_lockbits : Tdcam_lockbits ;
   dcam_unlockdata : Tdcam_unlockdata ;
   dcam_unlockbits : Tdcam_unlockbits ;
   dcam_setbitsinputlutrange : Tdcam_setbitsinputlutrange ;
   dcam_setbitsoutputlutrange  : Tdcam_setbitsoutputlutrange ;
   dcam_showpanel : Tdcam_showpanel ;
   dcam_firetrigger : Tdcam_firetrigger ;
   dcam_extended : Tdcam_extended ;

   LibraryLoaded : Boolean ;
   LibraryHnd : THandle ;

function DCAMAPI_LoadLibrary  : Boolean ;
{ ---------------------------------------------
  Load camera interface DLL library into memory
  ---------------------------------------------}
var
    LibFileName : string ;
begin


     Result := LibraryLoaded ;

     if LibraryLoaded then Exit ;


     { Load DLL camera interface library }
     LibFileName := 'dcamapi.dll' ;
     LibraryHnd := LoadLibrary( PChar(LibFileName));
     if LibraryHnd <= 0 then begin
        ShowMessage( 'DCAMAPI: ' + LibFileName + ' not found!' ) ;
        Exit ;
        end ;

     @dcam_firetrigger := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_firetrigger') ;
     @dcam_showpanel := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_showpanel') ;
     @dcam_setbitsoutputlutrange := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_setbitsoutputlutrange') ;
     @dcam_setbitsinputlutrange := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_setbitsinputlutrange') ;
     @dcam_unlockbits := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_unlockbits') ;
     @dcam_unlockdata := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_unlockdata') ;
     @dcam_lockbits := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_lockbits') ;
     @dcam_lockdata := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_lockdata') ;
     @dcam_releasebuffer := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_releasebuffer') ;
     @dcam_attachbuffer := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_attachbuffer') ;
     @dcam_freeframe := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_freeframe') ;
     @dcam_gettransferinfo := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_gettransferinfo') ;
     @dcam_getstatus := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_getstatus') ;
     @dcam_wait := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_wait') ;
     @dcam_idle := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_idle') ;
     @dcam_capture := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_capture') ;
     @dcam_getframecount := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_getframecount') ;
     @dcam_allocframe := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_allocframe') ;
     @dcam_getdataframebytes := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_getdataframebytes') ;
     @dcam_getdatarange := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_getdatarange') ;
     @dcam_precapture := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_precapture') ;
     @dcam_settriggerpolarity := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_settriggerpolarity') ;
     @dcam_settriggermode := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_settriggermode') ;
     @dcam_setexposuretime := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_setexposuretime') ;
     @dcam_setbinning := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_setbinning') ;
     @dcam_gettriggerpolarity := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_gettriggerpolarity') ;
     @dcam_gettriggermode := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_gettriggermode') ;
     @dcam_getexposuretime := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_getexposuretime') ;
     @dcam_getbinning := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_getbinning') ;
     @dcam_queryupdate := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_queryupdate') ;
     @dcam_getbitssize := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_getbitssize') ;
     @dcam_getdatasize := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_getdatasize') ;
     @dcam_setbitstype := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_setbitstype') ;
     @dcam_setdatatype := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_setdatatype') ;
     @dcam_getbitstype := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_getbitstype') ;
     @dcam_getdatatype := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_getdatatype') ;
     @dcam_getcapability := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_getcapability') ;
     @dcam_getstring := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_getstring') ;
     @dcam_close := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_close') ;
     @dcam_open := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_open') ;
     @dcam_getmodelinfo := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_getmodelinfo') ;
     @dcam_uninit := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_uninit') ;
     @dcam_init := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_init') ;
     @dcam_getlasterror := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_getlasterror') ;
     @dcam_extended := DCAMAPI_GetDLLAddress(LibraryHnd,'dcam_extended') ;

     LibraryLoaded := True ;
     Result := LibraryLoaded ;

     end ;


function DCAMAPI_GetDLLAddress(
         Handle : Integer ;
         const ProcName : string ) : Pointer ;
// -----------------------------------------
// Get address of procedure within DLL
// -----------------------------------------
begin
    Result := GetProcAddress(Handle,PChar(ProcName)) ;
    if Result = Nil then ShowMessage('dcamapi.dll: ' + ProcName + ' not found') ;
    end ;


function DCAMAPI_OpenCamera(
         var Session : TDCAMAPISession ;   // Camera session record
         var FrameWidthMax : Integer ;      // Returns camera frame width
         var FrameHeightMax : Integer ;     // Returns camera frame width
         var NumBytesPerPixel : Integer ;   // Returns bytes/pixel
         var PixelDepth : Integer ;         // Returns no. bits/pixel
         var PixelWidth : Single ;          // Returns pixel size (um)
         var BinFactorMax : Integer ;        // Max. bin factor
         CameraInfo : TStringList         // Returns Camera details
         ) : Boolean ;
// -------------------
// Open camera for use
// -------------------
const
    BufSize = 200 ;
var
    cBuf : Array[0..BufSize] of Char ;
    iBuf : Array[0..BufSize] of Integer ;
    iValue,iMax,iMin : Integer ;
    dwValue : DWord ;
    s : String ;
    InqParam : TDCAM_PARAM_FEATURE_INQ ;
    Err : Integer ;
begin

     // Load DLL camera control library
     if not DCAMAPI_LoadLibrary then Exit ;

 //    if dcam_init( {Application.Handle}0,Session.NumCameras,Nil ) then begin
 //       ShowMessage('DCAM: Unable to initialise DCAM Manager!') ;
//        Exit ;
//       end ;

        Err := dcam_init( 0,Session.NumCameras,Nil ) ;


     if Session.NumCameras <= 0 then begin
        ShowMessage(format('DCAM: No cameras available! %d',[Err])) ;
        Exit ;
        end ;

 //    if dcam_open( @Session.CamHandle, 0, nil ) then begin
 //       ShowMessage('DCAM: Unable to open camera!') ;
 //       Exit ;
 //       end ;
     dcam_open( @Session.CamHandle, 0, nil ) ;

     // Get camera information
     dcam_getmodelinfo( 0, DCAM_IDSTR_MODEL, cBuf, BufSize ) ;
     Session.CameraModel := cBuf ;
     s := format( ' Camera: %s',[cBuf] ) ;
     dcam_getmodelinfo(0, DCAM_IDSTR_CAMERAVERSION, cBuf, BufSize) ;
     s := s + format( ' V %s',[cBuf] ) ;
     dcam_getmodelinfo(0, DCAM_IDSTR_CAMERAID, cBuf, BufSize) ;
     s := s + format( ' (%s)',[cBuf] ) ;
     CameraInfo.Add( s ) ;

     // Get interface bus type
     dcam_getmodelinfo(0, DCAM_IDSTR_BUS, cBuf, BufSize) ;
     CameraInfo.Add(format( 'Interface type: (%s)',[cBuf] )) ;

     // Get driver information
     dcam_getmodelinfo(0, DCAM_IDSTR_DRIVERVERSION, cBuf, BufSize) ;
     CameraInfo.Add(format( 'Driver: %s',[cBuf] )) ;

     // Get module information
     dcam_getmodelinfo(0, DCAM_IDSTR_MODULEVERSION, cBuf, BufSize) ;
     CameraInfo.Add(format( 'Module: %s',[cBuf] )) ;

     // Get max. size of camera image
     dcam_getdatasize( Session.CamHandle, @iBuf ) ;
     FrameWidthMax := iBuf[0] ;
     FrameHeightMax := iBuf[1] ;

     // Get number of bits per pixel
     dcam_getdatarange( Session.CamHandle, iMax, iMin ) ;
     PixelDepth := 0 ;
     iMax := iMax + 1 ;
     while iMax > 1 do begin
        Inc(PixelDepth) ; ;
        iMax := iMax div 2 ;
        end ;

     dcam_getmodelinfo( 0, DCAM_IDSTR_MODEL, cBuf, BufSize ) ;
     if Pos('1024',cBuf) > 0 then PixelWidth := 13
     else if Pos('512',cBuf) > 0 then PixelWidth := 24
     else PixelWidth := 6.45 ;

     CameraInfo.Add( format('CCD: %d x %d x %.3gum pixels (%d bits/pixel)',
                     [FrameWidthMax,FrameHeightMax,PixelWidth,PixelDepth] )) ;

     // Get numbers of bytes / pixel
     dcam_getdatatype( Session.CamHandle, dwValue ) ;
     case dwValue of
        DCAM_DATATYPE_UINT8,DCAM_DATATYPE_INT8 : NumBytesPerPixel := 1 ;
        DCAM_DATATYPE_UINT16,DCAM_DATATYPE_INT16 : NumBytesPerPixel := 2 ;
        else NumBytesPerPixel := 4 ;
        end ;

     // Get pixel binning capabilities
     Session.NumBinFactors := 0 ;
     Session.BinFactors[Session.NumBinFactors] := 1 ;
     Inc(Session.NumBinFactors) ;

     dcam_getcapability( Session.CamHandle, dwValue, DCAM_QUERYCAPABILITY_FUNCTIONS ) ;
     s := 'Pixel Binning: ' ;
     if (dwValue AND DCAM_CAPABILITY_BINNING2) > 0 then begin
         s := s + '2x2, ';
         Session.BinFactors[Session.NumBinFactors] := 2 ;
         Inc(Session.NumBinFactors) ;
         end ;
     if (dwValue AND DCAM_CAPABILITY_BINNING4) > 0 then begin
        s := s + '4x4, ';
         Session.BinFactors[Session.NumBinFactors] := 4 ;
         Inc(Session.NumBinFactors) ;
         end ;
     if (dwValue AND DCAM_CAPABILITY_BINNING8) > 0 then begin
        s := s + '8x8 ';
         Session.BinFactors[Session.NumBinFactors] := 8 ;
         Inc(Session.NumBinFactors) ;
         end ;

     CameraInfo.Add( s ) ;

     BinFactorMax := Session.BinFactors[Session.NumBinFactors-1] ;

     if (dwValue AND DCAM_CAPABILITY_USERMEMORY) > 0 then
        CameraInfo.Add('Direct capture to user memory supported') ;

     // Get number of camera special features
    dcam_extended( Session.CamHandle,
                   DCAM_IDMSG_QUERYPARAMCOUNT,
                   @Session.NumFeatures,
                   SizeOf(Session.NumFeatures)) ;

     Session.FeatureID[0] := 0 ;
     // Get list of camera special feature IDs
     dcam_extended( Session.CamHandle,
                    DCAM_IDMSG_QUERYPARAMID,
                    @Session.FeatureID,
                    SizeOf(Session.FeatureID)) ;

     Session.CameraOpen := True ;
     Session.CapturingImages := False ;
     Result := Session.CameraOpen ;

     end ;


procedure DCAMAPI_CloseCamera(
          var Session : TDCAMAPISession // Session record
          ) ;
// ------------
// Close camera
// ------------
begin

    if not Session.CameraOpen then Exit ;

    // Stop capture if in progress
    if Session.CapturingImages then DCAMAPI_StopCapture( Session ) ;

    // Close camera
    dcam_close( Session.CamHandle ) ;

    // Close DCAM-API
    dcam_uninit( 0, Nil ) ;

    Session.CameraOpen := False ;

    end ;


procedure DCAMAPI_GetCameraGainList(
          var Session : TDCAMAPISession ;
          CameraGainList : TStringList
          ) ;
// --------------------
// Get camera gain list
// --------------------
var
    Param : TDCAM_PARAM_FEATURE_INQ ;
    Gain : Single ;
    Err : Integer ;
begin

    // Get gain information
    Param.HDR.Size := Sizeof(Param) ;
    Param.HDR.id := DCAM_IDPARAM_FEATURE_INQ ;
    Param.HDR.iFlag := dcamparam_featureinq_featureid or
                       dcamparam_featureinq_capflags or
                       dcamparam_featureinq_min or
                       dcamparam_featureinq_max or
                       dcamparam_featureinq_step or
                       dcamparam_featureinq_defaultvalue or
                       dcamparam_featureinq_units ;
    Param.HDR.oFlag := 0 ;

    Param.min := -1 ; // Set these to -1 as flags to indicate feature exists
    Param.max := -1 ;

    // Use sensitivity feature to set gain
    Session.GainID := DCAM_IDFEATURE_SENSITIVITY ;
    Param.featureid := Session.GainID ;
    dcam_extended( Session.CamHandle,
                   DCAM_IDMSG_GETPARAM,
                   @Param,
                   SizeOf(Param)) ;

   if (Param.min = -1) and (Param.max = -1) then begin
      // Use standard gain
      Session.GainID := DCAM_IDFEATURE_GAIN ;
      Param.featureid := Session.GainID ;
      Param.min := -1 ;
      Param.max := -1 ;
      dcam_extended( Session.CamHandle,
                     DCAM_IDMSG_GETPARAM,
                     @Param,
                     SizeOf(Param)) ;
      end ;

    outputdebugString(PChar(format('Err=%d',[Err])));
    // Set to gains available from list
    Session.NumGains := 0 ;
    CameraGainList.Clear ;
    Gain := Param.Min ;
    while (Gain <= Param.Max) and (Session.NumGains <= High(Session.Gains)) do begin
       Session.Gains[Session.NumGains] := Gain ;
       CameraGainList.Add( format( 'X%.1f',[Session.Gains[Session.NumGains]+1]));
       Inc( Session.NumGains ) ;
       Gain := Gain + Param.Step ;
       end ;

    end ;


procedure DCAMAPI_GetCameraReadoutSpeedList(
          var Session : TDCAMAPISession ;
          CameraReadoutSpeedList : TStringList
          ) ;
var
    Param : TDCAM_PARAM_SCANMODE_INQ ;
    i : Integer ;
    s : String ;
begin

    // Get maximum scan mode
    Param.HDR.Size := Sizeof(Param) ;
    Param.HDR.id := DCAM_IDPARAM_SCANMODE_INQ ;
    Param.HDR.iFlag := dcamparam_feature_featureid or
                       dcamparam_feature_flags or
                       dcamparam_feature_featurevalue ;
    Param.HDR.oFlag := 0 ;
    dcam_extended( Session.CamHandle,
                   DCAM_IDMSG_GETPARAM,
                   @Param,
                   SizeOf(Param)) ;

    CameraReadoutSpeedList.Clear ;
    for i := 1 to Param.SpeedMax do begin
       s := format( '%d',[i]) ;
       if i = 1 then s := s + ' (slow)' ;
       if i = Param.SpeedMax then s := s + ' (fast)' ;
       CameraReadoutSpeedList.Add( s );
       end ;

    end ;



function DCAMAPI_StartCapture(
         var Session : TDCAMAPISession ;   // Camera session record
         var InterFrameTimeInterval : Double ;      // Frame exposure time
         var AdditionalReadoutTime : Double ;
         AmpGain : Integer ;              // Camera amplifier gain index
         ReadoutSpeed : Integer ;         // Camera Read speed index number
         ExternalTrigger : Integer ;      // Trigger mode
         FrameLeft : Integer ;            // Left pixel in CCD readout area
         FrameRight : Integer ;           // Right pixel in CCD readout area
         FrameTop : Integer ;             // Top pixel in CCD eadout area
         FrameBottom : Integer ;          // Bottom pixel in CCD readout area
         BinFactor : Integer ;             // Binning factor (1,2,4,8,16)
         PFrameBuffer : Pointer ;         // Pointer to start of ring buffer
         NumFramesInBuffer : Integer ;    // No. of frames in ring buffer
         NumBytesPerFrame : Integer       // No. of bytes/frame
         ) : Boolean ;
var
    i,Err : Integer ;
    BufPointers : Array[0..999] of Pointer ;
    FrameWidth,FrameHeight : Integer ;
    ReadoutTime : Double ;
    Param : TDCAM_PARAM_FEATURE ;
    ExposureTime : Double ;
begin

    if not Session.CameraOpen then Exit ;

    // Set CCD readout region
    DCAMAPI_CheckROIBoundaries( Session,
                                ReadoutSpeed,
                                FrameLeft,
                                FrameRight,
                                FrameTop,
                                FrameBottom,
                                BinFactor,
                                FrameWidth,
                                FrameHeight,
                                InterFrameTimeInterval,
                                ReadoutTime) ;

    // Set exposure trigger mode
    if ExternalTrigger = camExtTrigger then begin
       if ANSIContainsText(Session.CameraModel, 'C9100') then begin
          dcam_settriggermode( Session.CamHandle, DCAM_TRIGMODE_SYNCREADOUT ) ;
          end
       else dcam_settriggermode( Session.CamHandle, DCAM_TRIGMODE_EDGE ) ;
       dcam_settriggerpolarity( Session.CamHandle, DCAM_TRIGPOL_POSITIVE ) ;
       end
    else begin
       dcam_settriggermode( Session.CamHandle, DCAM_TRIGMODE_INTERNAL ) ;
       end ;

    // Pre-capture camera initialisations
    dcam_precapture( Session.CamHandle, ccCapture_Sequence ) ;

   // Set camera gain

    Param.HDR.Size := Sizeof(Param) ;
    Param.HDR.id := DCAM_IDPARAM_FEATURE ;
    Param.HDR.iFlag := dcamparam_feature_featureid or
                       dcamparam_feature_flags or
                       dcamparam_feature_featurevalue ;
    Param.HDR.oFlag := 0 ;
    Param.FeatureID := Session.GainID ;
    Param.Flags := DCAM_FEATURE_FLAGS_MANUAL ;
    Param.featurevalue := Session.Gains[AmpGain] ;

    dcam_extended( Session.CamHandle,
                   DCAM_IDMSG_SETPARAM,
                   @Param,
                   SizeOf(Param)) ;

    // Set pointers to frames withi image capture buffer
    for i := 0 to NumFramesInBuffer-1 do
        Session.FramePointers[i] := Pointer(Cardinal(PFrameBuffer) + i*NumBytesPerFrame) ;

    // Attach image capture buffer
    Err := dcam_attachbuffer( Session.CamHandle, @Session.FramePointers,
                              NumFramesInBuffer*4) ;
 //   outputdebugString(PChar(format('dcam_attachbuffer Err=%d',[Err])));

    // Set exposure time
    if ExternalTrigger = camExtTrigger then begin
       if ANSIContainsText(Session.CameraModel, 'C9100') then begin
          dcam_settriggermode( Session.CamHandle, DCAM_TRIGMODE_SYNCREADOUT ) ;
          end
       else dcam_settriggermode( Session.CamHandle, DCAM_TRIGMODE_EDGE ) ;
       dcam_settriggerpolarity( Session.CamHandle, DCAM_TRIGPOL_POSITIVE ) ;
       // Subtract readout time for C4880 cameras, since these
       // camera can only have exposure times as multiples of readout time
       if ANSIContainsText(Session.CameraModel, 'C4880') then begin
          ExposureTime := ExposureTime - (Session.ReadoutTime+0.001) - AdditionalReadoutTime ;
          end
       else begin
           // Make exposure time 90% of frame interval
           ExposureTime := (InterFrameTimeInterval*0.9) - AdditionalReadoutTime ;
           end ;
       dcam_setexposuretime( Session.CAMHandle, ExposureTime ) ;
       end
    else begin
       dcam_settriggermode( Session.CamHandle, DCAM_TRIGMODE_INTERNAL ) ;
       dcam_setexposuretime( Session.CAMHandle, InterFrameTimeInterval ) ;
       dcam_getexposuretime( Session.CAMHandle, InterFrameTimeInterval ) ;
       end ;

    // Start capture
    Err := dcam_capture( Session.CamHandle ) ;
//    outputdebugString(PChar(format('dcam_capture Err=%d',[Err])));
    Session.CapturingImages := True ;
    Result := True ;

    end ;


procedure DCAMAPI_CheckROIBoundaries(
         var Session : TDCAMAPISession ;   // Camera session record
         var ReadoutSpeed : Integer ;         // Readout rate (index no.)
         var FrameLeft : Integer ;            // Left pixel in CCD readout area
         var FrameRight : Integer ;           // Right pixel in CCD eadout area
         var FrameTop : Integer ;             // Top of CCD readout area
         var FrameBottom : Integer ;          // Bottom of CCD readout area
         var BinFactor : Integer ;           // Pixel binning factor (In)
         var FrameWidth : Integer ;          // Image width
         var FrameHeight : Integer ;         // Image height
         var FrameInterval : Double ;        // Time interval between frames (s)
         var ReadoutTime : Double ) ;        // Frame readout time (s)
// ----------------------------------------------------------
// Check and set CCD ROI boundaries and return valid settings
// (Also calculates minimum readout time)
// -----------------------------------------------------------
var
    i,Err : Integer ;
    ParamSubArrayInq : TDCAM_PARAM_SUBARRAY_INQ ;
    ParamSubArray : TDCAM_PARAM_SUBARRAY ;
    ParamFrameReadoutTime : TDCAM_PARAM_FRAME_READOUT_TIME_INQ ;
    NumBYtes : cardinal ;
    Changed : Boolean ;
    MultipleofReadoutTime : Single ;
begin

    if not Session.CameraOpen then Exit ;

     // Save current settings
    Changed := False ;
    if Session.ReadoutSpeed <> ReadoutSpeed then Changed := True ;
    if Session.FrameLeft <> FrameLeft  then Changed := True ;
    if Session.FrameRight <> FrameRight  then Changed := True ;
    if Session.FrameTop <> FrameTop  then Changed := True ;
    if Session.FrameBottom <> FrameBottom  then Changed := True ;
    if Session.BinFactor <> BinFactor  then Changed := True ;
    if Session.FrameInterval <> FrameInterval then Changed := True ;
    if not Changed then begin
       FrameInterval := Session.FrameInterval ;
       ReadoutTime := Session.ReadoutTime ;
       FrameWidth := Session.FrameWidth ;
       FrameHeight := Session.FrameHeight ;
       Exit ;
       end ;

    // Set binning factor
    i := 0 ;
    while (BinFactor > Session.BinFactors[i]) and
           (i < Session.NumBinFactors) do Inc(i) ;
    BinFactor := Max( Session.BinFactors[i],1) ;
    dcam_setbinning( Session.CAMHandle, BinFactor ) ;

    // Get sub-array limits
    ParamSubArrayInq.HDR.Size := Sizeof(ParamSubArrayInq) ;
    ParamSubArrayInq.HDR.id := DCAM_IDPARAM_SUBARRAY_INQ ;
    ParamSubArrayInq.HDR.iFlag := dcamparam_subarrayinq_binning or
                                  dcamparam_subarrayinq_hmax or
                                  dcamparam_subarrayinq_vmax or
                                  dcamparam_subarrayinq_hposunit or
                                  dcamparam_subarrayinq_vposunit or
                                  dcamparam_subarrayinq_hunit or
                                  dcamparam_subarrayinq_vunit ;
    ParamSubArrayInq.HDR.oFlag := 0 ;
    ParamSubArrayInq.hposunit := 1 ;  // These values present in case
    ParamSubArrayInq.vposunit := 1 ;  // values in case not returned by DCAM
    ParamSubArrayInq.hunit := 1 ;
    ParamSubArrayInq.vunit := 1 ;
    ParamSubArrayInq.Binning := BinFactor ;
    dcam_extended( Session.CamHandle,
                   DCAM_IDMSG_GETPARAM,
                   @ParamSubArrayInq,
                   SizeOf(ParamSubArrayInq)) ;

    // Ensure sub-array limits are valid

    ParamSubArray.hpos := DCAMAPI_CheckStepSize(
                          FrameLeft div BinFactor,
                          ParamSubArrayInq.hposunit ) ;

    ParamSubArray.hsize := DCAMAPI_CheckStepSize(
                           (FrameRight - FrameLeft + 1) div BinFactor,
                           ParamSubArrayInq.hunit ) ;

    ParamSubArray.hsize := Max( ParamSubArray.hsize, ParamSubArrayInq.hunit ) ;
    ParamSubArray.vpos := DCAMAPI_CheckStepSize(
                          FrameTop div BinFactor,
                          ParamSubArrayInq.vposunit ) ;

    ParamSubArray.vsize := DCAMAPI_CheckStepSize(
                           (FrameBottom - FrameTop + 1) div BinFactor,
                           ParamSubArrayInq.vunit ) ;
    ParamSubArray.vsize := Max( ParamSubArray.vsize, ParamSubArrayInq.vunit ) ;

    // Set CCD sub-array limits
    ParamSubArray.HDR.Size := Sizeof(ParamSubArray) ;
    ParamSubArray.HDR.id := DCAM_IDPARAM_SUBARRAY ;
    ParamSubArray.HDR.iFlag := dcamparam_subarray_hpos or
                               dcamparam_subarray_vpos or
                               dcamparam_subarray_hsize or
                               dcamparam_subarray_vsize ;
    ParamSubArray.HDR.oFlag := 0 ;
    dcam_extended( Session.CamHandle,
                   DCAM_IDMSG_SETGETPARAM,
                   @ParamSubArray,
                   SizeOf(ParamSubArray)) ;

    // Update limits
    FrameLeft := ParamSubArray.hpos*BinFactor ;
    FrameTop := ParamSubArray.vpos*BinFactor ;
    FrameRight := ParamSubArray.hsize*BinFactor + FrameLeft - 1 ;
    FrameBottom := ParamSubArray.vsize*BinFactor + FrameTop - 1 ;
    FrameWidth := ParamSubArray.hsize ;
    FrameHeight := ParamSubArray.vsize ;

    // Get frame readout time
    ParamFrameReadoutTime.HDR.Size := Sizeof(ParamFrameReadoutTime) ;
    ParamFrameReadoutTime.HDR.id := DCAM_IDPARAM_FRAME_READOUT_TIME_INQ ;
    ParamFrameReadoutTime.HDR.iFlag := dcamparam_framereadouttimeinq_framereadouttime ;
    ParamFrameReadoutTime.HDR.oFlag := 0 ;
    dcam_extended( Session.CamHandle,
                   DCAM_IDMSG_GETPARAM,
                   @ParamFrameReadoutTime,
                   SizeOf(ParamFrameReadoutTime)) ;

    ReadoutTime := ParamFrameReadoutTime.framereadouttime ;
    if ANSIContainsText(Session.CameraModel, 'C4880') then begin
       MultipleofReadoutTime := Max(Round(FrameInterval/(ReadoutTime*2)),1)*(ReadoutTime*2) ;
       if MultipleofReadoutTime < (FrameInterval*0.99) then
          MultipleofReadoutTime := MultipleofReadoutTime + ReadOutTime*2 ;
       FrameInterval := MultipleofReadoutTime ;
       end
    else FrameInterval := Max( FrameInterval, ReadoutTime ) ;

     // Save current settings
     Session.ReadoutSpeed := ReadoutSpeed ;
     Session.FrameLeft := FrameLeft ;
     Session.FrameRight := FrameRight ;
     Session.FrameTop := FrameTop ;
     Session.FrameBottom := FrameBottom ;
     Session.BinFactor := BinFactor ;
     Session.FrameWidth := FrameWidth ;
     Session.FrameHeight := FrameHeight ;
     Session.FrameInterval := FrameInterval ;
     Session.ReadoutTime := ReadoutTime  ;

     end ;

function DCAMAPI_CheckStepSize( Value : Integer ;
                                StepSize : Integer ) : Integer ;
begin
    Result := (Value div StepSize) * StepSize ;
    end ;


function DCAMAPI_CheckFrameInterval(
          var Session : TDCAMAPISession ;   // Camera session record
          FrameLeft : Integer ;   // Left edge of capture region (In)
          FrameRight : Integer ;  // Right edge of capture region( In)
          FrameTop : Integer ;    // Top edge of capture region( In)
          FrameBottom : Integer ; // Bottom edge of capture region (In)
          BinFactor : Integer ;   // Pixel binning factor (In)
          Var FrameInterval : Double ;
          Var ReadoutTime : Double) : Boolean ;
begin
    end ;




procedure DCAMAPI_Wait( Delay : Single ) ;
begin
    end ;



procedure DCAMAPI_GetImage(
          var Session : TDCAMAPISession  // Camera session record
          ) ;
begin
    end ;

procedure DCAMAPI_StopCapture(
          var Session : TDCAMAPISession   // Camera session record
          ) ;
// ------------------
// Stop image capture
// ------------------
begin

    if not Session.CapturingImages then exit ;

    // Stop capture
    dcam_idle( Session.CamHandle ) ;

    // Release image transfer buffers
    dcam_releasebuffer( Session.CamHandle ) ;

    Session.CapturingImages := False ;

    end ;

procedure DCAMAPI_CheckError(
          FuncName : String ;   // Name of function called
          ErrNum : Integer      // Error # returned by function
          ) ;
begin
    end ;




end.
