unit FileIOUnit;
// =============================================================================
// WinFluor - Windows Fluorescence Imaging Program - Data file input/output methods
// (c) J. Dempster, University of Strathclyde, 2001-2003, All Rights Reserved
// =============================================================================
// 26.2.2002
// 29.6.2003 ... ROI shape added
// 7.7.2003 .... Import File added
// 31.7.2003 ... SaveFrameAsTIFF added
// 7.9.2003 .... Multiple PIC,TIF & STK files can now be imported
// 8.10.2003 ... MainFrm.ADCScanInterval now saved to INI file instead of ADCNumScansPerFrame
// 13.11.2003 .. Cam1.AmpGain added to initialisation list
// 14.03.2005 .. Lens magnification now stored in Cam1.LensMagnification
// 30.06.05 .... Save directories now checked for existence
// 01.09.05 .... Live A/D channel data now stored in MainFrm.ADCChannel
// 29.11.05 .... MainFrm.ADCRecordingTime added to ini file
// 12.05.06 .... LoadInitialisationFile now reads any size of INI file

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ImageFile, IDRFile, StrUtils ;



type

 // Region of interest definition record

  TFileIO = class(TDataModule)
    ImageFile: TImageFile;
    SaveDialog: TSaveDialog;
    OpenDialog: TOpenDialog;
    procedure SaveDialogTypeChange(Sender: TObject);
    procedure DataModuleCreate(Sender: TObject);
    procedure OpenDialogTypeChange(Sender: TObject);
  private
    { Private declarations }
    FileTypeNames : Array[1..10] of String ;
    FileTypeExts : Array[1..10] of String ;
    NumFileTypes : Integer ;
  public
    { Public declarations }
    procedure SaveInitialisationFile( FileName : String ) ;
    procedure LoadInitialisationFile( FileName : String ) ;
    procedure ImportFile ;
    procedure ExportFile ;
    procedure SaveFrameAsTIFF ;
  end;

var
  FileIO: TFileIO;

implementation

uses Main, shared, LabIOUnit, maths, AmpModule , RecUnit, ViewUnit,
  LightSourceUnit;

{$R *.DFM}


procedure TFileIO.DataModuleCreate(Sender: TObject);
// -----------------------------------
// Initialisations when module created
// -----------------------------------
var
     i : Integer ;
begin

     FileTypeNames[1] := 'BIORad PIC Files' ;
     FileTypeExts[1] := '.PIC' ;
     FileTypeNames[2] := 'TIFF Files' ;
     FileTypeExts[2] := '.TIF' ;
     FileTypeNames[3] := 'STK Files' ;
     FileTypeExts[3] := '.STK' ;
     FileTypeNames[4] := 'ICS Files' ;
     FileTypeExts[4] := '.ICS' ;
     NumFileTypes := 4 ;

     SaveDialog.Filter := '' ;
     for i := 1 to NumFileTypes do begin
         if i <> 1 then SaveDialog.Filter := SaveDialog.Filter + '|' ;
         SaveDialog.Filter := SaveDialog.Filter +
                              format( '%s (*%s)|*%s',
                                      [FileTypeNames[i],
                                       FileTypeExts[i],
                                       FileTypeExts[i]]) ;
         end ;
     SaveDialog.FilterIndex := 1 ;
     SaveDialog.DefaultExt := RightStr(FileTypeExts[SaveDialog.FilterIndex],3) ;
     SaveDialog.Options := [ofHideReadOnly,ofPathMustExist] ;

     OpenDialog.Filter := SaveDialog.Filter ;
     OpenDialog.FilterIndex := SaveDialog.FilterIndex ;
     OpenDialog.DefaultExt := SaveDialog.DefaultExt ;

     end;



procedure TFileIO.SaveInitialisationFile(
          FileName : String
          ) ;
// ------------------------
// Save initialisation file
// ------------------------
var
   Header : array[1..cNumIDRHeaderBytes] of char ;
   i,ch,INIFileHandle : Integer ;
begin

     INIFileHandle := FileCreate( FileName ) ;

     if INIFileHandle < 0 then begin
        MainFrm.StatusBar.SimpleText := 'Unable to create : ' + FileName ;
        Exit ;
        end ;

     // Initialise empty header buffer with zero bytes
     for i := 1 to sizeof(Header) do Header[i] := chr(0) ;

     // Camera settings
     AppendInt( Header, 'CAMTYPE=', MainFrm.Cam1.CameraType ) ;
     AppendFloat( Header, 'CAMFI=', MainFrm.Cam1.FrameInterval ) ;
     AppendInt( Header, 'CAMFL=', MainFrm.Cam1.FrameLeft ) ;
     AppendInt( Header, 'CAMFR=', MainFrm.Cam1.FrameRight ) ;
     AppendInt( Header, 'CAMFT=', MainFrm.Cam1.FrameTop ) ;
     AppendInt( Header, 'CAMFB=', MainFrm.Cam1.FrameBottom ) ;
     AppendInt( Header, 'CAMBIN=', MainFrm.Cam1.BinFactor ) ;
     AppendInt( Header, 'CAMRS=', MainFrm.Cam1.ReadoutSpeed ) ;
     AppendInt( Header, 'NFREQ=', MainFrm.NumFramesRequired ) ;
     AppendFloat( Header, 'ADCRECTIME=', MainFrm.ADCRecordingTime ) ;
     AppendInt( Header, 'CAMCOM=', MainFrm.Cam1.ComPort ) ;
     AppendInt( Header, 'CAMGN=', MainFrm.Cam1.AmpGain ) ;
     // Camera exposure/readout trigger offset
     AppendFloat( Header, 'CAMTRIGOFFSET=', MainFrm.CameraTriggerOffset ) ;


     AppendFloat( Header, 'LENSMAG=', MainFrm.Cam1.LensMagnification ) ;
     AppendFloat( Header, 'TLAPINT=', MainFrm.TimeLapseInterval ) ;

     AppendInt( Header, 'PAL=', Integer(MainFrm.PaletteType) ) ;
     AppendInt( Header, 'DZOOM=', MainFrm.DisplayZoomIndex ) ;

     // A/D channel settings
     AppendInt( Header, 'ADCNC=', MainFrm.ADCNumChannels ) ;
     AppendInt( Header, 'ADCINPUTMODE=', LabIO.ADCInputMode ) ;
     // NI interface API
     AppendInt( Header, 'NIDAQAPI=', LabIO.NIDAQAPI ) ;

     AppendFloat( Header, 'ADCSI=', MainFrm.ADCScanInterval ) ;
     AppendFloat( Header, 'ADCVR=', MainFrm.ADCVoltageRange ) ;
     AppendFloat( Header, 'ADCDW=', MainFrm.ADCDisplayWindow ) ;
     for ch := 0 to MainFrm.ADCNumChannels-1 do begin
         AppendInt( Header, format('CIN%d=',[ch]), MainFrm.ADCChannel[ch].ChannelOffset) ;
         AppendString( Header, format('CU%d=',[ch]), MainFrm.ADCChannel[ch].ADCUnits ) ;
         AppendString( Header, format('CN%d=',[ch]), MainFrm.ADCChannel[ch].ADCName ) ;
         AppendFloat( Header, format('CCF%d=',[ch]), MainFrm.ADCChannel[ch].ADCCalibrationFactor ) ;
         AppendFloat( Header, format('CSC%d=',[ch]), MainFrm.ADCChannel[ch].ADCScale) ;
         end ;


     // Patch clamp amplifier data
     AppendInt( Header, 'AMP=',Amplifier.TypeInUse ) ;
     AppendInt( Header, 'AMPCH=',Amplifier.TelegraphChannel ) ;

     // Command voltage settings
     AppendFloat( Header, 'VCDIV0=', MainFrm.VCommand[0].DivideFactor ) ;
     AppendFloat( Header, 'VCHOLD0=', MainFrm.VCommand[0].HoldingVoltage ) ;
     AppendFloat( Header, 'VCDIV1=', MainFrm.VCommand[1].DivideFactor ) ;
     AppendFloat( Header, 'VCHOLD1=', MainFrm.VCommand[1].HoldingVoltage ) ;

     // Light source
     AppendInt( Header, 'LSDEV=', LightSource.DeviceType ) ;
     AppendFloat( Header, 'LSW1=', LightSource.Wavelength1 ) ;
     AppendFloat( Header, 'LSV1=', LightSource.Voltage1 ) ;
     AppendFloat( Header, 'LSW2=', LightSource.Wavelength2 ) ;
     AppendFloat( Header, 'LSV2=', LightSource.Voltage2 ) ;

     // Excitation wavelength settings
     AppendLogical( Header, 'EXCSW=', MainFrm.EXCSingleWavelength ) ;
     AppendInt( Header, 'EXCSWN=', MainFrm.EXCSingleWavelengthNum ) ;
     AppendInt( Header, 'EXCNW=', MainFrm.EXCNumWavelengths ) ;
     for i := 0 to MainFrm.EXCNumWavelengths do
         AppendInt( Header, format('EXCSEQ%d=',[i]), MainFrm.EXCSequence[i]) ;
     for i := 0 to High(MainFrm.EXCWavelengths) do begin
         AppendInt( Header, format('EXCWC%d=',[i]), MainFrm.EXCWavelengths[i].Centre) ;
         AppendInt( Header, format('EXCWW%d=',[i]), MainFrm.EXCWavelengths[i].Width) ;
         end ;



     // Stimulus program file
     AppendString( Header, 'STIMFILE=', MainFrm.StimFileName) ;

     // Fluophore binding equation table
     for i := 0 to MainFrm.IDRFile.MaxEquations-1 do begin
        AppendLogical( Header, format('EQNUSE%d=',[i]), MainFrm.IDRFile.Equation[i].InUse) ;
        AppendString( Header, format('EQNION%d=',[i]), MainFrm.IDRFile.Equation[i].Ion) ;
        AppendString( Header, format('EQNUN%d=',[i]), MainFrm.IDRFile.Equation[i].Units) ;
        AppendString( Header, format('EQNNAM%d=',[i]), MainFrm.IDRFile.Equation[i].Name) ;
        AppendFloat( Header, format('EQNRMAX%d=',[i]), MainFrm.IDRFile.Equation[i].RMax) ;
        AppendFloat( Header, format('EQNRMIN%d=',[i]), MainFrm.IDRFile.Equation[i].RMin) ;
        AppendFloat( Header, format('EQNKEFF%d=',[i]), MainFrm.IDRFile.Equation[i].KEff) ;
        end ;

     AppendString( Header, 'DDIR=', MainFrm.DataDirectory ) ;

     // Printer page settings
     AppendInt( Header, 'PRTM=', MainFrm.PrinterTopMargin )  ;
     AppendInt( Header, 'PRBM=', MainFrm.PrinterBottomMargin )  ;
     AppendInt( Header, 'PRLM=', MainFrm.PrinterLeftMargin )  ;
     AppendInt( Header, 'PRRM=', MainFrm.PrinterRightMargin )  ;
     AppendInt( Header, 'PRLT=', MainFrm.PrinterLineThickness )  ;
     AppendInt( Header, 'PRMS=', MainFrm.PrinterMarkerSize )  ;
     AppendLogical( Header, 'PRUC=', MainFrm.PrinterUseColor ) ;
     AppendString( Header, 'PRFN=', MainFrm.PrinterFontName ) ;
     AppendInt( Header, 'PRFS=', MainFrm.PrinterFontSize )  ;

     for i := 0 to High(MainFrm.RecentFiles) do
         AppendString(Header,format('FILE%d=',[i]),MainFrm.RecentFiles[i]) ;

     // Input/output/control line configuration
     AppendInt( Header, 'IOADCI=', MainFrm.IOConfig.ADCIn ) ;
     AppendInt( Header, 'IOCAMS=', MainFrm.IOConfig.CameraStart ) ;
     AppendLogical( Header, 'IOCAMSAH=', MainFrm.IOConfig.CameraStartActiveHigh ) ;
     AppendInt( Header, 'IOVCOM0=', MainFrm.IOConfig.VCommand[0] ) ;
     AppendInt( Header, 'IOVCOM1=', MainFrm.IOConfig.VCommand[1] ) ;
     AppendInt( Header, 'IOLSSU=', MainFrm.IOConfig.LSShutter ) ;
     AppendLogical( Header, 'IOLSSUAH=', MainFrm.IOConfig.LSShutterActiveHigh ) ;
     AppendInt( Header, 'IOLSWS=', MainFrm.IOConfig.LSWavelengthStart ) ;
     AppendInt( Header, 'IOLSWE=', MainFrm.IOConfig.LSWavelengthEnd ) ;
     AppendInt( Header, 'IODSTA=', MainFrm.IOConfig.DigitalStimStart ) ;
     AppendInt( Header, 'IODEND=', MainFrm.IOConfig.DigitalStimEnd ) ;
     AppendInt( Header, 'IOCLKSYNC=', MainFrm.IOConfig.ClockSyncLine ) ;

     if FileWrite( INIFileHandle, Header, Sizeof(Header) ) <> Sizeof(Header) then
        MessageDlg( ' Initialisation file write failed ', mtWarning, [mbOK], 0 ) ;

     if INIFileHandle >= 0 then FileClose( INIFileHandle ) ;

     end ;


procedure TFileIO.LoadInitialisationFile(
          FileName : String
          ) ;
// ------------------------
// Load initialisation file
// ------------------------
var
   Header : array[1..cNumIDRHeaderBytes] of char ;
   i,ch,INIFileHandle : Integer ;
   iValue : Integer ;
   fValue : Single ;
   ADCChannel : TChannel ;
   Equation : TBindingEquation ;
begin

     if not FileExists( FileName ) then Exit ;

     INIFileHandle := FileOpen( FileName, fmOpenReadWrite ) ;

     if INIFileHandle < 0 then begin
        ShowMessage('Unable to open ' +  FileName ) ;
        Exit ;
        end ;

     // Fill with 0s
     for i := 0 to High(Header) do Header[i] := #0 ;

     // Read data from INI file
     FileRead( INIFileHandle, Header, Sizeof(Header) ) ;

     // Camera settings
     ReadInt( Header, 'CAMTYPE=', MainFrm.CameraType ) ;

     ReadInt( Header, 'CAMBIN=', iValue ) ;
     MainFrm.Cam1.BinFactor := iValue ;
     ReadInt( Header, 'CAMRS=', iValue ) ;
     MainFrm.Cam1.ReadoutSpeed := iValue ;
     ReadInt( Header, 'CAMCOM=', iValue ) ;
     MainFrm.Cam1.ComPort := iValue ;
     ReadInt( Header, 'CAMGN=', iValue ) ;
     MainFrm.Cam1.AmpGain := iValue ;

     ReadInt( Header, 'CAMFL=', iValue ) ;
     MainFrm.Cam1.FrameLeft := iValue ;
     ReadInt( Header, 'CAMFR=', iValue ) ;
     MainFrm.Cam1.FrameRight := iValue ;
     ReadInt( Header, 'CAMFT=', iValue ) ;
     MainFrm.Cam1.FrameTop := iValue ;
     ReadInt( Header, 'CAMFB=', iValue ) ;
     MainFrm.Cam1.FrameBottom := iValue ;

     // NOTE. Bin factor, readout speed and imaging area need to be set before
     // .FrameInterval to ensure camera will accept short intervals

     ReadFloat( Header, 'CAMFI=', fValue ) ;
     MainFrm.Cam1.FrameInterval := fValue ;

     // Camera exposure/readout trigger offset
     ReadFloat( Header, 'CAMTRIGOFFSET=', MainFrm.CameraTriggerOffset ) ;

     fValue := 0.0 ;
     ReadFloat( Header, 'LENSMAG=', fValue ) ;
     if fValue <> 0.0 then MainFrm.Cam1.LensMagnification := fValue ;

     ReadFloat( Header, 'TLAPINT=', MainFrm.TimeLapseInterval ) ;

     iValue := Integer(MainFrm.PaletteType) ;
     ReadInt( Header, 'PAL=', iValue ) ;
     MainFrm.PaletteType := TPaletteType(iValue) ;

     ReadInt( Header, 'DZOOM=', MainFrm.DisplayZoomIndex ) ;

     ReadInt( Header, 'NFREQ=', MainFrm.NumFramesRequired ) ;

     ReadFloat( Header, 'ADCRECTIME=', MainFrm.ADCRecordingTime ) ;

     // National Instruments interface library
     iValue := 0 ;
     ReadInt( Header, 'NIDAQAPI=', iValue ) ;
     LabIO.NIDAQAPI := iValue ;

     // A/D input mode (NRSE/Differential)
     iValue := 0 ;
     ReadInt( Header, 'ADCINPUTMODE=', iValue ) ;
     LabIO.ADCInputMode := iValue ;

     // A/D channel settings
     ReadInt( Header, 'ADCNC=', iValue  ) ;
     MainFrm.ADCNumChannels := iValue ;

     // A/D channel scanning interval
     ReadFloat( Header, 'ADCSI=', fValue ) ;
     MainFrm.ADCScanInterval := fValue ;

     ReadFloat( Header, 'ADCVR=', fValue ) ;
     MainFrm.ADCVoltageRange := fValue ;

     ReadFloat( Header, 'ADCDW=', MainFrm.ADCDisplayWindow ) ;

     for ch := 0 to MainFrm.ADCNumChannels-1 do begin
         // Get current channel settings
         ADCChannel := MainFrm.ADCChannel[ch] ;
         // Load parameters from INI file
         ReadInt( Header, format('CIN%d=',[ch]), ADCChannel.ChannelOffset ) ;
         ADCChannel.ChannelOffset := MainFrm.IDRFile.ADCNumChannels - ch - 1 ;
         ReadString( Header, format('CU%d=',[ch]), ADCChannel.ADCUnits ) ;
         ReadString( Header, format('CN%d=',[ch]), ADCChannel.ADCName ) ;
         ReadFloat( Header, format('CCF%d=',[ch]), ADCChannel.ADCCalibrationFactor ) ;
         ReadFloat( Header, format('CSC%d=',[ch]), ADCChannel.ADCScale) ;
         // Update channel
         MainFrm.ADCChannel[ch] := ADCChannel ;
         end ;

     // Patch clamp amplifier data
     ReadInt( Header, 'AMP=',Amplifier.TypeInUse ) ;
     ReadInt( Header, 'AMPCH=',Amplifier.TelegraphChannel ) ;

     // Patch clamp command voltage divide factor
     ReadFloat( Header, 'VCDIV=', MainFrm.VCommand[0].DivideFactor ) ;
     // Patch clamp holding potential
     ReadFloat( Header, 'VCHOLD=', MainFrm.VCommand[0].HoldingVoltage ) ;

     ReadFloat( Header, 'VCDIV0=', MainFrm.VCommand[0].DivideFactor ) ;
     ReadFloat( Header, 'VCHOLD0=', MainFrm.VCommand[0].HoldingVoltage ) ;
     ReadFloat( Header, 'VCDIV1=', MainFrm.VCommand[1].DivideFactor ) ;
     ReadFloat( Header, 'VCHOLD1=', MainFrm.VCommand[1].HoldingVoltage ) ;

     // Light source
     ReadInt( Header, 'LSDEV=', LightSource.DeviceType ) ;
     ReadFloat( Header, 'LSW1=', LightSource.Wavelength1 ) ;
     ReadFloat( Header, 'LSV1=', LightSource.Voltage1 ) ;
     ReadFloat( Header, 'LSW2=', LightSource.Wavelength2 ) ;
     ReadFloat( Header, 'LSV2=', LightSource.Voltage2 ) ;

     // Excitation wavelength settings
     ReadLogical( Header, 'EXCSW=', MainFrm.EXCSingleWavelength ) ;
     ReadInt( Header, 'EXCSWN=', MainFrm.EXCSingleWavelengthNum ) ;
     ReadInt( Header, 'EXCNW=', MainFrm.EXCNumWavelengths ) ;
     for i := 0 to MainFrm.EXCNumWavelengths do
         ReadInt( Header, format('EXCSEQ%d=',[i]), MainFrm.EXCSequence[i]) ;
     for i := 0 to High(MainFrm.EXCWavelengths) do begin
         ReadInt( Header, format('EXCWC%d=',[i]), MainFrm.EXCWavelengths[i].Centre) ;
         ReadInt( Header, format('EXCWW%d=',[i]), MainFrm.EXCWavelengths[i].Width) ;
         end ;

     // Stimulus program file
     ReadString( Header, 'STIMFILE=', MainFrm.StimFileName) ;

     // Fluophore binding equation table
     for i := 0 to MainFrm.IDRFile.MaxEquations-1 do begin
         ReadLogical( Header, format('EQNUSE%d=',[i]), Equation.InUse ) ;
         ReadString( Header, format('EQNION%d=',[i]), Equation.Ion) ;
         ReadString( Header, format('EQNUN%d=',[i]), Equation.Units) ;
         ReadString( Header, format('EQNNAM%d=',[i]), Equation.Name) ;
         ReadFloat( Header, format('EQNRMAX%d=',[i]), Equation.RMax) ;
         ReadFloat( Header, format('EQNRMIN%d=',[i]), Equation.RMin) ;
         ReadFloat( Header, format('EQNKEFF%d=',[i]), Equation.KEff) ;
         MainFrm.IDRFile.Equation[i] := Equation ;
         end ;

     // Printer page settings
     ReadInt( Header, 'PRTM=', MainFrm.PrinterTopMargin )  ;
     ReadInt( Header, 'PRBM=', MainFrm.PrinterBottomMargin )  ;
     ReadInt( Header, 'PRLM=', MainFrm.PrinterLeftMargin )  ;
     ReadInt( Header, 'PRRM=', MainFrm.PrinterRightMargin )  ;
     ReadInt( Header, 'PRLT=', MainFrm.PrinterLineThickness )  ;
     ReadInt( Header, 'PRMS=', MainFrm.PrinterMarkerSize )  ;
     ReadLogical( Header, 'PRUC=', MainFrm.PrinterUseColor ) ;
     ReadString( Header, 'PRFN=', MainFrm.PrinterFontName ) ;
     ReadInt( Header, 'PRFS=', MainFrm.PrinterFontSize )  ;

     ReadString( Header, 'DDIR=', MainFrm.DataDirectory ) ;

     for i := 0 to High(MainFrm.RecentFiles) do
         ReadString(Header,format('FILE%d=',[i]),MainFrm.RecentFiles[i]) ;

     // Input/output/control line configuration
     ReadInt( Header, 'IOADCI=', MainFrm.IOConfig.ADCIn ) ;
     ReadInt( Header, 'IOCAMS=', MainFrm.IOConfig.CameraStart ) ;
     ReadLogical( Header, 'IOCAMSAH=', MainFrm.IOConfig.CameraStartActiveHigh ) ;

     // Command voltage O/P lines 1 & 2
     ReadInt( Header, 'IOVCOM=', MainFrm.IOConfig.VCommand[0] ) ;
     // Above line for compatibility with versions earlier than V2.4.3
     ReadInt( Header, 'IOVCOM0=', MainFrm.IOConfig.VCommand[0] ) ;
     ReadInt( Header, 'IOVCOM1=', MainFrm.IOConfig.VCommand[1] ) ;

     ReadInt( Header, 'IOLSSU=', MainFrm.IOConfig.LSShutter ) ;
     ReadLogical( Header, 'IOLSSUAH=', MainFrm.IOConfig.LSShutterActiveHigh ) ;
     ReadInt( Header, 'IOLSWS=', MainFrm.IOConfig.LSWavelengthStart ) ;
     ReadInt( Header, 'IOLSWE=', MainFrm.IOConfig.LSWavelengthEnd ) ;
     ReadInt( Header, 'IODSTA=', MainFrm.IOConfig.DigitalStimStart ) ;
     ReadInt( Header, 'IODEND=', MainFrm.IOConfig.DigitalStimEnd ) ;
     ReadInt( Header, 'IOCLKSYNC=', MainFrm.IOConfig.ClockSyncLine ) ;

     FileClose( INIFileHandle ) ;

     end ;



procedure TFileIO.ImportFile ;
// --------------------------------------------
// Import images from another image file format
// --------------------------------------------
var
    PFrameBuf : Pointer ; // Image frame buffer pointer
    ImportFileName : String ;
    IDRFileName : String ; // Name of .IDR file to hold imported images
    FileNum : Integer ;    // Index into OpenDialog.Files list
    iFrame : Integer ;     // Frame counter
    NumFramesImported : Integer ;
begin


    // Setup Open Files dialog box
    OpenDialog.options := [ofPathMustExist,ofAllowMultiSelect] ;

    // Open last used data directory
     if DirectoryExists(MainFrm.DataDirectory) and
        (MainFrm.DataDirectory <> '') then begin
        OpenDialog.InitialDir := MainFrm.DataDirectory ;
        end
     else OpenDialog.InitialDir := MainFrm.DefaultDataDirectory ;

    OpenDialog.Title := 'Import File ' ;

    // Exit if no file name available
    if (not OpenDialog.Execute) or (OpenDialog.Files.Count <= 0) then Exit ;

    // Import frames from all files in list
    NumFramesImported := 0 ;
    for FileNum := 0 to OpenDialog.Files.Count-1 do begin

        // Open file
        ImportFileName := OpenDialog.Files[FileNum] ;
        if not ImageFile.OpenFile( ImportFileName ) then begin
           MainFrm.StatusBar.SimpleText := 'Unable to open : ' + ImportFileName ;
           Exit ;
           end ;

        // Create IDR file to hold images
        if FileNum = 0 then begin
           IDRFileName := ChangeFileExt( ImportFileName, '.IDR' ) ;
           if FileExists(IDRFileName) then begin
              if MessageDlg( 'File ' + IDRFileName +' already exists! Overwrite it?',
                 mtWarning,[mbYes,mbNo], 0 ) = mrYes then begin
                 DeleteFile(IDRFileName) ;
                 end
              else begin
                 // Present user with standard Save File dialog box
                 MainFrm.SaveDialog.options := [ofOverwritePrompt,ofHideReadOnly,ofPathMustExist] ;
                 MainFrm.SaveDialog.DefaultExt := DataFileExtension ;
                 MainFrm.SaveDialog.FileName := '' ;
                 MainFrm.SaveDialog.Filter := format(' %s Files (*.%s)|*.%s',
                             [DataFileExtension,DataFileExtension,DataFileExtension]);
                 MainFrm.SaveDialog.Title := 'Name for imported file' ;
                 MainFrm.SaveDialog.InitialDir := OpenDialog.InitialDir ;
                 if MainFrm.SaveDialog.execute then begin
                    IDRFileName := MainFrm.SaveDialog.FileName ;
                    end
                 else begin
                    ImageFile.CloseFile ;
                    Exit ;
                    end ;
                 end ;
              end ;

           // Create empty IDR data file to hold imported images
           MainFrm.IDRFile.CreateFile( IDRFileName) ;

           // Get image properties from import file
           MainFrm.IDRFile.FrameWidth := ImageFile.FrameWidth ;
           MainFrm.IDRFile.FrameHeight := ImageFile.FrameHeight ;
           MainFrm.IDRFile.PixelDepth := ImageFile.PixelDepth ;
           MainFrm.IDRFile.XResolution := ImageFile.XResolution ;
           MainFrm.IDRFile.ResolutionUnits := ImageFile.ResolutionUnit ;

           MainFrm.IDRFile.NumFrameTypes := ImageFile.ComponentsPerPixel ;
           MainFrm.IDRFile.FrameInterval := 1.0 ;
           MainFrm.IDRFile.ADCScanInterval := 1.0 ;
           MainFrm.IDRFile.ADCNumChannels := 0 ;

           // Allocate frame buffer
           GetMem( PFrameBuf,MainFrm.IDRFile.NumBytesPerFrame ) ;
           end
        else begin
           // Terminate import if frame size changed
           if (MainFrm.IDRFile.FrameWidth <> ImageFile.FrameWidth) or
              (MainFrm.IDRFile.FrameHeight <> ImageFile.FrameHeight) or
              (MainFrm.IDRFile.PixelDepth <> ImageFile.PixelDepth) then begin
              MainFrm.StatusBar.SimpleText := 'Import: Aborted at ' + ImportFileName ;
              ImageFile.CloseFile ;
              Break ;
              end ;
           end ;

        // Import frames from this file
        for iFrame := 1 to ImageFile.NumFrames do begin
           if ImageFile.LoadFrame( iFrame, PFrameBuf ) then begin
              Inc(NumFramesImported) ;
              MainFrm.IDRFile.SaveFrame( NumFramesImported, PFrameBuf ) ;
              MainFrm.StatusBar.SimpleText := format(
              'Importing %d/%d frames from %s',
              [iFrame,ImageFile.NumFrames,ImportFileName]) ;
              Application.ProcessMessages ;
              end ;
           end ;

        // Close this import file
        ImageFile.CloseFile ;

        end ;

    // Display in program title bar
    MainFrm.Caption := 'WinFluor : ' + MainFrm.IDRFile.FileName ;
    if MainFrm.IDRFile.NumFrames > 0 then MainFrm.mnViewImages.Click ;

{    edPixelWidth.Value := ImageFile.XResolution ;
    edPixelWidth.Units := ImageFile.ResolutionUnit ;
    edPixelUnits.Text := ImageFile.ResolutionUnit ;
    edPixelAspectRatio.Value := ImageFile.YResolution / ImageFile.XResolution ;
    edCalBarWidth.Value := 0.2*FrameWidth*edPixelWidth.Value ;
    edCalBarWidth.Units := ImageFile.ResolutionUnit ;}

     FreeMem( PFrameBuf ) ;
     end ;


procedure TFileIO.ExportFile ;
// --------------------------------------------
// Export images to another image file format
// --------------------------------------------
var
    FrameNum : Integer ; // Frame counter
    PFrameBuf : Pointer ; // Image frame buffer pointer
    Done : Boolean ;
    OK : Boolean ;
begin

     // Open last used data directory
     if DirectoryExists(MainFrm.DataDirectory) and (MainFrm.DataDirectory <> '') then begin
        SaveDialog.InitialDir := MainFrm.DataDirectory ;
        end
     else SaveDialog.InitialDir := MainFrm.DefaultDataDirectory ;

     SaveDialog.Title := 'Export to File' ;
     SaveDialog.FileName := ChangeFileExt( MainFrm.IDRFile.FileName,
                                           FileTypeExts[ SaveDialog.FilterIndex] ) ;

     Done := not SaveDialog.Execute ;
     OK := False ;
     while not Done do begin
        // Ensure that file has selected extension
        SaveDialog.FileName := ChangeFileExt( SaveDialog.FileName,
                                              FileTypeExts[ SaveDialog.FilterIndex] ) ;
        // Check if file exists
        if not FileExists(SaveDialog.FileName) then OK := True
        else if MessageDlg( format(
                'File %s already exists! DO you want to overwrite it? ',
                [SaveDialog.FileName]),mtWarning,[mbYes,mbNo], 0 )
                 = mrYes then OK := True ;

        // Let user choose another name
        if OK then Done := True
              else Done := not SaveDialog.Execute ;

        end ;

     // Exit if a suitable file name is not available
     if not OK then Exit ;

     OK := ImageFile.CreateFile( SaveDialog.FileName,
                                 MainFrm.IDRFile.FrameWidth,
                                 MainFrm.IDRFile.FrameHeight,
                                 MainFrm.IDRFile.NumBytesPerPixel*8,
                                 1,
                                 False ) ;

     if not OK then begin
        MainFrm.StatusBar.SimpleText := 'Unable to create : ' + SaveDialog.FileName ;
        Exit ;
        end ;

    // Allocate frame buffer
    GetMem( PFrameBuf, MainFrm.IDRFile.NumBytesPerFrame ) ;

    try
       for FrameNum := 1 to MainFrm.IDRFile.NumFrames do begin
           // Read frame from file
           OK := MainFrm.IDRFile.LoadFrame( FrameNum, PFrameBuf ) ;
           // Save file to export file
           if OK then begin
              ImageFile.SaveFrame( FrameNum, PFrameBuf ) ;
              MainFrm.StatusBar.SimpleText := format(
              'Exporting %d/%d frames',[FrameNum,MainFrm.IDRFile.NumFrames]) ;
              Application.ProcessMessages ;

              end ;
           end ;
    finally
       FreeMem( PFrameBuf ) ;
       ImageFile.CloseFile ;

       MainFrm.StatusBar.SimpleText := format(
       'Export: %d frames exported to %s',[MainFrm.IDRFile.NumFrames,SaveDialog.FileName]) ;

       end ;

{    edPixelWidth.Value := ImageFile.XResolution ;
    edPixelWidth.Units := ImageFile.ResolutionUnit ;
    edPixelUnits.Text := ImageFile.ResolutionUnit ;
    edPixelAspectRatio.Value := ImageFile.YResolution / ImageFile.XResolution ;
    edCalBarWidth.Value := 0.2*FrameWidth*edPixelWidth.Value ;
    edCalBarWidth.Units := ImageFile.ResolutionUnit ;}

       end ;


procedure TFileIO.SaveFrameAsTIFF ;
// ----------------------------------------
// Save currently active image to TIFF file
// ----------------------------------------
var
     PInFrameBuf : PIntArray ; // Pointer to image source buffer
     POutFrameBuf : Pointer ;  // Pointer to image save buffer
     TIFPixelDepth : Integer ;
     OK : Boolean ;
     i : Integer ;
begin

     // Present user with standard Save File dialog box
     SaveDialog.FilterIndex := 2 ;
     SaveDialog.DefaultExt := RightStr(FileTypeExts[SaveDialog.FilterIndex],3) ;
     SaveDialog.FileName := ChangeFileExt( MainFrm.IDRFile.FileName,
                                           FileTypeExts[SaveDialog.FilterIndex] ) ;
     SaveDialog.Title := 'Save as TIFF File' ;

     // Open last used data directory
     if DirectoryExists(MainFrm.DataDirectory) and (MainFrm.DataDirectory <> '') then begin
        SaveDialog.InitialDir := MainFrm.DataDirectory ;
        end
     else SaveDialog.InitialDir := MainFrm.DefaultDataDirectory ;

     if not SaveDialog.Execute then Exit ;

     if MainFrm.ActiveMDIChild.Name = 'IntegrateFrm' then begin
        // Save current image from integrated image module
        //if IntegrateFrm.ImageAvailable then
        end
     else if MainFrm.ActiveMDIChild.Name = 'RecordFrm' then begin
        // Save current image from Record Sequence module
        if RecordFrm.ImageAvailable then begin
           PInFrameBuf := RecordFrm.PDisplayBufs[RecordFrm.SelectedFrameType] ;
           if MainFrm.Cam1.PixelDepth > 8 then TIFPixelDepth := 16
                                          else TIFPixelDepth := 8 ;
           OK := ImageFile.CreateFile( SaveDialog.FileName,
                                       MainFrm.Cam1.FrameWidth,
                                       MainFrm.Cam1.FrameHeight,
                                       TIFPixelDepth,
                                       1,
                                       True ) ;
           end ;
        end
     else if MainFrm.ActiveMDIChild.Name = 'ViewFrm' then begin
        // Save current image from View Frames module
        if ViewFrm.ImageAvailable then begin
           PInFrameBuf := ViewFrm.PImageBufs[ViewFrm.SelectedFrameType] ;
           OK := ImageFile.CreateFile( SaveDialog.FileName,
                                       MainFrm.IDRFile.FrameWidth,
                                       MainFrm.IDRFile.FrameHeight,
                                       MainFrm.IDRFile.NumBytesPerPixel*8,
                                       1,
                                       True ) ;
           end ;
        end ;

     // Save image to TIFF file
     if OK then begin

        // Allocate buffer
        GetMem( POutFrameBuf, ImageFile.NumBytesPerFrame ) ;

        if ImageFile.PixelDepth <= 8 then begin
           // 8 bit image
           for i := 0 to ImageFile.NumPixelsPerFrame-1 do
               PByteArray(POutFrameBuf)^[i] := Byte(PInFrameBuf^[i]) ;
           end
        else begin
           // 16 bit image
           for i := 0 to ImageFile.NumPixelsPerFrame-1 do
               PWordArray(POutFrameBuf)^[i] := Word(PInFrameBuf^[i]) ;
           end ;
        ImageFile.SaveFrame( 1, POutFrameBuf ) ;
        ImageFile.CloseFile ;

        FreeMem( POutFrameBuf ) ;

        end ;

     end;


procedure TFileIO.SaveDialogTypeChange(Sender: TObject);
// -------------------------------------------------
// Update default extension when filter type changed
// -------------------------------------------------
begin
     SaveDialog.FileName := ChangeFileExt(SaveDialog.FileName,FileTypeExts[SaveDialog.FilterIndex]) ;
     SaveDialog.DefaultExt := RightStr(FileTypeExts[SaveDialog.FilterIndex],3) ;
     end;


procedure TFileIO.OpenDialogTypeChange(Sender: TObject);
// -------------------------------------------------
// Update default extension when filter type changed
// -------------------------------------------------
begin
     OpenDialog.DefaultExt := RightStr(FileTypeExts[OpenDialog.FilterIndex],3) ;
     end;




end.
